/*
 * @(#)GetEnvironment.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.sys;

import java.io.File;
import java.util.ArrayList;

import com.JRcServer.commons.exception.BaseException;
import com.JRcServer.commons.exception.ExecutionException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.exception.NotExistException;
import com.JRcServer.commons.exception.UnsupportException;
import com.JRcServer.commons.sys.cmd.NativeCmd;
import com.JRcServer.commons.thread.Synchronized;
import com.JRcServer.commons.util.ArrayTable;
import com.JRcServer.commons.util.UtilCom;

/**
 * 環境変数取得オブジェクト.
 * <BR><BR>
 * 環境変数を取得します.
 *
 * @version     1.00, 2004/05/13
 * @author      Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class GetEnvironment
{
    
    /**
     * 環境変数取得コマンド : windowsNT系.
     */
    private static final String[] WINNT_ENVCMD = {
        "cmd", "/c", "set"
    } ;
    
    /**
     * 環境変数取得コマンド : windows9x系
     */
    private static final String[] WIN9X_ENVCMD = {
        "command.com", "/c", "set"
    } ;
    
    /**
     * 環境変数取得コマンド : os/2系
     */
    private static final String[] OS_2_ENVCMD = {
        "cmd", "/c", "set"
    } ;
    
    /**
     * 環境変数取得コマンド : unix系その１.
     * まず１つ、このコマンド読み取りを実施.
     *
     * unix.
     * z/os.
     */
    private static final String[] UNIX1_ENVCMD = {
        "/bin/env"
    } ;
    
    /**
     * 環境変数取得コマンド : unix系その２.
     * まず１つ、このコマンド読み取りを実施.
     *
     * unix.
     * z/os.
     */
    private static final String[] UNIX2_ENVCMD = {
        "/usr/bin/env"
    } ;
    
    /**
     * 環境変数取得コマンド : unix系その３.
     * まず１つ、このコマンド読み取りを実施.
     * (PATH指定されたコマンドとして、実行).
     *
     * unix.
     * z/os.
     */
    private static final String[] UNIX3_ENVCMD = {
        "env"
    } ;
    
    /**
     * 環境変数取得コマンド : netware系
     *
     * netware.
     * os/400.
     */
    private static final String[] NETWARE_ENVCMD = {
        "env"
    } ;
    
    /**
     * 環境変数取得コマンド : openvms系.
     * 取得方法が、他と異なる.
     */
    private static final String[] VMS_ENVCMD = {
        "show", "logical"
    } ;
    
    
    /**
     * 生成フラグ.
     */
    private volatile boolean m_createFlg = false ;
    
    /**
     * 環境変数情報管理.
     */
    private final ArrayTable m_table = new ArrayTable() ;
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    /**
     * シングルトン.
     */
    private static final GetEnvironment SNGL = new GetEnvironment() ;
    
    
    
    /**
     * コンストラクタ.
     */
    private GetEnvironment()
    {
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * シングルトンオブジェクトを取得.
     * <BR><BR>
     * シングルトンオブジェクトを取得します.
     * <BR>
     * @return GetEnvironment シングルトンオブジェクトが返されます.
     */
    public static final GetEnvironment getInstance()
    {
        //SNGL.reCreate() ;
        return SNGL ;
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 環境変数を取得します。
     * <BR>
     * @exception InputException 入力例外.
     * @exception ExecutionException 実行例外.
     * @exception UnsupportException サポート外例外.<BR>
     *                               この例外が検知された場合、
     *                               環境変数取得はサポートされて
     *                               いません.
     */
    public final void create()
        throws InputException,ExecutionException,UnsupportException
    {
        this.clear() ;
        
        try{
            
            m_sync.create() ;
            
            synchronized( m_sync.get() ){
                
                switch( Os.getInstance().getOS() ){
                    case Os.OS_OS_2 :
                    case Os.OS_WINNT :
                    case Os.OS_WIN9X :
                    case Os.OS_Z_OS :
                    case Os.OS_UNIX :
                    case Os.OS_NETWARE :
                    case Os.OS_OS_400 :
                        GetEnvironment.getEnv( m_table,GetEnvironment.getCmd() ) ;
                        break ;
                    case Os.OS_OPENVMS :
                        GetEnvironment.getVmsEnv( m_table,GetEnvironment.getCmd() ) ;
                        break ;
                    default :
                        
                        throw new UnsupportException(
                            "対象のOS(" +
                            Os.getInstance().getName( Os.getInstance().getOS() ) +
                            ")では、環境変数取得処理を対応していません"
                        ) ;
                        
                }
                
            }
            
        }catch( InputException in ){
            this.clear() ;
            throw new ExecutionException( in ) ;
        }catch( UnsupportException ue ){
            this.clear() ;
            throw ue ;
        }catch( ExecutionException ee ){
            this.clear() ;
            throw ee ;
        }catch( Exception t ){
            this.clear() ;
        }
        
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public final void clear()
    {
        
        m_sync.clear() ;
        m_createFlg = false ;
        m_table.clear() ;
        
    }
    
    /**
     * テーブルオブジェクトの取得.
     * <BR><BR>
     * 環境変数が格納されているテーブルオブジェクトを取得します.
     * <BR>
     * @return ArrayTable テーブルオブジェクトが返されます.
     */
    public final ArrayTable getTable()
    {
        ArrayTable ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_table ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 環境変数取得処理.
     * <BR><BR>
     * 対象の環境変数情報を取得します.
     * <BR>
     * @param name 対象の環境変数名を設定します.
     * @return String 環境変数内に格納されている情報を取得します.<BR>
     *                情報が存在しない場合[null]が返されます.
     */
    public final String get( String name )
    {
        String ret = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                ret = ( String )m_table.get( name,0 ) ;
            }
            
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 環境変数取得処理.
     * <BR><BR>
     * 対象の環境変数情報を取得します.
     * <BR>
     * @param name 対象の環境変数名を設定します.
     * @param no 格納されている環境変数名の情報項番を設定します.
     * @return String 環境変数内に格納されている情報を取得します.<BR>
     *                情報が存在しない場合[null]が返されます.
     */
    public final String get( String name,int no )
    {
        String ret = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                ret = ( String )m_table.get( name,no ) ;
            }
            
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 環境変数取得処理.
     * <BR><BR>
     * 対象の環境変数情報を取得します.
     * <BR>
     * @param name 対象の環境変数名を設定します.
     * @return int 環境変数内に格納されている情報を取得します.<BR>
     *             情報が存在しない場合[0]が返されます.
     */
    public final int getNumber( String name )
    {
        int ret = 0 ;
        
        try{
            
            synchronized( m_sync.get() ){
                ret = Integer.parseInt(
                    ( String )m_table.get( name,0 )
                ) ;
            }
            
        }catch( Exception t ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 環境変数取得処理.
     * <BR><BR>
     * 対象の環境変数情報を取得します.
     * <BR>
     * @param name 対象の環境変数名を設定します.
     * @param no 格納されている環境変数名の情報項番を設定します.
     * @return int 環境変数内に格納されている情報を取得します.<BR>
     *                情報が存在しない場合[0]が返されます.
     */
    public final int getNumber( String name,int no )
    {
        int ret = 0 ;
        
        try{
            
            synchronized( m_sync.get() ){
                ret = Integer.parseInt(
                    ( String )m_table.get( name,no )
                ) ;
            }
            
        }catch( Exception t ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 環境変数取得処理.
     * <BR><BR>
     * 対象の環境変数情報を取得します.
     * <BR>
     * @param name 対象の環境変数名を設定します.
     * @return double 環境変数内に格納されている情報を取得します.<BR>
     *                情報が存在しない場合[0.0f]が返されます.
     */
    public final double getFloat( String name )
    {
        double ret = 0.0f ;
        
        try{
            
            synchronized( m_sync.get() ){
                ret = Double.parseDouble(
                    ( String )m_table.get( name,0 )
                ) ;
            }
            
        }catch( Exception t ){
            ret = 0.0f ;
        }
        
        return ret ;
    }
    
    /**
     * 環境変数取得処理.
     * <BR><BR>
     * 対象の環境変数情報を取得します.
     * <BR>
     * @param name 対象の環境変数名を設定します.
     * @param no 格納されている環境変数名の情報項番を設定します.
     * @return double 環境変数内に格納されている情報を取得します.<BR>
     *                情報が存在しない場合[0.0f]が返されます.
     */
    public final double getFloat( String name,int no )
    {
        double ret = 0.0f ;
        
        try{
            
            synchronized( m_sync.get() ){
                ret = Double.parseDouble(
                    ( String )m_table.get( name,no )
                ) ;
            }
            
        }catch( Exception t ){
            ret = 0.0f ;
        }
        
        return ret ;
    }
    
    /**
     * 環境変数名に対する情報数の取得.
     * <BR><BR>
     * 環境変数名に対する情報数が返されます.
     * <BR>
     * @param name 対象の環境変数名を設定します.
     * @return int 環境変数名に対する情報数が返されます.
     *             [0]が返された場合情報は存在しません.
     */
    public int getElements( String name )
    {
        int ret = 0 ;
        
        try{
            
            synchronized( m_sync.get() ){
                ret = m_table.getElements( name ) ;
            }
            
        }catch( Exception t ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 環境変数名一覧の取得.
     * <BR><BR>
     * 対象の環境変数名一覧を取得します.
     * <BR>
     * @return String[] 環境変数名一覧が返されます.
     *                  情報が存在しない場合[null]が返されます.
     */
    public final String[] getNames()
    {
        String[] ret = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                ret = m_table.getNames() ;
            }
            
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * 情報生成処理.
     */
    private synchronized final void reCreate()
    {
        try{
            
            if( m_createFlg == false ){
                
                this.create() ;
                m_createFlg = true ;
                
            }
            
        }catch( UnsupportException ue ){
            // 処理が失敗した場合でも、
            // 処理を成功したこととする.
            m_createFlg = true ;
        }catch( Exception t ){
        }
    }
    
    /**
     * 環境変数取得処理.
     * <BR><BR>
     * 環境変数情報を取得します.
     * <BR>
     * @param out 環境変数情報を格納するためのオブジェクトをセットします.
     * @param cmd 環境変数を取得するためのコマンドをセットします.
     * @exception InputException 入力例外.
     * @exception ExecutionExcepiton 実行例外.
     */
    private static final void getEnv( ArrayTable out,String[] cmd )
        throws InputException,ExecutionException
    {
        int i,j ;
        int len ;
        int lenJ ;
        int index ;
        
        String cat = null ;
        String key = null ;
        String oneStdout = null ;
        String[] stdOut = null ;
        
        ArrayList lst = null ;
        
        if( out == null || cmd == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        cat = System.getProperty( "path.separator" ) ;
        
        try{
            
            if( ( stdOut = GetEnvironment.executionCmd( cmd ) ) == null ){
                throw new ExecutionException(
                    "対象のコマンド(" + cmd[ 0 ] + ")から情報は取得できませんでした"
                ) ;
            }
            lst = new ArrayList() ;
            len = stdOut.length ;
            for( i = 0 ; i < len ; i ++ ){
                
                key = null ;
                oneStdout = null ;
                
                oneStdout = stdOut[ i ] ;
                stdOut[ i ] = null ;
                
                if( ( index = oneStdout.indexOf( '=' ) ) == -1 ){
                    continue ;
                }
                else if( index+1 >= oneStdout.length() ){
                    key = oneStdout.substring( 0,index ) ;
                    out.add( key,"" ) ;
                    continue ;
                }
                
                key = oneStdout.substring( 0,index ) ;
                UtilCom.pauseString( lst,oneStdout.substring( index+1 ),cat ) ;
                
                if( ( lenJ = lst.size() ) > 0 ){
                    
                    for( j = 0 ; j < lenJ ; j ++ ){
                        out.add( key,lst.get( j ) ) ;
                    }
                    
                }
                else{
                    out.add( key,"" ) ;
                }
                
                lst.clear() ;
                
            }
            
        }catch( InputException in ){
            in.printStackTrace() ;
            throw in ;
        }catch( ExecutionException ee ){
            ee.printStackTrace() ;
            throw ee ;
        }catch( Exception e ){
            e.printStackTrace() ;
        }finally{
            
            if( lst != null ){
                lst.clear() ;
            }
            
            cat = null ;
            key = null ;
            oneStdout = null ;
            stdOut = null ;
            lst = null ;
        }
        
    }
    
    /**
     * 環境変数取得処理( OpenVms用 ).
     * <BR><BR>
     * 環境変数情報( OpenVms用 )を取得します.
     * <BR>
     * @param out 環境変数情報を格納するためのオブジェクトをセットします.
     * @param cmd 環境変数を取得するためのコマンドをセットします.
     * @exception InputException 入力例外.
     * @exception ExecutionExcepiton 実行例外.
     */
    private static final void getVmsEnv( ArrayTable out,String[] cmd )
        throws InputException,ExecutionException
    {
        int i ;
        int len ;
        int index ;
        
        String logName = null ;
        String logValue = null ;
        String newLogName = null ;
        
        String line = null ;
        String[] stdOut = null ;
        
        if( out == null || cmd == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            if( ( stdOut = GetEnvironment.executionCmd( cmd ) ) == null ){
                throw new ExecutionException(
                    "対象のコマンド(" + cmd[ 0 ] + ")から情報は取得できませんでした"
                ) ;
            }
            len = stdOut.length ;
            for( i = 0 ; i < len ; i ++ ){
                
                line = null ;
                line = stdOut[ i ] ;
                stdOut[ i ] = null ;
                
                if( line.startsWith( "\t=" ) == true ) {
                    if( logName != null ){
                        out.add( logName,line.substring( 4, line.length() - 1 ) ) ;
                    }
                }
                else if( line.startsWith( "  \"" ) == true ){
                    if( logName != null ){
                        out.add( logName,logValue ) ;
                    }
                    index = line.indexOf( '=' ) ;
                    newLogName = line.substring( 3, index - 2 ) ;
                    
                    if( out.getElements( newLogName ) != 0 ){
                        logName = null ;
                    }
                    else{
                        logName = newLogName ;
                        logValue = line.substring( index + 3, line.length() - 1 ) ;
                    }
                }
                
            }
            if( logName != null ){
                out.add( logName,logValue ) ;
            }
            
        }catch( InputException in ){
            throw in ;
        }catch( ExecutionException ee ){
            throw ee ;
        }finally{
            
            logName = null ;
            logValue = null ;
            newLogName = null ;
            line = null ;
            stdOut = null ;
            
        }
        
    }
    
    
    /**
     * OSにあわせたコマンド取得.
     * <BR><BR>
     * OSにあわせたコマンドを取得します.
     * <BR>
     * @return String[] コマンド情報が返されます.
     * @exception UnsupportException サポート外例外.
     */
    private static final String[] getCmd()
        throws UnsupportException
    {
        String[] ret = null ;
        
        switch( Os.getInstance().getOS() ){
            case Os.OS_OS_2 :
                ret = GetEnvironment.OS_2_ENVCMD ;
                break ;
            case Os.OS_WINNT :
                ret = GetEnvironment.WINNT_ENVCMD ;
                break ;
            case Os.OS_WIN9X :
                ret = GetEnvironment.WIN9X_ENVCMD ;
                break ;
            case Os.OS_Z_OS :
            case Os.OS_UNIX :
                if( GetEnvironment.isRead( GetEnvironment.UNIX1_ENVCMD[ 0 ] ) == true ){
                    ret = GetEnvironment.UNIX1_ENVCMD ;
                }else if( GetEnvironment.isRead( GetEnvironment.UNIX2_ENVCMD[ 0 ] ) == true ){
                    ret = GetEnvironment.UNIX2_ENVCMD ;
                }else {
                    ret = GetEnvironment.UNIX3_ENVCMD ;
                }
                break ;
            case Os.OS_NETWARE :
            case Os.OS_OS_400 :
                ret = GetEnvironment.NETWARE_ENVCMD ;
                break ;
            case Os.OS_OPENVMS :
                ret = GetEnvironment.VMS_ENVCMD ;
                break ;
            default :
                throw new UnsupportException(
                    "対象のOS(" +
                    Os.getInstance().getName( Os.getInstance().getOS() ) +
                    ")では、環境変数取得処理を対応していません"
                ) ;
        }
        
        return ret ;
    }
    
    /**
     * 対象処理をコマンド実行.
     * <BR><BR>
     * 対象の処理をコマンドにより実行します.
     * <BR>
     * @param cmd 実行されるコマンド情報を指定します.
     * @return String[] コマンド実行された結果が返されます.
     * @exception InputException 入力例外.
     * @exception NotExistException 情報存在例外.
     * @exception ExecutionException 実行例外.
     */
    private static final String[] executionCmd( String[] cmd )
        throws InputException,ExecutionException
    {
        
        int i ;
        int len ;
        
        NativeCmd cmdLine = null ;
        String[] ret = null ;
        
        if( cmd == null || ( len = cmd.length ) <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        cmdLine = new NativeCmd() ;
        
        try{
            
            if( ( ret = cmdLine.execution( false,cmd ) ) == null ){
                throw new ExecutionException(
                    "取得情報は存在しません( cmd : " +
                    cmdLine.getCommand() +
                    ")"
                ) ;
            }
            
// DEBUG-OUT.
/*
len = ret.length ;
for( i = 0 ; i < len ; i ++ ){
    System.out.println( "cmd[" + i + "]:" + ret[ i ] ) ;
}
*/

        }catch( ExecutionException ee ){
            ret = null ;
            throw ee ;
        }catch( BaseException be ){
            ret = null ;
            throw new ExecutionException( be ) ;
        }finally{
            
            try{
                cmdLine.clear() ;
            }catch( Exception t ){
            }
            cmdLine = null ;
            
        }
        
        return ret ;
    }
    
    /**
     * 指定ファイル名が読み取り可能かチェック.
     */
    public static final boolean isRead( String name )
    {
        boolean ret ;
        
        File file = null ;
        
        if( name == null || name.length() <= 0 ){
            return false ;
        }
        
        try{
            
            file = new File( name ) ;
            ret = ( file.exists() == true ) ?
                (
                    ( file.isDirectory() == false ) ?
                        file.canRead() : false
                ) 
                : false ;
            
        }catch( Exception e ){
            ret = false ;
        }finally{
            file = null ;
        }
        
        return ret ;
    }
    
}

