/*
 * @(#)AutoUserPasswd.java
 *
 * Copyright (c) 2004 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.sys;

import java.net.Authenticator;
import java.net.InetAddress;
import java.net.PasswordAuthentication;
import java.net.URL;

import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.thread.Synchronized;
import com.JRcServer.commons.util.ArrayTable;
import com.JRcServer.commons.util.UtilCom;

/**
 * JRcCommons用Authenticator.
 * <BR><BR>
 * JRcCommons用Authenticatorを提供します.<BR>
 * また、このオブジェクトはURLでのユーザ名/パスワード要求の際に
 * 自動的にログオンする事が可能となります.
 * また、このオブジェクトを有効にする場合[AutoUserPasswd.mount()]を
 * 実行する必要があります.
 *
 * @version 1.00, 2004/12/05
 * @author  Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class AutoUserPasswd extends Authenticator
{
    
    /**
     * 管理格納テーブル.
     */
    private final ArrayTable m_table = new ArrayTable() ;
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    /**
     * 一時設定のユーザ名.
     */
    private String m_tmpUser = null ;
    
    /**
     * 一時設定のパスワード.
     */
    private String m_tmpPasswd = null ;
    
    /**
     * 一時設定のプロトコル名.
     */
    private String m_tmpProtocol = null ;
    
    /**
     * シングルトン.
     */
    private static final AutoUserPasswd SNGL = new AutoUserPasswd() ;
    
    
    
    /**
     * コンストラクタ.
     */
    private AutoUserPasswd()
    {
        m_sync.create() ;
    }
    
    /**
     * オブジェクト情報を取得.
     * <BR><BR>
     * オブジェクト情報を取得します.
     * <BR>
     * @return AutoUserPasswd オブジェクト情報が返されます.
     */
    public static final AutoUserPasswd getInstance()
    {
        return SNGL ;
    }
    
    /**
     * 情報登録.
     * <BR><BR>
     * オブジェクトを登録します.
     * <BR>
     * @return boolean 処理結果が返されます.<BR>
     *                 [true]が返された場合処理は成功しました.<BR>
     *                 [false]が返された場合処理は失敗しました.
     */
    public static final boolean mount()
    {
        return SNGL.flush() ;
    }
    
    /**
     * 登録情報クリア.
     * <BR><BR>
     * 登録されている情報を全てクリアします.
     */
    public final void clear()
    {
        m_table.clear() ;
        m_sync.clear() ;
    }
    
    /**
     * 情報登録.
     * <BR><BR>
     * オブジェクト情報を登録します.
     * <BR>
     * @return boolean 処理結果が返されます.<BR>
     *                 [true]が返された場合処理は成功しました.<BR>
     *                 [false]が返された場合処理は失敗しました.
     */
    protected final boolean flush()
    {
        boolean ret ;
        
        try{
            
            if( m_sync.isUse() == false ){
                m_sync.create() ;
            }
            
            synchronized( m_sync.get() ){
                Authenticator.setDefault( this ) ;
            }
            ret = true ;
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 要求ユーザ名/パスワード情報を返します.
     * <BR><BR>
     * 要求のユーザ名/パスワード情報を返します.
     * <BR>
     * @return PasswordAuthentication 要求のユーザ名/パスワード情報が返されます.<BR>
     *                                対象条件が存在しない場合[null]が返されます.
     */
    protected PasswordAuthentication getPasswordAuthentication()
    {
        int i ;
        int len ;
        
        String host = null ;
        String protocol = null ;
        String scheme = null ;
        String prompt = null ;
        String tmpUser = null ;
        String tmpPasswd = null ;
        ArrayTable tbl = null ;
        AutoUserPasswdPause pause = null ;
        PasswordAuthentication ret = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                protocol = this.getRequestingProtocol() ;
                
                // 一時設定が存在して有効な場合.
                if(
                    m_tmpUser != null && m_tmpPasswd != null && m_tmpProtocol != null &&
                    protocol.equals( m_tmpProtocol ) == true
                )
                {
                    tmpUser = m_tmpUser ;
                    tmpPasswd = m_tmpPasswd ;
                    
                    m_tmpUser = null ;
                    m_tmpPasswd = null ;
                    m_tmpProtocol = null ;
                    
                    ret = new PasswordAuthentication( tmpUser,tmpPasswd.toCharArray() ) ;
                    
                }
                else{
                    
                    // 要求ホスト名を生成.
                    host = AutoUserPasswd.getHost(
                        this.getRequestingSite(),
                        this.getRequestingPort(),
                        protocol
                    ) ;
                    
                    tbl = m_table ;
                    
                    // 要求ホスト名が存在する場合.
                    if( ( len = tbl.getElements( host ) ) > 0 ){
                        
                        // 要求詳細情報を取得.
                        scheme = this.getRequestingScheme() ;
                        prompt = this.getRequestingPrompt() ;
                        
                        // 詳細情報あり(scheme,prompt)で検索.
                        for( i = 0 ; i < len ; i ++ ){
                            
                            pause = ( AutoUserPasswdPause )tbl.get( host,i ) ;
                            
                            if( pause.isCheck( scheme,prompt ) == true ){
                                ret = pause.get() ;
                                break ;
                            }
                            
                        }
                        
                        // 詳細条件なし(scheme,prompt)で検索.
                        if( ret == null ){
                            
                            // 詳細条件なしで検索.
                            for( i = 0 ; i < len ; i ++ ){
                                
                                pause = ( AutoUserPasswdPause )tbl.get( host,i ) ;
                                
                                if( pause.isCheck( null,null ) == true ){
                                    ret = pause.get() ;
                                    break ;
                                }
                                
                            }
                            
                        }
                        
                    }
                }
            }
        }catch( Exception t ){
            ret = null ;
        }finally{
            host = null ;
            scheme = null ;
            prompt = null ;
            tbl = null ;
            pause = null ;
            tmpUser = null ;
            tmpPasswd = null ;
            protocol = null ;
        }
        
        return ret ;
    }
    
    /**
     * 一時利用のユーザ名/パスワードを設定.
     * <BR><BR>
     * 一時利用のユーザ名/パスワードを設定します.<BR>
     * この情報は対象のプロトコル処理で１度だけ有効となります.
     * <BR>
     * @param protocol 対象のプロトコル名を設定します.
     * @param user 対象のユーザ名を設定します.
     * @param passwd 対象のパスワードを設定します.
     */
    public final void setTmpUserPasswd( String protocol,String user,String passwd )
    {
        try{
            if( protocol != null && user != null && passwd != null ){
                
                synchronized( m_sync.get() ){
                    
                    m_tmpUser = user ;
                    m_tmpPasswd = passwd ;
                    m_tmpProtocol = protocol ;
                    
                }
                
            }
        }catch( Exception t ){
        }
    }
    
    /**
     * 対象の条件でユーザ名/パスワードを追加.
     * <BR><BR>
     * 対象の条件でユーザ名/パスワードを追加します.<BR>
     * この条件指定の場合、対象のプロトコル+ホスト名+ポート番号からの
     * 認証要求を受けた場合の共通条件として設定します.<BR>
     * つまり、対象のスキーム(認証)条件やプロンプト条件が要求された場合に
     * 格納情報が対象条件と一致しない場合にここで設定したユーザ名と
     * パスワードを指定します.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     * @param passwd 対象のパスワードを設定します.
     * @param addr 対象のアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @param protocol 認証対象のプロトコル名を設定します.
     * @exception InputException 入力例外.
     */
    public final void put(
        String user,String passwd,InetAddress addr,int port,String protocol
    )
        throws InputException
    {
        this.put( user,passwd,addr,port,protocol,null,null ) ;
    }
    
    /**
     * 対象の条件でユーザ名/パスワードを追加.
     * <BR><BR>
     * 対象の条件でユーザ名/パスワードを追加します.
     * この条件指定の場合、対象のプロトコル+ホスト名+ポート番号+
     * スキーム+プロンプトが一致した場合のユーザ名、パスワードを
     * 設定します.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     * @param passwd 対象のパスワードを設定します.
     * @param addr 対象のアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @param protocol 認証対象のプロトコル名を設定します.
     * @param scheme 対象の認証方式を設定します.
     * @param prompt 対象のプロンプト情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void put(
        String user,String passwd,InetAddress addr,int port,String protocol,
        String scheme,String prompt
    )
        throws InputException
    {
        int i ;
        int len ;
        boolean flg ;
        
        String host = null ;
        ArrayTable tbl = null ;
        AutoUserPasswdPause pause = null ;
        
        
        // 要求ホスト名を生成.
        host = AutoUserPasswd.getHost( addr,port,protocol ) ;
        
        if(
            user == null || passwd == null || host == null ||
            ( scheme != null && prompt == null ) ||
            ( scheme == null && prompt != null )
        ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            if( m_sync.isUse() == false ){
                m_sync.create() ;
            }
            
            synchronized( m_sync.get() ){
                
                tbl = m_table ;
                flg = false ;
                
                // 同一の条件が存在する場合上書き.
                if( ( len = tbl.getElements( host ) ) > 0 ){
                    
                    for( i = 0 ; i < len ; i ++ ){
                        
                        pause = ( AutoUserPasswdPause )tbl.get( host,i ) ;
                        
                        if( pause.isCheck( scheme,prompt ) == true ){
                            
                            pause.create( user,passwd ) ;
                            flg = true ;
                            
                            break ;
                        }
                    }
                    
                }
                
                // 同一条件が存在しない場合.
                if( flg == false ){
                    
                    pause = new AutoUserPasswdPause( user,passwd,scheme,prompt ) ;
                    tbl.add( host,pause ) ;
                    
                }
                
            }
            
        }catch( Exception t ){
        }finally{
            host = null ;
            tbl = null ;
            pause = null ;
        }
    }
    
    /**
     * 対象の条件でユーザ名/パスワードを削除.
     * <BR><BR>
     * 対象の条件でユーザ名/パスワードを削除します.
     * <BR>
     * @param addr 対象のアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @param protocol 認証対象のプロトコル名を設定します.
     */
    public final void remove(
        InetAddress addr,int port,String protocol
    )
    {
        this.remove( addr,port,protocol,null,null ) ;
    }
    
    /**
     * 対象の条件でユーザ名/パスワードを削除.
     * <BR><BR>
     * 対象の条件でユーザ名/パスワードを削除します.
     * <BR>
     * @param addr 対象のアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @param protocol 対象のプロトコル名を設定します.
     * @param scheme 対象の認証方式を設定します.
     * @param prompt 対象のプロンプト情報を設定します.
     */
    public final void remove(
        InetAddress addr,int port,String protocol,String scheme,String prompt
    )
    {
        int i ;
        int len ;
        
        String host = null ;
        ArrayTable tbl = null ;
        AutoUserPasswdPause pause = null ;
        
        host = AutoUserPasswd.getHost( addr,port,protocol ) ;
        
        if(
            host == null ||
            ( scheme != null && prompt == null ) ||
            ( scheme == null && prompt != null )
        ){
            return ;
        }
        
        try{
            
            synchronized( m_sync.get() ){
                
                tbl = m_table ;
                
                // 削除対象条件が存在するかチェック.
                if( ( len = tbl.getElements( host ) ) > 0 ){
                    
                    for( i = 0 ; i < len ; i ++ ){
                        
                        pause = ( AutoUserPasswdPause )tbl.get( host,i ) ;
                        
                        if( pause.isCheck( scheme,prompt ) == true ){
                            
                            pause.clear() ;
                            tbl.remove( host,i ) ;
                            
                            break ;
                            
                        }
                    }
                    
                }
                
            }
            
        }catch( Exception t ){
        }finally{
            host = null ;
            tbl = null ;
            pause = null ;
        }
    }
    
    /**
     * 対象の条件でユーザ名/パスワードを全て削除.
     * <BR><BR>
     * 対象の条件でユーザ名/パスワードを全て削除します.
     * <BR>
     * @param addr 対象のアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @param protocol 認証対象のプロトコル名を設定します.
     */
    public final void removeAll(
        InetAddress addr,int port,String protocol
    )
    {
        String host = null ;
        
        host = AutoUserPasswd.getHost( addr,port,protocol ) ;
        
        try{
            synchronized( m_sync.get() ){
                m_table.remove( host ) ;
            }
        }catch( Exception t ){
        }finally{
            host = null ;
        }
    }
    
    /**
     * 要素名数を取得.
     * <BR><BR>
     * 要素名数を取得します.
     * <BR>
     * @return int 要素名数が返されます.
     */
    public final int size()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_table.getKeySize() ;
            }
        }catch( Exception t ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 要素名群を取得.
     * <BR><BR>
     * 対象要素名群を取得します.
     * <BR>
     * @return URL[] 対象要素群が返されます.
     */
    public final URL[] getURLs()
    {
        int i ;
        int len ;
        
        URL[] ret = null ;
        String[] tmp = null ;
        
        tmp = m_table.getNames() ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                len = tmp.length ;
                ret = new URL[ len ] ;
                
                for( i = 0 ; i < len ; i ++ ){
                    ret[ i ] = new URL( tmp[ i ] ) ;
                }
                
            }
            
        }catch( Exception t ){
            ret = null ;
        }finally{
            tmp = null ;
        }
        
        return ret ;
    }
    
    /**
     * 要素内情報数を取得.
     * <BR><BR>
     * 要素内の情報数を取得します.
     * <BR>
     * @param url 対象の要素URLを設定します.
     * @return int 要素内情報数が返されます.
     */
    public final int getElements( URL url )
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = this.getElements(
                    InetAddress.getByName( url.getHost() ),
                    url.getPort(),
                    url.getProtocol()
                ) ;
            }
        }catch( Exception t ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 要素内情報数を取得.
     * <BR><BR>
     * 要素内の情報数を取得します.
     * <BR>
     * @param addr 対象のアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @param protocol 認証対象のプロトコル名を設定します.
     * @return int 要素内情報数が返されます.
     */
    public final int getElements( InetAddress addr,int port,String protocol )
    {
        int ret ;
        
        String host = null ;
        
        try{
            host = AutoUserPasswd.getHost( addr,port,protocol ) ;
            synchronized( m_sync.get() ){
                ret = m_table.getElements( host ) ;
            }
        }catch( Exception t ){
            ret = 0 ;
        }finally{
            host = null ;
        }
        
        return ret ;
    }
    
    /**
     * 要素内スキーム名を取得.
     * <BR><BR>
     * 要素内スキーム名を取得します.
     * <BR>
     * @param no 対象の情報項番を設定します.
     * @param url 対象の要素URLを設定します.
     * @return String 対象のスキーム名が返されます.
     */
    public final String getScheme( int no,URL url )
    {
        String ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = this.getScheme(
                    no,
                    InetAddress.getByName( url.getHost() ),
                    url.getPort(),
                    url.getProtocol()
                ) ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 要素内スキーム名を取得.
     * <BR><BR>
     * 要素内スキーム名を取得します.
     * <BR>
     * @param no 対象の情報項番を設定します.
     * @param addr 対象のアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @param protocol 認証対象のプロトコル名を設定します.
     * @return String 対象のスキーム名が返されます.
     */
    public final String getScheme( int no,InetAddress addr,int port,String protocol )
    {
        String ret = null ;
        String host = null ;
        
        try{
            host = AutoUserPasswd.getHost( addr,port,protocol ) ;
            synchronized( m_sync.get() ){
                ret = ( ( AutoUserPasswdPause )m_table.get( host,no ) ).getScheme() ;
            }
        }catch( Exception t ){
            ret = null ;
        }finally{
            host = null ;
        }
        
        return ret ;
    }
    
    /**
     * 要素内プロンプト名を取得.
     * <BR><BR>
     * 要素内プロンプト名を取得します.
     * <BR>
     * @param no 対象の情報項番を設定します.
     * @param url 対象の要素URLを設定します.
     * @return String 対象のプロンプト名が返されます.
     */
    public final String getPrompt( int no,URL url )
    {
        String ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = this.getPrompt(
                    no,
                    InetAddress.getByName( url.getHost() ),
                    url.getPort(),
                    url.getProtocol()
                ) ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 要素内プロンプト名を取得.
     * <BR><BR>
     * 要素内プロンプト名を取得します.
     * <BR>
     * @param no 対象の情報項番を設定します.
     * @param addr 対象のアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @param protocol 認証対象のプロトコル名を設定します.
     * @return String 対象のプロンプト名が返されます.
     */
    public final String getPrompt( int no,InetAddress addr,int port,String protocol )
    {
        String ret = null ;
        String host = null ;
        
        try{
            host = AutoUserPasswd.getHost( addr,port,protocol ) ;
            synchronized( m_sync.get() ){
                ret = ( ( AutoUserPasswdPause )m_table.get( host,no ) ).getPrompt() ;
            }
        }catch( Exception t ){
            ret = null ;
        }finally{
            host = null ;
        }
        
        return ret ;
    }
    
    /**
     * 同期オブジェクト.
     * <BR><BR>
     * 同期オブジェクトを取得します.
     * <BR>
     * @return Synchronized 同期オブジェクトが返されます.
     */
    public final Synchronized getSynchronized()
    {
        return m_sync ;
    }
    
    
    
    /**
     * ホスト名を生成.
     */
    private static final String getHost( InetAddress addr,int port,String protocol )
    {
        StringBuffer buf = null ;
        String ret = null ;
        
        try{
            
            buf = new StringBuffer() ;
            
            buf.append( UtilCom.trimPlus( protocol ) ) ;
            buf.append( "://" ) ;
            buf.append( addr.getHostAddress() ) ;
            buf.append( ":" ) ;
            buf.append( port ) ;
            buf.append( "/" ) ;
            
            ret = buf.toString() ;
            
        }catch( Exception t ){
            ret = null ;
        }finally{
            buf = null ;
        }
        
        return ret ;
    }
    
}

/**
 * ユーザ名/パスワード条件要素.
 */
class AutoUserPasswdPause
{
    
    /**
     * ユーザ名.
     */
    private String m_user = null ;
    
    /**
     * パスワード.
     */
    private String m_passwd = null ;
    
    /**
     * 認証条件.
     */
    private String m_code = null ;
    
    
    
    /**
     * コンストラクタ.
     */
    public AutoUserPasswdPause(){
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 条件を設定します.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     * @param passwd 対象のパスワードを設定します.
     * @exception InputException 入力例外.
     */
    public AutoUserPasswdPause( String user,String passwd )
        throws InputException
    {
        this.create( user,passwd,null,null ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 条件を設定します.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     * @param passwd 対象のパスワードを設定します.
     * @param scheme 対象の認証方式を設定します.
     * @param prompt 対象のプロンプト情報を設定します.
     * @exception InputException 入力例外.
     */
    public AutoUserPasswdPause( String user,String passwd,String scheme,String prompt )
        throws InputException
    {
        this.create( user,passwd,scheme,prompt ) ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 条件を設定します.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     * @param passwd 対象のパスワードを設定します.
     * @exception InputException 入力例外.
     */
    public final void create( String user,String passwd )
        throws InputException
    {
        this.create( user,passwd,null,null ) ;
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 条件を設定します.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     * @param passwd 対象のパスワードを設定します.
     * @param scheme 対象の認証方式を設定します.
     * @param prompt 対象のプロンプト情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void create( String user,String passwd,String scheme,String prompt )
        throws InputException
    {
        this.clear() ;
        m_user = UtilCom.trimPlus( user ) ;
        m_passwd = UtilCom.trimPlus( passwd ) ;
        
        m_code = AutoUserPasswdPause.getCode( scheme,prompt ) ;
    }
    
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public final void clear()
    {
        m_user = null ;
        m_passwd = null ;
        m_code = null ;
    }
    
    /**
     * 認証情報を取得.
     * <BR><BR>
     * 認証情報を取得します.
     * <BR>
     * @return PasswordAuthentication 認証結果のユーザ名/パスワード情報が返されます.<BR>
     *                                [null]が返された場合条件の情報は存在しません.
     */
    public final PasswordAuthentication get()
    {
        PasswordAuthentication ret = null ;
        
        try{
            ret = new PasswordAuthentication( m_user,m_passwd.toCharArray() ) ;
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 認証情報を取得.
     * <BR><BR>
     * 認証情報を取得します.
     * <BR>
     * @param scheme 対象の認証方式を設定します.
     * @param prompt 対象のプロンプト情報を設定します.
     * @return PasswordAuthentication 認証結果のユーザ名/パスワード情報が返されます.<BR>
     *                                [null]が返された場合条件の情報は存在しません.
     */
    public final PasswordAuthentication get( String scheme,String prompt )
    {
        PasswordAuthentication ret = null ;
        
        try{
            if( this.isCheck( scheme,prompt ) == true ){
                ret = new PasswordAuthentication( m_user,m_passwd.toCharArray() ) ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 認証文字コードからスキーム名を取得.
     * <BR><BR>
     * 認証文字からスキーム名を取得します.
     * <BR>
     * @return String スキーム名が返されます.
     */
    public final String getScheme()
    {
        String ret = null ;
        
        if( m_code != null ){
            ret = m_code.substring( 0,m_code.indexOf( "@" ) ) ;
        }
        
        return ret ;
    }
    
    /**
     * 認証文字コードからプロンプト名を取得.
     * <BR><BR>
     * 認証文字からプロンプト名を取得します.
     * <BR>
     * @return String プロンプト名が返されます.
     */
    public final String getPrompt()
    {
        String ret = null ;
        
        if( m_code != null ){
            ret = m_code.substring( m_code.indexOf( "@" ) ) ;
        }
        
        return ret ;
    }
    
    /**
     * チェック.
     * <BR><BR>
     * 対象条件をチェックします.
     * <BR>
     * @param scheme 対象の認証方式を設定します.
     * @param prompt 対象のプロンプト情報を設定します.
     * @return boolean チェック結果が返されます.
     */
    public final boolean isCheck( String scheme,String prompt )
    {
        boolean ret ;
        
        if( scheme == null && prompt == null && m_code == null ){
            return true ;
        }
        
        try{
            ret = m_code.equals( AutoUserPasswdPause.getCode( scheme,prompt ) ) ;
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 認証文字コード変換.
     */
    private static final String getCode( String scheme,String prompt )
        throws InputException
    {
        StringBuffer buf = null ;
        String ret = null ;
        
        
        if( scheme != null && prompt != null ){
            
            buf = new StringBuffer() ;
            
            buf.append( UtilCom.trimPlus( scheme ) ) ;
            buf.append( "@" ) ;
            buf.append( UtilCom.trimPlus( prompt ) ) ;
            
            ret = buf.toString() ;
            buf = null ;
            
        }
        
        return ret ;
    }
    
}

