/*
 * @(#)UseSectorFlags.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.resource.cache;

import java.io.Serializable;
import java.util.Arrays;

import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.serialize.SerializeUtil;

/**
 * 有効セクターフラグ管理オブジェクト.
 * <BR><BR>
 * 有効なセクターフラグ情報を管理するオブジェクトです.
 * 
 * @version 2006/07/28
 * @author  masahito suzuki
 * @since  JRcCommons 1.00
 */
class UseSectorFlags implements Serializable {
    
    static {
        serialVersionUID = SerializeUtil.serialVersionUID(
            UseSectorFlags.class.getName()
        ) ;
    }
    
    /**
     * シリアライズUID.
     */
    private static final long serialVersionUID ;
    
    /**
     * １データ内要素数.
     */
    private static final int ELEMENT_LENGTH = 64 ;
    
    /**
     * １データ内要素内単位.
     */
    private static final int ELEMENT_MASK = ELEMENT_LENGTH - 1 ;
    
    /**
     * データビット数.
     */
    private static final int ELEMENT_BITCOUNT = 6 ;
    
    
    
    /**
     * sectorフラグ管理.
     */
    private long[] sector64Flags = null ;
    
    /**
     * sectorフラグ個数.
     */
    private int maxSectorSize = 0 ;
    
    /**
     * 有効sectorフラグ個数.
     */
    private int useSectorSize = 0 ;
    
    
    
    /**
     * コンストラクタ.
     */
    public UseSectorFlags() {
        
        
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 管理対象のフラグ管理長を設定します.
     * <BR>
     * @param size 管理対象のフラグ長を設定します.
     * @exception InputException 入力例外.
     */
    public UseSectorFlags( int size )
        throws InputException {
        
        this.create( size ) ;
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception {
        
        try {
            this.clear() ;
        } catch( Exception t ) {
        }
        
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 管理対象のフラグ管理長を設定します.
     * <BR>
     * @param size 管理対象のフラグ長を設定します.
     * @exception InputException 入力例外.
     */
    public void create( int size )
        throws InputException {
        
        int no ;
        
        if ( size < 0 ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.clear() ;
        
        no = ( ( ( size & UseSectorFlags.ELEMENT_MASK ) != 0 ) ? 1 : 0 ) +
            ( ( size & (~UseSectorFlags.ELEMENT_MASK) ) >> UseSectorFlags.ELEMENT_BITCOUNT ) ;
        
        sector64Flags = new long[ no ] ;
        Arrays.fill( sector64Flags,0L ) ;
        
        maxSectorSize = no * UseSectorFlags.ELEMENT_LENGTH ;
        useSectorSize = 0 ;
        
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public void clear() {
        
        sector64Flags = null ;
        maxSectorSize = 0 ;
        useSectorSize = 0 ;
        
    }
    
    /**
     * 対象要素を有効/無効に設定.
     * <BR><BR>
     * 対象の要素を有効/無効に設定します.
     * <BR>
     * @param flg 対象の条件を設定します.<BR>
     *            [true]を設定した場合有効となります.<BR>
     *            [false]を設定した場合無効となります.
     * @param no 設定対象の項番を設定します.
     */
    public void put( boolean flg,int no ) {
        
        int indNo ;
        int indOff ;
        
        long[] flags = null ;
        
        flags = sector64Flags ;
        
        if ( flags == null || no < 0 || no >= maxSectorSize ) {
            return ;
        }
        
        indNo = ( no & (~UseSectorFlags.ELEMENT_MASK) ) >> UseSectorFlags.ELEMENT_BITCOUNT ;
        indOff = no & UseSectorFlags.ELEMENT_MASK ;
        
        if ( flg == true ) {
            
            if ( ( flags[ indNo ] & ( 1L << indOff ) ) == 0 ) {
                useSectorSize ++ ;
            }
            
            flags[ indNo ] |= ( ( 1L << indOff ) & 0xffffffffffffffffL ) ;
            
        }
        else {
            
            if ( ( flags[ indNo ] & ( 1L << indOff ) ) != 0 ) {
                useSectorSize -- ;
            }
            
            flags[ indNo ] &= ( ( ~( 1L << indOff ) ) & 0xffffffffffffffffL ) ;
            
        }
        
    }
    
    /**
     * 対象要素を取得.
     * <BR><BR>
     * 対象の要素を取得します.
     * <BR>
     * @param no 取得対象の要素番号を設定します.
     * @return boolean 取得された内容が返されます.<BR>
     *                 [true]が返された場合有効です.<BR>
     *                 [false]が返された場合無効です.
     */
    public boolean get( int no ) {
        
        long[] flags = null ;
        
        flags = sector64Flags ;
        
        if ( flags == null || no < 0 || no >= maxSectorSize ) {
            return false ;
        }
        
        return (
            (
                flags[ ( no & (~UseSectorFlags.ELEMENT_MASK) ) >> UseSectorFlags.ELEMENT_BITCOUNT ] &
                ( 1L << ( no & UseSectorFlags.ELEMENT_MASK ) ) 
            ) == 0
        ) ? false : true ;
    }
    
    /**
     * 有効領域を検索.
     * <BR><BR>
     * 有効領域を検索します.
     * <BR>
     * @return int 有効な項番が返されます.<BR>
     *             空き領域が存在しない場合[-1]が返されます.
     */
    public int useSearchSector() {
        
        return this.useSearchSector( 0 ) ;
        
    }
    
    /**
     * 有効領域を検索.
     * <BR><BR>
     * 有効領域を検索します.
     * <BR>
     * @param index 検索開始位置を設定します.
     * @return int 有効な項番が返されます.<BR>
     *             空き領域が存在しない場合[-1]が返されます.
     */
    public int useSearchSector( int index ) {
        
        int i,j ;
        int len ;
        int max ;
        int size ;
        boolean flg ;
        int ret ;
        
        long[] flags = null ;
        
        ret = -1 ;
        size = 0 ;
        
        size = maxSectorSize ;
        flags = sector64Flags ;
        
        if( flags == null ){
            return -1 ;
        }
        
        if ( index <= 0 ) {
            
            index = 0 ;
            
        }
        else {
            
            index = ( index & (~UseSectorFlags.ELEMENT_MASK) ) >>
                UseSectorFlags.ELEMENT_BITCOUNT ;
            
        }
        
        len = UseSectorFlags.ELEMENT_LENGTH ;
        max = flags.length ;
        
        // 情報を検索.
        for ( i = index,ret = -1,flg = false ; ( i < max && flg == false ) ; i ++ ) {
            
            // 空き領域が存在する場合.
            if ( flags[ i ] != 0xffffffffffffffffL ) {
                
                // 空き条件を調べる.
                for ( j = 0 ; j < len ; j ++ ) {
                    
                    // 空きデータが検知された場合.
                    if ( ( flags[ i ] & ( 1L << j ) ) == 0 ) {
                        
                        ret = ( i * UseSectorFlags.ELEMENT_LENGTH ) + j ;
                        
                        if ( ret >= size ) {
                            ret = -1 ;
                        }
                        else {
                            flg = true ;
                        }
                        
                        break ;
                        
                    }
                    
                }
                
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * 管理フラグ個数を取得.
     * <BR><BR>
     * 管理されているフラグ個数を取得します.
     * <BR>
     * @return int 管理されているフラグ個数が返されます.
     */
    public int getMaxSize() {
        
        return maxSectorSize ;
        
    }
    
    /**
     * 有効となっているフラグ個数を取得.
     * <BR><BR>
     * 有効となっているフラグ個数を取得します.
     * <BR>
     * @return int 有効となっているフラグ個数が返されます.
     */
    public int getUseSize() {
        
        return useSectorSize ;
        
    }
    
    /**
     * 利用可能チェック.
     * <BR><BR>
     * オブジェクトが利用可能であるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合利用可能です.<BR>
     *                 [false]が返された場合利用不可能です.
     */
    public boolean isUseObject() {
        
        return ( sector64Flags != null ) ? true : false ;
        
    }
    
}

