/*
 * @(#)Cache.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.resource.cache;

import com.JRcServer.commons.beans.BaseBean;
import com.JRcServer.commons.exception.AccessException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.sys.Initializer;
import com.JRcServer.commons.sys.Manager;
import com.JRcServer.commons.sys.NamingManager;
import com.JRcServer.commons.util.BigTableSync;
import com.JRcServer.commons.util.SearchPortion;

/**
 * キャッシュマネージャ.
 *  
 * @version 2006/08/29
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
public class CacheManager implements Initializer {
    
    /**
     * キャッシュオブジェクト管理名.
     */
    public static final String MANAGER_NAME = "cache@manager" ;
    
    /**
     * ネーミングマネージャ登録拡張子.
     */
    public static final String NAMING_PLUS = "@cache.manager" ;
    
    /**
     * 同期オブジェクト.
     */
    private static final Object SYNC = new Object() ;
    
    /**
     * コンストラクタ.
     */
    public CacheManager() {
        
    }
    
    /**
     * 初期処理.
     * <BR><BR>
     * 初期処理を実施します.<BR>
     * この処理はこのオブジェクトを利用する場合、１度呼び出す必要があります.<BR>
     * また、１度呼び出した場合２度目に呼び出しても効果がありません.<BR>
     * しかし、一度オブジェクトを破棄 Initializer.destroy() した場合、
     * 再び呼び出す事が可能となります.<BR>
     * また、このメソッドの場合、デフォルト値での初期化処理になります.
     * <BR>
     * @exception AccessException アクセス例外.
     */
    public synchronized void init() throws AccessException {
        this.init( null ) ;
    }
    
    /**
     * 初期処理.
     * <BR><BR>
     * 初期処理を実施します.<BR>
     * この処理はこのオブジェクトを利用する場合、１度呼び出す必要があります.<BR>
     * また、１度呼び出した場合２度目に呼び出しても効果がありません.<BR>
     * しかし、一度オブジェクトを破棄 Initializer.destroy() した場合、
     * 再び呼び出す事が可能となります.
     * <BR>
     * @param bean 設定値が設定されているBeanオブジェクトを設定します.
     * @exception AccessException アクセス例外.
     */
    public synchronized void init( BaseBean bean ) throws AccessException {
        
        BigTableSync table = null ;
        
        if( Manager.getInstance().isData( MANAGER_NAME ) == true ) {
            throw new AccessException( CacheManager.class.getName() + " は既に存在します" ) ;
        }
        
        try {
            table = new BigTableSync() ;
            Manager.getInstance().put( MANAGER_NAME,table ) ;
        }catch( Exception e ) {
            this.destroy() ;
            throw new AccessException( e ) ;
        }
        
    }
    
    /**
     * 終了処理.
     * <BR><BR>
     * 終了処理を実施します.<BR>
     * また、この処理を実施した場合、再び Initializer.init() を
     * 呼び出さないとオブジェクトの利用が出来なくなります.
     */
    public synchronized void destroy() {
        
        BigTableSync table = null ;
        
        if( ( table = ( BigTableSync )Manager.getInstance().get( MANAGER_NAME ) ) == null ) {
            return ;
        }
        
        table.clear() ;
        table = null ;
        
    }
    
    /**
     * キャッシュ情報を生成.
     * <BR><BR>
     * キャッシュ情報を生成します.
     * <BR>
     * @param cacheFileName セクタデータを保持するファイル名を設定します.
     * @param size セクタサイズを設定します.<BR>
     *             設定可能な最小値は[CacheDef.MIN_SECTOR]です.<BR>
     *             設定可能な最大値は[CacheDef.MAX_SECTOR]です.
     * @return long 生成されたキャッシュ情報に対するユニークIDが返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final long create( String cacheFileName,int size )
        throws InputException,AccessException {
        
        long ret = -1 ;
        
        BigTableSync table = null ;
        SectorIO io = null ;
        CacheIOImple imple = null ;
        
        synchronized( SYNC ) {
            
            if( ( table = ( BigTableSync )Manager.getInstance().get( MANAGER_NAME ) ) == null ) {
                throw new AccessException( CacheManager.class.getName() + " は存在しません" ) ;
            }
            
            try {
                
                io = new SectorIO( cacheFileName,size ) ;
                imple = new CacheIOImple( io ) ;
                
                ret = imple.getUniqueID() ;
                
                for( ;; ) {
                    
                    if( table.isData( ret ) == false ) {
                        break ;
                    }
                    
                    ret = ( long )( ( ret + 1L ) & 0x3fffffffffffffffL ) ;
                    
                }
                
                imple.setUniqueID( ret ) ;
                table.add( ret,imple ) ;
                
            } catch( Exception e ) {
                throw new AccessException( e ) ;
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * キャッシュオブジェクトを削除.
     * <BR><BR>
     * キャッシュオブジェクトを削除します.
     * <BR>
     * @param uniqueID 削除対象のユニークIDを設定します.
     */
    public static final void delete( Long uniqueID ) {
        
        if( uniqueID == null ) {
            return ;
        }
        
        CacheManager.delete( uniqueID.longValue() ) ;
        
    }
    
    /**
     * キャッシュオブジェクトを削除.
     * <BR><BR>
     * キャッシュオブジェクトを削除します.
     * <BR>
     * @param uniqueID 削除対象のユニークIDを設定します.
     */
    public static final void delete( long uniqueID ) {
        
        CacheIOImple io = null ;
        BigTableSync table = null ;
        
        synchronized( SYNC ) {
            
            if( ( table = ( BigTableSync )Manager.getInstance().get( MANAGER_NAME ) ) == null ) {
                return ;
            }
            
            if( ( io = ( CacheIOImple )table.remove( uniqueID ) ) != null ) {
                io.destroy() ;
            }
            
        }
        
    }
    
    /**
     * キャッシュオブジェクトを取得.
     * <BR><BR>
     * キャッシュオブジェクトを取得します.
     * <BR>
     * @param uniqueID 対象のユニークIDを設定します.
     * @return CacheIO 対象のキャッシュI/Oオブジェクトが返されます.
     */
    public static final CacheIO get( Long uniqueID ) {
        
        if( uniqueID == null ) {
            return null ;
        }
        
        return CacheManager.get( uniqueID.longValue() ) ;
        
    }
    
    /**
     * キャッシュオブジェクトを取得.
     * <BR><BR>
     * キャッシュオブジェクトを取得します.
     * <BR>
     * @param uniqueID 対象のユニークIDを設定します.
     * @return CacheIO 対象のキャッシュI/Oオブジェクトが返されます.
     */
    public static final CacheIO get( long uniqueID ) {
        
        CacheIOImple ret = null ;
        BigTableSync table = null ;
        
        synchronized( SYNC ) {
            
            if( ( table = ( BigTableSync )Manager.getInstance().get( MANAGER_NAME ) ) == null ) {
                ret = null ;
            }
            else {
                ret = ( CacheIOImple )table.get( uniqueID ) ;
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * キャッシュ情報を生成してネーミングマネージャに登録.
     * <BR><BR>
     * キャッシュ情報を生成してネーミングマネージャに登録します.
     * <BR>
     * @param naming ネーミングマネージャ登録対象の名前を設定します.
     * @param cacheFileName セクタデータを保持するファイル名を設定します.
     * @param size セクタサイズを設定します.<BR>
     *             設定可能な最小値は[CacheDef.MIN_SECTOR]です.<BR>
     *             設定可能な最大値は[CacheDef.MAX_SECTOR]です.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final void putNamingManagerByCacheIO( String naming,String cacheFileName,int size )
        throws InputException,AccessException {
        
        long uniqueID = -1 ;
        
        if( naming == null || ( naming = naming.trim() ).length() <= 0 ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        synchronized( SYNC ) {
            uniqueID = CacheManager.create( cacheFileName,size ) ;
            CacheManager.putNamingManager( naming,uniqueID ) ;
        }
        
    }
    
    /**
     * 取得UniqueIDをネーミングマネージャに登録.
     * <BR><BR>
     * [CacheManager.create()]で取得したUniqueIDをネーミングマネージャに登録します.
     * <BR>
     * @param name 対象の名前を設定します.
     * @param uniqueID 対象のユニークIDを設定します.
     */
    public static final void putNamingManager( String name,Long uniqueID ) {
        
        if( uniqueID == null ) {
            return ;
        }
        
        CacheManager.putNamingManager( name,uniqueID.longValue() ) ;
        
    }
    
    /**
     * 取得UniqueIDをネーミングマネージャに登録.
     * <BR><BR>
     * [CacheManager.create()]で取得したUniqueIDをネーミングマネージャに登録します.
     * <BR>
     * @param name 対象の名前を設定します.
     * @param uniqueID 対象のユニークIDを設定します.
     */
    public static final void putNamingManager( String name,long uniqueID ) {
        
        if(
            name == null || ( name = name.trim().toLowerCase() ).length() <= 0 ||
            uniqueID < 0
        ) {
            return ;
        }
        
        synchronized( SYNC ) {
            name = new StringBuffer().append( name ).append( NAMING_PLUS ).toString() ;
            NamingManager.add( name,new Long( uniqueID ) ) ;
        }
        
    }
    
    /**
     * ネーミングマネージャに登録されている内容を破棄し、
     * キャッシュマネージャに登録されている内容も破棄.
     * <BR><BR>
     * ネーミングマネージャに登録されている内容を破棄し、
     * キャッシュマネージャに登録されている内容も破棄します.
     * <BR>
     * @param name 破棄対象の名前を設定します.
     */
    public static final void removeNamingManagerByCacheIO( String name ) {
        
        long uniqueID = -1L ;
        
        synchronized( SYNC ) {
            
            uniqueID = removeNamingManager( name ) ;
            
            if( uniqueID != -1L ) {
                CacheManager.delete( uniqueID ) ;
            }
            
        }
        
    }
    
    /**
     * ネーミングマネージャに登録されているUniqueIDを削除.
     * <BR><BR>
     * ネーミングマネージャに登録されているUniqueIDを削除します.
     * <BR>
     * @param name 対象の名前を設定します.
     * @return long ユニークIDが返されます.
     */
    public static final long removeNamingManager( String name ) {
        
        long ret = -1L ;
        Object o = null ;
        
        if( name == null || ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            return -1L ;
        }
        
        name = new StringBuffer().append( name ).append( NAMING_PLUS ).toString() ;
        
        synchronized( SYNC ) {
            
            o = NamingManager.get( name ) ;
            
            if( o != null && ( o instanceof Long ) == true ) {
                
                ret = ( ( Long )o ).longValue() ;
                NamingManager.remove( name ) ;
                
            }
            
        }
        
        return ret ;
        
    }
    
    
    /**
     * ネーミングマネージャに登録されているUniqueIDに対するCacheI/Oを取得.
     * <BR><BR>
     * ネーミングマネージャに登録されているUniqueIDに対するCacheI/Oを取得します.
     * <BR>
     * @param name 対象の名前を設定します.
     * @return CacheIO 対象のCacheI/Oが返されます.
     */
    public static final CacheIO getNamingManagerByCacheIO( String name ) {
        
        long uniqueID = -1L ;
        CacheIO ret = null ;
        
        synchronized( SYNC ) {
            
            uniqueID = getNamingManager( name ) ;
            
            if( uniqueID != -1L ) {
                ret = CacheManager.get( uniqueID ) ;
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * ネーミングマネージャに登録されているUniqueIDを取得.
     * <BR><BR>
     * ネーミングマネージャに登録されているUniqueIDを取得します.
     * <BR>
     * @param name 対象の名前を設定します.
     * @return long ユニークIDが返されます.
     */
    public static final long getNamingManager( String name ) {
        
        long ret = -1L ;
        Object o = null ;
        
        if( name == null || ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            return -1L ;
        }
        
        name = new StringBuffer().append( name ).append( NAMING_PLUS ).toString() ;
        
        synchronized( SYNC ) {
            
            o = NamingManager.get( name ) ;
            
            if( o != null && ( o instanceof Long ) == true ) {
                
                ret = ( ( Long )o ).longValue() ;
                
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * 登録されているキャッシュオブジェクト名一覧を取得.
     * <BR><BR>
     * 登録されているキャッシュオブジェクト名一覧を取得します.
     * <BR>
     * @return String[] キャッシュオブジェクト名一覧が返されます.
     */
    public static final String[] getNames() {
        
        String[] ret = null ;
        
        synchronized( SYNC ) {
            ret = SearchPortion.searchString(
                NamingManager.getNames(),
                new StringBuffer().append( "*" ).append( NAMING_PLUS ).toString() ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * 管理されているキャッシュオブジェクト群を取得.
     * <BR><BR>
     * 管理されているキャッシュオブジェクト群を取得します.
     * <BR>
     * @return int 管理されているキャッシュオブジェクト群が返されます.
     */
    public static final int size() {
        
        BigTableSync table = null ;
        
        synchronized( SYNC ) {
            
            if( ( table = ( BigTableSync )Manager.getInstance().get( MANAGER_NAME ) ) == null ) {
                return 0 ;
            }
            
            return table.size() ;
            
        }
        
    }
    
    /**
     * 管理されているキャッシュオブジェクトのユニークID群を取得.
     * <BR><BR>
     * 管理されているキャッシュオブジェクトのユニークID群を取得します.
     * <BR>
     * @return long[] 管理されているキャッシュオブジェクトのユニークID群が返されます.
     */
    public static final long[] getUniqueIDs() {
        
        BigTableSync table = null ;
        
        synchronized( SYNC ) {
            
            if( ( table = ( BigTableSync )Manager.getInstance().get( MANAGER_NAME ) ) == null ) {
                return null ;
            }
            
            return table.getNumbers() ;
            
        }
        
    }
    
    /**
     * 管理オブジェクトが有効であるかチェック.
     * <BR><BR>
     * 管理オブジェクトが有効であるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、有効です.<BR>
     *                 [false]が返された場合、無効です.
     */
    public static final boolean isManager() {
        
        synchronized( SYNC ) {
            
            return Manager.getInstance().isData( MANAGER_NAME ) ;
            
        }
        
    }
    
}

