/*
 * @(#)Cache.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.resource.cache;

import com.JRcServer.commons.exception.AccessException;


/**
 * キャッシュインターフェイス.
 *  
 * @version 2006/08/29
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
public interface Cache {
    
    /**
     * オブジェクトクローズ.
     * <BR><BR>
     * オブジェクトをクローズします.
     */
    public void close() ;
    
    /**
     * リセット.
     * <BR><BR>
     * 有効データ長を0にリセットします.
     */
    public void reset() ;
    
    /**
     * データ読み込み.
     * <BR><BR>
     * データを読み込みます.
     * <BR>
     * @param seek 読み込みポイント開始位置を設定します.
     * @return byte[] 読み込まれた情報が返されます.
     * @exception AccessException アクセス例外.
     */
    public byte[] read( int seek ) throws AccessException ;
    
    /**
     * データ読み込み.
     * <BR><BR>
     * データを読み込みます.
     * <BR>
     * @param out 読み込み対象のデータを設定します.
     * @param seek 読み込みポイント開始位置を設定します.
     * @return int 読み込まれた情報長が返されます.
     * @exception AccessException アクセス例外.
     */
    public int read( byte[] out,int seek )
        throws AccessException ;
    
    /**
     * データ読み込み.
     * <BR><BR>
     * データを読み込みます.
     * <BR>
     * @param out 読み込み対象のデータを設定します.
     * @param seek 読み込みポイント開始位置を設定します.
     * @param offset 読み込み格納のオフセット値を設定します.
     * @return int 読み込まれた情報長が返されます.
     * @exception AccessException アクセス例外.
     */
    public int read( byte[] out,int seek,int offset )
        throws AccessException ;
    
    /**
     * データ読み込み.
     * <BR><BR>
     * データを読み込みます.
     * <BR>
     * @param out 読み込み対象のデータを設定します.
     * @param seek 読み込みポイント開始位置を設定します.
     * @param offset 読み込み格納のオフセット値を設定します.
     * @param length 取得データ長を設定します.
     * @return int 読み込まれた情報長が返されます.
     * @exception AccessException アクセス例外.
     */
    public int read( byte[] out,int seek,int offset,int length )
        throws AccessException ;
    
    /**
     * データ書き込み.
     * <BR><BR>
     * 対象データを書き込みます.
     * <BR>
     * @param in 書き込み対象のデータを設定します.
     * @param seek 書き込みポイント開始位置を設定します.
     * @return int 書き込まれた情報長が返されます.
     * @exception NotUseSectorException セクタ取得失敗例外.
     * @exception AccessException アクセス例外.
     */
    public int write( byte[] in,int seek )
        throws NotUseSectorException,AccessException ;
    
    /**
     * データ書き込み.
     * <BR><BR>
     * 対象データを書き込みます.
     * <BR>
     * @param in 書き込み対象のデータを設定します.
     * @param seek 書き込みポイント開始位置を設定します.
     * @param offset 書き込み開始のオフセット値を設定します.
     * @return int 書き込まれた情報長が返されます.
     * @exception NotUseSectorException セクタ取得失敗例外.
     * @exception AccessException アクセス例外.
     */
    public int write( byte[] in,int seek,int offset )
        throws NotUseSectorException,AccessException ;
    
    /**
     * データ書き込み.
     * <BR><BR>
     * 対象データを書き込みます.
     * <BR>
     * @param in 書き込み対象のデータを設定します.
     * @param seek 書き込みポイント開始位置を設定します.
     * @param offset 書き込み開始のオフセット値を設定します.
     * @param length 書き込みデータ長を設定します.
     * @return int 書き込まれた情報長が返されます.
     * @exception NotUseSectorException セクタ取得失敗例外.
     * @exception AccessException アクセス例外.
     */
    public int write( byte[] in,int seek,int offset,int length )
        throws NotUseSectorException,AccessException ;
    
    /**
     * 現在の有効データ長を取得.
     * <BR><BR>
     * 現在の有効データ長を取得します.
     * <BR>
     * @return int 現在の有効データ長を取得します.
     */
    public int getLength() ;
    
    /**
     * 現在確保しているセクタ数を取得.
     * <BR><BR>
     * 現在確保しているセクタ数を取得します.
     * <BR>
     * @return int 確保しているセクタ数が返されます.
     */
    public int getSector() ;
}

