/*
 * @(#)ResourceType.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.resource ;

import java.io.Serializable;

import com.JRcServer.commons.io.IOCom;
import com.JRcServer.commons.resource.cache.CacheManager;
import com.JRcServer.commons.serialize.SerializeUtil;

/**
 * リソースタイプ.
 * <BR><BR>
 * リソースのタイプを定義するオブジェクトです.
 *
 * @version     1.00, 2005/04/06
 * @author      Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class ResourceType implements Serializable
{
    
    static {
        serialVersionUID = SerializeUtil.serialVersionUID(
            ResourceType.class.getName()
        ) ;
    }
    
    /**
     * シリアライズUID.
     */
    private static final long serialVersionUID ;
    
    /**
     * リソースタイプ : メモリー.
     */
    public static final int RESOURCE_TYPE_MEMORY = 0x00000001 ;
    
    /**
     * リソースタイプ : テンポラリファイル.
     */
    public static final int RESOURCE_TYPE_FILE = 0x00000002 ;
    
    /**
     * リソースタイプ : キャッシュファイル.
     */
    public static final int RESOURCE_TYPE_CACHE = 0x00000003 ;
    
    
    
    /**
     * リソースタイプ.
     */
    private int m_type = 0 ;
    
    /**
     * リソースタイプ:RESOURCE_TYPE_FILE の一時ディレクトリ.
     */
    private String m_tmpDir = null ;
    
    /**
     * リソースタイプ:RESOURCE_TYPE_CACHE のID.
     */
    private long m_cacheID = -1L ;
    
    /**
     * コンストラクタ.
     */
    public ResourceType()
    {
        m_type = BinResource.BIN_RESOURCE_TYPE_MEMORY ;
        m_tmpDir = null ;
        m_cacheID = -1L ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * リソースタイプを設定して生成します.
     * <BR>
     * @param type リソースタイプを設定します.<BR>
     *             [ResourceType#RESOURCE_TYPE_MEMORY]を設定した場合、
     *             メモリータイプリソースです.<BR>
     *             [ResourceType#RESOURCE_TYPE_FILE]を設定した場合、
     *             ファイルタイプリソースです.<BR>
     *             [ResourceType#RESOURCE_TYPE_CACHE]を設定した場合、
     *             キャッシュファイルタイプリソースです.
     * @param dir リソースタイプが ResourceType#RESOURCE_TYPE_FILE の場合の
     *            一時ディレクトリ名を設定します.
     */
    public ResourceType( int type,String dir )
    {
        this( type,dir,-1L ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * リソースタイプを設定して生成します.
     * <BR>
     * @param type リソースタイプを設定します.<BR>
     *             [ResourceType#RESOURCE_TYPE_MEMORY]を設定した場合、
     *             メモリータイプリソースです.<BR>
     *             [ResourceType#RESOURCE_TYPE_FILE]を設定した場合、
     *             ファイルタイプリソースです.<BR>
     *             [ResourceType#RESOURCE_TYPE_CACHE]を設定した場合、
     *             キャッシュファイルタイプリソースです.
     * @param cacheID リソースタイプが ResourceType#RESOURCE_TYPE_CACHE の場合の
     *                 キャッシュユニークIDを設定します.
     */
    public ResourceType( int type,long cacheID )
    {
        this( type,null,cacheID ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * リソースタイプを設定して生成します.
     * <BR>
     * @param type リソースタイプを設定します.<BR>
     *             [ResourceType#RESOURCE_TYPE_MEMORY]を設定した場合、
     *             メモリータイプリソースです.<BR>
     *             [ResourceType#RESOURCE_TYPE_FILE]を設定した場合、
     *             ファイルタイプリソースです.<BR>
     *             [ResourceType#RESOURCE_TYPE_CACHE]を設定した場合、
     *             キャッシュファイルタイプリソースです.
     * @param dir リソースタイプが ResourceType#RESOURCE_TYPE_FILE の場合の
     *            一時ディレクトリ名を設定します.
     * @param cacheID リソースタイプが ResourceType#RESOURCE_TYPE_CACHE の場合の
     *                 キャッシュユニークIDを設定します.
     */
    public ResourceType( int type,String dir,long cacheID )
    {
        if(
            type != ResourceType.RESOURCE_TYPE_MEMORY && 
            type != ResourceType.RESOURCE_TYPE_FILE && 
            type != ResourceType.RESOURCE_TYPE_CACHE
        )
        {
            type = ResourceType.RESOURCE_TYPE_MEMORY ;
        }
        
        if( type == ResourceType.RESOURCE_TYPE_MEMORY ) {
            dir = null ;
            cacheID = -1L ;
        }
        else if( type == ResourceType.RESOURCE_TYPE_FILE ) {
            
            cacheID = -1L ;
            
            if( dir != null ){
                try{
                    IOCom.mkdirs( dir ) ;
                }catch( Exception e ){
                    dir = null ;
                    type = ResourceType.RESOURCE_TYPE_MEMORY ;
                }
            }
            else {
                dir = null ;
                type = ResourceType.RESOURCE_TYPE_MEMORY ;
            }
            
        }
        else if( type == ResourceType.RESOURCE_TYPE_CACHE ) {
            
            dir = null ;
            
            if( CacheManager.get( cacheID ) == null ) {
                type = ResourceType.RESOURCE_TYPE_MEMORY ;
                cacheID = -1L ;
            }
            
        }
        
        m_type = type ;
        m_tmpDir = dir ;
        m_cacheID = cacheID ;
        
    }
    
    /**
     * 設定されているリソースタイプを取得.
     * <BR><BR>
     * 設定されているリソースタイプが返されます.
     * <BR>
     * @return int リソースタイプが返されます.<BR>
     *             [ResourceType#RESOURCE_TYPE_MEMORY]が返された場合、
     *             メモリータイプリソースです.<BR>
     *             [ResourceType#RESOURCE_TYPE_FILE]が返された場合、
     *             ファイルタイプリソースです.<BR>
     *             [ResourceType#RESOURCE_TYPE_CACHE]が返された場合、
     *             CACHEファイルタイプリソースです.
     */
    public final int getType()
    {
        return m_type ;
    }
    
    /**
     * 設定されている一時ディレクトリを取得.
     * <BR><BR>
     * 設定されている一時ディレクトリを取得します.
     * <BR>
     * @return String 一時ディレクトリが返されます.
     */
    public final String getTmpDir()
    {
        return m_tmpDir ;
    }
    
    /**
     * 設定されている対象キャッシュIDを取得.
     * <BR><BR>
     * 設定されている対象キャッシュIDを取得します.
     * <BR>
     * @return long 設定されている対象キャッシュIDが返されます.
     */
    public final long getCacheID()
    {
        return m_cacheID ;
    }
    
}
