/*
 * @(#)Resource.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.resource ;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;

import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.resource.cache.CacheIO;
import com.JRcServer.commons.resource.cache.CacheManager;

/**
 * リソースユーティリティ.
 *
 * @version     1.00, 2005/04/01
 * @author      Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class Resource
{
    private Resource(){}
    
    /**
     * リソース同期.
     */
    public static final Object SYNC = new Object() ;
    
    /**
     * コピーバッファ長.
     */
    private static final int BUF_LEN = 1024 ;
    
    /**
     * タイプに対するバイナリリソースを生成.
     * <BR><BR>
     * タイプに対するバイナリリソースを生成します.
     * <BR>
     * @param rs バイナリリソースステータスを設定します.
     * @param size リソースサイズを設定します.
     * @return BinResource 生成されたリソース情報が返されます.
     */
    public static final BinResource createBinResource( ResourceType rs,int size )
    {
        BinResource ret = null ;
        
        try{
            
            if( rs == null ){
                if( size <= 0 ){
                    ret = new BinMemoryResource( 1 ) ;
                }
                else{
                    ret = new BinMemoryResource( size ) ;
                }
            }
            else{
                ret = Resource.createBinResource( rs.getType(),rs.getTmpDir(),rs.getCacheID(),size ) ;
            }
            
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * タイプに対するバイナリリソースを生成.
     * <BR><BR>
     * タイプに対するバイナリリソースを生成します.
     * <BR>
     * @param type リソースタイプを設定します.<BR>
     *             [BinResource#BIN_RESOURCE_TYPE_MEMORY]を設定した場合、
     *             [com.JRcServer.commons.resource.BinMemoryResource]オブジェクトです.<BR>
     *             [BinResource#BIN_RESOURCE_TYPE_FILE]を設定した場合、
     *             [com.JRcServer.commons.resource.BinFileResource]オブジェクトです.<BR>
     *             [BinResource#BIN_RESOURCE_TYPE_CACHE]を設定した場合、
     *             [com.JRcServer.commons.resource.BinCacheResource]オブジェクトです.
     * @param tmpDir リソースタイプが BinResource#BIN_RESOURCE_TYPE_FILE の場合の
     *               一時ディレクトリ名を設定します.<BR>
     *               [null]を設定した場合、BinResource#BIN_RESOURCE_TYPE_MEMORYのタイプになります.
     * @param size リソースサイズを設定します.
     * @return BinResource 生成されたリソース情報が返されます.<BR>
     *                     生成が失敗した場合[null]が返されます.
     */
    public static final BinResource createBinResource( int type,String tmpDir,int size )
    {
        return Resource.createBinResource( type,tmpDir,-1L,size ) ;
    }
    
    /**
     * タイプに対するバイナリリソースを生成.
     * <BR><BR>
     * タイプに対するバイナリリソースを生成します.
     * <BR>
     * @param type リソースタイプを設定します.<BR>
     *             [BinResource#BIN_RESOURCE_TYPE_MEMORY]を設定した場合、
     *             [com.JRcServer.commons.resource.BinMemoryResource]オブジェクトです.<BR>
     *             [BinResource#BIN_RESOURCE_TYPE_FILE]を設定した場合、
     *             [com.JRcServer.commons.resource.BinFileResource]オブジェクトです.<BR>
     *             [BinResource#BIN_RESOURCE_TYPE_CACHE]を設定した場合、
     *             [com.JRcServer.commons.resource.BinCacheResource]オブジェクトです.
     * @param tmpDir リソースタイプが BinResource#BIN_RESOURCE_TYPE_FILE の場合の
     *               一時ディレクトリ名を設定します.<BR>
     *               [null]を設定した場合、BinResource#BIN_RESOURCE_TYPE_MEMORYのタイプになります.
     * @param cacheID 対象のキャッシュIDを設定します.<BR>
     *                 [-1]を設定した場合、BinResource#BIN_RESOURCE_TYPE_MEMORYのタイプになります.
     * @param size リソースサイズを設定します.
     * @return BinResource 生成されたリソース情報が返されます.<BR>
     *                     生成が失敗した場合[null]が返されます.
     */
    public static final BinResource createBinResource( int type,String tmpDir,long cacheID,int size )
    {
        BinResource ret = null ;
        
        size = ( size <= 0 ) ? 1 : size ;
        
        try{
            
            switch( type ){
                case BinResource.BIN_RESOURCE_TYPE_MEMORY :
                    ret = new BinMemoryResource( size ) ;
                    break ;
                case BinResource.BIN_RESOURCE_TYPE_FILE :
                    if( tmpDir != null ){
                        ret = new BinFileResource( tmpDir,size ) ;
                    }
                    else{
                        ret = new BinMemoryResource( size ) ;
                    }
                    break ;
                case BinResource.BIN_RESOURCE_TYPE_CACHE :
                    if( cacheID >= 0 && CacheManager.get( cacheID ) != null ){
                        ret = new BinCacheResource( size,cacheID ) ;
                    }
                    else{
                        ret = new BinMemoryResource( size ) ;
                    }
                    break ;
                default :
                    ret = new BinMemoryResource( size ) ;
                    break ;
            }
            
            ret.reset() ;
            
        }catch( Exception e ){
            try{
                if( size <= 0 ){
                    ret = new BinMemoryResource( 1 ) ;
                }
                else{
                    ret = new BinMemoryResource( size ) ;
                }
            }catch( Exception ee ){
                ret = null ;
            }
        }
        
        return ret ;
        
    }
    
    /**
     * タイプに対するバイナリリソースを生成.
     * <BR><BR>
     * タイプに対するバイナリリソースを生成します.
     * <BR>
     * @param rs バイナリリソースステータスを設定します.
     * @param bin 設定対象のバイナリ情報を設定します.
     * @return BinResource 生成されたリソース情報が返されます.<BR>
     *                     生成が失敗した場合[null]が返されます.
     */
    public static final BinResource createBinResource( ResourceType rs,byte[] bin )
    {
        BinResource ret = null ;
        
        try{    
                
            if( rs == null ){
                if( bin == null || bin.length <= 0 ){
                    ret = new BinMemoryResource( 1 ) ;
                }
                else{
                    ret = new BinMemoryResource( bin ) ;
                }
            }
            else{
                ret = Resource.createBinResource( rs.getType(),rs.getTmpDir(),rs.getCacheID(),bin ) ;
            }
            
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * タイプに対するバイナリリソースを生成.
     * <BR><BR>
     * タイプに対するバイナリリソースを生成します.
     * <BR>
     * @param type リソースタイプを設定します.<BR>
     *             [BinResource#BIN_RESOURCE_TYPE_MEMORY]を設定した場合、
     *             [com.JRcServer.commons.resource.BinMemoryResource]オブジェクトです.<BR>
     *             [BinResource#BIN_RESOURCE_TYPE_FILE]を設定した場合、
     *             [com.JRcServer.commons.resource.BinFileResource]オブジェクトです.<BR>
     *             [BinResource#BIN_RESOURCE_TYPE_CACHE]を設定した場合、
     *             [com.JRcServer.commons.resource.BinCacheResource]オブジェクトです.
     * @param tmpDir リソースタイプが BinResource#BIN_RESOURCE_TYPE_FILE の場合の
     *               一時ディレクトリ名を設定します.<BR>
     *               [null]を設定した場合、BinResource#BIN_RESOURCE_TYPE_MEMORYのタイプになります.
     * @param bin 設定対象のバイナリ情報を設定します.
     * @return BinResource 生成されたリソース情報が返されます.<BR>
     *                     生成が失敗した場合[null]が返されます.
     */
    public static final BinResource createBinResource( int type,String tmpDir,byte[] bin )
    {
        return Resource.createBinResource( type,tmpDir,-1L,bin ) ;
    }
    
    /**
     * タイプに対するバイナリリソースを生成.
     * <BR><BR>
     * タイプに対するバイナリリソースを生成します.
     * <BR>
     * @param type リソースタイプを設定します.<BR>
     *             [BinResource#BIN_RESOURCE_TYPE_MEMORY]を設定した場合、
     *             [com.JRcServer.commons.resource.BinMemoryResource]オブジェクトです.<BR>
     *             [BinResource#BIN_RESOURCE_TYPE_FILE]を設定した場合、
     *             [com.JRcServer.commons.resource.BinFileResource]オブジェクトです.<BR>
     *             [BinResource#BIN_RESOURCE_TYPE_CACHE]を設定した場合、
     *             [com.JRcServer.commons.resource.BinCacheResource]オブジェクトです.
     * @param tmpDir リソースタイプが BinResource#BIN_RESOURCE_TYPE_FILE の場合の
     *               一時ディレクトリ名を設定します.<BR>
     *               [null]を設定した場合、BinResource#BIN_RESOURCE_TYPE_MEMORYのタイプになります.
     * @param cacheID 対象のキャッシュIDを設定します.<BR>
     *                 [-1]を設定した場合、BinResource#BIN_RESOURCE_TYPE_MEMORYのタイプになります.
     * @param bin 設定対象のバイナリ情報を設定します.
     * @return BinResource 生成されたリソース情報が返されます.<BR>
     *                     生成が失敗した場合[null]が返されます.
     */
    public static final BinResource createBinResource( int type,String tmpDir,long cacheID,byte[] bin )
    {
        BinResource ret = null ;
        
        try{
            
            switch( type ){
                case BinResource.BIN_RESOURCE_TYPE_MEMORY :
                    ret = ( bin == null || bin.length <= 0 ) ?
                        new BinMemoryResource( 1 ) :
                        new BinMemoryResource( bin ) ;
                    break ;
                case BinResource.BIN_RESOURCE_TYPE_FILE :
                    if( tmpDir != null ){
                        if( bin != null && bin.length > 0 ){
                            ret = new BinFileResource( tmpDir,bin.length ) ;
                            ret.setBinary( 0,bin ) ;
                        }
                        else{
                            ret = new BinFileResource( tmpDir,1 ) ;
                        }
                    }
                    else{
                        ret = ( bin == null || bin.length <= 0 ) ?
                                new BinMemoryResource( 1 ) :
                                new BinMemoryResource( bin ) ;
                    }
                    break ;
                case BinResource.BIN_RESOURCE_TYPE_CACHE :
                    if( cacheID >= 0 && CacheManager.get( cacheID ) != null ){
                        if( bin != null && bin.length > 0 ){
                            ret = new BinCacheResource( bin.length,cacheID ) ;
                            ret.setBinary( 0,bin ) ;
                        }
                        else{
                            ret = new BinCacheResource( 1,cacheID ) ;
                        }
                    }
                    else{
                        ret = ( bin == null || bin.length <= 0 ) ?
                            new BinMemoryResource( 1 ) :
                            new BinMemoryResource( bin ) ;
                        break ;
                    }
                    break ;
                default :
                    ret = ( bin == null || bin.length <= 0 ) ?
                        new BinMemoryResource( 1 ) :
                        new BinMemoryResource( bin ) ;
                    break ;
            }
            
        }catch( Exception e ){
            try{
                if( bin == null || bin.length <= 0 ){
                    ret = new BinMemoryResource( 1 ) ;
                }
                else{
                    ret = new BinMemoryResource( bin ) ;
                }
            }catch( Exception ee ){
                ret = null ;
            }
        }
        
        return ret ;
        
    }
    
    /**
     * バイナリ情報をコピー.
     * <BR><BR>
     * バイナリ情報をコピーします.
     * <BR>
     * @param src コピー元のバイナリオブジェクト情報を設定します.
     * @param src_pnt コピー元の位置を設定します.
     * @param dst コピー先のバイナリオブジェクト情報を設定します.
     * @param dst_pnt コピー先の位置を設定します.
     * @param len コピーするバイナリ長を設定します.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public static final void arraycopy( BinResource src,int src_pnt,BinResource dst,int dst_pnt,int len )
        throws ArrayIndexOutOfBoundsException
    {
        try{
            dst.setBinary( dst_pnt,src,src_pnt,len ) ;
        }catch( ArrayIndexOutOfBoundsException ai ){
            throw ai ;
        }catch( Exception e ){
            throw new ArrayIndexOutOfBoundsException( e.getMessage() ) ;
        }
    }
    
    /**
     * バイナリ情報をコピー.
     * <BR><BR>
     * バイナリ情報をコピーします.
     * <BR>
     * @param src コピー元のバイナリ情報を設定します.
     * @param src_pnt コピー元の位置を設定します.
     * @param dst コピー先のバイナリオブジェクト情報を設定します.
     * @param dst_pnt コピー先の位置を設定します.
     * @param len コピーするバイナリ長を設定します.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public static final void arraycopy( byte[] src,int src_pnt,BinResource dst,int dst_pnt,int len )
        throws ArrayIndexOutOfBoundsException
    {
        try{
            dst.setBinary( dst_pnt,src,src_pnt,len ) ;
        }catch( ArrayIndexOutOfBoundsException ai ){
            throw ai ;
        }catch( Exception e ){
            throw new ArrayIndexOutOfBoundsException( e.getMessage() ) ;
        }
    }
    
    /**
     * バイナリ情報をコピー.
     * <BR><BR>
     * バイナリ情報をコピーします.
     * <BR>
     * @param src コピー元のバイナリオブジェクト情報を設定します.
     * @param src_pnt コピー元の位置を設定します.
     * @param dst コピー先のバイナリ情報を設定します.
     * @param dst_pnt コピー先の位置を設定します.
     * @param len コピーするバイナリ長を設定します.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public static final void arraycopy( BinResource src,int src_pnt,byte[] dst,int dst_pnt,int len )
        throws ArrayIndexOutOfBoundsException
    {
        BinMemoryResource b = null ;
        
        try{
            
            b = new BinMemoryResource( dst ) ;
            b.setBinary( dst_pnt,src,src_pnt,len ) ;
            
        }catch( ArrayIndexOutOfBoundsException ai ){
            throw ai ;
        }catch( Exception e ){
            throw new ArrayIndexOutOfBoundsException( e.getMessage() ) ;
        }finally{
            if( b != null ){
                b.clear() ;
            }
            b = null ;
        }
    }
    
    /**
     * バイナリリソースを文字列に変換.
     * <BR><BR>
     * バイナリリソースを文字列に変換します.
     * <BR>
     * @param bin 対象のバイナリリソースを設定します.
     * @return String 変換結果が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportedEncodingException 変換失敗例外.
     */
    public static final String convertBinResourceByString( BinResource bin )
        throws InputException,UnsupportedEncodingException
    {
        return Resource.convertBinResourceByString( bin,0,-1,null ) ;
    }
    
    /**
     * バイナリリソースを文字列に変換.
     * <BR><BR>
     * バイナリリソースを文字列に変換します.
     * <BR>
     * @param bin 対象のバイナリリソースを設定します.
     * @param charset 対象のキャラクターセットを設定します.
     * @return String 変換結果が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportedEncodingException 変換失敗例外.
     */
    public static final String convertBinResourceByString( BinResource bin,String charset )
        throws InputException,UnsupportedEncodingException
    {
        return Resource.convertBinResourceByString( bin,0,-1,charset ) ;
    }
    
    /**
     * バイナリリソースを文字列に変換.
     * <BR><BR>
     * バイナリリソースを文字列に変換します.
     * <BR>
     * @param bin 対象のバイナリリソースを設定します.
     * @param offset 対象のバイナリオフセット値を設定します.
     * @param length 対象のバイナリデータ長を設定します.
     * @return String 変換結果が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportedEncodingException 変換失敗例外.
     */
    public static final String convertBinResourceByString( BinResource bin,int offset,int length )
        throws InputException,UnsupportedEncodingException
    {
        return Resource.convertBinResourceByString( bin,offset,length,null ) ;
    }
    
    /**
     * バイナリリソースを文字列に変換.
     * <BR><BR>
     * バイナリリソースを文字列に変換します.
     * <BR>
     * @param bin 対象のバイナリリソースを設定します.
     * @param offset 対象のバイナリオフセット値を設定します.
     * @param length 対象のバイナリデータ長を設定します.
     * @param charset 対象のキャラクターセットを設定します.
     * @return String 変換結果が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportedEncodingException 変換失敗例外.
     */
    public static final String convertBinResourceByString( BinResource bin,int offset,int length,String charset )
        throws InputException,UnsupportedEncodingException
    {
        int len ;
        int blen ;
        
        char[] tmp = null ;
        BufferedReader br = null ;
        StringBuffer buf = null ;
        String ret = null ;
        
        if(
             bin == null || bin.isUse() == false || ( blen = bin.size() ) <= 0 ||
            offset < 0 || ( length != -1 && length < 0 )
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        
        charset = ( charset == null || charset.length() <= 0 ) ? null : charset ;
        length = ( length == -1 ) ? blen - offset : length ;
        
        try{
            
            if( charset == null ){
                
                br = new BufferedReader(
                    new InputStreamReader(
                        new BinResourceInputStream(
                            false,bin,offset,length
                        )
                    )
                ) ;
                
            }
            else{
                
                br = new BufferedReader(
                    new InputStreamReader(
                        new BinResourceInputStream(
                            false,bin,offset,length
                        ),charset
                    )
                ) ;
                
            }
            
            buf = new StringBuffer() ;
            
            tmp = new char[ Resource.BUF_LEN ] ;
            while( ( len = br.read( tmp,0,Resource.BUF_LEN ) ) >= 0 ){
                buf.append( tmp,0,len ) ;
            }
            
            ret = buf.toString() ;
            
        }catch( InputException in ){
            throw in ;
        }catch( UnsupportedEncodingException ue ){
            throw ue ;
        }catch( Exception e ){
            ret = null ;
        }finally{
            
            try{
                br.close() ;
            }catch( Exception ee ){
            }
            
            buf = null ;
            tmp = null ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * 文字列をバイナリリソースに変換.
     * <BR><BR>
     * 文字列をバイナリリソースに変換します.
     * <BR>
     * @param out 戻り値となるバイナリリソースを設定します.
     * @param str 対象の文字列を設定します.
     * @return int 変換データ長が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportedEncodingException 変換失敗例外.
     */
    public static final int convertStringByBinResource( BinResource out,String str )
        throws InputException,UnsupportedEncodingException
    {
        return Resource.convertStringByBinResource( out,0,str,0,-1,null ) ;
    }
    
    /**
     * 文字列をバイナリリソースに変換.
     * <BR><BR>
     * 文字列をバイナリリソースに変換します.
     * <BR>
     * @param out 戻り値となるバイナリリソースを設定します.
     * @param str 対象の文字列を設定します.
     * @param charset 対象のキャラクターセットを設定します.
     * @return int 変換データ長が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportedEncodingException 変換失敗例外.
     */
    public static final int convertStringByBinResource( BinResource out,String str,String charset )
        throws InputException,UnsupportedEncodingException
    {
        return Resource.convertStringByBinResource( out,0,str,0,-1,charset ) ;
    }
    
    /**
     * 文字列をバイナリリソースに変換.
     * <BR><BR>
     * 文字列をバイナリリソースに変換します.
     * <BR>
     * @param out 戻り値となるバイナリリソースを設定します.
     * @param str 対象の文字列を設定します.
     * @param offset 対象の文字列オフセット値を設定します.
     * @param length 対象の文字列データ長を設定します.
     * @return int 変換データ長が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportedEncodingException 変換失敗例外.
     */
    public static final int convertStringByBinResource( BinResource out,String str,int offset,int length )
        throws InputException,UnsupportedEncodingException
    {
        return Resource.convertStringByBinResource( out,0,str,offset,length,null ) ;
    }
    
    /**
     * 文字列をバイナリリソースに変換.
     * <BR><BR>
     * 文字列をバイナリリソースに変換します.
     * <BR>
     * @param out 戻り値となるバイナリリソースを設定します.
     * @param str 対象の文字列を設定します.
     * @param offset 対象の文字列オフセット値を設定します.
     * @param length 対象の文字列データ長を設定します.
     * @param charset 対象のキャラクターセットを設定します.
     * @return int 変換データ長が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportedEncodingException 変換失敗例外.
     */
    public static final int convertStringByBinResource( BinResource out,String str,int offset,int length,String charset )
        throws InputException,UnsupportedEncodingException
    {
        return Resource.convertStringByBinResource( out,0,str,offset,length,null ) ;
    }
    
    /**
     * 文字列をバイナリリソースに変換.
     * <BR><BR>
     * 文字列をバイナリリソースに変換します.
     * <BR>
     * @param out 戻り値となるバイナリリソースを設定します.
     * @param no バイナリリソース書き込み開始位置を設定します.
     * @param str 対象の文字列を設定します.
     * @return int 変換データ長が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportedEncodingException 変換失敗例外.
     */
    public static final int convertStringByBinResource( BinResource out,int no,String str )
        throws InputException,UnsupportedEncodingException
    {
        return Resource.convertStringByBinResource( out,no,str,0,-1,null ) ;
    }
    
    /**
     * 文字列をバイナリリソースに変換.
     * <BR><BR>
     * 文字列をバイナリリソースに変換します.
     * <BR>
     * @param out 戻り値となるバイナリリソースを設定します.
     * @param no バイナリリソース書き込み開始位置を設定します.
     * @param str 対象の文字列を設定します.
     * @param charset 対象のキャラクターセットを設定します.
     * @return int 変換データ長が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportedEncodingException 変換失敗例外.
     */
    public static final int convertStringByBinResource( BinResource out,int no,String str,String charset )
        throws InputException,UnsupportedEncodingException
    {
        return Resource.convertStringByBinResource( out,no,str,0,-1,charset ) ;
    }
    
    /**
     * 文字列をバイナリリソースに変換.
     * <BR><BR>
     * 文字列をバイナリリソースに変換します.
     * <BR>
     * @param out 戻り値となるバイナリリソースを設定します.
     * @param no バイナリリソース書き込み開始位置を設定します.
     * @param str 対象の文字列を設定します.
     * @param offset 対象の文字列オフセット値を設定します.
     * @param length 対象の文字列データ長を設定します.
     * @return int 変換データ長が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportedEncodingException 変換失敗例外.
     */
    public static final int convertStringByBinResource( BinResource out,int no,String str,int offset,int length )
        throws InputException,UnsupportedEncodingException
    {
        return Resource.convertStringByBinResource( out,no,str,offset,length,null ) ;
    }
    
    /**
     * 文字列をバイナリリソースに変換.
     * <BR><BR>
     * 文字列をバイナリリソースに変換します.
     * <BR>
     * @param out 戻り値となるバイナリリソースを設定します.
     * @param no バイナリリソース書き込み開始位置を設定します.
     * @param str 対象の文字列を設定します.
     * @param offset 対象の文字列オフセット値を設定します.
     * @param length 対象の文字列データ長を設定します.
     * @param charset 対象のキャラクターセットを設定します.
     * @return int 変換データ長が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportedEncodingException 変換失敗例外.
     */
    public static final int convertStringByBinResource( BinResource out,int no,String str,int offset,int length,String charset )
        throws InputException,UnsupportedEncodingException
    {
        int len ;
        int slen ;
        int ret ;
        
        BufferedWriter bw = null ;
        
        if(
            out == null || out.isUse() == false || str == null || ( slen = str.length() ) <= 0 ||
            offset < 0 || ( length != -1 && length < 0 )
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        no = ( no <= 0 ) ? 0 : no ;
        charset = ( charset == null || charset.length() <= 0 ) ? null : charset ;
        length = ( length == -1 ) ? slen - offset : length ;
        
        try{
            
            if( charset == null ){
                
                bw = new BufferedWriter(
                    new OutputStreamWriter(
                        new BinResourceOutputStream(
                            false,out,no
                        )
                    )
                ) ;
                
            }
            else{
                
                bw = new BufferedWriter(
                    new OutputStreamWriter(
                        new BinResourceOutputStream(
                            false,out,no
                        ),charset
                    )
                ) ;
                
            }
            
            bw.write( str,offset,length ) ;
            ret = out.size() ;
            
        }catch( InputException in ){
            if( out != null ){
                out.clear() ;
            }
            throw in ;
        }catch( UnsupportedEncodingException ue ){
            if( out != null ){
                out.clear() ;
            }
            throw ue ;
        }catch( Exception e ){
            if( out != null ){
                out.clear() ;
            }
            ret = -1 ;
        }finally{
            
            try{
                bw.close() ;
            }catch( Exception ee ){
            }
            
            bw = null ;
            
        }
        
        return ( ret <= 0 ) ? -1 : ret ;
        
    }
    
    /**
     * バイナリリソースをキャラクターリソースに変換.
     * <BR><BR>
     * バイナリリソースをキャラクターリソースに変換します.
     * <BR>
     * @param out 戻り値となるキャラクターリソースを設定します.
     * @param bin 対象のバイナリリソースを設定します.
     * @return int 変換データ長が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportedEncodingException 変換失敗例外.
     */
    public static final int convertBinResourceByCharResource( CharResource out,BinResource bin )
        throws InputException,UnsupportedEncodingException
    {
        return Resource.convertBinResourceByCharResource( out,bin,0,-1,null ) ;
    }
    
    /**
     * バイナリリソースをキャラクターリソースに変換.
     * <BR><BR>
     * バイナリリソースをキャラクターリソースに変換します.
     * <BR>
     * @param out 戻り値となるキャラクターリソースを設定します.
     * @param bin 対象のバイナリリソースを設定します.
     * @param charset 対象のキャラクターセットを設定します.
     * @return int 変換データ長が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportedEncodingException 変換失敗例外.
     */
    public static final int convertBinResourceByCharResource( CharResource out,BinResource bin,String charset )
        throws InputException,UnsupportedEncodingException
    {
        return Resource.convertBinResourceByCharResource( out,bin,0,-1,charset ) ;
    }
    
    /**
     * バイナリリソースをキャラクターリソースに変換.
     * <BR><BR>
     * バイナリリソースをキャラクターリソースに変換します.
     * <BR>
     * @param out 戻り値となるキャラクターリソースを設定します.
     * @param bin 対象のバイナリリソースを設定します.
     * @param offset 対象のバイナリオフセット値を設定します.
     * @param length 対象のバイナリデータ長を設定します.
     * @return int 変換データ長が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportedEncodingException 変換失敗例外.
     */
    public static final int convertBinResourceByCharResource( CharResource out,BinResource bin,int offset,int length )
        throws InputException,UnsupportedEncodingException
    {
        return Resource.convertBinResourceByCharResource( out,bin,offset,length,null ) ;
    }
    
    /**
     * バイナリリソースをキャラクターリソースに変換.
     * <BR><BR>
     * バイナリリソースをキャラクターリソースに変換します.
     * <BR>
     * @param out 戻り値となるキャラクターリソースを設定します.
     * @param bin 対象のバイナリリソースを設定します.
     * @param offset 対象のバイナリオフセット値を設定します.
     * @param length 対象のバイナリデータ長を設定します.
     * @param charset 対象のキャラクターセットを設定します.
     * @return int 変換データ長が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportedEncodingException 変換失敗例外.
     */
    public static final int convertBinResourceByCharResource( CharResource out,BinResource bin,int offset,int length,String charset )
        throws InputException,UnsupportedEncodingException
    {
        int len ;
        int blen ;
        int pnt ;
        int ret ;
        
        char[] tmp = null ;
        BufferedReader br = null ;
        
        if(
            out == null || out.isUse() == false || bin == null || bin.isUse() == false || ( blen = bin.size() ) <= 0 ||
            offset < 0 || ( length != -1 && length < 0 )
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        
        out.reset() ;
        charset = ( charset == null || charset.length() <= 0 ) ? null : charset ;
        length = ( length == -1 ) ? blen - offset : length ;
        
        try{
            
            if( charset == null ){
                
                br = new BufferedReader(
                    new InputStreamReader(
                        new BinResourceInputStream(
                            false,bin,offset,length
                        )
                    )
                ) ;
                
            }
            else{
                
                br = new BufferedReader(
                    new InputStreamReader(
                        new BinResourceInputStream(
                            false,bin,offset,length
                        ),charset
                    )
                ) ;
                
            }
            
            tmp = new char[ Resource.BUF_LEN ] ;
            pnt = 0 ;
            while( ( len = br.read( tmp,0,Resource.BUF_LEN ) ) >= 0 ){
                out.setChars( pnt,tmp,0,len ) ;
                pnt += len ;
            }
            
            ret = out.size() ;
            
        }catch( InputException in ){
            if( out != null ){
                out.clear() ;
            }
            throw in ;
        }catch( UnsupportedEncodingException ue ){
            if( out != null ){
                out.clear() ;
            }
            throw ue ;
        }catch( Exception e ){
            if( out != null ){
                out.clear() ;
            }
            ret = -1 ;
        }finally{
            
            try{
                br.close() ;
            }catch( Exception ee ){
            }
            
            tmp = null ;
            
        }
        
        return ( ret <= 0 ) ? -1 : ret ;
        
    }
    
    /**
     * キャラクターリソースをバイナリリソースに変換.
     * <BR><BR>
     * キャラクターリソースをバイナリリソースに変換します.
     * <BR>
     * @param out 戻り値となるバイナリリソースを設定します.
     * @param chr 対象のキャラクターリソースを設定します.
     * @return int 変換データ長が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportedEncodingException 変換失敗例外.
     */
    public static final int convertCharResourceByBinResource( BinResource out,CharResource chr )
        throws InputException,UnsupportedEncodingException
    {
        return Resource.convertCharResourceByBinResource( out,chr,0,-1,null ) ;
    }
    
    /**
     * キャラクターリソースをバイナリリソースに変換.
     * <BR><BR>
     * キャラクターリソースをバイナリリソースに変換します.
     * <BR>
     * @param out 戻り値となるバイナリリソースを設定します.
     * @param chr 対象のキャラクターリソースを設定します.
     * @param charset 対象のキャラクターセットを設定します.
     * @return int 変換データ長が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportedEncodingException 変換失敗例外.
     */
    public static final int convertCharResourceByBinResource( BinResource out,CharResource chr,String charset )
        throws InputException,UnsupportedEncodingException
    {
        return Resource.convertCharResourceByBinResource( out,chr,0,-1,charset ) ;
    }
    
    /**
     * キャラクターリソースをバイナリリソースに変換.
     * <BR><BR>
     * キャラクターリソースをバイナリリソースに変換します.
     * <BR>
     * @param out 戻り値となるバイナリリソースを設定します.
     * @param chr 対象のキャラクターリソースを設定します.
     * @param offset 対象のキャラクターリソースオフセット値を設定します.
     * @param length 対象のキャラクターリソースデータ長を設定します.
     * @return int 変換データ長が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportedEncodingException 変換失敗例外.
     */
    public static final int convertCharResourceByBinResource( BinResource out,CharResource chr,int offset,int length )
        throws InputException,UnsupportedEncodingException
    {
        return Resource.convertCharResourceByBinResource( out,chr,offset,length,null ) ;
    }
    
    /**
     * キャラクターリソースをバイナリリソースに変換.
     * <BR><BR>
     * キャラクターリソースをバイナリリソースに変換します.
     * <BR>
     * @param out 戻り値となるバイナリリソースを設定します.
     * @param chr 対象のキャラクターリソースを設定します.
     * @param offset 対象のキャラクターリソースオフセット値を設定します.
     * @param length 対象のキャラクターリソースデータ長を設定します.
     * @param charset 対象のキャラクターセットを設定します.
     * @return int 変換データ長が返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportedEncodingException 変換失敗例外.
     */
    public static final int convertCharResourceByBinResource( BinResource out,CharResource chr,int offset,int length,String charset )
        throws InputException,UnsupportedEncodingException
    {
        int len ;
        int slen ;
        int pnt ;
        int readLen ;
        int endLen ;
        int ret ;
        
        char[] tmp = null ;
        BufferedWriter bw = null ;
        
        if(
            out == null || out.isUse() == false || chr == null || chr.isUse() == false || ( slen = chr.size() ) <= 0 ||
            offset < 0 || ( length != -1 && length < 0 )
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        out.reset() ;
        charset = ( charset == null || charset.length() <= 0 ) ? null : charset ;
        length = ( length == -1 ) ? slen - offset : length ;
        
        try{
            
            if( charset == null ){
                
                bw = new BufferedWriter(
                    new OutputStreamWriter(
                        new BinResourceOutputStream(
                            false,out
                        )
                    )
                ) ;
                
            }
            else{
                
                bw = new BufferedWriter(
                    new OutputStreamWriter(
                        new BinResourceOutputStream(
                            false,out
                        ),charset
                    )
                ) ;
                
            }
            
            tmp = new char[ Resource.BUF_LEN ] ;
            
            for( pnt = offset,endLen = 0 ;; ){
                
                if( length <= 0 ){
                    break ;
                }
                else if( ( readLen = chr.getChars( tmp,pnt,0,Resource.BUF_LEN ) ) <= 0 ){
                    break ;
                }
                
                
                bw.write( tmp,0,( readLen >= length ) ? length : readLen ) ;
                
                length -= readLen ;
                pnt += readLen ;
                
            }
            
            ret = out.size() ;
            
        }catch( InputException in ){
            if( out != null ){
                out.clear() ;
            }
            throw in ;
        }catch( UnsupportedEncodingException ue ){
            if( out != null ){
                out.clear() ;
            }
            throw ue ;
        }catch( Exception e ){
            if( out != null ){
                out.clear() ;
            }
            ret = -1 ;
        }finally{
            
            try{
                bw.close() ;
            }catch( Exception ee ){
            }
            
            bw = null ;
            tmp = null ;
            
        }
        
        return ( ret <= 0 ) ? -1 : ret ;
        
    }
    
    /**
     * ロールデータ再生成サイズを取得.
     * <BR><BR>
     * ロールデータ再生成サイズを取得します.<BR>
     * このメソッドはロールデータに追加する場合、前回の最大長が
     * 今回のデータ長とかけ離れている場合の再定義を行うきっかけとなる
     * 値を取得します.
     * @param length 対象データ長を設定します.
     * @return int 再生成サイズのきっかけ値が返されます.
     */
    public static final int getRenewRollValueCode( int length )
    {
        length = ( int )( length * ResourceDef.ROLL_RENEW_VALUE_CODE ) ;
        
        return ( int )(
            (
                ( ( length & (~CacheIO.ELEMENT_MASK) ) >> CacheIO.ELEMENT_BITCOUNT ) +
                ( ( ( length & CacheIO.ELEMENT_BITCOUNT ) != 0 ) ? 1 : 0 )
            ) * CacheIO.ELEMENT_LENGTH
        ) ;
    } 
}
