/*
 * @(#)CharResource.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.resource ;

/**
 * キャラクターリソース.
 * <BR><BR>
 * キャラクタリソース情報を扱うオブジェクトを提供します.
 *
 * @version     1.00, 2005/04/13
 * @author      Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class CharResource
{
    
    /**
     * 1データ変換サイズ.
     */
    private static final int ONE_CONVERT_SIZE = 512 ;
    
    /**
     * 管理キャラクタリソース.
     */
    private BinResource m_bin = null ;
    
    /**
     * コンストラクタ.
     */
    public CharResource(){}
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * リソースタイプを設定してオブジェクトを生成します.
     * <BR>
     * @param type 対象のリソースタイプを設定します.
     * @param size オブジェクト生成サイズを設定します.
     */
    public CharResource( ResourceType type,int size )
    {
        this.create( type,size ) ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * リソースタイプを設定してオブジェクトを生成します.
     * <BR>
     * @param type 対象のリソースタイプを設定します.
     * @param size オブジェクト生成サイズを設定します.
     */
    public final void create( ResourceType type,int size )
    {
        m_bin = Resource.createBinResource( type,size * 2 ) ;
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public final void clear()
    {
        
        if( m_bin != null ){
            m_bin.clear() ;
        }
        
        m_bin = null ;
        
    }
    
    /**
     * リセット処理.
     * <BR><BR>
     * 有効データ長をリセットします.
     */
    public final void reset()
    {
        if( m_bin != null ){
            m_bin.reset() ;
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 対象条件に情報を設定します.
     * <BR>
     * @param no 設定対象項番を設定します.
     * @param c 設定対象のキャラクター情報を設定します.
     */
    public final void set( int no,char c )
    {
        byte[] tmp = null ;
        
        if( m_bin == null || m_bin.isUse() == false ){
            return ;
        }
        tmp = new byte[ 2 ] ;
        tmp[ 0 ] = ( byte )( c & 0x000000ff ) ;
        tmp[ 1 ] = ( byte )( ( c & 0x0000ff00 ) >> 8 ) ;
        
        m_bin.setBinary( no*2,tmp ) ;
        tmp = null ;
        
    }
    
    /**
     * キャラクタ情報設定.
     * <BR><BR>
     * 対象のキャラクタ情報を設定します.
     * <BR>
     * @param no 設定開始位置となる項番を設定します.
     * @param chr 設定対象のキャラクタ情報を設定します.
     * @return int 設定されたキャラクタ長が返されます.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public final int setChars( int no,char[] chr )
        throws ArrayIndexOutOfBoundsException
    {
        int len ;
        
        if( chr == null ){
            throw new ArrayIndexOutOfBoundsException( "引数は不正です" ) ;
        }
        else if( ( len = chr.length ) < 0 ){
            return 0 ;
        }
        
        return this.setChars( no,chr,0,len ) ;
    }
    
    /**
     * キャラクタ情報設定.
     * <BR><BR>
     * 対象のキャラクタ情報を設定します.
     * <BR>
     * @param no 設定開始位置となる項番を設定します.
     * @param chr 設定対象のキャラクタ情報を設定します.
     * @param off 設定対象のオフセット値を設定します.
     * @param len 設定対象のキャラクタ長を設定します.
     * @return int 設定されたキャラクタ長が返されます.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public final int setChars( int no,char[] chr,int off,int len )
        throws ArrayIndexOutOfBoundsException
    {
        int i,j,k ;
        int lenI ;
        int etc ;
        int chrLen ;
        int pnt ;
        int pntAdd ;
        int noPnt ;
        int ret ;
        
        byte[] tmp = null ;
        BinResource bin = null ;
        
        if( m_bin == null || m_bin.isUse() == false ){
            return -1 ;
        }
        if( chr == null ){
            throw new ArrayIndexOutOfBoundsException( "引数は不正です" ) ;
        }
        else if( ( chrLen = chr.length ) < 0 ){
            return 0 ;
        }
        
        try{
            
            bin = m_bin ;
            len = ( ( off + len ) >= chrLen ) ? chrLen - off : len ;
            
            tmp = new byte[ CharResource.ONE_CONVERT_SIZE ] ;
            
            lenI = ( len*2 ) / CharResource.ONE_CONVERT_SIZE ;
            etc = ( len*2 ) % CharResource.ONE_CONVERT_SIZE ;
            
            noPnt = no * 2 ;
            
            pnt = off ;
            pntAdd = CharResource.ONE_CONVERT_SIZE / 2 ;
            
            for( i = 0,ret = 0 ; i < lenI ; i ++ ){
                
                for( j = 0,k = pnt ; j < CharResource.ONE_CONVERT_SIZE ; j += 2,k ++ ){
                    
                    tmp[ j ] = ( byte )( chr[ k ] & 0x000000ff ) ;
                    tmp[ j+1 ] = ( byte )( ( chr[ k ] & 0x0000ff00 ) >> 8 ) ;
                    
                }
                
                bin.setBinary( noPnt,tmp ) ;
                
                pnt += pntAdd ;
                ret += pntAdd ;
                noPnt += CharResource.ONE_CONVERT_SIZE ;
                
            }
            
            if( etc != 0 ){
                
                for( j = 0,k = pnt ; j < etc ; j += 2,k ++ ){
                    
                    tmp[ j ] = ( byte )( chr[ k ] & 0x000000ff ) ;
                    tmp[ j+1 ] = ( byte )( ( chr[ k ] & 0x0000ff00 ) >> 8 ) ;
                    
                }
                
                bin.setBinary( noPnt,tmp,0,etc ) ;
                ret += etc / 2 ;
                
            }
            
        }catch( Exception e ){
            ret = -1 ;
        }finally{
            tmp = null ;
            bin = null ;
        }
        
        return ret ;
    }
    
    /**
     * 情報取得.
     * <BR><BR>
     * 対象条件の情報を取得します.
     * <BR>
     * @param no 取得対象項番を設定します.
     * @return char 対象のキャラクタ情報が返されます.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public final char get( int no ) throws ArrayIndexOutOfBoundsException
    {
        char ret ;
        byte[] tmp = null ;
        
        if( m_bin == null || m_bin.isUse() == false ){
            return 0 ;
        }
        tmp = m_bin.getBinary( no*2,2 ) ;
        
        ret = ( char )(
            ( tmp[ 0 ] & 0x000000ff ) |
            ( ( tmp[ 1 ] & 0x000000ff ) << 8 )
        ) ;
        tmp = null ;
        
        return ret ;
    }
    
    /**
     * キャラクタ情報を取得.
     * <BR><BR>
     * 対象のキャラクタ情報を取得します.
     * <BR>
     * @param no 取得開始位置となる項番を設定します.
     * @return char[] 取得されたキャラクタ情報が返されます.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public final char[] getChars( int no )
        throws ArrayIndexOutOfBoundsException
    {
        if( no < 0 ){
            return null ;
        }
        
        return this.getChars( no,this.size() - no ) ;
    }
    
    /**
     * キャラクタ情報を取得.
     * <BR><BR>
     * 対象のキャラクタ情報を取得します.
     * <BR>
     * @param no 取得開始位置となる項番を設定します.
     * @param len 取得対象のキャラクタ長を設定します.
     * @return char[] 取得されたキャラクタ情報が返されます.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public final char[] getChars( int no,int len )
        throws ArrayIndexOutOfBoundsException
    {
        int i,j,k ;
        int lenI ;
        int etc ;
        int binLen ;
        int noPnt ;
        int pnt ;
        int pntAdd ;
        
        BinResource bin = null ;
        byte[] tmp = null ;
        char[] ret = null ;
        
        if( m_bin == null || m_bin.isUse() == false ){
            return null ;
        }
        if( no < 0 || len < 0 ){
            throw new ArrayIndexOutOfBoundsException( "引数は不正です" ) ;
        }
        
        try{
            
            noPnt = no * 2 ;
            binLen = len * 2 ;
            bin = m_bin ;
            
            if( bin.size() <= noPnt ){
                throw new ArrayIndexOutOfBoundsException(
                    "指定no(" + no + ")は範囲を超しています"
                ) ;
            }
            
            binLen = ( bin.size() <= noPnt + binLen ) ? bin.size() - noPnt : binLen ;
            len = binLen / 2 ;
            binLen = len * 2 ;
            
            lenI = binLen / CharResource.ONE_CONVERT_SIZE ;
            etc = binLen % CharResource.ONE_CONVERT_SIZE ;
            
            pntAdd = CharResource.ONE_CONVERT_SIZE / 2 ;
            
            ret = new char[ len ] ;
            tmp = new byte[ CharResource.ONE_CONVERT_SIZE ] ;
            
            for( i = 0,pnt = 0 ; i < lenI ; i ++ ){
                
                bin.getBinary( tmp,noPnt ) ;
                
                for( j = 0,k = pnt ; j < CharResource.ONE_CONVERT_SIZE ; j += 2,k ++ ){
                    
                    ret[ k ] = ( char )(
                        ( tmp[ j ] & 0x000000ff ) |
                        ( ( tmp[ j+1 ] & 0x000000ff ) << 8 )
                    ) ;
                    
                }
                
                pnt += pntAdd ;
                noPnt += CharResource.ONE_CONVERT_SIZE ;
                
            }
            
            if( etc != 0 ){
                
                bin.getBinary( tmp,noPnt,0,etc ) ;
                
                for( j = 0,k = pnt ; j < etc ; j += 2,k ++ ){
                    
                    ret[ k ] = ( char )(
                        ( tmp[ j ] & 0x000000ff ) |
                        ( ( tmp[ j+1 ] & 0x000000ff ) << 8 )
                    ) ;
                    
                }
                
            }
            
        }catch( ArrayIndexOutOfBoundsException ai ){
            throw ai ;
        }catch( Exception e ){
            ret = null ;
        }finally{
            bin = null ;
            tmp = null ;
        }
        
        return ret ;
    }
    
    /**
     * キャラクタ情報を取得.
     * <BR><BR>
     * 格納されているキャラクタ情報を取得します.
     * <BR>
     * @param out 取得対象のキャラクタ情報が返されます.
     * @return int 取得されたキャラクタ長が返されます.
     */
    public final int getChars( char[] out )
    {
        int len ;
        
        if( out == null ){
            return -1 ;
        }
        else if( ( len = out.length ) <= 0 ){
            return 0 ;
        }
        
        return this.getChars( out,0,0,len ) ;
    }
    
    /**
     * キャラクタ情報を取得.
     * <BR><BR>
     * 対象のキャラクタ情報を取得します.
     * <BR>
     * @param out 取得対象のキャラクタ情報が返されます.
     * @param no 取得開始位置となる項番を設定します.
     * @return int 取得されたキャラクタ長が返されます.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public final int getChars( char[] out,int no )
        throws ArrayIndexOutOfBoundsException
    {
        int len ;
        
        if( no < 0 || out == null ){
            return -1 ;
        }
        else if( ( len = out.length ) <= 0 ){
            return 0 ;
        }
        
        return this.getChars( out,no,0,len ) ;
    }
    
    /**
     * キャラクタ情報を取得.
     * <BR><BR>
     * 対象のキャラクタ情報を取得します.
     * <BR>
     * @param out 取得対象のキャラクタ情報が返されます.
     * @param no 取得開始位置となる項番を設定します.
     * @param off 取得対象のキャラクタオフセット値を設定します.
     * @param len 取得対象のキャラクタ長を設定します.
     * @return int 取得されたキャラクタ長が返されます.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public final int getChars( char[] out,int no,int off,int len )
        throws ArrayIndexOutOfBoundsException
    {
        int i,j,k ;
        int lenI ;
        int etc ;
        int chrLen ;
        int binLen ;
        int noPnt ;
        int pnt ;
        int pntAdd ;
        int ret ;
        
        BinResource bin = null ;
        byte[] tmp = null ;
        
        if( m_bin == null || m_bin.isUse() == false ){
            return -1 ;
        }
        if( out == null || ( chrLen = out.length ) < 0 || no < 0 || off < 0 || len < 0 ){
            throw new ArrayIndexOutOfBoundsException( "引数は不正です" ) ;
        }
        
        try{
            
            len = ( ( off + len ) >= chrLen ) ? chrLen - off : len ;
            
            noPnt = no * 2 ;
            binLen = len * 2 ;
            bin = m_bin ;
            
            if( bin.size() <= noPnt ){
                throw new ArrayIndexOutOfBoundsException(
                    "指定no(" + no + ")は範囲を超しています"
                ) ;
            }
            
            binLen = ( bin.size() <= noPnt + binLen ) ? bin.size() - noPnt : binLen ;
            len = binLen / 2 ;
            binLen = len * 2 ;
            
            lenI = binLen / CharResource.ONE_CONVERT_SIZE ;
            etc = binLen % CharResource.ONE_CONVERT_SIZE ;
            
            pnt = off ;
            pntAdd = CharResource.ONE_CONVERT_SIZE / 2 ;
            
            tmp = new byte[ CharResource.ONE_CONVERT_SIZE ] ;
            
            for( i = 0,ret = 0 ; i < lenI ; i ++ ){
                
                bin.getBinary( tmp,noPnt ) ;
                
                for( j = 0,k = pnt ; j < CharResource.ONE_CONVERT_SIZE ; j += 2,k ++ ){
                    
                    out[ k ] = ( char )(
                        ( tmp[ j ] & 0x000000ff ) |
                        ( ( tmp[ j+1 ] & 0x000000ff ) << 8 )
                    ) ;
                    
                }
                
                pnt += pntAdd ;
                ret += pntAdd ;
                noPnt += CharResource.ONE_CONVERT_SIZE ;
                
            }
            
            if( etc != 0 ){
                
                bin.getBinary( tmp,noPnt,0,etc ) ;
                
                for( j = 0,k = pnt ; j < etc ; j += 2,k ++ ){
                    
                    out[ k ] = ( char )(
                        ( tmp[ j ] & 0x000000ff ) |
                        ( ( tmp[ j+1 ] & 0x000000ff ) << 8 )
                    ) ;
                    
                }
                
                ret += etc / 2 ;
            }
            
        }catch( ArrayIndexOutOfBoundsException ai ){
            throw ai ;
        }catch( Exception e ){
            ret = -1 ;
        }finally{
            bin = null ;
            tmp = null ;
        }
        
        return ret ;
    }
    
    /**
     * バイナリリソースを取得.
     * <BR><BR>
     * 利用中のバイナリリソースが返されます.
     * <BR>
     * @return BinResource バイナリリソースが返されます.
     */
    public final BinResource getBinResource()
    {
        return m_bin ;
    }
    
    /**
     * 現在の有効キャラクタ長を取得.
     * <BR><BR>
     * 現在の有効なキャラクタ長を取得します.
     * <BR>
     * @return int 現在の有効なキャラクタ長が返されます.
     */
    public final int size()
    {
        int ret ;
        
        try{
            ret = m_bin.size() / 2 ;
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 現在のキャラクタ長を取得.
     * <BR><BR>
     * 現在のキャラクタ長を取得します.
     * <BR>
     * @return int 現在のキャラクタ長が返されます.
     */
    public final int getAllSize()
    {
        int ret ;
        
        try{
            ret = m_bin.getAllSize() / 2 ;
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * オブジェクトタイプを取得.
     * <BR><BR>
     * オブジェクトタイプを取得します.
     * <BR>
     * @return int オブジェクトタイプが返されます.<BR>
     *             [ResourceType#RESOURCE_TYPE_MEMORY]が返された場合、
     *             メモリータイプリソースです.<BR>
     *             [ResourceType#RESOURCE_TYPE_FILE]が返された場合、
     *             ファイルタイプリソースです.<BR>
     *             [ResourceType#RESOURCE_TYPE_CACHE]が返された場合、
     *             CACHEファイルタイプリソースです.<BR>
     *             その他の値の場合オブジェクトは無効です.
     */
    public final int getType()
    {
        int ret ;
        
        try{
            ret = m_bin.getType() ;
        }catch( Exception e ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 対象情報を文字列に変換.
     * <BR><BR>
     * 対象情報を文字列に変換します.
     * <BR>
     * @return String 変換された文字列が返されます.
     */
    public final String toString()
    {
        char[] tmp = null ;
        String ret = null ;
        
        if( m_bin == null || m_bin.isUse() == false ){
            return "" ;
        }
        try{
            tmp = this.getChars( 0 ) ;
            ret = new String( tmp ) ;
            tmp = null ;
        }catch( Exception e ){
            ret = "" ;
        }finally{
            tmp = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象情報を文字列に変換.
     * <BR><BR>
     * 対象情報を文字列に変換します.
     * <BR>
     * @param offset 対象のオフセット値を設定します.
     * @param length 対象の文字列長を設定します.
     * @return String 変換された文字列が返されます.
     */
    public final String toString( int offset,int length )
    {
        char[] tmp = null ;
        String ret = null ;
        
        if( m_bin == null || m_bin.isUse() == false ){
            return "" ;
        }
        try{
            tmp = this.getChars( offset,length ) ;
            ret = new String( tmp ) ;
            tmp = null ;
        }catch( Exception e ){
            ret = "" ;
        }finally{
            tmp = null ;
        }
        
        return ret ;
    }
    
    /**
     * オブジェクト有効チェック.
     * <BR><BR>
     * オブジェクトが有効であるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合有効です.<BR>
     *                 [false]が返された場合無効です.
     */
    public final boolean isUse()
    {
        boolean ret ;
        
        try{
            ret = m_bin.isUse() ;
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
}

