/*
 * @(#)BufferedBinResource.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.resource ;

import java.util.Arrays;

import com.JRcServer.commons.exception.InputException;

/**
 * バイナリリソースをバッファ処理.
 * <BR><BR>
 * バイナリリソースをバッファ処理します.<BR>
 * バイナリリソースのデータI/Oをbyte単位で読み書きする場合、
 * このオブジェクトを利用してください.
 *
 * @version     1.00, 2005/04/01
 * @author      Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class BufferedBinResource
{
    
    /**
     * バッファ情報.
     */
    private byte[] m_buffer = null ;
    
    /**
     * バッファサイズ.
     */
    private int m_bufLen = -1 ;
    
    /**
     * バッファ条件位置.
     */
    private int m_pos = -1 ;
    
    /**
     * バッファ条件に対する有効データ長.
     */
    private int m_posLen = -1 ;
    
    /**
     * 対象バッファリソースオブジェクト.
     */
    private BinResource m_resource = null ;
    
    /**
     * 現在リソース長.
     */
    private int m_resourceLen = -1 ;
    
    /**
     * 書き込みフラグ.
     */
    private boolean m_writeFlg = false ;
    
    /**
     * コンストラクタ.
     */
    private BufferedBinResource(){}
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象バイナリリソースを設定します.
     * <BR>
     * @param resource 対象バイナリリソースを設定します.
     * @exception InputException 入力例外.
     */
    public BufferedBinResource( BinResource resource )
        throws InputException
    {
        this( resource,ResourceDef.BUFFER_LENGTH ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象バイナリリソースを設定します.
     * <BR>
     * @param resource 対象バイナリリソースを設定します.
     * @param size 対象バッファサイズを設定します.
     * @exception InputException 入力例外.
     */
    public BufferedBinResource( BinResource resource,int size )
        throws InputException
    {
        if( resource == null || resource.isUse() == false ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        size = ( size < ResourceDef.BUFFER_LENGTH ) ? ResourceDef.BUFFER_LENGTH : size ;
        
        m_bufLen = size ;
        m_buffer = new byte[ size ] ;
        m_resource = resource ;
        m_resourceLen = resource.size() ;
        m_writeFlg = false ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public final void clear()
    {
        
        try{
            this.flush() ;
        }catch( Error e ){
            throw e ;
        }catch( Throwable t ){
        }finally{
            
            m_buffer = null ;
            m_bufLen = -1 ;
            m_pos = -1 ;
            m_posLen = -1 ;
            m_resource = null ;
            m_resourceLen = -1 ;
            m_writeFlg = false ;
            
        }
    }
    
    /**
     * バッファ条件をフラッシュ.
     * <BR><BR>
     * バッファ条件をフラッシュします.<BR>
     * また、書き込み処理を行った場合、最後にこのメソッドか[clear]メソッドを
     * 呼び出さない限り、有効になりません.
     */
    public final void flush()
    {
        try{
            if(
                m_pos != -1 && m_posLen > 0 &&
                m_resource != null && m_resource.isUse() == true &&
                m_writeFlg == true
            )
            {
                m_resource.setBinary( m_pos,m_buffer,0,m_posLen ) ;
                m_resourceLen = m_resource.size() ;
                
            }
        }catch( Exception e ){
        }finally{
            m_writeFlg = false ;
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 対象の情報を設定します.
     * <BR>
     * @param no 設定対象項番を設定します.
     * @param value 設定対象情報を設定します.
     */
    public final void set( int no,int value )
    {
        if( no < 0 ){
            return ;
        }
        
        no = this.newBuffer( no ) ;
        m_buffer[ no ] = ( byte )( value & 0x000000ff ) ;
        m_posLen = ( m_posLen < no+1 ) ? no+1 : m_posLen ;
        m_writeFlg = true ;
    }
    
    /**
     * 情報取得.
     * <BR><BR>
     * 対象情報を取得します.
     * <BR>
     * @param no 取得対象項番を設定します.
     * @return int 対象位置内容が返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public final int get( int no )
    {
        if( no < 0 ){
            return -1 ;
        }
        
        no = this.newBuffer( no ) ;
        return ( m_posLen <= no ) ? -1 : ( int )( m_buffer[ no ] & 0x000000ff ) ;
    }
    
    /**
     * リソースデータ長を取得.
     * <BR><BR>
     * 対象リソースのデータ長を取得します.
     * <BR>
     * @return int リソースデータ長が返されます.
     */
    public final int size()
    {
        return m_resourceLen ;
    }
    
    /**
     * 対象バッファ長を取得.
     * <BR><BR>
     * 対象バッファ長を取得します.
     * <BR>
     * @return int 対象バッファ長が返されます.
     */
    public final int getBufferLength()
    {
        return m_bufLen ;
    }
    
    /**
     * バイナリリソースオブジェクトを取得.
     * <BR><BR>
     * 設定されているバイナリリソースオブジェクトを取得します.
     * <BR>
     * @return BinResource 設定されているバイナリリソースオブジェクトが返されます.
     */
    public final BinResource getResource()
    {
        return m_resource ;
    }
    
    
    
    /**
     * バッファ条件が一致しない場合取得.
     */
    private final int newBuffer( int no )
    {
        int pos ;
        int posLen ;
        
        pos = m_pos ;
        posLen = m_posLen ;
        
        if( pos != -1 && no >= pos && no < pos + m_bufLen ){
            return no - pos ;
        }
        else if( pos != -1 && posLen > 0 ){
            m_resource.setBinary( pos,m_buffer,0,posLen ) ;
        }
        
        Arrays.fill( m_buffer,( byte )0x00000000 ) ;
        
        try{
            m_posLen = m_resource.getBinary( m_buffer,no ) ;
            m_pos = no ;
        }catch( Exception e ){
            m_posLen = 0 ;
            m_pos = no ;
        }
        
        m_resourceLen = m_resource.size() ;
        
        return 0 ;
    }
}
