/*
 * @(#)BinResourceRollEx.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.resource ;

import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.util.Roll;

/**
 * バイナリリソースロール拡張版.
 * <BR><BR>
 * バイナリリソース用のロール管理を行う拡張版オブジェクトです.
 *
 * @version     1.00, 2005/07/27
 * @author      Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class BinResourceRollEx implements Roll
{
    /**
     * デフォルト管理数.
     */
    private static final int DEF_SIZE = 32 ;
    
    
    
    /**
     * 送信ロール配列.
     */
    private BinResourceValue[] m_roll = null ;
    
    /**
     * リソースタイプ.
     */
    private ResourceType m_resType = null ;
    
    /**
     * ステータスオブジェクト名.
     */
    private Class m_stateClass = null ;
    
    /**
     * ロール管理最大数.
     */
    private int m_max = 0 ;
    
    /**
     * 現在位置.
     */
    private int m_now = 0 ;
    
    /**
     * 格納数.
     */
    private int m_nowLen = 0 ;
    
    /**
     * 追加予約フラグ.
     */
    private int m_addResCnt = 0 ;
    
    /**
     * コンストラクタ.
     */
    private BinResourceRollEx(){}
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * ロール管理数を設定して生成します.
     * <BR>
     * @param resType 対象のリソースタイプを設定します.
     * @param name 拡張条件で生成されるオブジェクト名[パッケージ名+クラス名]を
     *             設定します.
     * @param max ロール管理数を設定します.<BR>
     *            設定値は[32]以下を設定した場合、その値となります.
     * @exception InputException 入力例外.
     */
    public BinResourceRollEx( ResourceType resType,String name,int max )
        throws InputException
    {
        int i ;
        Object obj = null ;
        Class cls = null ;
        BinResourceValue[] roll = null ;
        
        if( resType == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            cls = Class.forName( name ) ;
            obj = cls.newInstance() ;
            if( ( obj instanceof ResourceRollState ) == false ){
                throw new InputException(
                    "対象名[" + name +
                    "]は(com.JRcServer.commons.resource.ResourceRollState" +
                    ")ではありません"
                ) ;
            }
            
            max = ( max <= DEF_SIZE ) ? DEF_SIZE : max ;
            
            roll = new BinResourceValue[ max ] ;
            
            for( i = 0 ; i < max ; i ++ ){
                roll[ i ] = new BinResourceValue() ;
                roll[ i ].RES = null ;
                roll[ i ].STA = ( ResourceRollState )cls.newInstance() ;
            }
            
            m_roll = roll ;
            m_resType = resType ;
            m_stateClass = cls ;
            m_max = max ;
            m_now = 0 ;
            m_nowLen = 0 ;
            m_addResCnt = 0 ;
            
        }catch( InputException in ){
            this.destroy() ;
            throw in ;
        }catch( Exception e ){
            this.destroy() ;
            throw new InputException( e ) ;
        }finally{
            cls = null ;
            obj = null ;
            roll = null ;
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        this.destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public final void destroy()
    {
        int i ;
        int len ;
        
        BinResourceValue[] roll = null ;
        
        len = m_max ;
        
        if( len > 0 ){
            
            roll = m_roll ;
            
            for( i = 0 ; i < len ; i ++ ){
                if( roll[ i ] != null ){
                    if( roll[ i ].RES != null ){
                        roll[ i ].RES.clear() ;
                    }
                    if( roll[ i ].STA != null ){
                        roll[ i ].STA.clear() ;
                    }
                }
                roll[ i ] = null ;
            }
            
        }
        
        m_roll = null ;
        m_resType = null ;
        m_stateClass = null ;
        m_max = 0 ;
        m_now = 0 ;
        m_nowLen = 0 ;
        m_addResCnt = 0 ;
    }
    
    /**
     * 追加予約をONにセット.
     * <BR><BR>
     * 追加予約をONにセットします.
     */
    public final void addReservationByON()
    {
        synchronized( this ){
            m_addResCnt ++ ;
        }
    }
    
    /**
     * 追加予約をOFFにセット.
     * <BR><BR>
     * 追加予約をOFFにセットします.
     */
    public final void addReservationByOFF()
    {
        synchronized( this ){
            m_addResCnt = ( m_addResCnt <= 0 ) ? 0 : m_addResCnt - 1 ;
        }
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象情報を追加します.<BR>
     * この処理は最後のロールに情報を追加します.
     * <BR>
     * @param value 対象の情報を設定します.
     * @param state 対象のステータス情報を設定します.
     * @return boolean 設定の合否が返されます.<BR>
     *                 [true]が返された場合、正しく設定されました.<BR>
     *                 [false]が返された場合、空き情報が存在しないことから、
     *                 正しく設定できませんでした.
     * @exception InputException 入力例外.
     */
    public final boolean add( byte[] value,ResourceRollState state )
        throws InputException
    {
        return this.addTo( false,value,state ) ;
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象情報を追加します.<BR>
     * この処理は最後のロールに情報を追加します.
     * <BR>
     * @param value 対象の情報を設定します.
     * @param state 対象のステータス情報を設定します.
     * @return boolean 設定の合否が返されます.<BR>
     *                 [true]が返された場合、正しく設定されました.<BR>
     *                 [false]が返された場合、空き情報が存在しないことから、
     *                 正しく設定できませんでした.
     * @exception InputException 入力例外.
     */
    public final boolean add( BinResource value,ResourceRollState state )
        throws InputException
    {
        return this.addTo( false,value,state ) ;
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象情報を追加します.<BR>
     * この処理は最初のロールに情報を追加します.
     * <BR>
     * @param value 対象の情報を設定します.
     * @param state 対象のステータス情報を設定します.
     * @return boolean 設定の合否が返されます.<BR>
     *                 [true]が返された場合、正しく設定されました.<BR>
     *                 [false]が返された場合、空き情報が存在しないことから、
     *                 正しく設定できませんでした.
     * @exception InputException 入力例外.
     */
    public final boolean addHead( byte[] value,ResourceRollState state )
        throws InputException
    {
        return this.addTo( true,value,state ) ;
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象情報を追加します.<BR>
     * この処理は最初のロールに情報を追加します.
     * <BR>
     * @param value 対象の情報を設定します.
     * @param state 対象のステータス情報を設定します.
     * @return boolean 設定の合否が返されます.<BR>
     *                 [true]が返された場合、正しく設定されました.<BR>
     *                 [false]が返された場合、空き情報が存在しないことから、
     *                 正しく設定できませんでした.
     * @exception InputException 入力例外.
     */
    public final boolean addHead( BinResource value,ResourceRollState state )
        throws InputException
    {
        return this.addTo( true,value,state ) ;
    }
    
    /**
     * 情報取得.
     * <BR><BR>
     * 対象の情報をバイナリ(byte[])で取得します.
     * <BR>
     * @param out 格納先のオブジェクトを設定します.
     * return byte[] バイナリ情報が返されます.<BR>
     *               [null]が返された場合、情報は存在しません.
     */
    public final byte[] getBinary( ResourceRollState out )
    {
        int pnt ;
        boolean flg ;
        
        BinResourceValue[] roll = null ;
        byte[] ret = null ;
        
        // ステータスオブジェクト型チェック.
        try{
            this.checkResourceRollState( out ) ;
            flg = true ;
        }catch( Exception e ){
            ret = null ;
            flg = false ;
        }
        
        if( flg == true ){
            
            synchronized( this ){
                
                if( m_nowLen <= 0 ){
                    ret = null ;
                }
                else{
                    
                    roll = m_roll ;
                    
                    if( ( pnt = m_now - m_nowLen ) < 0 ){
                        pnt = ( m_max + pnt ) ;
                    }
                    
                    // ロールデータを取得.
                    ret = roll[ pnt ].RES.getBinary() ;
                    
                    // ステータス情報を取得.
                    out.set( roll[ pnt ].STA ) ;
                    roll[ pnt ].STA.clear() ;
                    
                    m_nowLen -- ;
                    
                }
                
            }
            
        }
        
        return ret ;
    }
    
    /**
     * 情報取得.
     * <BR><BR>
     * 対象の情報をバイナリリソース(BinResource)で取得します.
     * <BR>
     * @param out 格納先のオブジェクトを設定します.
     * return BinResource バイナリリソース情報が返されます.<BR>
     *                    [null]が返された場合、情報は存在しません.
     */
    public final BinResource getBinResource( ResourceRollState out )
    {
        int pnt ;
        boolean flg ;
        
        BinResourceValue[] roll = null ;
        BinResource ret = null ;
        
        // ステータスオブジェクト型チェック.
        try{
            this.checkResourceRollState( out ) ;
            flg = true ;
        }catch( Exception e ){
            ret = null ;
            flg = false ;
        }
        
        if( flg == true ){
            
            synchronized( this ){
                
                if( m_nowLen <= 0 ){
                    ret = null ;
                }
                else{
                    
                    roll = m_roll ;
                    
                    if( ( pnt = m_now - m_nowLen ) < 0 ){
                        pnt = ( m_max + pnt ) ;
                    }
                    
                    // ロールデータを取得.
                    ret = roll[ pnt ].RES ;
                    roll[ pnt ].RES = null ;
                    
                    // ステータス情報を取得.
                    out.set( roll[ pnt ].STA ) ;
                    roll[ pnt ].STA.clear() ;
                    
                    m_nowLen -- ;
                    
                }
                
            }
            
        }
        
        return ret ;
    }
    
    /**
     * 対象の管理サイズを取得.
     * <BR><BR>
     * 対象の管理サイズが返されます.
     * <BR>
     * @return int 管理サイズが返されます.
     */
    public final int getMax()
    {
        int ret ;
        
        synchronized( this ){
            ret = m_max ;
        }
        
        return ret ;
    }
    
    /**
     * 現在の格納数を取得.
     * <BR><BR>
     * 現在の格納数を取得します.
     * <BR>
     * @return int 現在の格納数が返されます.
     */
    public final int getSize()
    {
        int ret ;
        
        synchronized( this ){
            ret = m_nowLen ;
        }
        
        return ret ;
    }
    
    /**
     * データが追加できるかチェック.
     * <BR><BR>
     * データが追加できるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、設定可能です.<BR>
     *                 [false]が返された場合、ロールは満杯のため設定はできません.
     */
    public final boolean isAdd()
    {
        boolean ret ;
        
        synchronized( this ){
            if( ( m_nowLen + m_addResCnt ) >= m_max ){
                ret = false ;
            }
            ret = true ;
        }
        
        return ret ;
    }
    
    /**
     * 追加予約が行われているかチェック.
     * <BR><BR>
     * 追加予約が行われているかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、追加予約されています.<BR>
     *                 [false]が返された場合追加予約はされていません.
     */
    public final boolean isAddReservation()
    {
        boolean ret ;
        
        synchronized( this ){
            ret = ( m_addResCnt > 0 ) ? true : false ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * 条件を追加.
     */
    private final boolean addTo( boolean hlMode,byte[] value,ResourceRollState state )
        throws InputException
    {
        int next ;
        boolean ret ;
        
        BinResourceValue[] roll = null ;
        
        if( value == null || value.length <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        // ステータスオブジェクト型チェック.
        this.checkResourceRollState( state ) ;
        
        ret = false ;
        
        synchronized( this ){
            
            // データ格納領域が存在しない場合.
            if( ( m_nowLen + m_addResCnt ) >= m_max ){
                ret = false ;
            }
            else{
                
                // 追加ポイントを取得.
                next = this.getAddPoint( hlMode ) ;
                
                // ロールオブジェクトを取得.
                roll = m_roll ;
                
                // 前回のデータが存在しないか、前回のデータ長と
                // 今回設定するデータ長*係数より大きい場合.
                if(
                    roll[ next ].RES == null ||
                    roll[ next ].RES.getAllSize() > Resource.getRenewRollValueCode( value.length )
                )
                {
                    // 前回のデータが存在する場合.
                    if( roll[ next ].RES != null ){
                        roll[ next ].RES.clear() ;
                        roll[ next ].RES = null ;
                    }
                    // 新たに生成.
                    roll[ next ].RES = Resource.createBinResource( m_resType,value ) ;
                }
                // 再利用可能な場合.
                else{
                    roll[ next ].RES.reset() ;
                    roll[ next ].RES.setBinary( 0,value ) ;
                }
                
                // ステータス情報を設定.
                roll[ next ].STA.set( state ) ;
                
                ret = true ;
            }
            
        }
        
        return ret ;
    }
    
    /**
     * 条件を追加.
     */
    private final boolean addTo( boolean hlMode,BinResource value,ResourceRollState state )
        throws InputException
    {
        int next ;
        boolean ret ;
        
        BinResourceValue[] roll = null ;
        
        if( value == null || value.isUse() == false || value.size() <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        // ステータスオブジェクト型チェック.
        this.checkResourceRollState( state ) ;
        
        ret = false ;
        
        synchronized( this ){
            
            // データ格納領域が存在しない場合.
            if( ( m_nowLen + m_addResCnt ) >= m_max ){
                ret = false ;
            }
            else{
                
                // 追加ポイントを取得.
                next = this.getAddPoint( hlMode ) ;
                
                // ロールオブジェクトを取得.
                roll = m_roll ;
                
                // ロールデータをセット.
                if( roll[ next ].RES != null ){
                    roll[ next ].RES.clear() ;
                    roll[ next ].RES = null ;
                }
                roll[ next ].RES = value ;
                
                // ステータス情報を設定.
                roll[ next ].STA.set( state ) ;
                
                ret = true ;
            }
            
        }
        
        return ret ;
    }
    
    /**
     * 追加位置を取得.
     */
    private final int getAddPoint( boolean hlMode )
    {
        int ret ;
        
        ret = m_now ;
        
        // 先頭に追加する場合.
        if( hlMode == true ){
            
            // データが存在する場合.
            if( m_nowLen > 0 ){
                if( ( ret = m_now - m_nowLen ) < 0 ){
                    ret = ( m_max + ret ) ;
                }
                ret -- ;
                if( ret < 0 ){
                    ret = m_max - 1 ;
                }
            }
            // データが存在しない場合.
            else{
                if( ret >= m_max ){
                    ret = 0 ;
                }
                m_now = ret + 1 ;
            }
            
            m_nowLen ++ ;
            
        }
        // 最後に追加する場合.
        else{
            
            if( ret >= m_max ){
                ret = 0 ;
            }
            
            m_nowLen ++ ;
            m_now = ret + 1 ;
            
        }
        
        return ret ;
    }
    
    /**
     * 対象ステータスオブジェクトをチェック.
     */
    private final void checkResourceRollState( ResourceRollState obj )
        throws InputException
    {
        // 対象引数が設定されていない場合.
        if( obj == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        // 対象引数が規定のオブジェクトと一致しない場合.
        else if( m_stateClass.isAssignableFrom( obj.getClass() ) == false ){
            throw new InputException(
                "指定ステータスオブジェクトは(" + m_stateClass.getName() +
                ")ではありません"
            ) ;
        }
        
    }
}

/**
 * バイナリリソース要素.
 */
class BinResourceValue
{
    BinResource RES = null ;
    ResourceRollState STA = null ;
}
