/*
 * @(#)BinCacheResource.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.resource ;

import com.JRcServer.commons.exception.AccessException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.resource.cache.Cache;
import com.JRcServer.commons.resource.cache.CacheIO;
import com.JRcServer.commons.resource.cache.CacheManager;
import com.JRcServer.commons.thread.Synchronized;

/**
 * スペース領域を利用したバイナリオブジェクト.
 * <BR><BR>
 * スペース領域を利用したバイナリオブジェクトを生成します.
 *
 * @version     1.00, 2005/04/01
 * @author      Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class BinCacheResource implements BinResource
{
    
    /**
     * バイナリ単位.
     */
    private static final int BINARY_LENGTH = 512 ;
    
    /**
     * バイナリ情報.
     */
    private Cache m_fp = null ;
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    /**
     * コンストラクタ.
     */
    public BinCacheResource()
    {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * スペースファイルを利用してバイナリオブジェクトを生成します.
     * <BR>
     * @param size 利用可能な領域を指定します.
     * @param cacheID 対象のキャッシュIDを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public BinCacheResource( int size,long cacheID )
        throws InputException,AccessException
    {
        this.create( size,cacheID ) ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * スペースファイルを利用してバイナリオブジェクトを生成します.
     * <BR>
     * @param size 利用可能な領域を指定します.
     * @param cacheID 対象のキャッシュIDを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final void create( int size,long cacheID )
        throws InputException,AccessException
    {
        int len ;
        CacheIO io = null ;
        
        if( size <= 0 || cacheID < 0 || ( io = CacheManager.get( cacheID ) ) == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.clear() ;
        m_sync.create() ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                m_fp = io.newCacheObject( size ) ;
                
            }
            
        }catch( AccessException ac ){
            this.clear() ;
            throw ac ;
        }catch( Exception e ){
            this.clear() ;
            throw new AccessException( e ) ;
        }
        
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * バイナリ情報をクリアします.
     */
    public final void clear()
    {
        try{
            synchronized( m_sync.get() ){
                if( m_fp != null ) {
                    m_fp.close() ;
                }
            }
        }catch( Exception e ){
        }
        
        m_sync.clear() ;
        m_fp = null ;
    }
    
    /**
     * リセット処理.
     * <BR><BR>
     * 有効データ長をリセットします.
     */
    public final void reset()
    {
        try{
            synchronized( m_sync.get() ){
                m_fp.reset() ;
            }
        }catch( Exception e ){
        }
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 対象条件に情報を設定します.
     * <BR>
     * @param no 設定対象項番を設定します.
     * @param b 設定対象のバイト情報を設定します.
     */
    public final void set( int no,int b )
    {
        byte[] bin = null ;
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_fp == null ){
                    return ;
                }
                
                bin[ 0 ] = ( byte )( b & 0x000000ff ) ;
                m_fp.write( bin,no ) ;
                
            }
        }catch( Exception in ){
        }finally{
            bin = null ;
        }
    }
    
    /**
     * バイナリ情報設定.
     * <BR><BR>
     * 対象のバイナリ情報を設定します.
     * <BR>
     * @param no 設定開始位置となる項番を設定します.
     * @param bin 設定対象のバイナリ情報を設定します.
     * @return int 設定されたバイナリ長が返されます.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public final int setBinary( int no,byte[] bin ) throws ArrayIndexOutOfBoundsException
    {
        if( bin == null || no < 0 ){
            if( bin == null ){
                throw new ArrayIndexOutOfBoundsException( "設定バイナリデータは不正です" ) ;
            }
            throw new ArrayIndexOutOfBoundsException(
                "指定位置(" + no + "は範囲外です"
            ) ;
        }
        else if( bin.length <= 0 ){
            return 0 ;
        }
        
        return this.setBinary( no,bin,0,bin.length ) ;
    }
    
    /**
     * バイナリ情報設定.
     * <BR><BR>
     * 対象のバイナリ情報を設定します.
     * <BR>
     * @param no 設定開始位置となる項番を設定します.
     * @param bin 設定対象のバイナリ情報を設定します.
     * @param off 設定対象のオフセット値を設定します.
     * @param len 設定対象のバイナリ長を設定します.
     * @return int 設定されたバイナリ長が返されます.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public final int setBinary( int no,byte[] bin,int off,int len ) throws ArrayIndexOutOfBoundsException
    {
        int inputLen ;
        int ret ;
        
        if( bin == null || off < 0 || len <= 0 || no < 0 ){
            if( bin == null ){
                throw new ArrayIndexOutOfBoundsException( "設定バイナリデータは不正です" ) ;
            }
            else if( no < 0 ){
                throw new ArrayIndexOutOfBoundsException(
                    "指定位置(" + no + "は範囲外です"
                ) ;
            }
            else{
                throw new ArrayIndexOutOfBoundsException(
                    "設定条件(off:" + off + " len:" + len + ")は不正です"
                ) ;
            }
        }
        else if( ( inputLen = bin.length ) <= 0 ){
            return 0 ;
        }
        else if( off >= inputLen ){
            throw new ArrayIndexOutOfBoundsException(
                "設定条件(off:" + off +
                ")は設定バイナリ長(" + inputLen +
                ")を越しています"
            ) ;
        }
        
        try{
            
            synchronized( m_sync.get() ){
                
                if( m_fp == null ){
                    return -1 ;
                }
                
                inputLen = ( ( off + len ) > inputLen ) ? inputLen - off : len ;
                ret = m_fp.write( bin,no,off,inputLen ) ;
                
            }
        }catch( ArrayIndexOutOfBoundsException ai ){
            throw ai ;
        }catch( Exception in ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * バイナリ情報設定.
     * <BR><BR>
     * 対象のバイナリ情報を設定します.
     * <BR>
     * @param no 設定開始位置となる項番を設定します.
     * @param bin 設定対象のバイナリオブジェクトを設定します.
     * @return int 設定されたバイナリ長が返されます.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public final int setBinary( int no,BinResource bin ) throws ArrayIndexOutOfBoundsException
    {
        if( bin == null || bin.isUse() == false || no < 0 ){
            if( bin == null || bin.isUse() == false ){
                throw new ArrayIndexOutOfBoundsException( "設定バイナリデータは不正です" ) ;
            }
            throw new ArrayIndexOutOfBoundsException(
                "指定位置(" + no + "は範囲外です"
            ) ;
        }
        else if( bin.size() <= 0 ){
            return 0 ;
        }
        
        return this.setBinary( no,bin,0,bin.size() ) ;
    }
    
    /**
     * バイナリ情報設定.
     * <BR><BR>
     * 対象のバイナリ情報を設定します.
     * <BR>
     * @param no 設定開始位置となる項番を設定します.
     * @param bin 設定対象のバイナリオブジェクトを設定します.
     * @param off 設定対象のオフセット値を設定します.
     * @param len 設定対象のバイナリ長を設定します.
     * @return int 設定されたバイナリ長が返されます.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public final int setBinary( int no,BinResource bin,int off,int len ) throws ArrayIndexOutOfBoundsException
    {
        int i ;
        int lenI ;
        int inputLen ;
        int etc ;
        int pnt ;
        int bpnt ;
        int wlen ;
        int ret ;
        
        byte[] buf = null ;
        
        if( bin == null || bin.isUse() == false || off < 0 || len <= 0 || no < 0 ){
            if( bin == null || bin.isUse() == false ){
                throw new ArrayIndexOutOfBoundsException( "設定バイナリデータは不正です" ) ;
            }
            else if( no < 0 ){
                throw new ArrayIndexOutOfBoundsException(
                    "指定位置(" + no + "は範囲外です"
                ) ;
            }
            else{
                throw new ArrayIndexOutOfBoundsException(
                    "設定条件(off:" + off + " len:" + len + ")は不正です"
                ) ;
            }
        }
        else if( ( inputLen = bin.size() ) <= 0 ){
            return 0 ;
        }
        else if( off >= inputLen ){
            throw new ArrayIndexOutOfBoundsException(
                "設定条件(off:" + off +
                ")は設定バイナリ長(" + inputLen +
                ")を越しています"
            ) ;
        }
        
        
        try{
            
            synchronized( m_sync.get() ){
                
                if( m_fp == null ){
                    return -1 ;
                }
                
                inputLen = ( ( off + len ) > inputLen ) ? inputLen - off : len ;
                
                lenI = inputLen / ResourceDef.BUFFER_LENGTH ;
                etc = inputLen % ResourceDef.BUFFER_LENGTH ;
                
                buf = new byte[ ResourceDef.BUFFER_LENGTH ] ;
                for( i = 0,pnt = no,bpnt = off,ret = 0 ; i < lenI ; i ++ ){
                    
                    wlen = bin.getBinary( buf,bpnt ) ;
                    m_fp.write( buf,pnt,0,wlen ) ;
                    
                    pnt += wlen ;
                    bpnt += wlen ;
                    ret += wlen ;
                    
                }
                
                if( etc != 0 ){
                    wlen = bin.getBinary( buf,bpnt,0,etc ) ;
                    m_fp.write( buf,pnt,0,wlen ) ;
                    ret += wlen ;
                }
                
            }
        }catch( ArrayIndexOutOfBoundsException ai ){
            throw ai ;
        }catch( Exception in ){
            ret = 0 ;
        }finally{
            buf = null ;
        }
        
        return ret ;
    }
    
    /**
     * 情報取得.
     * <BR><BR>
     * 対象条件の情報を取得します.
     * <BR>
     * @param no 取得対象項番を設定します.
     * @return int 対象のバイナリ情報が返されます.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public final int get( int no ) throws ArrayIndexOutOfBoundsException
    {
        int ret ;
        byte[] tmp = null ;
        
        if( no < 0 ){
            throw new ArrayIndexOutOfBoundsException(
                "指定位置(" + no + "は範囲外です"
            ) ;
        }
        
        try{
            
            tmp = new byte[ 1 ] ;
            
            synchronized( m_sync.get() ){
                
                if( m_fp == null ){
                    return -1 ;
                }
                
                m_fp.read( tmp,no,0,1 ) ;
                ret = ( int )( tmp[ 0 ] & 0x000000ff ) ;
                
                
            }
        }catch( NullPointerException nul ){
            ret = -1 ;
        }catch( Exception e ){
            throw new ArrayIndexOutOfBoundsException( e.getMessage() ) ;
        }finally{
            tmp = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * バイナリ情報を取得.
     * <BR><BR>
     * 格納されているバイナリ情報を取得します.
     * <BR>
     * @return byte[] 設定されているバイナリ情報が返されます.
     */
    public final byte[] getBinary()
    {
        byte[] ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_fp == null ){
                    return null ;
                }
                
                ret = m_fp.read( 0 ) ;
                
            }
        }catch( Exception in ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * バイナリ情報を取得.
     * <BR><BR>
     * 対象のバイナリ情報を取得します.
     * <BR>
     * @param no 取得開始位置となる項番を設定します.
     * @return byte[] 取得されたバイナリ情報が返されます.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public final byte[] getBinary( int no ) throws ArrayIndexOutOfBoundsException
    {
        int useLen ;
        byte[] ret = null ;
        
        if( no < 0 ){
            throw new ArrayIndexOutOfBoundsException(
                "指定位置(" + no + "は範囲外です"
            ) ;
        }
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_fp == null ){
                    return null ;
                }
                
                useLen = m_fp.getLength() ;
                
                if( useLen > 0 ){
                    ret = new byte[ useLen - no ] ;
                    m_fp.read( ret,no,0,useLen-no ) ;
                }
                else if( useLen <= no ){
                    throw new ArrayIndexOutOfBoundsException(
                        "指定位置(" + no +
                        ")は有効データ長(" + useLen +
                        ")を越しています"
                    ) ;
                }
                else{
                    ret = null ;
                }
                
            }
        }catch( ArrayIndexOutOfBoundsException ai ){
            throw ai ;
        }catch( Exception in ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * バイナリ情報を取得.
     * <BR><BR>
     * 対象のバイナリ情報を取得します.
     * <BR>
     * @param no 取得開始位置となる項番を設定します.
     * @param len 取得対象のバイナリ長を設定します.
     * @return byte[] 取得されたバイナリ情報が返されます.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public final byte[] getBinary( int no,int len ) throws ArrayIndexOutOfBoundsException
    {
        int useLen ;
        byte[] ret = null ;
        
        if( no < 0 ){
            throw new ArrayIndexOutOfBoundsException(
                "指定位置(" + no + "は範囲外です"
            ) ;
        }
        else if( len <= 0 ){
            return null ;
        }
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_fp == null ){
                    return null ;
                }
                useLen = m_fp.getLength() ;
                
                if( useLen > 0 ){
                    len = ( useLen < no + len ) ? useLen - no : len ;
                    
                    ret = new byte[ len ] ;
                    m_fp.read( ret,no,0,len ) ;
                }
                else if( useLen <= no ){
                    throw new ArrayIndexOutOfBoundsException(
                        "指定位置(" + no +
                        ")は有効データ長(" + useLen +
                        ")を越しています"
                    ) ;
                }
                else{
                    ret = null ;
                }
                
            }
        }catch( ArrayIndexOutOfBoundsException ai ){
            throw ai ;
        }catch( Exception in ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * バイナリ情報を取得.
     * <BR><BR>
     * 格納されているバイナリ情報を取得します.
     * <BR>
     * @param out 取得対象のバイナリ情報が返されます.
     * @return int 取得されたバイナリ長が返されます.
     */
    public final int getBinary( byte[] out )
    {
        int inputLen ;
        
        if( out == null ){
            throw new ArrayIndexOutOfBoundsException( "設定バイナリデータは不正です" ) ;
        }
        else if( ( inputLen = out.length ) <= 0 ){
            return 0 ;
        }
        
        return this.getBinary( out,0,0,inputLen ) ;
    }
    
    /**
     * バイナリ情報を取得.
     * <BR><BR>
     * 対象のバイナリ情報を取得します.
     * <BR>
     * @param out 取得対象のバイナリ情報が返されます.
     * @param no 取得開始位置となる項番を設定します.
     * @return int 取得されたバイナリ長が返されます.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public final int getBinary( byte[] out,int no ) throws ArrayIndexOutOfBoundsException
    {
        int inputLen ;
        
        if( out == null ){
            throw new ArrayIndexOutOfBoundsException( "設定バイナリデータは不正です" ) ;
        }
        else if( ( inputLen = out.length ) <= 0 ){
            return 0 ;
        }
        
        return this.getBinary( out,no,0,inputLen ) ;
    }
    
    /**
     * バイナリ情報を取得.
     * <BR><BR>
     * 対象のバイナリ情報を取得します.
     * <BR>
     * @param out 取得対象のバイナリ情報が返されます.
     * @param no 取得開始位置となる項番を設定します.
     * @param off 取得対象のバイナリオフセット値を設定します.
     * @param len 取得対象のバイナリ長を設定します.
     * @return int 取得されたバイナリ長が返されます.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public final int getBinary( byte[] out,int no,int off,int len ) throws ArrayIndexOutOfBoundsException
    {
        int useLen ;
        int inputLen ;
        int ret ;
        
        if( out == null || off < 0 || len <= 0 || no < 0 ){
            if( out == null ){
                throw new ArrayIndexOutOfBoundsException( "設定バイナリデータは不正です" ) ;
            }
            else if( no < 0 ){
                throw new ArrayIndexOutOfBoundsException(
                    "指定位置(" + no + "は範囲外です"
                ) ;
            }
            else{
                throw new ArrayIndexOutOfBoundsException(
                    "設定条件(off:" + off + " len:" + len + ")は不正です"
                ) ;
            }
        }
        else if( ( inputLen = out.length ) <= 0 ){
            return 0 ;
        }
        
        try{
            
            synchronized( m_sync.get() ){
                
                if( m_fp == null ){
                    return -1 ;
                }
                
                useLen = m_fp.getLength() ;
                inputLen = ( ( off + len ) > inputLen ) ? inputLen - off : len ;
                inputLen = ( useLen < no + inputLen ) ? useLen - no : inputLen ;
                ret = m_fp.read( out,no,off,inputLen ) ;
                
            }
        }catch( ArrayIndexOutOfBoundsException ai ){
            throw ai ;
        }catch( Exception in ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * バイナリ情報を取得.
     * <BR><BR>
     * 格納されているバイナリ情報を取得します.
     * <BR>
     * @param out 取得対象のバイナリ情報が返されます.
     * @return int 取得されたバイナリ長が返されます.
     */
    public final int getBinary( BinResource out )
    {
        int inputLen ;
        
        if( out == null || out.isUse() == false ){
            throw new ArrayIndexOutOfBoundsException( "設定バイナリデータは不正です" ) ;
        }
        else if( ( inputLen = out.getAllSize() ) <= 0 ){
            return 0 ;
        }
        
        return this.getBinary( out,0,0,inputLen ) ;
    }
    
    /**
     * バイナリ情報を取得.
     * <BR><BR>
     * 対象のバイナリ情報を取得します.
     * <BR>
     * @param out 取得対象のバイナリ情報が返されます.
     * @param no 取得開始位置となる項番を設定します.
     * @return int 取得されたバイナリ長が返されます.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public final int getBinary( BinResource out,int no ) throws ArrayIndexOutOfBoundsException
    {
        int inputLen ;
        
        if( out == null || out.isUse() == false ){
            throw new ArrayIndexOutOfBoundsException( "設定バイナリデータは不正です" ) ;
        }
        else if( ( inputLen = out.getAllSize() ) <= 0 ){
            return 0 ;
        }
        
        return this.getBinary( out,no,0,inputLen ) ;
    }
    
    /**
     * バイナリ情報を取得.
     * <BR><BR>
     * 対象のバイナリ情報を取得します.
     * <BR>
     * @param out 取得対象のバイナリ情報が返されます.
     * @param no 取得開始位置となる項番を設定します.
     * @param off 取得対象のバイナリオフセット値を設定します.
     * @param len 取得対象のバイナリ長を設定します.
     * @return int 取得されたバイナリ長が返されます.
     * @exception ArrayIndexOutOfBoundsException 不正インデックス例外.
     */
    public final int getBinary( BinResource out,int no,int off,int len ) throws ArrayIndexOutOfBoundsException
    {
        int useLen ;
        int inputLen ;
        int ret ;
        
        if( out == null || out.isUse() == false || off < 0 || len <= 0 || no < 0 ){
            if( out == null || out.isUse() == false ){
                throw new ArrayIndexOutOfBoundsException( "設定バイナリデータは不正です" ) ;
            }
            else if( no < 0 ){
                throw new ArrayIndexOutOfBoundsException(
                    "指定位置(" + no + "は範囲外です"
                ) ;
            }
            else{
                throw new ArrayIndexOutOfBoundsException(
                    "設定条件(off:" + off + " len:" + len + ")は不正です"
                ) ;
            }
        }
        else if( ( inputLen = out.getAllSize() ) <= 0 ){
            return 0 ;
        }
        
        try{
            
            synchronized( m_sync.get() ){
                
                if( m_fp == null ){
                    return -1 ;
                }
                
                useLen = m_fp.getLength() ;
                //inputLen = ( ( off + len ) > inputLen ) ? inputLen - off : len ;
                //inputLen = ( useLen < no + inputLen ) ? useLen - no : inputLen ;
                inputLen = ( useLen < no + len ) ? useLen - no : len ;
                ret = out.setBinary( off,this,no,inputLen ) ;
                
            }
        }catch( ArrayIndexOutOfBoundsException ai ){
            throw ai ;
        }catch( Exception in ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 現在の有効バイナリ長を取得.
     * <BR><BR>
     * 現在の有効なバイナリ長を取得します.
     * <BR>
     * @return int 現在の有効なバイナリ長が返されます.
     */
    public final int size()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_fp == null ){
                    return -1 ;
                }
                
                ret = m_fp.getLength() ;
                
            }
        }catch( Exception e ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 現在のバイナリ長を取得.
     * <BR><BR>
     * 現在のバイナリ長を取得します.
     * <BR>
     * @return int 現在のバイナリ長が返されます.
     */
    public final int getAllSize()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_fp == null ){
                    return -1 ;
                }
                
                ret = m_fp.getSector() * CacheIO.ELEMENT_LENGTH ;
                
            }
        }catch( Exception e ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * オブジェクトタイプを取得.
     * <BR><BR>
     * オブジェクトタイプを取得します.
     * <BR>
     * @return int オブジェクトタイプが返されます.<BR>
     *             [BinResource#BIN_RESOURCE_TYPE_MEMORY]が返された場合、
     *             [com.JRcServer.commons.resource.BinMemoryResource]オブジェクトです.<BR>
     *             [BinResource#BIN_RESOURCE_TYPE_FILE]が返された場合、
     *             [com.JRcServer.commons.resource.BinFileResource]オブジェクトです.<BR>
     *             [BinResource#BIN_RESOURCE_TYPE_CACHE]が返された場合、
     *             [com.JRcServer.commons.resource.BinCacheResource]オブジェクトです.
     */
    public final int getType()
    {
        return BinResource.BIN_RESOURCE_TYPE_CACHE ;
    }
    
    /**
     * オブジェクト有効チェック.
     * <BR><BR>
     * オブジェクトが有効であるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合有効です.<BR>
     *                 [false]が返された場合無効です.
     */
    public final boolean isUse()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = ( m_fp == null ) ? false : true ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
}
