/*
 * @(#)UdpSndResultThread.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.net.uio;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.JRcServer.commons.exception.BaseException;
import com.JRcServer.commons.exception.ExecutionException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.net.ConnectTimeoutException;
import com.JRcServer.commons.net.UdpProtocol;
import com.JRcServer.commons.thread.ExecutionThread;
import com.JRcServer.commons.thread.LoopThread;
import com.JRcServer.commons.thread.Synchronized;

/**
 * UDP送信結果情報取得スレッド.
 * <BR><BR>
 * 送信UDPプロトコルに対して送信結果が返された処理をスレッドで行います.
 *
 * @version 1.00, 2005/03/24
 * @author  Masahito Suzuki
 * @since   JRcCommons 1.00
 */
class UdpSndResultThread extends ExecutionThread
{
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( UdpSndResultThread.class ) ;
    
    /**
     * UDP受信タイムアウト値.
     */
    private static final int UDP_RECV_TIMEOUT = 500 ;
    
    /**
     * UDPプロトコル.
     */
    private UdpProtocol m_udp = null ;
    
    /**
     * UDP送信結果テーブル.
     */
    private UdpSndResultTable m_table = null ;
    
    /**
     * ループスレッド.
     */
    private final LoopThread m_thread = new LoopThread() ;
    
    /**
     * 同期処理.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    /**
     * 電文送信用同期処理.
     */
    private Synchronized m_sndSync = null ;
    
    
    
    /**
     * コンストラクタ.
     */
    private UdpSndResultThread(){}
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 基本条件を渡してスレッド実行します.
     * <BR>
     * @param udp 送信処理を行うUDPプロトコルを設定します.
     * @param tbl 送信結果を格納するテーブルを設定します.
     * @param sndSync 送信処理用同期オブジェクトを設定します.
     * @exception InputException 入力例外.
     */
    public UdpSndResultThread( UdpProtocol udp,UdpSndResultTable tbl,Synchronized sndSync )
        throws InputException
    {
        if(
            udp == null || udp.isOpen() == false ||
            tbl == null || sndSync == null ||
            sndSync.isUse() == false
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.destroy() ;
        m_sync.create() ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                m_udp = udp ;
                m_table = tbl ;
                m_sndSync = sndSync ;
                
            }
            
            m_thread.create( this ) ;
            m_thread.startThread() ;
            
        }catch( Exception e ){
            this.destroy() ;
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public final void destroy()
    {
        m_sync.clear() ;
        m_thread.clear() ;
        
        m_udp = null ;
        m_table = null ;
        m_sndSync = null ;
    }
    
    /**
     * スレッド状態を取得.
     * <BR><BR>
     * スレッド状態を取得します.
     * <BR>
     * @return boolean スレッド状態が返されます.<BR>
     *                 [true]が返された場合、スレッドは実行中です.<BR>
     *                 [false]が返された場合、スレッドは停止中です.
     */
    public final boolean isThread()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_thread.isThread() ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * 実行初期化処理をサポートします.
     * <BR><BR>
     * 実行初期化処理をサポートします.<BR>
     * この処理は、スレッド処理が開始された時に呼び出されます.
     * <BR>
     * @param obj 実行開始時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void init( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行終了化処理をサポートします.
     * <BR><BR>
     * 実行終了化処理をサポートします.<BR>
     * この処理は、スレッド処理が終了された時に呼び出されます.
     * <BR>
     * @param obj 実行終了時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void exit( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * ストップ処理をサポートします。
     * <BR><BR>
     * ストップ処理をサポートします。<BR>
     * この処理は、スレッドでのストップ処理に対して呼び出し実行されます.
     * <BR>
     * @param obj ストップ時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void stop( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行処理をサポートします。
     * <BR><BR>
     * 実行処理をサポートします。<BR>
     * この処理は、スレッドでの実行処理に対して呼び出し実行されます.
     * <BR>
     * @param obj 実行時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void execution( Object obj )
        throws ExecutionException
    {
        byte[] rcv = null ;
        UdpProtocol udp = null ;
        UdpSndResultTable tbl = null ;
        Synchronized sndSync = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                udp = m_udp ;
                tbl = m_table ;
                sndSync = m_sndSync ;
            }
            
            try{
                
                rcv = udp.receive( null,UdpSndResultThread.UDP_RECV_TIMEOUT ) ;
                
                // 受信データが正しい場合.
                if( rcv != null && rcv.length > 0 && UdpIOCommon.isPacket( rcv ) == true ){
                    synchronized( sndSync.get() ){
                        tbl.put( rcv ) ;
                    }
                }
                else{
                    rcv = null ;
                }
                
            }catch( ConnectTimeoutException et ){
            }catch( InputException e ){
            }finally{
                rcv = null ;
            }
            
        }catch( NullPointerException nul ){
            throw new ExecutionException(
                nul,ExecutionException.LEVEL_STOP
            ) ;
        }catch( BaseException be ){
            LOG.error( "エラーが発生しました", be ) ;
        }catch( Exception e ){
            LOG.error( "エラーが発生しました", e ) ;
        }finally{
            
            rcv = null ;
            udp = null ;
            tbl = null ;
            sndSync = null ;
            
        }
        
    }
    
}

