/*
 * @(#)UdpSndResultTable.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.net.uio;

import com.JRcServer.commons.exception.ExecutionException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.resource.ResourceType;
import com.JRcServer.commons.thread.Synchronized;
import com.JRcServer.commons.util.NumberTable;
import com.JRcServer.commons.util.SearchInt;
import com.JRcServer.commons.util.UtilCom;

/**
 * UDP送信結果格納テーブル.
 *
 * @version 1.00, 2005/03/25
 * @author  Masahito Suzuki
 * @since   JRcCommons 1.00
 */
class UdpSndResultTable
{
    
    /**
     * 受信電文格納用.
     */
    private final NumberTable m_table = new NumberTable() ;
    
    /**
     * 送信予約ID.
     */
    private final SearchInt m_resID = new SearchInt() ;
    
    /**
     * リソースタイプ.
     */
    private ResourceType m_resType = null ;
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    
    
    /**
     * コンストラクタ.
     */
    public UdpSndResultTable(){
        this.clear() ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 対象テーブルを生成します.
     */
    public final void create()
    {
        this.create( null ) ;
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 対象テーブルを生成します.
     * <BR>
     * @param resType 対象のリソースタイプを設定します.
     */
    public final void create( ResourceType resType )
    {
        this.clear() ;
        m_sync.create() ;
        m_resID.create() ;
        
        m_resType = ( resType == null ) ? UdpIODef.DEF_RESOURCE_TYPE : resType ;
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public final void clear()
    {
        m_resID.clear() ;
        m_table.clear() ;
        m_sync.clear() ;
        m_resType = null ;
    }
    
    /**
     * 送信予約IDを設定.
     * <BR><BR>
     * 送信予約IDを設定します.<BR>
     * このIDを設定しない場合、受信データは全て破棄されます.
     * <BR>
     * @param id 送信予約IDを設定します.
     */
    public final void reservation( int id )
    {
        try{
            synchronized( m_sync.get() ){
                m_resID.add( id ) ;
            }
        }catch( Exception e ){
        }
    }
    
    /**
     * 送信予約IDを破棄.
     * <BR><BR>
     * 送信予約IDを破棄します.<BR>
     * 送信処理が完了した場合に呼び出す必要があります.<BR>
     * また、テーブルに格納されている全ての条件も削除されます.
     * <BR>
     * @param id 破棄対象の送信予約IDを設定します.
     */
    public final void cancell( int id )
    {
        UdpSndResultChild ch = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                m_resID.remove( id ) ;
                
                if( ( ch = ( UdpSndResultChild )m_table.remove( id ) ) != null ){
                    ch.clear() ;
                }
                
            }
            
        }catch( Exception e ){
        }finally{
            ch = null ;
        }
    }
    
    /**
     * 送信結果データ設定.
     * <BR><BR>
     * 送信結果データを設定します.
     * <BR>
     * @param binary 送信先のバイナリ情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void put( byte[] binary )
        throws InputException
    {
        int id ;
        int sub ;
        int rnd ;
        
        UdpSndResultChild ch = null ;
        
        if( binary == null || binary.length <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            // パケットデータであるかチェック.
            if( UdpIOCommon.isPacket( binary ) == true ){
                
                // 各データを取得.
                id = UdpIOCommon.getPacketToID( binary ) ;
                sub = UdpIOCommon.getPacketToSubID( binary ) ;
                rnd = UdpIOCommon.getPacketToRandID( binary ) ;
                
                synchronized( m_sync.get() ){
                    
                    // 受信IDが予約IDに存在する場合.
                    if( id >= 0L && m_resID.isData( id ) == true ){
                        
                        // バイナリ条件を追加.
                        if( ( ch = ( UdpSndResultChild )m_table.get( id ) ) == null ){
                            ch = new UdpSndResultChild() ;
                            m_table.add( id,ch ) ;
                        }
                        
                        ch.add( sub,rnd,binary ) ;
                        
                    }
                    
                }
                
            }
            
        }catch( Exception e ){
        }finally{
            ch = null ;
        }
        
    }
    
    /**
     * 送信結果データを取得.
     * <BR><BR>
     * 送信結果データを取得します.<BR>
     * また、取得後テーブルから破棄されます.
     * <BR>
     * @param id 取得対象のIDを設定します.
     * @param subID 取得対象のサブIDを設定します.
     * @param rndID 取得対象のランダムIDを設定します.
     * @return byte[] 対象のバイナリ情報が返されます.
     */
    public final byte[] get( int id,int subID,int rndID )
    {
        
        byte[] ret = null ;
        UdpSndResultChild ch = null ;
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_resID.isData( id ) == true ){
                    
                    if( ( ch = ( UdpSndResultChild )m_table.get( id ) ) != null ){
                        ret = ch.get( subID,rndID ) ;
                    }
                    
                }
                
            }
        }catch( Exception e ){
            ret = null ;
        }finally{
            ch = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 送信結果データを取得.
     * <BR><BR>
     * 送信結果データを取得します.<BR>
     * この処理はタイムアウトを設けて情報を取得します.<BR>
     * また、取得後テーブルから破棄されます.
     * <BR>
     * @param id 取得対象のIDを設定します.
     * @param subID 取得対象のサブIDを設定します.
     * @param rndID 取得対象のランダムIDを設定します.
     * @param timeout 取得タイムアウトを設定します.
     * @return byte[] 対象のバイナリ情報が返されます.
     * @exception ExecutionException 実行例外.
     */
    public final byte[] get( int id,int subID,int rndID,int timeout )
        throws ExecutionException
    {
        
        long now ;
        byte[] ret = null ;
        
        timeout = ( timeout <= 0 ) ? 0 : timeout ;
        
        try{
            
            if( timeout == 0 ){
                ret = this.get( id,subID,rndID ) ;
            }
            else{
                
                now = System.currentTimeMillis() + timeout ;
                
                for( ;; ){
                    
                    ret = this.get( id,subID,rndID ) ;
                    
                    if( ret != null || now <= System.currentTimeMillis() ){
                        break ;
                    }
                    
                    UtilCom.idleTime() ;
                    
                }
                
            }
            
        }catch( ExecutionException ee ){
            ret = null ;
            throw ee ;
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 送信結果ヘッダデータを取得.
     * <BR><BR>
     * 送信結果ヘッダデータを取得します.<BR>
     * また、取得後テーブルから破棄されます.
     * <BR>
     * @param id 取得対象のIDを設定します.
     * @param rndID 取得対象のランダムIDを設定します.
     * @return byte[] 対象のバイナリ情報が返されます.
     */
    public final byte[] getHeader( int id,int rndID )
    {
        return this.get( id,UdpIODef.PACKET_HEADER_SUBID,rndID ) ;
    }
    
    /**
     * 送信結果ヘッダデータを取得.
     * <BR><BR>
     * 送信結果ヘッダデータを取得します.<BR>
     * この処理はタイムアウトを設けて情報を取得します.<BR>
     * また、取得後テーブルから破棄されます.
     * <BR>
     * @param id 取得対象のIDを設定します.
     * @param rndID 取得対象のランダムIDを設定します.
     * @param timeout 取得タイムアウトを設定します.
     * @return byte[] 対象のバイナリ情報が返されます.
     * @exception ExecutionException 実行例外.
     */
    public final byte[] getHeader( int id,int rndID,int timeout )
        throws ExecutionException
    {
        return this.get( id,UdpIODef.PACKET_HEADER_SUBID,rndID,timeout ) ;
    }
    
    /**
     * 対象IDに対するランダムIDを取得.
     * <BR><BR>
     * 対象IDに対するランダムIDを取得します.
     * <BR>
     * @param id 対象のID情報を取得します.
     * @return int 対象IDに対するランダムIDが返されます.<BR>
     *             [Integer.MAX_VALUE]が返された場合情報は存在しません.
     */
    public final int getRandomID( int id )
    {
        int ret = Integer.MAX_VALUE ;
        UdpSndResultChild ch = null ;
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_resID.isData( id ) == true ){
                    
                    if( ( ch = ( UdpSndResultChild )m_table.get( id ) ) != null ){
                        ret = ch.getRandomID() ;
                    }
                    
                }
                
            }
        }catch( Exception e ){
            ret = Integer.MAX_VALUE ;
        }finally{
            ch = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象IDに対するサブID群を取得.
     * <BR><BR>
     * 対象IDに対するサブID群を取得します.
     * <BR>
     * @param id 対象のID情報を取得します.
     * @return int[] 対象IDに対するサブID群が返されます.
     */
    public final int[] subIDs( int id )
    {
        int[] ret = null ;
        UdpSndResultChild ch = null ;
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_resID.isData( id ) == true ){
                    
                    if( ( ch = ( UdpSndResultChild )m_table.get( id ) ) != null ){
                        ret = ch.subIDs() ;
                    }
                    
                }
                
            }
        }catch( Exception e ){
            ret = null ;
        }finally{
            ch = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象IDに対するサブID数を取得.
     * <BR><BR>
     * 対象IDに対するサブID数を取得します.
     * <BR>
     * @param id 対象のID情報を取得します.
     * @return int 対象IDに対するサブID数が返されます.
     */
    public final int size( int id )
    {
        int ret = 0 ;
        UdpSndResultChild ch = null ;
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_resID.isData( id ) == true ){
                    
                    if( ( ch = ( UdpSndResultChild )m_table.get( id ) ) != null ){
                        ret = ch.size() ;
                    }
                    
                }
                
            }
        }catch( Exception e ){
            ret = 0 ;
        }finally{
            ch = null ;
        }
        
        return ret ;
    }
    
    /**
     * リソースタイプを取得.
     * <BR><BR>
     * リソースタイプを取得します.
     * <BR>
     * @return ResourceType 設定されているリソースタイプが返されます.
     */
    public final ResourceType getResourceType()
    {
        ResourceType ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_resType ;
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象条件が存在するかチェック.
     * <BR><BR>
     * 対象条件が存在するかチェックします.
     * <BR>
     * @param id 取得対象のIDを設定します.
     * @param subID 取得対象のサブIDを設定します.
     * @param rndID 取得対象のランダムIDを設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、情報は存在します.<BR>
     *                 [false]が返された場合、情報は存在しません.
     */
    public final boolean isData( int id,int subID,int rndID )
    {
        boolean ret = false ;
        UdpSndResultChild ch = null ;
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_resID.isData( id ) == true ){
                    
                    if( ( ch = ( UdpSndResultChild )m_table.get( id ) ) != null ){
                        ret = ch.isData( subID,rndID ) ;
                    }
                    
                }
                
            }
        }catch( Exception e ){
            ret = false ;
        }finally{
            ch = null ;
        }
        
        return ret ;
    }
    
}

/**
 * 要素格納データ.
 */
class UdpSndResultChild
{
    
    /**
     * 格納テーブル.
     */
    private NumberTable m_table = null ;
    
    /**
     * ランダムID.
     */
    private int m_randID = Integer.MAX_VALUE ;
    
    /**
     * コンストラクタ.
     */
    public UdpSndResultChild(){
        m_table = new NumberTable() ;
    }
    
    /**
     * 情報クリア.
     */
    public final void clear()
    {
        if( m_table != null ){
            m_table.clear() ;
        }
        
        m_table = null ;
        m_randID = Integer.MAX_VALUE ;
    }
    
    /**
     * 要素追加.
     */
    public final void add( int sub,int rnd,byte[] binary )
    {
        if( binary == null || binary.length <= 0 || m_table == null ){
            return ;
        }
        else if( m_randID == Integer.MAX_VALUE ){
            m_randID = rnd ;
        }
        else if( m_randID != rnd ){
            return ;
        }
        
        m_table.add( sub,binary ) ;
    }
    
    /**
     * 情報取得.
     */
    public final byte[] get( int sub,int rnd )
    {
        if( m_table == null ){
            return null ;
        }
        else if( m_randID != rnd ){
            return null ;
        }
        
        return ( byte[] )m_table.remove( sub ) ;
        
    }
    
    /**
     * 格納されているランダムIDを取得.
     */
    public final int getRandomID()
    {
        return m_randID ;
    }
    
    /**
     * 格納されているサブIDを取得.
     */
    public final int[] subIDs()
    {
        if( m_table == null ){
            return null ;
        }
        
        return m_table.getNumbers() ;
    }
    
    /**
     * 格納されているサブID数を取得.
     */
    public final int size()
    {
        if( m_table == null ){
            return 0 ;
        }
        
        return m_table.size() ;
    }
    
    /**
     * 存在チェック.
     */
    public final boolean isData( int sub,int rnd )
    {
        if(  m_table == null ){
            return false ;
        }
        else if( m_randID != rnd ){
            return false ;
        }
        
        return m_table.isData( sub ) ;
    }
}

