/*
 * @(#)UdpRcvConnectThread.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.net.uio;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.JRcServer.commons.exception.BaseException;
import com.JRcServer.commons.exception.ExecutionException;
import com.JRcServer.commons.exception.ExistException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.net.ConnectAddress;
import com.JRcServer.commons.net.ConnectTimeoutException;
import com.JRcServer.commons.net.TelegramResourceRoll;
import com.JRcServer.commons.net.UdpProtocol;
import com.JRcServer.commons.thread.ExecutionThread;
import com.JRcServer.commons.thread.LoopThread;
import com.JRcServer.commons.thread.Synchronized;
import com.JRcServer.commons.util.ByteUtil;

/**
 * UDP受信コネクション情報取得スレッド.
 * <BR><BR>
 * UDPプロトコルの受信処理をスレッドで行います.
 *
 * @version 1.00, 2005/03/24
 * @author  Masahito Suzuki
 * @since   JRcCommons 1.00
 */
class UdpRcvConnectThread extends ExecutionThread
{
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( UdpRcvConnectThread.class ) ;
    
    /**
     * UDP受信タイムアウト値.
     */
    private static final int UDP_RECV_TIMEOUT = 500 ;
    
    /**
     * UDPプロトコル.
     */
    private UdpProtocol m_udp = null ;
    
    /**
     * UDP受信コネクションテーブル.
     */
    private UdpRcvConnectTable m_table = null ;
    
    /**
     * 受信完了データ格納テーブル.
     */
    private TelegramResourceRoll m_roll = null ;
    
    /**
     * ループスレッド.
     */
    private final LoopThread m_thread = new LoopThread() ;
    
    /**
     * 同期処理.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    /**
     * 電文受信用同期処理.
     */
    private Synchronized m_rcvSync = null ;
    
    /**
     * コネクションデータ格納用.
     */
    private final ConnectAddress m_tconn = new ConnectAddress() ;
    
    /**
     * 戻り電文テンポラリ領域.
     */
    private final ByteUtil m_tbuf = new ByteUtil() ;
    
    
    
    /**
     * コンストラクタ.
     */
    private UdpRcvConnectThread(){}
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 基本条件を渡してスレッド実行します.
     * <BR>
     * @param udp 受信処理を行うUDPプロトコルを設定します.
     * @param tbl 受信結果を格納するテーブルを設定します.
     * @param roll 対象のロールオブジェクトを設定します.
     * @param rcvSync 受信処理用同期オブジェクトを設定します.
     * @exception InputException 入力例外.
     */
    public UdpRcvConnectThread( UdpProtocol udp,UdpRcvConnectTable tbl,TelegramResourceRoll roll,Synchronized rcvSync )
        throws InputException
    {
        if(
            udp == null || udp.isOpen() == false ||
            roll == null || tbl == null || rcvSync == null ||
            rcvSync.isUse() == false
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.destroy() ;
        m_sync.create() ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                m_udp = udp ;
                m_table = tbl ;
                m_roll = roll ;
                m_rcvSync = rcvSync ;
                
            }
            
            m_thread.create( this ) ;
            m_thread.startThread() ;
            
        }catch( Exception e ){
            this.destroy() ;
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public final void destroy()
    {
        m_sync.clear() ;
        m_thread.clear() ;
        m_tconn.clear() ;
        m_tbuf.clear() ;
        
        m_udp = null ;
        m_table = null ;
        m_roll = null ;
        m_rcvSync = null ;
        
    }
    
    /**
     * スレッド状態を取得.
     * <BR><BR>
     * スレッド状態を取得します.
     * <BR>
     * @return boolean スレッド状態が返されます.<BR>
     *                 [true]が返された場合、スレッドは実行中です.<BR>
     *                 [false]が返された場合、スレッドは停止中です.
     */
    public final boolean isThread()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_thread.isThread() ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * 実行初期化処理をサポートします.
     * <BR><BR>
     * 実行初期化処理をサポートします.<BR>
     * この処理は、スレッド処理が開始された時に呼び出されます.
     * <BR>
     * @param obj 実行開始時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void init( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行終了化処理をサポートします.
     * <BR><BR>
     * 実行終了化処理をサポートします.<BR>
     * この処理は、スレッド処理が終了された時に呼び出されます.
     * <BR>
     * @param obj 実行終了時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void exit( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * ストップ処理をサポートします。
     * <BR><BR>
     * ストップ処理をサポートします。<BR>
     * この処理は、スレッドでのストップ処理に対して呼び出し実行されます.
     * <BR>
     * @param obj ストップ時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void stop( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行処理をサポートします。
     * <BR><BR>
     * 実行処理をサポートします。<BR>
     * この処理は、スレッドでの実行処理に対して呼び出し実行されます.
     * <BR>
     * @param obj 実行時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void execution( Object obj )
        throws ExecutionException
    {
        int id ;
        int subID ;
        int rndID ;
        int max ;
        int packetLen ;
        int length ;
        int hdType ;
        
        byte[] rcv = null ;
        UdpProtocol udp = null ;
        UdpRcvConnectTable tbl = null ;
        TelegramResourceRoll roll = null ;
        Synchronized rcvSync = null ;
        ConnectAddress tconn = null ;
        ByteUtil tbuf = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                udp = m_udp ;
                tbl = m_table ;
                roll = m_roll ;
                rcvSync = m_rcvSync ;
                tconn = m_tconn ;
                tbuf = m_tbuf ;
            }
            
            // デフォルト設定.
            hdType = UdpIODef.NOT_STATE_CODE ;
            id = -1 ;
            rndID = -1 ;
            
            try{
                
                // 受信処理.
                rcv = udp.receive( tconn,UdpRcvConnectThread.UDP_RECV_TIMEOUT ) ;
                
                // 対象パケットがタイプである場合.
                if( UdpIOCommon.isPacket( rcv ) == true ){
                    
                    // パケットから、ID,SubID,randID を取得.
                    id = UdpIOCommon.getPacketToID( rcv ) ;
                    subID = UdpIOCommon.getPacketToSubID( rcv ) ;
                    rndID = UdpIOCommon.getPacketToRandID( rcv ) ;
                    
                    /////////////////////
                    // ヘッダ電文の場合.
                    /////////////////////
                    if( subID == UdpIODef.PACKET_HEADER_SUBID ){
                        
                        // ヘッダタイプを取得.
                        hdType = UdpIOCommon.getPacketHeaderToCode( rcv ) ;
                        
                        ////////////////////////////
                        // 送信電文開始ヘッダの場合.
                        ////////////////////////////
                        if( hdType == UdpIODef.FIRST_STATE_CODE ){
                            
                            // 最大パケット長、最大電文長を取得.
                            max = UdpIOCommon.getPacketHeaderToMaxPacket( rcv ) ;
                            length = UdpIOCommon.getPacketHeaderToTelegramLength( rcv ) ;
                            
                            // 対象ロール条件が追加可能かチェック.
                            if( roll.isAdd() == false ){
                                
                                //<><><><><><><><><><><><><><><><><>
                                // 対象ロール条件が追加不可能な場合.
                                //<><><><><><><><><><><><><><><><><>
                                
                                // ロールフルステータスコードを送信.
                                UdpRcvConnectThread.sendPacketState(
                                    tbuf,tconn,udp,id,max,rndID,length,-1,UdpIODef.ROLL_FULL_STATE_CODE
                                ) ;
                                
                            }
                            // 確認電文の場合.
                            else if( max == UdpIODef.PACKET_CCONN_MAX_PACKET ){
                                
                                // 確認電文の場合、lengthに対して受信中の数をセット.
                                length = tbl.size() ;
                                
                                // 通知確認OKの電文を返信.
                                UdpRcvConnectThread.sendPacketState(
                                    tbuf,tconn,udp,id,max,rndID,length,-1,
                                    UdpIODef.FIRST_NOTICE_OK_STATE_CODE
                                ) ;
                                
                            }
                            // 対象ロール条件が追加可能な場合.
                            else{
                                
                                //<><><><><><><><><><><><><><><><><><><><>
                                // 受信可能にするためにパケット情報を予約.
                                //<><><><><><><><><><><><><><><><><><><><>
                                
                                // １パケット長を取得.
                                packetLen = UdpIOCommon.getPacketHeaderToOnePacket( rcv ) ;
                                
                                // パケット受信をテーブルに予約.
                                tbl.reservation( id,rndID,max,packetLen,length,tconn.getAddress(),tconn.getPort() ) ;
                                
                                // パケットデータをテーブルに追加.
                                tbl.put( tconn.getAddress(),tconn.getPort(),id,subID,rndID,rcv ) ;
                                
                                // 正常開始の電文を返信.
                                UdpRcvConnectThread.sendPacketState(
                                    tbuf,tconn,udp,id,max,rndID,length,-1,UdpIODef.FIRST_SUCCESS_STATE_CODE
                                ) ;
                                
                            }
                            
                        }
                        ///////////////////////////////////////
                        // 送信電文開始以外のヘッダステータス.
                        ///////////////////////////////////////
                        else if( ( hdType & UdpIODef.STATE_CHECK ) == UdpIODef.STATE_CHECK ){
                            
                            // パケットデータをテーブルに追加.
                            tbl.put( tconn.getAddress(),tconn.getPort(),id,subID,rndID,rcv ) ;
                            
                        }
                        
                    }
                    /////////////////////////
                    // ヘッダ電文で無い場合.
                    /////////////////////////
                    else{
                        
                        // 対象パケット条件がテーブルに存在する場合.
                        if( tbl.isData( id,rndID ) == true ){
                            
                            // 対象サブIDが範囲内の場合.
                            if( subID >= 0 && subID < tbl.getMaxPacket( id,rndID ) ){
                                
                                // パケットデータをテーブルに追加.
                                tbl.put( tconn.getAddress(),tconn.getPort(),id,subID,rndID,rcv ) ;
                                
                            }
                            
                        }
                        
                    }
                    
                }
                else{
                    rcv = null ;
                }
                
            }catch( ConnectTimeoutException et ){
            }catch( ExistException ee ){
                
                //<><><><><><><><><><><><><><><><><><><><><><><>
                // 送信電文開始ヘッダのIDランダムIDが既に他から
                // 設定されているとき、エラーを送信する.
                //<><><><><><><><><><><><><><><><><><><><><><><>
                
                // 最大パケット長、最大電文長を取得.
                max = UdpIOCommon.getPacketHeaderToMaxPacket( rcv ) ;
                length = UdpIOCommon.getPacketHeaderToTelegramLength( rcv ) ;
                
                // 送信電文開始ヘッダの場合.
                if( hdType == UdpIODef.FIRST_STATE_CODE && id != -1 ){
                    
                    // 開始エラーを送信.
                    UdpRcvConnectThread.sendPacketState(
                        tbuf,tconn,udp,id,max,rndID,length,-1,UdpIODef.FIRST_ERROR_STATE_CODE
                    ) ;
                    
                }
                // 送信電文開始以外のヘッダ情報の場合.
                else if( ( hdType & UdpIODef.STATE_CHECK ) == UdpIODef.STATE_CHECK && id != -1 ){
                    
                    // エラーステータスコードを送信.
                    UdpRcvConnectThread.sendPacketState(
                        tbuf,tconn,udp,id,max,rndID,length,-1,UdpIODef.ERROR_STATE_CODE
                    ) ;
                    
                }
                
            }catch( InputException e ){
            }finally{
                rcv = null ;
            }
            
        }catch( NullPointerException nul ){
            throw new ExecutionException(
                nul,ExecutionException.LEVEL_STOP
            ) ;
        }catch( BaseException be ){
            LOG.error( "エラーが発生しました", be ) ;
        }catch( Exception e ){
            LOG.error( "エラーが発生しました", e ) ;
        }finally{
            
            rcv = null ;
            udp = null ;
            tbl = null ;
            roll = null ;
            rcvSync = null ;
            tconn = null ;
            tbuf = null ;
            
        }
        
    }
    
    /**
     * パケットステータスを送信.
     */
    private static final void sendPacketState( ByteUtil tbuf,ConnectAddress addr,UdpProtocol udp,int id,int max,int rndID,int length,int packetLen,int state )
    {
        byte[] snd = null ;
        
        try{
            
            // パケットヘッダを送信.
            UdpIOCommon.createPacketHeader( tbuf,id,max,rndID,length,packetLen,state ) ;
            snd = tbuf.getToClear() ;
            udp.send( snd,addr ) ;
            snd = null ;
            
        }catch( Exception e ){
        }finally{
            snd = null ;
        }
    }
    
}

