/*
 * @(#)UdpRcvConnectTable.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.net.uio;

import java.net.InetAddress;
import java.util.Arrays;

import com.JRcServer.commons.exception.ExecutionException;
import com.JRcServer.commons.exception.ExistException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.net.ConnectAddress;
import com.JRcServer.commons.resource.BinResource;
import com.JRcServer.commons.resource.Resource;
import com.JRcServer.commons.resource.ResourceType;
import com.JRcServer.commons.thread.Synchronized;
import com.JRcServer.commons.util.BigTable;
import com.JRcServer.commons.util.SearchInt;
import com.JRcServer.commons.util.UtilCom;
import com.JRcServer.commons.util.array.IntArray;
import com.JRcServer.commons.util.array.LongArray;

/**
 * UDP受信データ格納テーブル.
 *
 * @version 1.00, 2005/03/24
 * @author  Masahito Suzuki
 * @since   JRcCommons 1.00
 */
class UdpRcvConnectTable
{
    
    /**
     * 受信電文格納テーブル.
     * ID+RandomIDをキーとする.
     * key =
     * (long)((long)ID & 0x00000000ffffffffL ) |
     * ( ((long)RandomID & 0x00000000ffffffffL) << 32L ) ;
     */
    private final BigTable m_connTable = new BigTable() ;
    
    /**
     * 受信処理実行ID発行.
     */
    private final LongArray m_rcvGetID = new LongArray() ;
    
    /**
     * リソースタイプ.
     */
    private ResourceType m_resType = null ;
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    
    /**
     * コンストラクタ.
     */
    public UdpRcvConnectTable(){
        this.clear() ;
    }
    
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 対象テーブルを生成します.
     */
    public final void create()
    {
        this.create( null ) ;
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 対象テーブルを生成します.
     * <BR>
     * @param resType 対象のリソースタイプを設定します.
     */
    public final void create( ResourceType resType )
    {
        this.clear() ;
        m_sync.create() ;
        
        m_resType = ( resType == null ) ? UdpIODef.DEF_RESOURCE_TYPE : resType ;
    }
    
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public final void clear()
    {
        m_connTable.clear() ;
        m_rcvGetID.clear() ;
        m_sync.clear() ;
    }
    
    /**
     * 受信受理を予約.
     * <BR><BR>
     * 受信受理を予約します.<BR>
     * この処理を呼び出さない限り、対象の通信情報を受信出来ません.
     * <BR>
     * @param id 対象の受信受理IDを設定します.
     * @param randID 対象の受信受理ランダムIDを設定します.
     * @param max 受信パケット長を設定します.
     * @param packetLen 対象パケット長を設定します.
     * @param length 受信電文長を設定します.
     * @param addr 対象の受信先IPアドレスを設定します.
     * @param port 対象の受信先ポート番号を設定します.
     * @exception InputException 入力例外.
     * @exception ExistException 存在例外.
     */
    public final void reservation( int id,int rndID,int max,int packetLen,int length,InetAddress addr,int port )
        throws InputException,ExistException
    {
        long key ;
        UdpRcvConnectChild ch = null ;
        
        if( id < 0 || max <= 0 || packetLen <= 0 || length < 0 || addr == null || port < 0 || port > 65535 ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        // IDとランダムIDから、キーを生成.
        key = UdpRcvConnectTable.convertIDToRndIDByLong( id,rndID ) ;
        
        try{
            synchronized( m_sync.get() ){
                
                // 条件が既に存在する場合.
                if( m_connTable.isData( key ) == true ){
                    
                    // 存在条件を取得.
                    ch = this.getChild( key ) ;
                    
                    // 存在条件が今回設定する条件と一致しない場合.
                    if(
                        addr.equals( ch.getAddress() ) == false ||
                        port != ch.getPort()
                    )
                    {
                        
                        // エラーとする.
                        throw new ExistException(
                            "対象キー(id:" + id +
                            " randID:" + rndID +
                            ")は既に存在します"
                        ) ;
                        
                    }
                    
                }
                // 条件が存在しない場合.
                else{
                    
                    // 条件を新規に生成.
                    ch = new UdpRcvConnectChild() ;
                    ch.create( max,packetLen,length,addr,port,m_resType ) ;
                    
                    // 新規条件を追加.
                    m_connTable.add( key,ch ) ;
                    
                    // 新規の場合、受信開始IDを設定.
                    m_rcvGetID.add( key ) ;
                    
                }
                
            }
        }catch( NullPointerException nul ){
            this.cancell( id,rndID ) ;
        }catch( ExistException ex ){
            throw ex ;
        }catch( InputException in ){
            this.cancell( id,rndID ) ;
            throw in ;
        }finally{
            ch = null ;
        }
        
    }
    
    /**
     * 受信受理を破棄.
     * <BR><BR>
     * 対象の受信受理を破棄します.<BR>
     * 受信処理が完了した場合に呼び出す必要があります.<BR>
     * また、テーブルに格納されている全ての条件も削除されます.
     * <BR>
     * @param id 破棄対象の受信受理IDを設定します.
     * @param randID 破棄対象の受信受理ランダムIDを設定します.
     */
    public final void cancell( int id,int rndID )
    {
        long key ;
        UdpRcvConnectChild ch = null ;
        
        // IDとランダムIDから、キーを生成.
        key = UdpRcvConnectTable.convertIDToRndIDByLong( id,rndID ) ;
        
        try{
            synchronized( m_sync.get() ){
                
                // 削除対象条件を取得.
                ch = this.getChild( key ) ;
                
                // 削除条件が存在する場合、全てクリア.
                if( ch != null ){
                    ch.clear() ;
                    m_connTable.remove( key ) ;
                }
                
            }
        }catch( Exception e ){
        }finally{
            ch = null ;
        }
    }
    
    /**
     * 受信開始IDを取得.
     * <BR><BR>
     * 受信開始IDを取得します.<BR>
     * この処理は、受信処理が受理された場合にテーブルにセットされ、
     * このメソッドで取得した場合、その条件は破棄されます.
     * <BR>
     * これにより、受信電文に対する取得処理を判別する事が可能で、
     * 受信処理スレッドがこの処理を呼び出します.
     * <BR>
     * @return long 受信開始IDが返されます.<BR>
     *              この内容をIDとランダムIDに区別する場合、メソッド
     *              [UdpIOCommon#getReceiveID( long )]で、パケットID情報を取得し、
     *              [UdpIOCommon#getReceiveRandomID( long )]で、パケットランダムID情報を
     *              取得します.<BR>
     *              また、[-1L]が返された場合、情報は存在しません.
     */
    public final long getReceiveStartID()
    {
        long ret = -1L ;
        
        try{
            synchronized( m_sync.get() ){
                if( m_rcvGetID.size() > 0 ){
                    ret = m_rcvGetID.remove( 0 ) ;
                }
            }
        }catch( Exception e ){
            ret = -1L ;
        }
        
        return ret ;
    }
    
    /**
     * 受信パケット電文を設定.
     * <BR><BR>
     * 受信されたパケット電文を設定します.
     * <BR>
     * @param addr 対象の受信先IPアドレスを設定します.
     * @param port 対象の受信先ポート番号を設定します.
     * @param id 対象のID情報を設定します.
     * @param subID 対象のサブIDを設定します.
     * @param rndID 対象のランダムIDを設定します.
     * @param packet 対象のパケット電文を設定します.
     * @exception InputException 入力例外.
     * @exception ExistException 存在例外.
     */
    public final void put( InetAddress addr,int port,int id,int subID,int rndID,byte[] packet )
        throws InputException,ExistException
    {
        long key ;
        
        UdpRcvConnectChild ch = null ;
        
        if( addr == null || port < 0 || port > 65535 || packet == null || packet.length <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        // テーブルキーを取得.
        key = UdpRcvConnectTable.convertIDToRndIDByLong( id,rndID ) ;
        
        try{
            synchronized( m_sync.get() ){
                
                // ID存在チェック.
                if( ( ch = this.getChild( key ) ) == null ){
                    throw new InputException(
                        "対象条件(id:" + id +
                        " randID:" + rndID +
                        ")は存在しません"
                    ) ;
                }
                
                // 条件追加.
                ch.put( addr,port,subID,packet ) ;
                
            }
        }catch( InputException in ){
            throw in ;
        }catch( ExistException ex ){
            throw ex ;
        }finally{
            ch = null ;
        }
        
    }
    
    /**
     * 受信パケットヘッダ電文を取得.
     * <BR><BR>
     * 受信されたパケットヘッダ電文を取得します.<BR>
     * また、パケットヘッダは取得後削除されます.
     * <BR>
     * @param id 取得対象のIDを設定します.
     * @param rndID 取得対象のランダムIDを設定します.
     * @return byte[] 対象のバイナリ情報が返されます.
     */
    public final byte[] getHeader( int id,int rndID )
    {
        UdpRcvConnectChild ch = null ;
        byte[] ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ch = this.getChild( id,rndID ) ;
                ret = ch.removeHeader() ;
            }
        }catch( Exception e ){
            ret = null ;
        }finally{
            ch = null ;
        }
        
        return ret ;
    }
    
    /**
     * 受信パケットヘッダ電文を取得.
     * <BR><BR>
     * 受信されたパケットヘッダ電文を取得します.<BR>
     * この処理はタイムアウトを設けて情報を取得します.<BR>
     * また、パケットヘッダは取得後削除されます.
     * <BR>
     * @param id 取得対象のIDを設定します.
     * @param rndID 取得対象のランダムIDを設定します.
     * @param timeout 取得タイムアウトを設定します.
     * @return byte[] 対象のバイナリ情報が返されます.
     * @exception ExecutionException 実行例外.
     */
    public final byte[] getHeader( int id,int rndID,int timeout )
        throws ExecutionException
    {
        long now ;
        byte[] ret = null ;
        
        timeout = ( timeout <= 0 ) ? 0 : timeout ;
        
        try{
            
            if( timeout == 0 ){
                ret = this.getHeader( id,rndID ) ;
            }
            else{
                
                now = System.currentTimeMillis() + timeout ;
                
                for( ;; ){
                    
                    ret = this.getHeader( id,rndID ) ;
                    
                    if( ret != null || now <= System.currentTimeMillis() ){
                        break ;
                    }
                    
                    UtilCom.idleTime() ;
                    
                }
                
            }
            
        }catch( ExecutionException ee ){
            ret = null ;
            throw ee ;
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象受信電文を取得.
     * <BR><BR>
     * 対象受信電文を取得します.
     * <BR>
     * @param id 取得対象のIDを設定します.
     * @param rndID 取得対象のランダムIDを設定します.
     * @return byte[] 受信電文が返されます.<BR>
     *                情報が存在しない場合[null]が返されます.
     */
    public final byte[] getTelegram( int id,int rndID )
    {
        UdpRcvConnectChild ch = null ;
        byte[] ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ch = this.getChild( id,rndID ) ;
                ret = ch.getTelegram() ;
            }
        }catch( Exception e ){
            ret = null ;
        }finally{
            ch = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象受信電文を取得.
     * <BR><BR>
     * 対象受信電文を取得します.
     * <BR>
     * @param id 取得対象のIDを設定します.
     * @param rndID 取得対象のランダムIDを設定します.
     * @return BinResource 受信電文が返されます.<BR>
     *                     情報が存在しない場合[null]が返されます.
     */
    public final BinResource getTelegramByResource( int id,int rndID )
    {
        UdpRcvConnectChild ch = null ;
        BinResource ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ch = this.getChild( id,rndID ) ;
                ret = ch.getTelegramByResource() ;
            }
        }catch( Exception e ){
            if( ret != null ){
                ret.clear() ;
            }
            ret = null ;
        }finally{
            ch = null ;
        }
        
        return ret ;
    }
    
    /**
     * ロストパケット項番を取得.
     * <BR><BR>
     * ロストパケット項番を取得します.
     * <BR>
     * @param id 取得対象のIDを設定します.
     * @param rndID 取得対象のランダムIDを設定します.
     * @return IntArray ロストパケット項番が返されます.<BR>
     *                  [null]が返された場合情報は存在しません.
     */
    public final IntArray getLostPacket( int id,int rndID )
    {
        UdpRcvConnectChild ch = null ;
        IntArray ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ch = this.getChild( id,rndID ) ;
                ret = ch.getLostPacketID() ;
            }
        }catch( Exception e ){
            ret = null ;
        }finally{
            ch = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象格納テーブル(ID+ランダムID)群を取得.
     * <BR><BR>
     * 対象格納テーブル(ID+ランダムID)群を取得します.
     * <BR>
     * @return long[] 現在格納されている格納テーブル(ID+ランダムID)群が返されます.<BR>
     *                情報が存在しない場合[null]が返されます.
     */
    public final long[] getNumbers()
    {
        long[] ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_connTable.getNumbers() ;
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象格納テーブル(ID+ランダムID)数を取得.
     * <BR><BR>
     * 対象格納テーブル(ID+ランダムID)数を取得します.
     * <BR>
     * @return int 現在格納されている格納テーブル(ID+ランダムID)数が返されます.
     */
    public final int size()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_connTable.size() ;
            }
        }catch( Exception e ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 受信パケット項番群を取得.
     * <BR><BR>
     * 受信パケット項番群を取得します.
     * <BR>
     * @param id 取得対象のIDを設定します.
     * @param rndID 取得対象のランダムIDを設定します.
     * @return int[] 受信パケット項番が返されます.<BR>
     *               受信パケットが存在しない場合[null]が返されます.
     */
    public final int[] subIDs( int id,int rndID )
    {
        UdpRcvConnectChild ch = null ;
        int[] ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ch = this.getChild( id,rndID ) ;
                ret = ch.subIDs() ;
            }
        }catch( Exception e ){
            ret = null ;
        }finally{
            ch = null ;
        }
        
        return ret ;
    }
    
    /**
     * 受信パケット数を取得.
     * <BR><BR>
     * 受信パケット数を取得します.
     * <BR>
     * @param id 取得対象のIDを設定します.
     * @param rndID 取得対象のランダムIDを設定します.
     * @return int 現在受信されているパケット数が返されます.
     */
    public final int subIDSize( int id,int rndID )
    {
        int ret ;
        
        UdpRcvConnectChild ch = null ;
        
        try{
            synchronized( m_sync.get() ){
                ch = this.getChild( id,rndID ) ;
                ret = ch.size() ;
            }
        }catch( Exception e ){
            ret = 0 ;
        }finally{
            ch = null ;
        }
        
        return ret ;
    }
    
    /**
     * 受信アドレスを取得.
     * <BR><BR>
     * 受信元のアドレス情報を取得します.
     * <BR>
     * @param id 取得対象のIDを設定します.
     * @param rndID 取得対象のランダムIDを設定します.
     * @return InetAddress 受信元のアドレス情報が返されます.<BR>
     *                     情報が存在しない場合[null]が返されます.
     */
    public final InetAddress getAddress( int id,int rndID )
    {
        UdpRcvConnectChild ch = null ;
        InetAddress ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ch = this.getChild( id,rndID ) ;
                ret = ch.getAddress() ;
            }
        }catch( Exception e ){
            ret = null ;
        }finally{
            ch = null ;
        }
        
        return ret ;
    }
    
    /**
     * 受信ポート番号を取得.
     * <BR><BR>
     * 受信元のポート番号を取得します.
     * <BR>
     * @param id 取得対象のIDを設定します.
     * @param rndID 取得対象のランダムIDを設定します.
     * @return int 受信元のポート番号が返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public final int getPort( int id,int rndID )
    {
        int ret ;
        
        UdpRcvConnectChild ch = null ;
        
        try{
            synchronized( m_sync.get() ){
                ch = this.getChild( id,rndID ) ;
                ret = ch.getPort() ;
            }
        }catch( Exception e ){
            ret = -1 ;
        }finally{
            ch = null ;
        }
        
        return ret ;
    }
    
    /**
     * 受信アドレス+ポート番号を取得.
     * <BR><BR>
     * 受信元のアドレス+ポート番号を取得します.
     * <BR>
     * @param id 取得対象のIDを設定します.
     * @param rndID 取得対象のランダムIDを設定します.
     * @return ConnectAddress 対象のアドレス+ポート番号が返されます.<BR>
     *                        情報が存在しない場合[null]が返されます.
     */
    public final ConnectAddress getConnect( int id,int rndID )
    {
        UdpRcvConnectChild ch = null ;
        ConnectAddress ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ch = this.getChild( id,rndID ) ;
                ret = ch.getConnect() ;
            }
        }catch( Exception e ){
            ret = null ;
        }finally{
            ch = null ;
        }
        
        return ret ;
    }
    
    /**
     * 最大パケット数を取得.
     * <BR><BR>
     * 受信に対して必要なパケット数が返されます.
     * <BR>
     * @param id 取得対象のIDを設定します.
     * @param rndID 取得対象のランダムIDを設定します.
     * @return int 最大パケット数が返されます.
     */
    public final int getMaxPacket( int id,int rndID )
    {
        int ret ;
        
        UdpRcvConnectChild ch = null ;
        
        try{
            synchronized( m_sync.get() ){
                ch = this.getChild( id,rndID ) ;
                ret = ch.getMaxPacket() ;
            }
        }catch( Exception e ){
            ret = 0 ;
        }finally{
            ch = null ;
        }
        
        return ret ;
    }
    
    /**
     * 最大電文長を取得.
     * <BR><BR>
     * 受信電文で必要な電文長が返されます.
     * <BR>
     * @param id 取得対象のIDを設定します.
     * @param rndID 取得対象のランダムIDを設定します.
     * @return int 最大電文長が返されます.
     */
    public final int getMaxTelegramLength( int id,int rndID )
    {
        int ret ;
        
        UdpRcvConnectChild ch = null ;
        
        try{
            synchronized( m_sync.get() ){
                ch = this.getChild( id,rndID ) ;
                ret = ch.getMaxLength() ;
            }
        }catch( Exception e ){
            ret = 0 ;
        }finally{
            ch = null ;
        }
        
        return ret ;
    }
    
    /**
     * 最終書き込み時間を取得.
     * <BR><BR>
     * 最終書き込み時間が返されます.
     * <BR>
     * @param id 取得対象のIDを設定します.
     * @param rndID 取得対象のランダムIDを設定します.
     * @return long 最終書き込み時間が返されます.<BR>
     *              情報が存在しない場合[-1L]が返されます.
     */
    public final long getLastTime( int id,int rndID )
    {
        long ret ;
        
        UdpRcvConnectChild ch = null ;
        
        try{
            synchronized( m_sync.get() ){
                ch = this.getChild( id,rndID ) ;
                ret = ch.getLastUpdate() ;
            }
        }catch( Exception e ){
            ret = -1L ;
        }finally{
            ch = null ;
        }
        
        return ret ;
    }
    
    /**
     * リソースタイプを取得.
     * <BR><BR>
     * リソースタイプを取得します.
     * <BR>
     * @return ResourceType 設定されているリソースタイプが返されます.
     */
    public final ResourceType getResourceType()
    {
        ResourceType ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_resType ;
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * ロストパケット存在チェック.
     * <BR><BR>
     * ロストパケットが存在するかチェックします.
     * <BR>
     * @param id 対象のID情報を設定します.
     * @param rndID 対象のランダムIDを設定します.
     * @return boolean ロストパケットが存在するかチェックします.<BR>
     *                 [true]が返された場合、ロストパケットが存在します.<BR>
     *                 [false]が返された場合、ロストパケットが存在しません.
     */
    public final boolean isLost( int id,int rndID )
    {
        boolean ret ;
        
        UdpRcvConnectChild ch = null ;
        
        try{
            synchronized( m_sync.get() ){
                ch = this.getChild( id,rndID ) ;
                ret = ch.isLost() ;
            }
        }catch( Exception e ){
            ret = false ;
        }finally{
            ch = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象情報存在チェック.
     * <BR><BR>
     * 対象の情報が存在するかチェックします.
     * <BR>
     * @param id 対象のID情報を設定します.
     * @param rndID 対象のランダムIDを設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合情報は存在します.<BR>
     *                 [false]が返された場合情報は存在しません.
     */
    public final boolean isData( int id,int rndID )
    {
        long key ;
        boolean ret ;
        
        key = UdpRcvConnectTable.convertIDToRndIDByLong( id,rndID ) ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_connTable.isData( key ) ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 対象情報存在チェック.
     * <BR><BR>
     * 対象の情報が存在するかチェックします.
     * <BR>
     * @param id 対象のID情報を設定します.
     * @param subID 対象のサブIDを設定します.
     * @param rndID 対象のランダムIDを設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合情報は存在します.<BR>
     *                 [false]が返された場合情報は存在しません.
     */
    public final boolean isData( int id,int subID,int rndID )
    {
        boolean ret ;
        
        UdpRcvConnectChild ch = null ;
        
        try{
            synchronized( m_sync.get() ){
                ch = this.getChild( id,rndID ) ;
                ret = ch.isData( subID ) ;
            }
        }catch( Exception e ){
            ret = false ;
        }finally{
            ch = null ;
        }
        
        return ret ;
    }
    
    /**
     * 要素格納データを取得.
     */
    private final UdpRcvConnectChild getChild( long key )
    {
        UdpRcvConnectChild ret = null ;
        
        ret = ( UdpRcvConnectChild )m_connTable.get( key ) ;
        
        return ret ;
    }
    
    /**
     * 要素格納データを取得.
     */
    private final UdpRcvConnectChild getChild( int id,int rndID )
    {
        long key ;
        UdpRcvConnectChild ret = null ;
        
        key = UdpRcvConnectTable.convertIDToRndIDByLong( id,rndID ) ;
        ret = ( UdpRcvConnectChild )m_connTable.get( key ) ;
        
        return ret ;
    }
    
    /**
     * IDとrndIDをlongに変換.
     */
    private static final long convertIDToRndIDByLong( int id,int rndID )
    {
        return ( long )(
            ( ( long )id & 0x00000000ffffffffL ) |
            ( ( ( long )rndID & 0x00000000ffffffffL ) << 32L )
        ) ;
    }
    
}

/**
 * 要素格納データ.
 */
class UdpRcvConnectChild
{
    
    /**
     * ID管理.
     */
    private SearchInt m_subIds = null ;
    
    /**
     * パケットデータ管理オブジェクト.
     */
    private BinResource m_packets = null ;
    
    /**
     * 受信コネクション先.
     */
    private final ConnectAddress m_conn = new ConnectAddress() ;
    
    /**
     * 受信パケット長.
     */
    private int m_maxPacket = -1 ;
    
    /**
     * 受信電文長.
     */
    private int m_length = -1 ;
    
    /**
     * １パケット長.
     */
    private int m_packetLen = -1 ;
    
    /**
     * ラストアップデート.
     */
    private long m_lastUpdate = -1L ;
    
    /**
     * ヘッダパケット.
     */
    private byte[] m_headerPacket = null ;
    
    /**
     * リソースタイプ.
     */
    private ResourceType m_resType = null ;
    
    
    
    /**
     * コンストラクタ.
     */
    public UdpRcvConnectChild()
    {
        this.clear() ;
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 受信された条件を設定します.
     * <BR>
     * @param max 受信パケット長を設定します.
     * @param packetLen 対象パケット長を設定します.
     * @param length 受信電文長を設定します.
     * @param addr 対象のIPアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @param resType 対象のリソースタイプを設定します.
     * @exception InputException 入力例外.
     */
    public final void create( int max,int packetLen,int length,InetAddress addr,int port,ResourceType resType )
        throws InputException
    {
        if( max <= 0 || addr == null || port < 0 || port > 65535 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.clear() ;
        
        m_resType = ( resType == null ) ? UdpIODef.DEF_RESOURCE_TYPE : resType ;
        m_subIds = new SearchInt() ;
        m_packets = Resource.createBinResource(
            m_resType,packetLen * max
        ) ;
        
        m_conn.create( addr,port ) ;
        m_maxPacket = max ;
        m_length = length ;
        m_packetLen = packetLen ;
        
        m_lastUpdate = System.currentTimeMillis() ;
        
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public final void clear()
    {
        if( m_subIds != null ){
            m_subIds.clear() ;
        }
        if( m_packets != null ){
            m_packets.clear() ;
        }
        
        m_subIds = null ;
        m_packets = null ;
        m_headerPacket = null ;
        m_resType = null ;
        
        m_conn.clear() ;
        m_maxPacket = -1 ;
        m_length = -1 ;
        m_packetLen = -1 ;
        
        m_lastUpdate = -1L ;
    }
    
    /**
     * 受信データ設定.
     * <BR><BR>
     * 受信されたデータを設定します.
     * <BR>
     * @param addr 受信されたIPアドレスを設定します.
     * @param port 受信されたポート番号を設定します.
     * @param subID 対象のサブIDを設定します.
     * @param packet 対象の受信パケット情報を設定します.
     * @exception InputException 入力例外.
     * @exception ExistException 存在例外.
     */
    public final void put( InetAddress addr,int port,int subID,byte[] packet )
        throws InputException,ExistException
    {
        if( m_subIds == null ){
            return ;
        }
        else if(
            m_conn.getAddress().equals( addr ) == false || m_conn.getPort() != port ||
            packet == null || packet.length <= 0 || packet.length > m_packetLen
        )
        {
            if( packet == null || packet.length <= 0 || packet.length > m_packetLen ){
                throw new InputException( "引数は不正です" ) ;
            }
            throw new ExistException(
                "対象の受信条件(addr:" + addr +
                " port:" + port +
                ")は不正な条件です"
            ) ;
        }
        
        if( subID == UdpIODef.PACKET_HEADER_SUBID ){
            m_headerPacket = packet ;
        }
        else{
            Resource.arraycopy( packet,0,m_packets,subID * m_packetLen,packet.length ) ;
        }
        
        if( m_subIds.isData( subID ) == false ){
            m_subIds.add( subID ) ;
        }
        
        m_lastUpdate = System.currentTimeMillis() ;
    }
    
    /**
     * ヘッダパケット情報取得.
     * <BR><BR>
     * 対象のヘッダパケット情報を取得します.
     * <BR>
     * @return byte[] 対象のヘッダパケット情報が返されます.
     */
    public final byte[] getHeader()
    {
        if( m_subIds == null ){
            return null ;
        }
        return m_headerPacket ;
    }
    
    /**
     * ヘッダパケット削除.
     * <BR><BR>
     * 対象のヘッダパケット情報を削除して取得します.
     * <BR>
     * @return byte[] 対象のヘッダパケット情報が返されます.
     */
    public final byte[] removeHeader()
    {
        byte[] ret = null ;
        
        if( m_subIds == null ){
            return null ;
        }
        ret = m_headerPacket ;
        m_headerPacket = null ;
        m_subIds.remove( UdpIODef.PACKET_HEADER_SUBID ) ;
        
        return ret ;
    }
    
    /**
     * パケット群を電文に変換.
     * <BR><BR>
     * パケット群を電文に変換します.<BR>
     * また、格納されているパケット情報は削除されます.
     * <BR>
     * @return byte[] 電文情報が返されます.
     */
    public final byte[] getTelegram()
    {
        int i ;
        int len ;
        int pnt ;
        int nowLen ;
        int one ;
        int cpLen ;
        
        BinResource packets = null ;
        byte[] ret = null ;
        
        if( m_subIds == null || this.isLost() == true ){
            return null ;
        }
        
        try{
            
            len = m_maxPacket ;
            one = m_packetLen ;
            nowLen = m_length ;
            
            packets = m_packets ;
            
            ret = new byte[ m_length ] ;
            
            for( i = 0,pnt = UdpIODef.PACKET_HEADER_LENGTH ; i < len ; i ++ ){
                
                cpLen = ( nowLen <= one ) ? nowLen : one-UdpIODef.PACKET_HEADER_LENGTH ;
                nowLen -= cpLen ;
                
                Resource.arraycopy( packets,pnt,ret,0,cpLen ) ;
                
                pnt += cpLen + UdpIODef.PACKET_HEADER_LENGTH ;
                
            }
            
        }catch( Exception e ){
            ret = null ;
        }finally{
            m_subIds.clear() ;
            m_packets.clear() ;
            packets = null ;
        }
        
        return ret ;
    }
    
    /**
     * パケット群を電文に変換.
     * <BR><BR>
     * パケット群を電文に変換します.<BR>
     * また、格納されているパケット情報は削除されます.
     * <BR>
     * @return BinResource 電文情報が返されます.
     */
    public final BinResource getTelegramByResource()
    {
        int i,j ;
        int len ;
        int pnt ;
        int nowLen ;
        int one ;
        int pkLen ;
        int cpLen ;
        
        BinResource packets = null ;
        BinResource ret = null ;
        
        if( m_subIds == null || this.isLost() == true ){
            return null ;
        }
        
        try{
            
            len = m_maxPacket ;
            one = m_packetLen ;
            pkLen = one - UdpIODef.PACKET_HEADER_LENGTH ;
            nowLen = m_length ;
            
            packets = m_packets ;
            
            ret = Resource.createBinResource(
                m_resType,m_length
            ) ;
            
            for( i = 0,j = 0,pnt = UdpIODef.PACKET_HEADER_LENGTH ; i < len ; i ++,j += pkLen ){
                
                cpLen = ( nowLen <= one ) ? nowLen : one-UdpIODef.PACKET_HEADER_LENGTH ;
                nowLen -= cpLen ;
                Resource.arraycopy( packets,pnt,ret,j,cpLen ) ;
                
                pnt += cpLen + UdpIODef.PACKET_HEADER_LENGTH ;
                
            }
            
        }catch( Exception e ){
            ret = null ;
        }finally{
            if( m_packets != null ){
                m_packets.clear() ;
            }
            m_subIds.clear() ;
            packets = null ;
        }
        
        return ret ;
    }
    
    /**
     * 格納されているサブID群を取得.
     * <BR><BR>
     * 格納されているサブID群を取得します.
     * <BR>
     * @return int[] 格納されているサブID群が返されます.
     */
    public final int[] subIDs()
    {
        if( m_subIds == null ){
            return null ;
        }
        return ( int[] )m_subIds.getAll() ;
    }
    
    /**
     * 格納されているサブID数を取得.
     * <BR><BR>
     * 格納されているサブID数を取得します.
     * <BR>
     * @return int 格納されているサブID数が返されます.
     */
    public final int size()
    {
        if( m_subIds == null ){
            return 0 ;
        }
        return m_subIds.size() ;
    }
    
    /**
     * 受信IPアドレスを取得.
     * <BR><BR>
     * 受信されたIPアドレスを取得します.
     * <BR>
     * @return InetAddress 受信IPアドレスが返されます.
     */
    public final InetAddress getAddress()
    {
        if( m_subIds == null ){
            return null ;
        }
        return m_conn.getAddress() ;
    }
    
    /**
     * 受信ポート番号を取得.
     * <BR><BR>
     * 受信されたポート番号を取得します.
     * <BR>
     * @return int 受信ポート番号が返されます.
     */
    public final int getPort()
    {
        if( m_subIds == null ){
            return -1 ;
        }
        return m_conn.getPort() ;
    }
    
    /**
     * 受信IPアドレス+ポート番号を取得.
     * <BR><BR>
     * 受信IPアドレス+ポート番号を取得します.
     * <BR>
     * @return ConnectAddress 受信元のコネクション条件が返されます.
     */
    public final ConnectAddress getConnect()
    {
        if( m_subIds == null ){
            return null ;
        }
        return m_conn ;
    }
    
    /**
     * １パケット長を取得.
     * <BR><BR>
     * 対象電文の１パケット長を取得します.
     * <BR>
     * @return int １パケット長が返されます.
     */
    public final int getOnePakcet()
    {
        if( m_subIds == null ){
            return 0 ;
        }
        return m_packetLen ;
    }
    
    /**
     * 最大パケット長を取得.
     * <BR><BR>
     * 最大パケット長を取得します.
     * <BR>
     * @return int 受信に必要な最大パケット長が返されます.
     */
    public final int getMaxPacket()
    {
        if( m_subIds == null ){
            return 0 ;
        }
        return m_maxPacket ;
    }
    
    /**
     * 受信電文長を取得.
     * <BR><BR>
     * 受信される電文長を取得します.
     * <BR>
     * @return int 受信される電文長が返されます.
     */
    public final int getMaxLength()
    {
        if( m_subIds == null ){
            return 0 ;
        }
        return m_length ;
    }
    
    /**
     * ラストアップデート時間を取得.
     * <BR><BR>
     * ラストアップデート時間を取得します.
     * <BR>
     * @return long ラストアップデート時間が返されます.
     */
    public final long getLastUpdate()
    {
        if( m_subIds == null ){
            return -1L ;
        }
        return m_lastUpdate ;
    }
    
    /**
     * ロスト電文ID取得.
     * <BR><BR>
     * 現在ロスト電文が存在する場合、取得します.
     * <BR>
     * @return IntArray ロスト電文サブIDが格納されます.<BR>
     *                  ロスト電文が存在しない場合[null]が返されます.
     */
    public final IntArray getLostPacketID()
    {
        int i,j,k ;
        int len ;
        int lenJ ;
        int before ;
        
        int[] now = null ;
        IntArray ret = null ;
        
        if( m_subIds == null ){
            return null ;
        }
        
        ret = new IntArray() ;
        
        now = m_subIds.getAll() ;
        
        // 一部のみロストしている場合.
        if( now != null && ( len = now.length ) > 0 ){
            
            Arrays.sort( now ) ;
            
            for( i = 0,before = -1 ; i < len ; i ++ ){
                
                if( now[ i ] >= 0 ){
                    
                    if( now[ i ] != before+1 ){
                        
                        lenJ = ( now[ i ] - before ) - 1 ;
                        for( j = 0,k = before+1 ; j < lenJ ; j ++,k ++ ){
                            ret.add( k ) ;
                        }
                        
                    }
                    
                    before = now[ i ] ;
                    
                }
                
            }
            
            // 最後から比べて、条件が存在しない場合.
            if( m_maxPacket != before+1 ){
                
                lenJ = ( m_maxPacket - before ) - 1 ;
                for( j = 0,k = before+1 ; j < lenJ ; j ++,k ++ ){
                    ret.add( k ) ;
                }
                
            }
                
            
            now = null ;
            
            if( ( len = ret.size() ) == 0 ){
                ret = null ;
            }
            
        }
        // 全体がロストしている場合.
        else{
            
            len = m_maxPacket ;
            for( i = 0 ; i < len ; i ++ ){
                ret.add( i ) ;
            }
            
        }
        
        return ret ;
    }
    
    /**
     * ロストパケット存在チェック.
     * <BR><BR>
     * ロストパケットが存在するかチェックします.
     * <BR>
     * @return boolean ロストパケットが存在するかチェックします.<BR>
     *                 [true]が返された場合、ロストパケットが存在します.<BR>
     *                 [false]が返された場合、ロストパケットが存在しません.
     */
    public final boolean isLost()
    {
        int len ;
        
        len = ( m_subIds.isData( UdpIODef.PACKET_HEADER_SUBID ) == true ) ? m_subIds.size()-1 : m_subIds.size() ;
        return ( m_maxPacket != len ) ? true : false ;
        
    }
    
    /**
     * データ存在チェック.
     * <BR><BR>
     * 対象サブIDのデータが存在するかチェックします.
     * <BR>
     * @param subID 対象のサブIDを設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、存在します.<BR>
     *                 [false]が返された場合、存在しません.
     */
    public final boolean isData( int subID )
    {
        if( m_subIds == null ){
            return false ;
        }
        return m_subIds.isData( subID ) ;
    }
    
}

