/*
 * @(#)UdpIOSndThread.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.net.uio;

import java.net.InetAddress;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.JRcServer.commons.exception.BaseException;
import com.JRcServer.commons.exception.ExecutionException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.net.TelegramResourceRoll;
import com.JRcServer.commons.net.TelegramValue;
import com.JRcServer.commons.net.UdpProtocol;
import com.JRcServer.commons.resource.BinResource;
import com.JRcServer.commons.thread.ExecutionThread;
import com.JRcServer.commons.thread.LoopThread;
import com.JRcServer.commons.thread.Synchronized;
import com.JRcServer.commons.util.ByteUtil;
import com.JRcServer.commons.util.ConvertParam;
import com.JRcServer.commons.util.SequenceID;
import com.JRcServer.commons.util.UtilCom;
import com.JRcServer.commons.util.array.IntArray;

/**
 * UDP-I/O送信処理をスレッドで行います.
 *
 * @version 1.00, 2005/03/24
 * @author  Masahito Suzuki
 * @since   JRcCommons 1.00
 */
class UdpIOSndThread extends ExecutionThread
{
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( UdpIOSndThread.class ) ;
    
    /**
     * 送信パケット長.
     */
    private int m_packetLength = -1 ;
    
    /**
     * 送信結果コールバック.
     */
    private UIOResultCallback m_sendCallback = null ;
    
    /**
     * UDPプロトコル.
     */
    private UdpProtocol m_udp = null ;
    
    /**
     * UDP送信結果テーブル.
     */
    private UdpSndResultTable m_result = null ;
    
    /**
     * 電文ロール管理.
     */
    private TelegramResourceRoll m_roll = null ;
    
    /**
     * 電文受け取り要素.
     */
    private final TelegramValue m_rollValue = new TelegramValue() ;
    
    /**
     * 再送カウント規定値.
     */
    private int m_resendCount = -1 ;
    
    /**
     * ループスレッド.
     */
    private final LoopThread m_thread = new LoopThread() ;
    
    /**
     * 送信ID発版.
     */
    private SequenceID m_seq = null ;
    
    /**
     * 同期処理.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    /**
     * 電文送信用同期処理.
     */
    private Synchronized m_sndSync = null ;
    
    /**
     * 一時バイナリ格納オブジェクト.
     */
    private final ByteUtil m_tBuf = new ByteUtil() ;
    
    /**
     * ロストパケット用格納オブジェクト.
     */
    private final IntArray m_tLost = new IntArray() ;
    
    /**
     * ロール追加予約フラグ.
     */
    private volatile boolean m_sndResFlg = false ;
    
    
    
    /**
     * コンストラクタ.
     */
    public UdpIOSndThread(){}
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * UDP-I/O送信スレッドを生成します.
     * <BR>
     * @param scall 送信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param udp 送信処理を行うUDPプロトコルを設定します.
     * @param result 送信結果を取得するオブジェクトを設定します.
     * @param roll 送信電文格納オブジェクトを設定します.
     * @param sndSync 送信同期オブジェクトを設定します.
     * @param seq 送信ID発版オブジェクトを設定します.
     * @exception InputException 入力例外.
     */
    public final void create(
        UIOResultCallback scall,UdpProtocol udp,UdpSndResultTable result,TelegramResourceRoll roll,
        Synchronized sndSync,SequenceID seq
    )
        throws InputException
    {
        this.create( scall,udp,result,roll,sndSync,seq,UdpIODef.DEFAULT_PACKET,UdpIODef.DEF_RESEND_COUNT ) ;
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * UDP-I/O送信スレッドを生成します.
     * <BR>
     * @param scall 送信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param udp 送信処理を行うUDPプロトコルを設定します.
     * @param result 送信結果を取得するオブジェクトを設定します.
     * @param roll 送信電文格納オブジェクトを設定します.
     * @param sndSync 送信同期オブジェクトを設定します.
     * @param seq 送信ID発版オブジェクトを設定します.
     * @param packetLen 送信対象のパケット長を設定します.
     * @exception InputException 入力例外.
     */
    public final void create(
        UIOResultCallback scall,UdpProtocol udp,UdpSndResultTable result,TelegramResourceRoll roll,
        Synchronized sndSync,SequenceID seq,int packetLen
    )
        throws InputException
    {
        this.create( scall,udp,result,roll,sndSync,seq,packetLen,UdpIODef.DEF_RESEND_COUNT ) ;
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * UDP-I/O送信スレッドを生成します.
     * <BR>
     * @param scall 送信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param udp 送信処理を行うUDPプロトコルを設定します.
     * @param result 送信結果を取得するオブジェクトを設定します.
     * @param roll 送信電文格納オブジェクトを設定します.
     * @param sndSync 送信同期オブジェクトを設定します.
     * @param seq 送信ID発版オブジェクトを設定します.
     * @param packetLen 送信対象のパケット長を設定します.
     * @param resendCnt 再送カウントを設定します.<BR>
     *                   設定可能な最小値は[3]です.<BR>
     *                   また、これ以下を設定しても意味はありません.
     * @exception InputException 入力例外.
     */
    public final void create(
        UIOResultCallback scall,UdpProtocol udp,UdpSndResultTable result,TelegramResourceRoll roll,
        Synchronized sndSync,SequenceID seq,int packetLen,int resendCnt
    )
        throws InputException
    {
        if(
            udp == null || udp.isOpen() == false ||
            result == null || roll == null || sndSync == null ||
            sndSync.isUse() == false || seq == null
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.clear() ;
        m_sync.create() ;
        
        packetLen = ( packetLen <= UdpIODef.MIN_PACKET ) ?
            UdpIODef.MIN_PACKET :
            (
                ( packetLen >= UdpIODef.MAX_PACKET ) ?
                    UdpIODef.MAX_PACKET : packetLen
            ) ;
        
        resendCnt = ( resendCnt <= UdpIODef.MIN_RESEND_COUNT ) ?
            UdpIODef.MIN_RESEND_COUNT : resendCnt ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                m_sendCallback = scall ;
                m_udp = udp ;
                m_result = result ;
                m_roll = roll ;
                m_sndSync = sndSync ;
                m_seq = seq ;
                m_packetLength = packetLen ;
                m_sndResFlg = false ;
                m_resendCount = resendCnt ;
                
            }
            
            m_thread.create( this ) ;
            m_thread.startThread() ;
            
        }catch( Exception e ){
            this.clear() ;
        }
        
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報を破棄します.
     */
    public final void clear()
    {
        m_sync.clear() ;
        m_thread.clear() ;
        m_tBuf.clear() ;
        m_tLost.clear() ;
        m_rollValue.clear() ;
        
        m_sendCallback = null ;
        m_udp = null ;
        m_result = null ;
        m_roll = null ;
        m_sndSync = null ;
        m_seq = null ;
        m_packetLength = -1 ;
        m_sndResFlg = false ;
        
    }
    
    /**
     * 情報送信先を設定.
     * <BR><BR>
     * 情報送信先を設定します.
     * <BR>
     * @param addr 送信先のアドレス情報を設定します.
     * @param port 送信先のポート番号を設定します.
     * @param binary 送信先のバイナリ情報を設定します.<BR>
     *               送信可能なオブジェクトは(byte[])か(BinResource)です.
     * @return int 送信されたID情報が返されます.<BR>
     *             [-1]が返された場合送信に失敗しました.
     */
    public final int setData( InetAddress addr,int port,Object binary )
    {
        int i ;
        int len ;
        int id ;
        
        int ret ;
        
        byte[] b1 = null ;
        BinResource b2 = null ;
        
        // 指定許可のオブジェクト型でない場合.
        if(
            ( binary instanceof byte[] ) == false &&
            ( binary instanceof BinResource ) == false
        )
        {
            return -1 ;
        }
        
        // 対象オブジェクト型に変換.
        if( binary instanceof byte[] ){
            b1 = ( byte[] )binary ;
        }
        else{
            b2 = ( BinResource )binary ;
        }
        
        try{
            
            // 送信IDを取得.
            synchronized( m_sync.get() ){
                synchronized( m_sndSync.get() ){
                    id = m_seq.getID() ;
                }
            }
            
            // 待機カウントをセット.
            len = UdpIODef.WAIT_SET_SEND_ROLL_COUNT ;
            
            // 対象バイナリが(byte[])の場合.
            if( b1 != null ){
            
                // 送信テーブルに設定できるまで待機.
                // (ただし待機回数分設定できなかった場合をのぞく).
                for( i = 0,ret = -1 ; i < len ; i ++ ){
                    if( m_roll.add( id,b1,addr,port ) == true ){
                        ret = id ;
                        break ;
                    }
                    UtilCom.idleTime() ;
                }
                
            }
            // 対象バイナリが(BinResource)の場合.
            else{
                
                // 送信テーブルに設定できるまで待機.
                // (ただし待機回数分設定できなかった場合をのぞく).
                for( i = 0,ret = -1 ; i < len ; i ++ ){
                    if( m_roll.add( id,b2,addr,port ) == true ){
                        ret = id ;
                        break ;
                    }
                    UtilCom.idleTime() ;
                }
                
            }
            
        }catch( Exception e ){
            ret = -1 ;
        }finally{
            b1 = null ;
            b2 = null ;
        }
        
        return ret ;
    }
    
    /**
     * 接続可能をチェックする確認電文を送信.
     * <BR><BR>
     * 接続可能をチェックする確認電文を送信します.
     * <BR>
     * @param addr 接続先のアドレスを設定します.
     * @param port 接続先のポート番号を設定します.
     * @return int 現在受信側が接続中の数が返されます.<BR>
     *              [-1]が返された場合、確認電文の受信に失敗しました.
     */
    public final int checkConnect( InetAddress addr,int port )
    {
        return this.checkConnect( addr,port,UdpIODef.MIN_RESEND_COUNT ) ;
    }
    
    /**
     * 接続可能をチェックする確認電文を送信.
     * <BR><BR>
     * 接続可能をチェックする確認電文を送信します.
     * <BR>
     * @param addr 接続先のアドレスを設定します.
     * @param port 接続先のポート番号を設定します.
     * @param max 最大ループ回数を設定します.
     * @return int 現在受信側が接続中の数が返されます.<BR>
     *              [-1]が返された場合、確認電文の受信に失敗しました.
     */
    public final int checkConnect( InetAddress addr,int port,int max )
    {
        int i ;
        int id ;
        int randID ;
        int rcvCd ;
        int packLen ;
        int ret ;
        
        byte[] snd = null ;
        byte[] rcv = null ;
        ByteUtil tBuf = null ;
        UdpProtocol udp = null ;
        UdpSndResultTable result = null ;
        
        if( addr == null || port <= 0 || port >= 65535 ){
            return -1 ;
        }
        
        id = -1 ;
        ret = -1 ;
        max = ( max <= 0 ) ? UdpIODef.MIN_RESEND_COUNT : max ;
        
        try{
            
            // 基本データを取得.
            synchronized( m_sync.get() ){
                
                tBuf = m_tBuf ;
                udp = m_udp ;
                result = m_result ;
                packLen = m_packetLength ;
                
                // 送信IDを取得.
                synchronized( m_sndSync.get() ){
                    id = m_seq.getID() ;
                }
            }
            
            
            // 送信IDを予約.
            result.reservation( id ) ;
            
            // ランダムIDを生成.
            randID = UdpIOCommon.getRandomID() ;
            
            // 接続確認用ファーストパケットを生成.
            UdpIOCommon.createPacketHeader(
                tBuf,id,UdpIODef.PACKET_CCONN_MAX_PACKET,randID,0,packLen,
                UdpIODef.FIRST_STATE_CODE
            ) ;
            snd = tBuf.getToClear() ;
            
            /////////////////////////////////////
            // 相手先にファーストパケットを送信.
            /////////////////////////////////////
            for( i = 0 ; i < max ; i ++ ){
                
                // ファーストパケットを送信.
                udp.send( snd,addr,port ) ;
                
                // 送信確認を受信.
                rcv = result.getHeader(
                    id,
                    randID,
                    UdpIODef.FIRST_RECV_TIMEOUT
                ) ;
                
                // ヘッダ条件が受信された場合.
                if( UdpIOCommon.isPacketHeader( id,UdpIODef.PACKET_CCONN_MAX_PACKET,randID,rcv ) == true ){
                    
                    // 受信コードを取得.
                    rcvCd = UdpIOCommon.getPacketHeaderToCode( rcv ) ;
                    
                    ///////////////////////////////////
                    // ステータスが電文開始以外の場合.
                    ///////////////////////////////////
                    if( ( rcvCd & UdpIODef.FIRST_CHECK ) != UdpIODef.FIRST_CHECK ){
                        
                        // 受信時にエラーが発生した場合.
                        if( ( rcvCd & UdpIODef.ERROR_CHECK ) == UdpIODef.ERROR_CHECK ){
                            
                            ret = -1 ;
                            break ;
                            
                        }
                        // 正常系ステータスを除く不正なステータスの場合.
                        else if( ( rcvCd & UdpIODef.SUCCESS_CHECK ) != UdpIODef.SUCCESS_CHECK ){
                            
                            ret = -1 ;
                            break ;
                            
                        }
                        // それ以外の場合.
                        else{
                            
                            // 読み飛ばす.
                            i -- ;
                            continue ;
                            
                        }
                        
                    }
                    /////////////////////////////
                    // 通知確認OKが返された場合.
                    /////////////////////////////
                    else if( rcvCd == UdpIODef.FIRST_NOTICE_OK_STATE_CODE ){
                        
                        // 受信側の接続数をを取得.
                        ret = UdpIOCommon.getPacketHeaderToTelegramLength( rcv ) ;
                        break ;
                        
                    }
                    /////////////////////////////////
                    // ファーストコードが不正な場合.
                    /////////////////////////////////
                    else if( rcvCd == UdpIODef.FIRST_ERROR_STATE_CODE ){
                        
                        //<><><><><><><><><><><><><><><><><><><>
                        // 新たにランダムIDを発行した電文を作成.
                        //<><><><><><><><><><><><><><><><><><><>
                        
                        // ランダムIDを再生成.
                        randID = UdpIOCommon.getRandomID() ;
                        
                        // ファーストパケットを再生成.
                        UdpIOCommon.createPacketHeader(
                            tBuf,id,UdpIODef.PACKET_CCONN_MAX_PACKET,randID,0,packLen,
                            UdpIODef.FIRST_STATE_CODE
                        ) ;
                        snd = tBuf.getToClear() ;
                        
                        // 再送カウントを１デクリメント.
                        i -- ;
                        
                    }
                }
                
            }
            
        }catch( Exception e ){
            ret = -1 ;
        }finally{
            
            // 送信電文IDを予約している場合、解除.
            if( id != -1 && result != null ){
                result.cancell( id ) ;
            }
            if( tBuf != null ){
                tBuf.clear() ;
            }
            
            snd = null ;
            rcv = null ;
            tBuf = null ;
            udp = null ;
            result = null ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * スレッド状態を取得.
     * <BR><BR>
     * スレッド状態を取得します.
     * <BR>
     * @return boolean スレッド状態が返されます.<BR>
     *                 [true]が返された場合、スレッドは実行中です.<BR>
     *                 [false]が返された場合、スレッドは停止中です.
     */
    public final boolean isThread()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_thread.isThread() ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * 実行初期化処理をサポートします.
     * <BR><BR>
     * 実行初期化処理をサポートします.<BR>
     * この処理は、スレッド処理が開始された時に呼び出されます.
     * <BR>
     * @param obj 実行開始時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void init( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行終了化処理をサポートします.
     * <BR><BR>
     * 実行終了化処理をサポートします.<BR>
     * この処理は、スレッド処理が終了された時に呼び出されます.
     * <BR>
     * @param obj 実行終了時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void exit( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * ストップ処理をサポートします。
     * <BR><BR>
     * ストップ処理をサポートします。<BR>
     * この処理は、スレッドでのストップ処理に対して呼び出し実行されます.
     * <BR>
     * @param obj ストップ時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void stop( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行処理をサポートします。
     * <BR><BR>
     * 実行処理をサポートします。<BR>
     * この処理は、スレッドでの実行処理に対して呼び出し実行されます.
     * <BR>
     * @param obj 実行時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void execution( Object obj )
        throws ExecutionException
    {
        int i,j ;
        int len ;
        int allPacketLength ;
        int packLen ;
        int port ;
        int id ;
        int randID ;
        int rcvCd ;
        int resendCnt ;
        int maxResend ;
        boolean sndRollFlg ;
        boolean firstRcvFlg ;
        boolean rollbackFlg ;
        
        byte[] snd = null ;
        byte[] rcv = null ;
        byte[] esnd = null ;
        BinResource data = null ;
        ByteUtil tBuf = null ;
        UdpProtocol udp = null ;
        UdpSndResultTable result = null ;
        Synchronized sndSync = null ;
        TelegramResourceRoll roll = null ;
        TelegramValue rollVal = null ;
        InetAddress addr = null ;
        
        id = -1 ;
        port = -1 ;
        resendCnt = -1 ;
        firstRcvFlg = false ;
        rollbackFlg = false ;
        
        try{
            
            // 一時停止.
            UtilCom.idleTime() ;
            
            // 電文情報が存在しない場合.
            if( this.isSendTelegram() == true ){
                
                // 送信対象が0の場合何もしない.
                return ;
                
            }
            
            // 基本データを取得.
            synchronized( m_sync.get() ){
                
                tBuf = m_tBuf ;
                udp = m_udp ;
                result = m_result ;
                sndSync = m_sndSync ;
                roll = m_roll ;
                rollVal = m_rollValue ;
                packLen = m_packetLength ;
                maxResend = m_resendCount ;
                
                m_sndResFlg = false ;
                
                // 送信条件を取得.
                synchronized( roll ){
                    
                    // 情報が取得できた場合.
                    if( ( sndRollFlg = roll.get( rollVal ) ) == true ){
                        
                        // 追加予約をON.
                        roll.addReservationByON() ;
                        m_sndResFlg = true ;
                        
                    }
                    
                }
                
            }
            
            // 送信条件が存在する場合.
            if( sndRollFlg == true ){
                
                // 送信データを分解.
                id = rollVal.getID() ;
                addr = rollVal.getAddress() ;
                port = rollVal.getPort() ;
                resendCnt = rollVal.getResendCount() ;
                data = ( BinResource )rollVal.getBinary() ;
                
                rollVal.clear() ;
                rollVal = null ;
                
                // IDが正常な場合.
                if( id >= 0 ){
                    
                    // 再送カウントが規定値を上回っている場合.
                    if( resendCnt > maxResend ){
                        
                        // 警告をログに出力し、対象電文を破棄.
                        LOG.debug(
                            "電文情報(id:" + id +
                            " addr:" + addr +
                            " port:" + port +
                            " size:" + data.size() +
                            ")は、再送回数(" + maxResend +
                            ")を越したため破棄します"
                        ) ;
                        
                        // 電文破棄.
                        if( data != null ){
                            data.clear() ;
                            data = null ;
                        }
                        return ; 
                        
                    }
                    else{
                        // 再送カウントをインクリメント.
                        resendCnt ++ ;
                    }
                    
                    // 送信IDを予約.
                    result.reservation( id ) ;
                    
                    // 最大送信データ長を取得.
                    allPacketLength = UdpIOCommon.getMaxPacket( packLen,data ) ;
                    
                    // ランダムIDを生成.
                    randID = UdpIOCommon.getRandomID() ;
                    
                    // ファーストパケットを生成.
                    UdpIOCommon.createPacketHeader(
                        tBuf,id,allPacketLength,randID,data.size(),packLen,
                        UdpIODef.FIRST_STATE_CODE
                    ) ;
                    snd = tBuf.getToClear() ;
                    
                    // 送信開始の場合.
                    if( resendCnt == 1 ){
                        LOG.debug(
                            "電文情報(id:" + id +
                            " addr:" + addr +
                            " port:" + port +
                            " randCode:" + randID +
                            " resendCnt:" + resendCnt +
                            ")のファースト送信を開始します"
                        ) ;
                    }
                    
                    /////////////////////////////////////
                    // 相手先にファーストパケットを送信.
                    /////////////////////////////////////
                    for( j = 0 ; j < UdpIODef.FIRST_ERROR_RETRY ; j ++ ){
                        
                        // ファーストパケットを送信.
                        udp.send( snd,addr,port ) ;
                        
                        // 送信確認を受信.
                        rcv = result.getHeader(
                            id,
                            randID,
                            UdpIODef.FIRST_RECV_TIMEOUT
                        ) ;
                        
                        // ヘッダ条件が受信された場合.
                        if( UdpIOCommon.isPacketHeader( id,allPacketLength,randID,rcv ) == true ){
                            
                            // 受信コードを取得.
                            rcvCd = UdpIOCommon.getPacketHeaderToCode( rcv ) ;
                            
                            ///////////////////////////////////
                            // ステータスが電文開始以外の場合.
                            ///////////////////////////////////
                            if( ( rcvCd & UdpIODef.FIRST_CHECK ) != UdpIODef.FIRST_CHECK ){
                                
                                // 受信時にエラーが発生した場合.
                                if( ( rcvCd & UdpIODef.ERROR_CHECK ) == UdpIODef.ERROR_CHECK ){
                                    
                                    // 送信対象電文をロールバック.
                                    rollbackFlg = this.rollbackRoll( rollbackFlg,id,addr,port,data,resendCnt ) ;
                                    
                                    LOG.error(
                                        "電文情報(id:" + id +
                                        " addr:" + addr +
                                        " port:" + port +
                                        " randCode:" + randID +
                                        " resendCnt:" + resendCnt +
                                        ")のファースト送信で通信エラー(" +
                                        UdpIOCommon.getErrorCodeByString( rcvCd ) +
                                        ")が返されました"
                                    ) ;
                                    firstRcvFlg = true ;
                                    break ;
                                    
                                }
                                // 正常系ステータスを除く不正なステータスの場合.
                                else if( ( rcvCd & UdpIODef.SUCCESS_CHECK ) != UdpIODef.SUCCESS_CHECK ){
                                    
                                    // 送信対象電文をロールバック.
                                    rollbackFlg = this.rollbackRoll( rollbackFlg,id,addr,port,data,resendCnt ) ;
                                    
                                    LOG.error(
                                        "電文情報(id:" + id +
                                        " addr:" + addr +
                                        " port:" + port +
                                        " randCode:" + randID +
                                        " resendCnt:" + resendCnt +
                                        ")のファースト送信で不明なステータスコード(" +
                                        UdpIOCommon.getErrorCodeByString( rcvCd ) +
                                        ")が返されました"
                                    ) ;
                                    firstRcvFlg = true ;
                                    break ;
                                    
                                }
                                // それ以外の場合.
                                else{
                                    
                                    // 読み飛ばす.
                                    j -- ;
                                    continue ;
                                    
                                }
                            }
                            /////////////////////////////////////////////
                            // 正常にファーストコード送信が完了した場合.
                            /////////////////////////////////////////////
                            else if( rcvCd == UdpIODef.FIRST_SUCCESS_STATE_CODE ){
                                
                                snd = null ;
                                rcv = null ;
                                firstRcvFlg = true ;
                                break ;
                                
                            }
                            /////////////////////////////////
                            // ファーストコードが不正な場合.
                            /////////////////////////////////
                            else if( rcvCd == UdpIODef.FIRST_ERROR_STATE_CODE ){
                                
                                //<><><><><><><><><><><><><><><><><><><>
                                // 新たにランダムIDを発行した電文を作成.
                                //<><><><><><><><><><><><><><><><><><><>
                                
                                // ランダムIDを再生成.
                                randID = UdpIOCommon.getRandomID() ;
                                
                                // ファーストパケットを再生成.
                                UdpIOCommon.createPacketHeader(
                                    tBuf,id,allPacketLength,randID,data.size(),packLen,
                                    UdpIODef.FIRST_STATE_CODE
                                ) ;
                                snd = tBuf.getToClear() ;
                                
                                // 再送カウントを１デクリメント.
                                resendCnt -- ;
                                
                            }
                        }
                        
                    }
                    
                    /////////////////////////////////////////
                    // ファーストデータの送信に失敗した場合.
                    /////////////////////////////////////////
                    if( snd != null ){
                        
                        // ファースト受信が１度もない場合.
                        if( firstRcvFlg == false ){
                            
                            //<><><><><><><><><><><><><><><><><><><><><><><>
                            // この場合は、接続先が存在しない可能性が高い.
                            //<><><><><><><><><><><><><><><><><><><><><><><>
                            
                            // 送信対象電文をロールバック.
                            rollbackFlg = this.rollbackRoll( rollbackFlg,id,addr,port,data,resendCnt ) ;
                            
                        }
                        
                        return ;
                        
                    }
                    ///////////////////////////////////////////
                    // ファーストデータの送信に成功した場合、
                    // バイナリデータをパケット単位で送信する.
                    ///////////////////////////////////////////
                    else{
                        
                        // 全ての電文をパケット単位で送信.
                        for( i = 0 ; i < allPacketLength ; i ++ ){
                            
                            UdpIOCommon.createTelegram(
                                tBuf,id,i,randID,packLen,data
                            ) ;
                            
                            snd = tBuf.getToClear() ;
                            
                            // パケット電文送信.
                            udp.send( snd,addr,port ) ;
                            snd = null ;
                            
                        }
                        
                        // パケット終了を生成.
                        UdpIOCommon.createPacketHeader(
                            tBuf,id,allPacketLength,randID,-1,packLen,
                            UdpIODef.END_STATE_CODE
                        ) ;
                        
                        esnd = tBuf.getToClear() ;
                        
                        // 送信成功待ち.
                        for( j = 0 ; j < UdpIODef.PACKET_ERROR_RETRY ; j ++ ){
                            
                            // パケット終了を送信.
                            udp.send( esnd,addr,port ) ;
                            
                            // 全電文送信確認を受信.
                            rcv = result.getHeader(
                                id,
                                randID,
                                UdpIODef.PACKET_RECV_TIMEOUT
                            ) ;
                            
                            // 何らかの送信結果が返された場合.
                            if( rcv != null ){
                                
                                // 受信ヘッダパケットステータスコードを取得.
                                rcvCd = UdpIOCommon.getPacketHeaderToCode( rcv ) ;
                                
                                ///////////////////////////////
                                // ステータスが電文開始の場合.
                                ///////////////////////////////
                                if( ( rcvCd & UdpIODef.FIRST_CHECK ) == UdpIODef.FIRST_CHECK ){
                                    // 読み飛ばす.
                                    j -- ;
                                    continue ;
                                }
                                /////////////////////////////
                                // 受信データか完了した場合.
                                /////////////////////////////
                                else if( rcvCd == UdpIODef.END_STATE_CODE ){
                                    
                                    // 送信処理を完結させる.
                                    this.commitRoll( id ) ;
                                    
                                    LOG.debug(
                                        "電文情報(id:" + id +
                                        " addr:" + addr +
                                        " port:" + port +
                                        " randCode:" + randID +
                                        " resendCnt:" + resendCnt +
                                        ")の送信(" + allPacketLength +
                                        ")が正常に終了しました"
                                    ) ;
                                    
                                    // 電文を破棄.
                                    if( data != null ){
                                        data.clear() ;
                                        data = null ;
                                    }
                                    
                                    // 対象IDを無効にすることで、送信完了とする.
                                    id = -1 ;
                                    
                                    break ;
                                    
                                }
                                /////////////////////////////////////////////////////
                                // 受信データの中でロストしたパケットが存在する場合.
                                /////////////////////////////////////////////////////
                                else if( rcvCd == UdpIODef.RESEND_STATE_CODE ){
                                    
                                    // ロストしたパケットを再送する.
                                    this.reSendLostPacket(
                                        addr,port,id,randID,packLen,resendCnt,allPacketLength,data
                                    ) ;
                                    
                                }
                                /////////////////////////////////
                                // 受信時にエラーが発生した場合.
                                /////////////////////////////////
                                else if( ( rcvCd & UdpIODef.ERROR_CHECK ) == UdpIODef.ERROR_CHECK ){
                                    
                                    // 送信対象電文をロールバック.
                                    rollbackFlg = this.rollbackRoll( rollbackFlg,id,addr,port,data,resendCnt ) ;
                                    
                                    LOG.error(
                                        "電文情報(id:" + id +
                                        " addr:" + addr +
                                        " port:" + port +
                                        " randCode:" + randID +
                                        " resendCnt:" + resendCnt +
                                        ")の送信で通信エラー(" + UdpIOCommon.getErrorCodeByString( rcvCd ) +
                                        ")が返されました"
                                    ) ;
                                    break ;
                                    
                                }
                                /////////////////////////////////////////////////
                                // 正常系ステータスを除く不正なステータスの場合.
                                /////////////////////////////////////////////////
                                else if( ( rcvCd & UdpIODef.SUCCESS_CHECK ) != UdpIODef.SUCCESS_CHECK ){
                                    
                                    // 送信対象電文をロールバック.
                                    rollbackFlg = this.rollbackRoll( rollbackFlg,id,addr,port,data,resendCnt ) ;
                                    
                                    LOG.warn(
                                        "電文情報(id:" + id +
                                        " addr:" + addr +
                                        " port:" + port +
                                        " randCode:" + randID +
                                        " resendCnt:" + resendCnt +
                                        ")の送信で不明なステータスコード(" + UdpIOCommon.getErrorCodeByString( rcvCd ) +
                                        ")が返されました"
                                    ) ;
                                    break ;
                                    
                                }
                                
                                
                            }
                        }
                        
                        // 電文の送信に失敗した場合.
                        // ロールバック処理が１度も行われていない場合.
                        if( id != -1 && rollbackFlg == false ){
                            
                            // 送信対象電文をロールバック.
                            rollbackFlg = this.rollbackRoll( rollbackFlg,id,addr,port,data,resendCnt ) ;
                            
                            LOG.error(
                                "電文情報(id:" + id +
                                " addr:" + addr +
                                " port:" + port +
                                " randCode:" + randID +
                                " resendCnt:" + resendCnt +
                                ")のパケット送信(" + allPacketLength +
                                ")に失敗しました"
                            ) ;
                            
                        }
                        
                        
                    }
                    
                    
                }
                
            }
            
            
        }catch( NullPointerException nul ){
            throw new ExecutionException(
                nul,ExecutionException.LEVEL_STOP
            ) ;
        }catch( ExecutionException ee ){
            throw ee ;
        }catch( BaseException be ){
            
            // ロールバックを試みる.
            try{
                if( id > 0 && addr != null && port > 0 && data != null ){
                    this.rollbackRoll( rollbackFlg,id,addr,port,data,resendCnt ) ;
                }
            }catch( Exception ee ){
            }
            
            LOG.error( "エラーが発生しました",be ) ;
            
        }catch( Exception e ){
            
            // ロールバックを試みる.
            try{
                if( id > 0 && addr != null && port > 0 && data != null ){
                    this.rollbackRoll( rollbackFlg,id,addr,port,data,resendCnt ) ;
                }
            }catch( Exception ee ){
            }
            
            LOG.error( "エラーが発生しました",e ) ;
            
        }finally{
            
            // ロール予約の後始末.
            this.resetReservation() ;
            
            // 送信電文IDを予約している場合、解除.
            if( id != -1 && result != null ){
                result.cancell( id ) ;
            }
            if( tBuf != null ){
                tBuf.clear() ;
            }
            
            snd = null ;
            rcv = null ;
            esnd = null ;
            data = null ;
            tBuf = null ;
            udp = null ;
            result = null ;
            sndSync = null ;
            roll = null ;
            rollVal = null ;
            addr = null ;
            
        }
        
    }
    
    /**
     * ロストしたパケットを再送信.
     */
    private final void reSendLostPacket(
        InetAddress addr,int port,int id,int randID,
        int plen,int resendCnt,int allPacketLength,BinResource data
    )
        throws BaseException
    {
        int i,j ;
        int len ;
        int now ;
        int before ;
        
        int[] subIDs = null ;
        byte[] snd = null ;
        byte[] packet = null ;
        UdpProtocol udp = null ;
        UdpSndResultTable result = null ;
        IntArray iAry = null ;
        ByteUtil tBuf = null ;
        
        udp = m_udp ;
        result = m_result ;
        iAry = m_tLost ;
        tBuf = m_tBuf ;
        
        try{
            
            /////////////////////////////////////////////
            // パケットロストサブID格納パケット群を取得.
            /////////////////////////////////////////////
            for( i = 0 ; i < UdpIODef.PACKET_LOST_RETRY ; i ++ ){
                
                // 現在存在するパケット条件を取得.
                subIDs = result.subIDs( id ) ;
                
                // パケットロスト条件を取得.
                if( subIDs != null && ( len = subIDs.length ) > 0 ){
                    
                    iAry.clear() ;
                    
                    // 存在するパケット条件を解析して、ロストパケットサブIDを取得.
                    for( j = 0 ; j < len ; j ++ ){
                        
                        packet = result.get( id,subIDs[ j ],randID ) ;
                        
                        // パケットヘッダ以外の番号の場合.
                        if( subIDs[ j ] != UdpIODef.PACKET_HEADER_SUBID ){
                            
                            UdpIOSndThread.getLostPacketBySubIDs( iAry,packet ) ;
                            
                        }
                        
                        packet = null ;
                        
                    }
                    
                    subIDs = null ;
                    
                }
                
            }
            
            ///////////////////////////////////////////
            // パケットロストサブID項番が存在する場合.
            // ロストパケット要求条件を送信する.
            ///////////////////////////////////////////
            if( ( len = iAry.size() ) > 0 ){
                
                LOG.debug(
                    "電文情報(id:" + id +
                    " addr:" + addr +
                    " port:" + port +
                    " randCode:" + randID +
                    " resendCnt:" + resendCnt +
                    ")の送信(" + allPacketLength +
                    ")でロスト電文(" + len +
                    ")が検出されました"
                ) ;
                
                // パケットロストサブID項番をソート.
                iAry.sort() ;
                
                // パケットデータを再送する.
                for( i = 0,before = UdpIODef.PACKET_HEADER_SUBID ; i < len ; i ++ ){
                    
                    now = iAry.get( i ) ;
                    
                    // 前回送信したIDと同様のものが存在する場合、
                    // 送信しない.
                    if( before != now ){
                        
                        // 再送対象パケットを作成.
                        UdpIOCommon.createTelegram(
                            tBuf,id,now,randID,plen,data
                        ) ;
                        
                        // 再送パケットを送信.
                        snd = tBuf.getToClear() ;
                        udp.send( snd,addr,port ) ;
                        snd = null ;
                        
                    }
                    
                    before = now ;
                    
                }
                
            }
            
        }catch( BaseException be ){
            throw be ;
        }catch( Exception e ){
            throw new BaseException( e ) ;
        }finally{
            subIDs = null ;
            snd = null ;
            packet = null ;
            result = null ;
            iAry = null ;
            tBuf = null ;
        }
        
    }
    
    /**
     * ロストパケットデータからサブID群を取得.
     */
    private static final void getLostPacketBySubIDs( IntArray out,byte[] packet )
        throws InputException
    {
        int i ;
        int len ;
        int pnt ;
        
        if( out == null || UdpIOCommon.isPacket( packet ) == false ){
            return ;
        }
        
        pnt = UdpIODef.PACKET_HEADER_LENGTH ;
        len = ( int )ConvertParam.convertShort( pnt,packet ) ;
        pnt += 2 ;
        
        for( i = 0 ; i < len ; i ++ ){
            
            out.add( ConvertParam.convertInt( pnt,packet ) ) ;
            pnt += 4 ;
            
        }
        
    }
    
    /**
     * 対象ロールをコミット.
     */
    private final void commitRoll( int id )
    {
        UIOResultCallback call = null ;
        
        synchronized( m_sync.get() ){
            
            // 送信コールバックを取得.
            call = m_sendCallback ;
            
            // 追加予約をOFF.
            m_roll.addReservationByOFF() ;
            
            // 追加予約フラグをOFF,
            m_sndResFlg = false ;
            
            // 予約IDを破棄する.
            m_result.cancell( id ) ;
            
        }
        
        // 送信コールバックが存在する場合.
        if( call != null ){
            
            // 送信コールバックを呼び出す.
            call.callback( id ) ;
            
        }
        
    }
    
    /**
     * 対象ロールをロールバック.
     */
    private final boolean rollbackRoll( boolean flg,int id,InetAddress addr,int port,BinResource data,int resendCnt )
        throws InputException
    {
        synchronized( m_sync.get() ){
            
            // 追加予約フラグをOFF,
            m_sndResFlg = false ;
            
            // １度もロールバックが行われていなくて、データが不正で無い場合.
            if( flg == false && addr != null && data != null && data.isUse() == true ){
                
                // 対象条件をロールバック.
                synchronized( m_roll ){
                    m_roll.addReservationByOFF() ;
                    m_roll.add( id,data,addr,port,resendCnt ) ;
                }
                
                flg = true ;
                
            }
            
        }
        
        return flg ;
    }
    
    /**
     * 送信追加予約がONの後処理.
     */
    private final void resetReservation()
    {
        try{
            synchronized( m_sync.get() ){
                
                // 送信予約がONのままの場合.
                if( m_sndResFlg == true ){
                    
                    // 予約をOFF.
                    m_sndResFlg = false ;
                    m_roll.addReservationByOFF() ;
                    
                }
                
            }
        }catch( Exception e ){
        }
    }
    
    /**
     * 送信データが存在するかチェック.
     */
    private final boolean isSendTelegram()
    {
        // 送信対象数が0の場合.
        if( m_roll.getSize() <= 0 ){
            return true ;
        }
        
        return false ;
    }
    
}

