/*
 * @(#)UdpIORcvThread.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.net.uio;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.JRcServer.commons.exception.BaseException;
import com.JRcServer.commons.exception.ExecutionException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.net.ConnectAddress;
import com.JRcServer.commons.net.TelegramResourceRoll;
import com.JRcServer.commons.net.UdpProtocol;
import com.JRcServer.commons.resource.BinResource;
import com.JRcServer.commons.thread.ExecutionThread;
import com.JRcServer.commons.thread.LoopThread;
import com.JRcServer.commons.thread.Synchronized;
import com.JRcServer.commons.util.ByteUtil;
import com.JRcServer.commons.util.UtilCom;
import com.JRcServer.commons.util.array.IntArray;

/**
 * UDP-I/O受信処理をスレッドで行います.
 *
 * @version 1.00, 2005/03/24
 * @author  Masahito Suzuki
 * @since   JRcCommons 1.00
 */
class UdpIORcvThread extends ExecutionThread
{
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( UdpIORcvThread.class ) ;
    
    /**
     * 受信結果コールバック.
     */
    private UIOResultCallback m_rcvCallback = null ;
    
    /**
     * UDPプロトコル.
     */
    private UdpProtocol m_udp = null ;
    
    /**
     * UDP受信パケットテーブル.
     */
    private UdpRcvConnectTable m_rcvPacket = null ;
    
    /**
     * 受信完了データ格納テーブル.
     */
    private TelegramResourceRoll m_roll = null ;
    
    /**
     * 現在受信処理中ID.
     */
    private int m_nowID = -1 ;
    
    /**
     * 現在受信処理中ランダムID.
     */
    private int m_nowRandomID = -1 ;
    
    /**
     * 現在受信処理中パケット長.
     */
    private int m_nowRcvPacketLength = -1 ;
    
    /**
     * 受信電文タイムアウト.
     */
    private int m_rcvTimeout = -1 ;
    
    /**
     * ループスレッド.
     */
    private final LoopThread m_thread = new LoopThread() ;
    
    /**
     * 同期処理.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    /**
     * 電文受信用同期処理.
     */
    private Synchronized m_rcvSync = null ;
    
    /**
     * 一時バイナリ格納オブジェクト.
     */
    private final ByteUtil m_tBuf = new ByteUtil() ;
    
    
    
    /**
     * コンストラクタ.
     */
    public UdpIORcvThread(){}
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * UDP-I/O受信スレッドを生成します.
     * <BR>
     * @param rcall 受信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param udp 受信処理を行うUDPプロトコルを設定します.
     * @param rcvPacket 受信結果を取得するオブジェクトを設定します.
     * @param roll 受信電文ロールオブジェクトを設定します.
     * @param rcvSync 受信同期オブジェクトを設定します.
     * @exception InputException 入力例外.
     */
    public final void create(
        UIOResultCallback rcall,UdpProtocol udp,UdpRcvConnectTable rcvPacket,TelegramResourceRoll roll,
        Synchronized rcvSync
    )
        throws InputException
    {
        this.create(
            rcall,udp,rcvPacket,roll,rcvSync,
            UdpIODef.DEFAULT_PACKET,
            UdpIODef.RCV_PACKET_CLEAR_TIME
        ) ;
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * UDP-I/O受信スレッドを生成します.
     * <BR>
     * @param rcall 受信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param udp 受信処理を行うUDPプロトコルを設定します.
     * @param rcvPacket 受信結果を取得するオブジェクトを設定します.
     * @param roll 受信電文ロールオブジェクトを設定します.
     * @param rcvSync 受信同期オブジェクトを設定します.
     * @param packetLength 受信データ送信パケット長を設定します.
     * @exception InputException 入力例外.
     */
    public final void create(
        UIOResultCallback rcall,UdpProtocol udp,UdpRcvConnectTable rcvPacket,TelegramResourceRoll roll,
        Synchronized rcvSync,int packetLength
    )
        throws InputException
    {
        this.create(
            rcall,udp,rcvPacket,roll,rcvSync,packetLength,
            UdpIODef.RCV_PACKET_CLEAR_TIME
        ) ;
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * UDP-I/O受信スレッドを生成します.
     * <BR>
     * @param rcall 受信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param udp 受信処理を行うUDPプロトコルを設定します.
     * @param rcvPacket 受信結果を取得するオブジェクトを設定します.
     * @param roll 受信電文ロールオブジェクトを設定します.
     * @param rcvSync 受信同期オブジェクトを設定します.
     * @param packetLength 受信データ送信パケット長を設定します.
     * @param timeout 受信データ破棄タイムアウト値を設定します.<BR>
     *                受信電文が一定期間送信されない場合に破棄される時間を設定します.
     * @exception InputException 入力例外.
     */
    public final void create(
        UIOResultCallback rcall,UdpProtocol udp,UdpRcvConnectTable rcvPacket,TelegramResourceRoll roll,
        Synchronized rcvSync,int packetLength,int timeout
    )
        throws InputException
    {
        if(
            udp == null || udp.isOpen() == false ||
            rcvPacket == null || roll == null || rcvSync == null ||
            rcvSync.isUse() == false || timeout <= 0
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.clear() ;
        m_sync.create() ;
        
        packetLength = ( packetLength <= UdpIODef.MIN_PACKET ) ?
            UdpIODef.MIN_PACKET :
            (
                ( packetLength >= UdpIODef.MAX_PACKET ) ?
                    UdpIODef.MAX_PACKET : packetLength
            ) ;
        
        try{
            
            synchronized( m_sync.get() ){
                m_rcvCallback = rcall ;
                m_udp = udp ;
                m_rcvPacket = rcvPacket ;
                m_roll = roll ;
                m_rcvSync = rcvSync ;
                m_rcvTimeout = timeout ;
                m_nowRcvPacketLength = packetLength ;
            }
            
            m_thread.create( this ) ;
            m_thread.startThread() ;
            
        }catch( Exception e ){
            this.clear() ;
        }
        
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報を破棄します.
     */
    public final void clear()
    {
        m_sync.clear() ;
        m_thread.clear() ;
        m_tBuf.clear() ;
        
        m_rcvCallback = null ;
        m_udp = null ;
        m_rcvPacket = null ;
        m_roll = null ;
        m_rcvSync = null ;
        
        m_nowID = -1 ;
        m_nowRandomID = -1 ;
        m_nowRcvPacketLength = -1 ;
        m_rcvTimeout = -1 ;
        
    }
    
    /**
     * スレッド状態を取得.
     * <BR><BR>
     * スレッド状態を取得します.
     * <BR>
     * @return boolean スレッド状態が返されます.<BR>
     *                 [true]が返された場合、スレッドは実行中です.<BR>
     *                 [false]が返された場合、スレッドは停止中です.
     */
    public final boolean isThread()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_thread.isThread() ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * 実行初期化処理をサポートします.
     * <BR><BR>
     * 実行初期化処理をサポートします.<BR>
     * この処理は、スレッド処理が開始された時に呼び出されます.
     * <BR>
     * @param obj 実行開始時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void init( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行終了化処理をサポートします.
     * <BR><BR>
     * 実行終了化処理をサポートします.<BR>
     * この処理は、スレッド処理が終了された時に呼び出されます.
     * <BR>
     * @param obj 実行終了時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void exit( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * ストップ処理をサポートします。
     * <BR><BR>
     * ストップ処理をサポートします。<BR>
     * この処理は、スレッドでのストップ処理に対して呼び出し実行されます.
     * <BR>
     * @param obj ストップ時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void stop( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行処理をサポートします。
     * <BR><BR>
     * 実行処理をサポートします。<BR>
     * この処理は、スレッドでの実行処理に対して呼び出し実行されます.
     * <BR>
     * @param obj 実行時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void execution( Object obj )
        throws ExecutionException
    {
        int i ;
        
        int id ;
        int rndID ;
        int nowPacketLength ;
        int rcvTimeout ;
        int state ;
        long nowPacketTime ;
        long key ;
        boolean addResFlg ;
        
        byte[] rcv = null ;
        BinResource rcvBinResource = null ;
        UdpProtocol udp = null ;
        TelegramResourceRoll roll = null ;
        UdpRcvConnectTable rcvPacket = null ;
        Synchronized rcvSync = null ;
        ConnectAddress addr = null ;
        ByteUtil buf = null ;
        
        addResFlg = false ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                udp = m_udp ;
                roll = m_roll ;
                rcvPacket = m_rcvPacket ;
                rcvSync = m_rcvSync ;
                buf = m_tBuf ;
                
                rcvTimeout = m_rcvTimeout ;
                
                id = m_nowID ;
                rndID = m_nowRandomID ;
                nowPacketLength = m_nowRcvPacketLength ;
                
            }
            
            /////////////////////////////////////////
            // 受信対象IDが存在しない場合、取得する.
            /////////////////////////////////////////
            if( id == -1 ){
                
                // ID情報を取得.
                key = rcvPacket.getReceiveStartID() ;
                
                // ID情報が取得された場合.
                if( key != -1L ){
                    
                    id = UdpIOCommon.getReceiveID( key ) ;
                    rndID = UdpIOCommon.getReceiveRandomID( key ) ;
                    
                    // IDが有効な場合.
                    if( id >= 0 ){
                        
                        // 次回からのスレッド処理に対してIDを有効に設定.
                        synchronized( m_sync.get() ){
                            
                            m_nowID = id ;
                            m_nowRandomID = rndID ;
                            
                        }
                        
                    }
                    // IDが無効な場合.
                    else{
                        
                        // IDを無効にする.
                        id = -1 ;
                        rndID = -1 ;
                        
                    }
                    
                }
                
            }
            
            ////////////////////////////////////////////////////
            // 受信対象IDが存在する場合、受信ヘッダパケットから
            // 受信処理を行う.
            ////////////////////////////////////////////////////
            if( id != -1 && rcvPacket.isData( id,rndID ) == true ){
                
                // 受信ヘッダを取得.
                rcv = rcvPacket.getHeader( id,rndID,UdpIODef.PACKET_RECV_TIMEOUT ) ;
                
                //<><><><><><><><><><><><><><><>
                // ヘッダパケットが存在する場合.
                //<><><><><><><><><><><><><><><>
                if( rcv != null ){
                    
                    // 受信ヘッダステータスを取得.
                    state = UdpIOCommon.getPacketHeaderToCode( rcv ) ;
                    
                    // 受信ヘッダタイプにより処理を行う.
                    switch( state ){
                        
                        ////////////////////////////////////////
                        // 電文開始ステータスの場合
                        // 電文開始正常ステータスの場合.
                        // ロストパケット要求ステータスの場合.
                        ////////////////////////////////////////
                        case UdpIODef.FIRST_STATE_CODE :
                        case UdpIODef.FIRST_SUCCESS_STATE_CODE :
                        case UdpIODef.RESEND_STATE_CODE :
                            
                            // 無視.
                            break ;
                            
                        //////////////////////////////
                        // 電文エラーステータスの場合.
                        //////////////////////////////
                        case UdpIODef.ERROR_STATE_CODE :
                            
                            // エラーパケットをリフライ.
                            UdpIORcvThread.sendPacketState(
                                buf,rcvPacket,udp,id,rndID,-1,UdpIODef.ERROR_STATE_CODE
                            ) ;
                            
                            // 対象パケット条件を削除.
                            rcvPacket.cancell( id,rndID ) ;
                            break ;
                        
                        /////////////////////////////
                        // 送信終了ステータスの場合.
                        /////////////////////////////
                        case UdpIODef.END_STATE_CODE :
                            
                            //<><><><><><><><><><><><><><><><>
                            // ロストパッケージが存在する場合.
                            //<><><><><><><><><><><><><><><><>
                            if( rcvPacket.isLost( id,rndID ) == true ){
                                
                                // ロストパッケージ要求ヘッダパケットを送信.
                                UdpIORcvThread.sendLostPacket( udp,rcvPacket,buf,id,rndID,nowPacketLength ) ;
                                
                            }
                            //<><><><><><><><>
                            // 受信完了の場合.
                            //<><><><><><><><>
                            else{
                                
                                // ロールオブジェクトで同期.
                                synchronized( roll ){
                                    
                                    ////////////////////////////
                                    // データが設定不可能な場合.
                                    ////////////////////////////
                                    if( roll.isAdd() == false ){
                                        
                                        // ロールフルステータスコードを送信.
                                        UdpIORcvThread.sendPacketState(
                                            buf,rcvPacket,udp,id,rndID,-1,UdpIODef.ROLL_FULL_STATE_CODE
                                        ) ;
                                        
                                        // 情報の設定に失敗した場合.
                                        LOG.error(
                                            "対象の受信情報(id:" +
                                            id +
                                            " addr:" +
                                            addr.getAddress().getHostAddress() +
                                            " port:" +
                                            addr.getPort() +
                                            ")はロールフルであるため追加に失敗しました"
                                        ) ;
                                        
                                        // 対象パケット条件を削除.
                                        rcvPacket.cancell( id,rndID ) ;
                                        addResFlg = false ;
                                        
                                    }
                                    //////////////////////////
                                    // データが設定可能な場合.
                                    //////////////////////////
                                    else{
                                    
                                        // 受信データ完了ステータスを送信.
                                        UdpIORcvThread.sendPacketState(
                                            buf,rcvPacket,udp,id,rndID,-1,UdpIODef.END_STATE_CODE
                                        ) ;
                                        
                                        // 追加予約をONにセット.
                                        roll.addReservationByON() ;
                                        addResFlg = true ;
                                        
                                    }
                                    
                                }
                                
                                ///////////////////////
                                // 追加予約がONの場合.
                                ///////////////////////
                                if( addResFlg == true ){
                                    
                                    // パケットデータを合成.
                                    rcvBinResource = rcvPacket.getTelegramByResource( id,rndID ) ;
                                    
                                    // 受信データが存在する場合.
                                    if( rcvBinResource != null && rcvBinResource.isUse() == true ){
                                        
                                        // 対象IDのアドレスを取得.
                                        addr = rcvPacket.getConnect( id,rndID ) ;
                                        
                                        // ロールオブジェクトで同期.
                                        synchronized( roll ){
                                            
                                            // 追加予約をOFFにセット.
                                            roll.addReservationByOFF() ;
                                            addResFlg = false ;
                                            
                                            // 受信データを追加.
                                            roll.add( id,rcvBinResource,addr.getAddress(),addr.getPort() ) ;
                                            
                                        }
                                        
                                        // 電文情報が受信ロールに格納できた場合.
                                        // コールバックが存在する場合.
                                        if( m_rcvCallback != null ){
                                            
                                            // コールバック処理.
                                            m_rcvCallback.callback( id ) ;
                                            
                                        }
                                        
                                        // エラーを出力.
                                        LOG.debug(
                                            "対象の受信情報(id:" +
                                            id +
                                            " addr:" +
                                            addr.getAddress().getHostAddress() +
                                            " port:" +
                                            addr.getPort() +
                                            " size:" + rcvBinResource.size() +
                                            ")は完了しました"
                                        ) ;
                                        
                                        // 後始末.
                                        rcvBinResource = null ;
                                        rcv = null ;
                                        
                                        // 対象パケット条件を削除.
                                        rcvPacket.cancell( id,rndID ) ;
                                        
                                    }
                                    // 送信データが存在しない場合.
                                    else{
                                        
                                        // エラーを出力.
                                        LOG.error(
                                            "対象の受信情報(id:" +
                                            id +
                                            " addr:" +
                                            addr.getAddress().getHostAddress() +
                                            " port:" +
                                            addr.getPort() +
                                            ")は完了された受信電文の追加に失敗しました"
                                        ) ;
                                        
                                        // 対象パケット条件を削除.
                                        rcvPacket.cancell( id,rndID ) ;
                                        
                                    }
                                    
                                }
                                
                            }
                            
                            break ;
                        
                        // 不明ステータスの場合.
                        default :
                            
                            //<><><><><><><><><><><><><><><><><>
                            // ステータスを含むパケットヘッダは
                            // 送信しない.
                            //<><><><><><><><><><><><><><><><><>
                            
                            // 対象パケット条件を削除.
                            rcvPacket.cancell( id,rndID ) ;
                            
                            break ;
                            
                        
                    }
                
                }
                //<><><><><><><><><><><><><><><>
                // ヘッダ条件が存在しない場合.
                //<><><><><><><><><><><><><><><>
                else{
                    
                    // パケットテーブル更新時間をチェック.
                    nowPacketTime = rcvPacket.getLastTime( id,rndID ) + rcvTimeout ;
                    
                    // 更新時間がタイムアウトを上回った場合.
                    if( System.currentTimeMillis() >= nowPacketTime ){
                        
                        // 対象パケット条件を削除.
                        rcvPacket.cancell( id,rndID ) ;
                        
                    }
                    // 更新時間がタイムアウト未満の場合.
                    else{
                        
                        // 一時停止.
                        UtilCom.idleTime() ;
                        
                    }
                    
                }
                
            }
            
            // パケットテーブルに条件が存在しない場合.
            if( rcvPacket.isData( id,rndID ) == false ){
                
                synchronized( m_sync.get() ){
                    m_nowID = -1 ;
                    m_nowRandomID = -1 ;
                }
                
                UtilCom.idleTime() ;
            }
            
        }catch( NullPointerException nul ){
            throw new ExecutionException(
                nul,ExecutionException.LEVEL_STOP
            ) ;
        }catch( ExecutionException ee ){
            throw ee ;
        }catch( BaseException be ){
            LOG.error( "エラーが発生しました",be ) ;
        }catch( Exception e ){
            LOG.error( "エラーが発生しました",e ) ;
        }finally{
            
            // ロール追加予約がセットされたままの場合.
            if( addResFlg == true ){
                
                // ロール追加予約をOFF.
                if( roll != null ){
                    roll.addReservationByOFF() ;
                }
                
            }
            
            rcv = null ;
            rcvBinResource = null ;
            udp = null ;
            roll = null ;
            rcvPacket = null ;
            rcvSync = null ;
            addr = null ;
            buf = null ;
            
        }
        
    }
    
    /**
     * ロストパケットを再送.
     */
    private static final boolean sendLostPacket(
        UdpProtocol udp,UdpRcvConnectTable tbl,ByteUtil buf,
        int id,int rndID,int nowPacketLength
    )
    {
        int i ;
        int len ;
        int all ;
        boolean ret ;
        
        byte[] snd = null ;
        IntArray losts = null ;
        ConnectAddress addr = null ;
        
        try{
            
            // 送信先アドレスを取得.
            addr = tbl.getConnect( id,rndID ) ;
            
            // ロストパケット情報を設定.
            losts = tbl.getLostPacket( id,rndID ) ;
            
            // ロストパケット情報が存在する場合.
            if( losts != null && losts.size() > 0 ){
                
                // ロストパケット最大パケット数を取得.
                len = UdpIOCommon.getMaxLostPacket( nowPacketLength,losts ) ;
                
                // ロストパケット最大電文長を取得.
                all = UdpIOCommon.getMaxLostTelegramLength( nowPacketLength,losts ) ;
                
                // ロストパケットヘッダを生成して送信.
                UdpIOCommon.createPacketHeader( buf,id,len,rndID,all,nowPacketLength,UdpIODef.RESEND_STATE_CODE ) ;
                snd = buf.getToClear() ;
                
                // ヘッダ情報を送信.
                udp.send( snd,addr ) ;
                
                snd = null ;
                
                // ロスト電文IDパケットを送信.
                for( i = 0 ; i < len ; i ++ ){
                    
                    UdpIOCommon.createLostPacket( buf,id,i,rndID,nowPacketLength,losts ) ;
                    snd = buf.getToClear() ;
                    
                    udp.send( snd,addr ) ;
                    snd = null ;
                    
                }
                
                ret = true ;
                
            }
            // ロストパケット情報が存在しない場合.
            else{
                
                ret = false ;
                
            }
            
        }catch( Exception e ){
            ret = false ;
        }finally{
            if( losts != null ){
                losts.clear() ;
            }
            snd = null ;
            losts = null ;
            addr = null ;
        }
        
        return ret ;
    }
    
    /**
     * パケットステータスを送信.
     */
    private static final void sendPacketState( ByteUtil tbuf,UdpRcvConnectTable tbl,UdpProtocol udp,int id,int rndID,int packetLen,int state )
    {
        byte[] snd = null ;
        
        try{
            
            // パケットヘッダを送信.
            UdpIOCommon.createPacketHeader(
                tbuf,id,tbl.getMaxPacket( id,rndID ),rndID,
                tbl.getMaxTelegramLength( id,rndID ),packetLen,
                state
            ) ;
            
            snd = tbuf.getToClear() ;
            udp.send( snd,tbl.getConnect( id,rndID ) ) ;
            snd = null ;
            
        }catch( Exception e ){
        }finally{
            snd = null ;
        }
    }
    
}

