/*
 * @(#)UdpIOCommon.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.net.uio;

import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.resource.BinResource;
import com.JRcServer.commons.util.ByteUtil;
import com.JRcServer.commons.util.ConvertParam;
import com.JRcServer.commons.util.UtilCom;
import com.JRcServer.commons.util.array.IntArray;

/**
 * UDP-I/O共通関数.
 *
 * @version 1.00, 2005/03/24
 * @author  Masahito Suzuki
 * @since   JRcCommons 1.00
 */
class UdpIOCommon
{
    
    /**
     * ランダムID生成範囲.
     */
    private static final int RND_ID = 256 ;
    
    /**
     * 新しいランダムIDを生成.
     * <BR><BR>
     * 新しいランダムIDを生成します.
     * <BR>
     * @return int 新しいランダムIDが返されます.
     */
    public static final int getRandomID()
    {
        return ( int )(
            ( UtilCom.random( RND_ID ) & 0x000000ff ) |
            ( ( UtilCom.random( RND_ID ) & 0x000000ff ) << 8 ) |
            ( ( UtilCom.random( RND_ID ) & 0x000000ff ) << 16 ) |
            ( ( UtilCom.random( RND_ID ) & 0x000000ff ) << 24 )
        ) ;
    }
    
    /**
     * 共通データヘッダを生成.
     * <BR><BR>
     * 共通データヘッダを生成します.
     * <BR>
     * @param out 電文情報が返されます.
     * @param id 対象の電文IDを設定します.
     * @param subID 対象のサブIDを設定します.
     * @param randID ランダムIDを設定します.
     * @exception InputException 入力例外.
     */
    public static final void createCommonData( ByteUtil out,int id,int subID,int randID )
        throws InputException
    {
        
        out.clear() ;
        out.add( UdpIODef.PACKET_HEADER ) ;
        out.put( ConvertParam.convertInt( id ) ) ;
        out.put( ConvertParam.convertInt( subID ) ) ;
        out.put( ConvertParam.convertInt( randID ) ) ;
        
    }
    
    /**
     * パケットヘッダの生成.
     * <BR><BR>
     * パケットヘッダを生成します.<BR>
     * この情報は、電文送信時に最初に送られるコードです.
     * <BR>
     * @param out 電文情報が格納されます.
     * @param id 対象の電文IDを設定します.
     * @param max 最大パケット長を設定します.
     * @param randID ランダムIDを設定します.
     * @param all 送信電文長を設定します.
     * @param packetLen 対象１パケット長を設定します.
     * @param code ヘッダタイプを設定します.
     * @exception InputException 入力例外.
     */
    public static final void createPacketHeader( ByteUtil out,int id,int max,int randID,int all,int packetLen,int code )
        throws InputException
    {
        
        UdpIOCommon.createCommonData( out,id,UdpIODef.PACKET_HEADER_SUBID,randID ) ;
        out.put( ConvertParam.convertInt( max ) ) ;
        out.put( ConvertParam.convertInt( all ) ) ;
        out.put( ConvertParam.convertInt( packetLen ) ) ;
        out.put( ConvertParam.convertInt( code ) ) ;
        
    }
    
    /**
     * 送信電文を生成.
     * <BR><BR>
     * 送信電文を生成します.
     * <BR>
     * @param out 電文情報が格納されます.
     * @param id 対象の電文IDを設定します.
     * @param subID 対象のサブIDを設定します.
     * @param randID 対象のランダムIDを設定します.
     * @param plen 最大パケット長を設定します.
     * @raram data 送信対象のバイナリ情報を設定します.
     * @exception InputException 入力例外.
     */
    public static final void createTelegram( ByteUtil out,int id,int subID,int randID,int plen,BinResource data )
        throws InputException
    {
        int pnt ;
        int dlen ;
        int len ;
        
        byte[] packet = null ;
        
        dlen = data.size() ;
        len = plen - UdpIODef.PACKET_HEADER_LENGTH ;
        pnt = len * subID ;
        dlen -= pnt ;
        dlen = ( dlen >= len ) ? len : dlen ;
        
        UdpIOCommon.createCommonData( out,id,subID,randID ) ;
        
        packet = data.getBinary( pnt,dlen ) ;
        out.put( packet ) ;
        
        packet = null ;
        
    }
    
    /**
     * ロストしたパケット情報を再送する電文を生成.
     * <BR><BR>
     * ロストしたパケット情報を再送する電文を生成します.
     * <BR>
     * @param out 電文情報が格納されます.
     * @param id 対象の電文IDを設定します.
     * @param subID 対象のサブIDを設定します.
     * @param randID 対象のランダムIDを設定します.
     * @param plen 最大パケット長を設定します.
     * @param subIDs 再送要求する電文のサブID番号群を設定します.
     * @exception InputException 入力例外.
     */
    public static final void createLostPacket( ByteUtil out,int id,int subID,int randID,int plen,IntArray subIDs )
        throws InputException
    {
        int i,j ;
        int len ;
        int sLen ;
        int pnt ;
        int one ;
        
        if( subIDs == null || ( sLen = subIDs.size() ) <= 0 || plen <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        len = plen - UdpIODef.PACKET_HEADER_LENGTH ;
        one = ( len - 2 ) / 4 ;
        pnt = one * subID ;
        
        if( sLen <= pnt ){
            throw new InputException(
                "サブID位置(" + pnt +
                ")が、条件(" + sLen +
                ")を越しています"
            ) ;
        }
        
        UdpIOCommon.createCommonData( out,id,subID,randID ) ;
        
        len = sLen - pnt ;
        len = ( len >= one ) ? one : len ;
        
        out.add( ConvertParam.convertShort( ( short )len ) ) ;
        
        for( i = 0,j = pnt ; i < len ; i ++,j ++ ){
            
            out.add( ConvertParam.convertInt( subIDs.get( j ) ) ) ;
            
        }
        
    }
    
    /**
     * 電文情報に対するMaxSubID数を取得.
     * <BR><RB>
     * 電文情報に対するMaxSubID数を取得します.
     * <BR>
     * @param packetLength 対象のパケット長を設定します.
     * @param data 対象の電文情報を設定します.
     * @return int 対象電文情報に対するパケット長が返されます.
     */
    public static final int getMaxPacket( int packetLength,BinResource data )
    {
        
        int len ;
        int bLen ;
        int ret ;
        
        len = packetLength - UdpIODef.PACKET_HEADER_LENGTH ;
        bLen = data.size() ;
        ret = bLen / len ;
        return ( ( bLen % len ) != 0 ) ? ret + 1 : ret ;
        
    }
    
    /**
     * ロストパケット電文情報に対するMaxSubIDを取得.
     * <BR><BR>
     * ロストパケット電文情報に対するMaxSubID数を取得します.
     * <BR>
     * @param packetLength 対象のパケット長を設定します.
     * @param subIDs ロストパケット対象のサブID群を設定します.
     * @return int 対象ロストパケットに対するパケット長が返されます.
     */
    public static final int getMaxLostPacket( int packetLength,IntArray subIDs )
    {
        
        int len ;
        int sLen ;
        int one ;
        int ret ;
        
        if( subIDs == null || ( sLen = subIDs.size() ) <= 0 || packetLength <= 0 ){
            return -1 ;
        }
        
        len = packetLength - UdpIODef.PACKET_HEADER_LENGTH ;
        one = ( len - 2 ) / 4 ;
        ret = sLen / one ;
        ret += ( ( sLen % one ) != 0 ) ? 1 : 0 ;
        
        return ret ;
    }
    
    /**
     * ロストパケット電文情報に対する電文長を取得.
     * <BR><BR>
     * ロストパケット電文情報に対する電文長を取得します.
     * <BR>
     * @param packetLength 対象のパケット長を設定します.
     * @param subIDs ロストパケット対象のサブID群を設定します.
     * @return int 対象ロストパケットに対する電文長が返されます.
     */
    public static final int getMaxLostTelegramLength( int packetLength,IntArray subIDs )
    {
        
        int len ;
        int sLen ;
        int one ;
        int packet ;
        int etc ;
        int ret ;
        
        if( subIDs == null || ( sLen = subIDs.size() ) <= 0 || packetLength <= 0 ){
            return -1 ;
        }
        
        len = packetLength - UdpIODef.PACKET_HEADER_LENGTH ;
        one = ( len - 2 ) / 4 ;
        packet = sLen / one ;
        etc = sLen % one ;
        ret = packetLength * packet ;
        ret += ( etc != 0 ) ? ( ( etc * 4 ) + UdpIODef.PACKET_HEADER_LENGTH + 2 ) : 0 ;
        
        return ret ;
    }
    
    
    /**
     * パケット情報からメインIDを取得.
     * <BR><BR>
     * 対象のパケット情報から、メインIDを取得します.
     * <BR>
     * @param data 対象のパケットデータを設定します.
     * @return int パケット内のメインIDが返されます.<BR>
     *             [-1]が返された場合、不正です.
     */
    public static final int getPacketToID( byte[] data )
    {
        int ret ;
        
        try{
            ret = ConvertParam.convertInt( 2,data ) ;
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * パケット情報からサブIDを取得.
     * <BR><BR>
     * 対象のパケット情報から、サブIDを取得します.
     * <BR>
     * @param data 対象のパケットデータを設定します.
     * @return int パケット内のサブIDが返されます.<BR>
     *             [-1]が返された場合、不正か、ファーストパケットです.
     */
    public static final int getPacketToSubID( byte[] data )
    {
        int ret ;
        
        try{
            ret = ConvertParam.convertInt( 6,data ) ;
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * パケット情報からランダムIDを取得.
     * <BR><BR>
     * 対象のパケット情報から、ランダムIDを取得します.
     * <BR>
     * @param data 対象のパケットデータを設定します.
     * @return int パケット内のランダムIDが返されます.<BR>
     *             [-1]が返された場合、不正です.
     */
    public static final int getPacketToRandID( byte[] data )
    {
        int ret ;
        
        try{
            ret = ConvertParam.convertInt( 10,data ) ;
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    
    /**
     * パケットヘッダ電文から最大パケット長を取得.
     * <BR><BR>
     * 対象のパケットヘッダ電文から、最大パケット長を取得します.
     * <BR>
     * @param data 対象のパケットデータを設定します.
     * @return int パケット内の最大パケット長が返されます.<BR>
     *             [-1]が返された場合、不正です.
     */
    public static final int getPacketHeaderToMaxPacket( byte[] data )
    {
        int ret ;
        
        try{
            ret = ConvertParam.convertInt( 14,data ) ;
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * パケットヘッダ電文から電文長を取得.
     * <BR><BR>
     * 対象のパケットヘッダ電文から、電文長を取得します.<BR>
     * また、接続確認電文の場合は、受信側の接続数が返されます.
     * <BR>
     * @param data 対象のパケットデータを設定します.
     * @return int パケット内の最大パケット長が返されます.<BR>
     *             [-1]が返された場合、不正です.
     */
    public static final int getPacketHeaderToTelegramLength( byte[] data )
    {
        int ret ;
        
        try{
            ret = ConvertParam.convertInt( 18,data ) ;
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * パケットヘッダ電文から１パケット長を取得.
     * <BR><BR>
     * 対象のパケットヘッダ電文から、１パケット長を取得します.
     * <BR>
     * @param data 対象のパケットデータを設定します.
     * @return int パケット内の１パケット長が返されます.<BR>
     *             [-1]が返された場合、不正です.
     */
    public static final int getPacketHeaderToOnePacket( byte[] data )
    {
        int ret ;
        
        try{
            ret = ConvertParam.convertInt( 22,data ) ;
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * パケットヘッダ電文からヘッダタイプを取得.
     * <BR><BR>
     * 対象のパケットヘッダ電文から、ヘッダタイプを取得します.
     * <BR>
     * @param data 対象のパケットデータを設定します.
     * @return int パケット内のヘッダタイプが返されます.<BR>
     *             [UdpIODef.NOT_STATE_CODE]が返された場合、不正です.
     */
    public static final int getPacketHeaderToCode( byte[] data )
    {
        int ret ;
        
        try{
            ret = ConvertParam.convertInt( 26,data ) ;
        }catch( Exception e ){
            ret = UdpIODef.NOT_STATE_CODE ;
        }
        
        return ret ;
    }
    
    
    /**
     * 対象パケットが正常であるかチェック.
     * <BR><BR>
     * 対象のパケット情報が正常であるかチェックします.
     * <BR>
     * @param data チェック対象のパケット情報を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、パケットは正常です.
     *                 [false]が返された場合、パケットは異常です.
     */
    public static final boolean isPacket( byte[] data )
    {
        return (
            data == null ||
            data.length < UdpIODef.PACKET_HEADER_LENGTH ||
            data[ 0 ] != UdpIODef.PACKET_HEADER[ 0 ] ||
            data[ 1 ] != UdpIODef.PACKET_HEADER[ 1 ]
        ) ? false : true ;
    }
    
    /**
     * パケットヘッダをチェック.
     * <BR><BR>
     * パケットヘッダをチェックします.
     * <BR>
     * @param id IDを設定します.
     * @param max パケット数を設定します.
     * @param randID ランダムIDを設定します.
     * @param data チェック対象のパケットを設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、パケットヘッダです.<BR>
     *                 [false]が返された場合、パケットヘッダではありません.
     */
    public static final boolean isPacketHeader( int id,int max,int randID,byte[] data )
    {
        boolean ret ;
        
        try{
            ret =(
                data[ 0 ] == UdpIODef.PACKET_HEADER[ 0 ] &&
                data[ 1 ] == UdpIODef.PACKET_HEADER[ 1 ] &&
                ConvertParam.convertInt( 2,data ) == id &&
                ConvertParam.convertInt( 6,data ) == UdpIODef.PACKET_HEADER_SUBID &&
                ConvertParam.convertInt( 10,data ) == randID &&
                ConvertParam.convertInt( 14,data ) == max
            ) ? true : false ;
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 受信開始IDから、パケットIDを取得.
     * <BR><BR>
     * 受信開始IDから、パケットIDを取得します.
     * <BR>
     * @param rcvst 対象の受信開始IDを設定します.
     * @return int パケットIDが返されます.
     */
    public static final int getReceiveID( long rcvst )
    {
        if( rcvst == -1L ){
            return -1 ;
        }
        else{
            return ( int )( rcvst & 0x00000000ffffffffL ) ;
        }
    }
    
    /**
     * 受信開始IDから、パケットランダムIDを取得.
     * <BR><BR>
     * 受信開始IDから、パケットランダムIDを取得します.
     * <BR>
     * @param rcvst 対象の受信開始IDを設定します.
     * @return int パケットランダムIDが返されます.
     */
    public static final int getReceiveRandomID( long rcvst )
    {
        if( rcvst == -1L ){
            return -1 ;
        }
        else{
            return ( int )( ( rcvst & 0xffffffff00000000L ) >> 32L ) ;
        }
    }
    
    /**
     * 対象エラーコードを文字列に変換.
     * <BR><BR>
     * 対象エラーコードを文字列に変換します.
     * <BR>
     * @param code 対象のエラーコードを設定します.
     * @return String 変換された文字列が返されます.
     */
    public static final String getErrorCodeByString( int code )
    {
        String ret = null ;
        switch( code ){
            case UdpIODef.ERROR_STATE_CODE :
                ret = "エラー(0x" + Integer.toHexString( code ) + ")" ;
                break ;
            case UdpIODef.FIRST_STATE_CODE :
                ret = "開始電文(0x" + Integer.toHexString( code ) + ")" ;
                break ;
            case UdpIODef.FIRST_ERROR_STATE_CODE :
                ret = "開始電文エラー(0x" + Integer.toHexString( code ) + ")" ;
                break ;
            case UdpIODef.FIRST_SUCCESS_STATE_CODE :
                ret = "開始電文完了(0x" + Integer.toHexString( code ) + ")" ;
                break ;
            case UdpIODef.END_STATE_CODE :
                ret = "送信完了(0x" + Integer.toHexString( code ) + ")" ;
                break ;
            case UdpIODef.RESEND_STATE_CODE :
                ret = "ロストパッケージ要求(0x" + Integer.toHexString( code ) + ")" ;
                break ;
            case UdpIODef.ROLL_FULL_STATE_CODE :
                ret = "ロールフル(0x" + Integer.toHexString( code ) + ")" ;
                break ;
            case UdpIODef.FIRST_NOTICE_OK_STATE_CODE :
                ret = "通知確認OK(0x" + Integer.toHexString( code ) + ")" ;
                break ;
            default :
                ret = "不明(0x" + Integer.toHexString( code ) + ")" ;
                break ;
        }
        
        return ret ;
    }
    
}

