/*
 * @(#)UIO.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.net.uio;

import java.net.InetAddress;

import com.JRcServer.commons.exception.ExecutionException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.net.ConnectAddress;
import com.JRcServer.commons.net.ConnectTimeoutException;
import com.JRcServer.commons.net.TelegramResourceRoll;
import com.JRcServer.commons.net.TelegramValue;
import com.JRcServer.commons.net.UdpProtocol;
import com.JRcServer.commons.resource.BinResource;
import com.JRcServer.commons.resource.ResourceType;
import com.JRcServer.commons.thread.Synchronized;
import com.JRcServer.commons.util.SequenceID;
import com.JRcServer.commons.util.UtilCom;

/**
 * UDP-I/Oオブジェクト.
 * <BR><BR>
 * 非同期通信を行う処理をサポートします.<BR>
 * このオブジェクトはUDPプロトコルによる長文送信/受信を可能にした
 * 通信プロトコルです.通常UDPでは、通信到達を保証しませんがこのオブジェクトは
 * 通信到達を保証します.<BR>
 * しかしUDPであることから、到達順位は保証されません.<BR>
 * このオブジェクトの利点として、TCP/IPとは違い相互接続数が多い場合、コストが少なくて済みます.<BR>
 * 利用状況に応じて利用してください.<BR>
 * また、送信ポートと受信ポートをデフォルト値として設定した場合は、以下のポート番号が割り当てられます.<BR>
 * Default-Send-Port=[22055].<BR>
 * Default-Receive-Port=[22155].
 *
 * @version 1.00, 2005/03/30
 * @author  Masahito Suzuki
 * @since   JRcCommons 1.00
 */
public class UIO
{
    
    /**
     * 送信用UDPオブジェクト.
     */
    private final UdpProtocol m_sndPrt = new UdpProtocol() ;
    
    /**
     * 送信用受信パケット格納テーブル.
     */
    private final UdpSndResultTable m_sndResult = new UdpSndResultTable() ;
    
    /**
     * 送信データ管理オブジェクト.
     */
    private TelegramResourceRoll m_sndMan = null ;
    
    /**
     * 送信用同期オブジェクト.
     */
    private final Synchronized m_sndSync = new Synchronized() ;
    
    /**
     * 送信ID管理オブジェクト.
     */
    private final SequenceID m_sndSeq = new SequenceID() ;
    
    /**
     * 送信結果処理用スレッド.
     */
    private UdpSndResultThread[] m_sndResultThreads = null ;
    
    /**
     * 送信処理用スレッド.
     */
    private UdpIOSndThread[] m_sndThreads = null ;
    
    /**
     * 送信結果コールバック.
     */
    private UIOResultCallback m_sendCallback = null ;
    
    
    
    
    /**
     * 受信用UDPオブジェクト.
     */
    private final UdpProtocol m_rcvPrt = new UdpProtocol() ;
    
    /**
     * 受信パケット管理用テーブル.
     */
    private final UdpRcvConnectTable m_rcvPacket = new UdpRcvConnectTable() ;
    
    /**
     * 受信データ管理オブジェクト.
     */
    private TelegramResourceRoll m_rcvMan = null ;
    
    /**
     * 受信用同期オブジェクト.
     */
    private final Synchronized m_rcvSync = new Synchronized() ;
    
    /**
     * 受信パケット処理用スレッド.
     */
    private UdpRcvConnectThread[] m_rcvPacketThreads = null ;
    
    /**
     * 受信処理用スレッド.
     */
    private UdpIORcvThread[] m_rcvThreads = null ;
    
    /**
     * 受信結果コールバック.
     */
    private UIOResultCallback m_rcvCallback = null ;
    
    
    
    
    /**
     * パケット長.
     */
    private int m_packetLength = -1 ;
    
    /**
     * 受信パケット破棄時間.
     */
    private int m_destroyPacketTime = -1 ;
    
    /**
     * 送信フリップ値.
     */
    private int m_sendFlip = 0 ;
    
    /**
     * 通常同期.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    
    
    /**
     * コンストラクタ.
     */
    public UIO(){}
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 通信処理をオープンします.
     * <BR>
     * @param scall 送信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param rcall 受信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @exception InputException 入力例外.
     */
    public UIO( UIOResultCallback scall,UIOResultCallback rcall )
        throws InputException
    {
        try{
            this.open(
                scall,null,-1,UdpIODef.DEFAULT_CORE_THREAD_SIZE,UdpIODef.DEFAULT_THREAD_SIZE,UdpIODef.DEFAULT_UDP_BUFFER,
                rcall,null,-1,UdpIODef.DEFAULT_CORE_THREAD_SIZE,UdpIODef.DEFAULT_THREAD_SIZE,UdpIODef.DEFAULT_UDP_BUFFER,
                UdpIODef.RCV_PACKET_CLEAR_TIME,UdpIODef.DEFAULT_PACKET,-1,-1,null,null
            ) ;
        }catch( InputException in ){
            this.close() ;
            throw in ;
        }catch( Exception e ){
            this.close() ;
        }
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 通信処理をオープンします.<BR>
     * また、この処理の場合送信ポートと受信ポートはデフォルト値が
     * 設定されます.<BR>
     * デフォルトの送信ポートは[22055]です.<BR>
     * デフォルトの受信ポートは[22155]です.
     * <BR>
     * @param maxRcvRoll 受信ロール管理数を設定します.<BR>
     *                   設定可能な最小値は[32]です.これ以上小さな値を
     *                   設定しても意味を持ちません.<BR>
     *                   設定可能な最大数は[9999]です.
     * @exception InputException 入力例外.
     */
    public UIO( int maxRcvRoll )
        throws InputException
    {
        try{
            this.open(
                null,null,-1,UdpIODef.DEFAULT_CORE_THREAD_SIZE,UdpIODef.DEFAULT_THREAD_SIZE,UdpIODef.DEFAULT_UDP_BUFFER,
                null,null,-1,UdpIODef.DEFAULT_CORE_THREAD_SIZE,UdpIODef.DEFAULT_THREAD_SIZE,UdpIODef.DEFAULT_UDP_BUFFER,
                UdpIODef.RCV_PACKET_CLEAR_TIME,UdpIODef.DEFAULT_PACKET,-1,maxRcvRoll,null,null
            ) ;
        }catch( InputException in ){
            this.close() ;
            throw in ;
        }catch( Exception e ){
            this.close() ;
        }
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 通信処理をオープンします.
     * <BR>
     * @param scall 送信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param rcall 受信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param maxRcvRoll 受信ロール管理数を設定します.<BR>
     *                   設定可能な最小値は[32]です.これ以上小さな値を
     *                   設定しても意味を持ちません.<BR>
     *                   設定可能な最大数は[9999]です.
     * @exception InputException 入力例外.
     */
    public UIO( UIOResultCallback scall,UIOResultCallback rcall,int maxRcvRoll )
        throws InputException
    {
        try{
            this.open(
                scall,null,-1,UdpIODef.DEFAULT_CORE_THREAD_SIZE,UdpIODef.DEFAULT_THREAD_SIZE,UdpIODef.DEFAULT_UDP_BUFFER,
                rcall,null,-1,UdpIODef.DEFAULT_CORE_THREAD_SIZE,UdpIODef.DEFAULT_THREAD_SIZE,UdpIODef.DEFAULT_UDP_BUFFER,
                UdpIODef.RCV_PACKET_CLEAR_TIME,UdpIODef.DEFAULT_PACKET,-1,maxRcvRoll,null,null
            ) ;
        }catch( InputException in ){
            this.close() ;
            throw in ;
        }catch( Exception e ){
            this.close() ;
        }
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 通信処理をオープンします.
     * <BR>
     * @param sndPort 送信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param rcvPort 受信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param packetLen パケットサイズを設定します.<BR>
     *                  設定可能な最小値は[512]です.<BR>
     *                  設定可能な最大値は[8192]です.
     * @param maxRcvRoll 受信ロール管理数を設定します.<BR>
     *                   設定可能な最小値は[32]です.これ以上小さな値を
     *                   設定しても意味を持ちません.<BR>
     *                   設定可能な最大数は[9999]です.<BR>
     * @exception InputException 入力例外.
     */
    public UIO( int sndPort,int rcvPort,int packetLen,int maxRcvRoll )
        throws InputException
    {
        try{
            this.open(
                null,null,sndPort,UdpIODef.DEFAULT_CORE_THREAD_SIZE,UdpIODef.DEFAULT_THREAD_SIZE,UdpIODef.DEFAULT_UDP_BUFFER,
                null,null,rcvPort,UdpIODef.DEFAULT_CORE_THREAD_SIZE,UdpIODef.DEFAULT_THREAD_SIZE,UdpIODef.DEFAULT_UDP_BUFFER,
                UdpIODef.RCV_PACKET_CLEAR_TIME,packetLen,-1,maxRcvRoll,null,null
            ) ;
        }catch( InputException in ){
            this.close() ;
            throw in ;
        }catch( Exception e ){
            this.close() ;
        }
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 通信処理をオープンします.
     * <BR>
     * @param scall 送信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param sndPort 送信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param rcall 受信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param rcvPort 受信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param packetLen パケットサイズを設定します.<BR>
     *                  設定可能な最小値は[512]です.<BR>
     *                  設定可能な最大値は[8192]です.
     * @param maxRcvRoll 受信ロール管理数を設定します.<BR>
     *                   設定可能な最小値は[32]です.これ以上小さな値を
     *                   設定しても意味を持ちません.<BR>
     *                   設定可能な最大数は[9999]です.<BR>
     * @exception InputException 入力例外.
     */
    public UIO( UIOResultCallback scall,int sndPort,UIOResultCallback rcall,int rcvPort,int packetLen,int maxRcvRoll )
        throws InputException
    {
        try{
            this.open(
                scall,null,sndPort,UdpIODef.DEFAULT_CORE_THREAD_SIZE,UdpIODef.DEFAULT_THREAD_SIZE,UdpIODef.DEFAULT_UDP_BUFFER,
                rcall,null,rcvPort,UdpIODef.DEFAULT_CORE_THREAD_SIZE,UdpIODef.DEFAULT_THREAD_SIZE,UdpIODef.DEFAULT_UDP_BUFFER,
                UdpIODef.RCV_PACKET_CLEAR_TIME,packetLen,-1,maxRcvRoll,null,null
            ) ;
        }catch( InputException in ){
            this.close() ;
            throw in ;
        }catch( Exception e ){
            this.close() ;
        }
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 通信処理をオープンします.
     * <BR>
     * @param sndPort 送信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param rcvPort 受信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param packetLen パケットサイズを設定します.<BR>
     *                  設定可能な最小値は[512]です.<BR>
     *                  設定可能な最大値は[8192]です.
     * @param maxRcvRoll 受信ロール管理数を設定します.<BR>
     *                   設定可能な最小値は[32]です.これ以上小さな値を
     *                   設定しても意味を持ちません.<BR>
     *                   設定可能な最大数は[9999]です.<BR>
     * @param resourceType リソースタイプを設定します.
     * @exception InputException 入力例外.
     */
    public UIO( int sndPort,int rcvPort,int packetLen,int maxRcvRoll,ResourceType resourceType )
        throws InputException
    {
        try{
            this.open(
                null,null,sndPort,UdpIODef.DEFAULT_CORE_THREAD_SIZE,UdpIODef.DEFAULT_THREAD_SIZE,UdpIODef.DEFAULT_UDP_BUFFER,
                null,null,rcvPort,UdpIODef.DEFAULT_CORE_THREAD_SIZE,UdpIODef.DEFAULT_THREAD_SIZE,UdpIODef.DEFAULT_UDP_BUFFER,
                UdpIODef.RCV_PACKET_CLEAR_TIME,packetLen,-1,maxRcvRoll,resourceType,resourceType
            ) ;
        }catch( InputException in ){
            this.close() ;
            throw in ;
        }catch( Exception e ){
            this.close() ;
        }
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 通信処理をオープンします.
     * <BR>
     * @param scall 送信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param sndPort 送信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param rcall 受信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param rcvPort 受信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param packetLen パケットサイズを設定します.<BR>
     *                  設定可能な最小値は[512]です.<BR>
     *                  設定可能な最大値は[8192]です.
     * @param maxRcvRoll 受信ロール管理数を設定します.<BR>
     *                   設定可能な最小値は[32]です.これ以上小さな値を
     *                   設定しても意味を持ちません.<BR>
     *                   設定可能な最大数は[9999]です.<BR>
     * @param resourceType リソースタイプを設定します.
     * @exception InputException 入力例外.
     */
    public UIO(
        UIOResultCallback scall,int sndPort,UIOResultCallback rcall,int rcvPort,
        int packetLen,int maxRcvRoll,ResourceType resourceType
    )
        throws InputException
    {
        try{
            this.open(
                scall,null,sndPort,UdpIODef.DEFAULT_CORE_THREAD_SIZE,UdpIODef.DEFAULT_THREAD_SIZE,UdpIODef.DEFAULT_UDP_BUFFER,
                rcall,null,rcvPort,UdpIODef.DEFAULT_CORE_THREAD_SIZE,UdpIODef.DEFAULT_THREAD_SIZE,UdpIODef.DEFAULT_UDP_BUFFER,
                UdpIODef.RCV_PACKET_CLEAR_TIME,packetLen,-1,maxRcvRoll,resourceType,resourceType
            ) ;
        }catch( InputException in ){
            this.close() ;
            throw in ;
        }catch( Exception e ){
            this.close() ;
        }
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 通信処理をオープンします.
     * <BR>
     * @param sndPort 送信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param sndThread 送信スレッドサイズを設定します.<BR>
     *                  設定可能な最小値は[1]です.<BR>
     *                  設定可能な最大値は[8]です.
     * @param rcvPort 受信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param rcvThread 受信スレッドサイズを設定します.<BR>
     *                  設定可能な最小値は[1]です.<BR>
     *                  設定可能な最大値は[8]です.
     * @param packetLen パケットサイズを設定します.<BR>
     *                  設定可能な最小値は[512]です.<BR>
     *                  設定可能な最大値は[8192]です.
     * @param maxRcvRoll 受信ロール管理数を設定します.<BR>
     *                   設定可能な最小値は[32]です.これ以上小さな値を
     *                   設定しても意味を持ちません.<BR>
     *                   設定可能な最大数は[9999]です.<BR>
     * @exception InputException 入力例外.
     */
    public UIO( int sndPort,int sndThread,int rcvPort,int rcvThread,int packetLen,int maxRcvRoll )
        throws InputException
    {
        try{
            this.open(
                null,null,sndPort,( int )( ( double )sndThread / 2.0f + 0.5f ),sndThread,UdpIODef.DEFAULT_UDP_BUFFER,
                null,null,rcvPort,( int )( ( double )rcvThread / 2.0f + 0.5f ),rcvThread,UdpIODef.DEFAULT_UDP_BUFFER,
                UdpIODef.RCV_PACKET_CLEAR_TIME,packetLen,-1,maxRcvRoll,null,null
            ) ;
        }catch( InputException in ){
            this.close() ;
            throw in ;
        }catch( Exception e ){
            this.close() ;
        }
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 通信処理をオープンします.
     * <BR>
     * @param scall 送信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param sndPort 送信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param sndThread 送信スレッドサイズを設定します.<BR>
     *                  設定可能な最小値は[1]です.<BR>
     *                  設定可能な最大値は[8]です.
     * @param rcall 受信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param rcvPort 受信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param rcvThread 受信スレッドサイズを設定します.<BR>
     *                  設定可能な最小値は[1]です.<BR>
     *                  設定可能な最大値は[8]です.
     * @param packetLen パケットサイズを設定します.<BR>
     *                  設定可能な最小値は[512]です.<BR>
     *                  設定可能な最大値は[8192]です.
     * @param maxRcvRoll 受信ロール管理数を設定します.<BR>
     *                   設定可能な最小値は[32]です.これ以上小さな値を
     *                   設定しても意味を持ちません.<BR>
     *                   設定可能な最大数は[9999]です.<BR>
     * @exception InputException 入力例外.
     */
    public UIO(
        UIOResultCallback scall,int sndPort,int sndThread,UIOResultCallback rcall,int rcvPort,int rcvThread,
        int packetLen,int maxRcvRoll
    )
        throws InputException
    {
        try{
            this.open(
                scall,null,sndPort,( int )( ( double )sndThread / 2.0f + 0.5f ),sndThread,UdpIODef.DEFAULT_UDP_BUFFER,
                rcall,null,rcvPort,( int )( ( double )rcvThread / 2.0f + 0.5f ),rcvThread,UdpIODef.DEFAULT_UDP_BUFFER,
                UdpIODef.RCV_PACKET_CLEAR_TIME,packetLen,-1,maxRcvRoll,null,null
            ) ;
        }catch( InputException in ){
            this.close() ;
            throw in ;
        }catch( Exception e ){
            this.close() ;
        }
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 通信処理をオープンします.
     * <BR>
     * @param sndPort 送信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param sndThread 送信スレッドサイズを設定します.<BR>
     *                  設定可能な最小値は[1]です.<BR>
     *                  設定可能な最大値は[8]です.
     * @param rcvPort 受信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param rcvThread 受信スレッドサイズを設定します.<BR>
     *                  設定可能な最小値は[1]です.<BR>
     *                  設定可能な最大値は[8]です.
     * @param packetLen パケットサイズを設定します.<BR>
     *                  設定可能な最小値は[512]です.<BR>
     *                  設定可能な最大値は[8192]です.
     * @param maxRcvRoll 受信ロール管理数を設定します.<BR>
     *                   設定可能な最小値は[32]です.これ以上小さな値を
     *                   設定しても意味を持ちません.<BR>
     *                   設定可能な最大数は[9999]です.<BR>
     * @param resourceType リソースタイプを設定します.
     * @exception InputException 入力例外.
     */
    public UIO( int sndPort,int sndThread,int rcvPort,int rcvThread,int packetLen,int maxRcvRoll,ResourceType resourceType )
        throws InputException
    {
        try{
            this.open(
                null,null,sndPort,( int )( ( double )sndThread / 2.0f + 0.5f ),sndThread,UdpIODef.DEFAULT_UDP_BUFFER,
                null,null,rcvPort,( int )( ( double )rcvThread / 2.0f + 0.5f ),rcvThread,UdpIODef.DEFAULT_UDP_BUFFER,
                UdpIODef.RCV_PACKET_CLEAR_TIME,packetLen,-1,maxRcvRoll,resourceType,resourceType
            ) ;
        }catch( InputException in ){
            this.close() ;
            throw in ;
        }catch( Exception e ){
            this.close() ;
        }
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 通信処理をオープンします.
     * <BR>
     * @param scall 送信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param sndPort 送信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param sndThread 送信スレッドサイズを設定します.<BR>
     *                  設定可能な最小値は[1]です.<BR>
     *                  設定可能な最大値は[8]です.
     * @param rcall 受信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param rcvPort 受信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param rcvThread 受信スレッドサイズを設定します.<BR>
     *                  設定可能な最小値は[1]です.<BR>
     *                  設定可能な最大値は[8]です.
     * @param packetLen パケットサイズを設定します.<BR>
     *                  設定可能な最小値は[512]です.<BR>
     *                  設定可能な最大値は[8192]です.
     * @param maxRcvRoll 受信ロール管理数を設定します.<BR>
     *                   設定可能な最小値は[32]です.これ以上小さな値を
     *                   設定しても意味を持ちません.<BR>
     *                   設定可能な最大数は[9999]です.<BR>
     * @param resourceType リソースタイプを設定します.
     * @exception InputException 入力例外.
     */
    public UIO(
        UIOResultCallback scall,int sndPort,int sndThread,UIOResultCallback rcall,int rcvPort,int rcvThread,
        int packetLen,int maxRcvRoll,ResourceType resourceType
    )
        throws InputException
    {
        try{
            this.open(
                scall,null,sndPort,( int )( ( double )sndThread / 2.0f + 0.5f ),sndThread,UdpIODef.DEFAULT_UDP_BUFFER,
                rcall,null,rcvPort,( int )( ( double )rcvThread / 2.0f + 0.5f ),rcvThread,UdpIODef.DEFAULT_UDP_BUFFER,
                UdpIODef.RCV_PACKET_CLEAR_TIME,packetLen,-1,maxRcvRoll,resourceType,resourceType
            ) ;
        }catch( InputException in ){
            this.close() ;
            throw in ;
        }catch( Exception e ){
            this.close() ;
        }
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 通信処理をオープンします.
     * <BR>
     * @param sndAddr 送信バインド先アドレスを設定します.<BR>
     *                [null]を設定した場合、デフォルト値となります.
     * @param sndPort 送信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param sndCThread 送信コアスレッドサイズを設定します.<BR>
     *                   設定可能な最小値は[1]です.<BR>
     *                   設定可能な最大値は[8]です.
     * @param sndThread 送信スレッドサイズを設定します.<BR>
     *                  設定可能な最小値は[1]です.<BR>
     *                  設定可能な最大値は[8]です.
     * @param sndBuffer 送信バッファ長を設定します.<BR>
     *                  設定単位は[バイト]です.
     * @param rcvAddr 受信バインド先アドレスを設定します.<BR>
     *                [null]を設定した場合、デフォルト値となります.
     * @param rcvPort 受信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param rcvCThread 受信コアスレッドサイズを設定します.<BR>
     *                   設定可能な最小値は[1]です.<BR>
     *                   設定可能な最大値は[8]です.
     * @param rcvThread 受信スレッドサイズを設定します.<BR>
     *                  設定可能な最小値は[1]です.<BR>
     *                  設定可能な最大値は[8]です.
     * @param rcvBuffer 受信バッファ長を設定します.<BR>
     *                  設定単位は[バイト]です.
     * @param rcvDestroyTime 受信パケット削除時間を設定します.<BR>
     *                       この値は0以上を指定する必要があります.<BR>
     *                       また、設定単位は[ミリ秒]です.
     * @param packetLen パケットサイズを設定します.<BR>
     *                  設定可能な最小値は[512]です.<BR>
     *                  設定可能な最大値は[8192]です.
     * @param maxSndRoll 送信ロール管理数を設定します.<BR>
     *                   設定可能な最小値は[32]です.これ以上小さな値を
     *                   設定しても意味を持ちません.<BR>
     *                   設定可能な最大数は[9999]です.<BR>
     * @param maxRcvRoll 受信ロール管理数を設定します.<BR>
     *                   設定可能な最小値は[32]です.これ以上小さな値を
     *                   設定しても意味を持ちません.<BR>
     *                   設定可能な最大数は[9999]です.<BR>
     * @param sndResourceType 送信リソースタイプを設定します.
     * @param rcvResourceType 受信リソースタイプを設定します.
     * @exception InputException 入力例外.
     */
    public UIO(
        InetAddress sndAddr,int sndPort,int sndCThread,int sndThread,int sndBuffer,
        InetAddress rcvAddr,int rcvPort,int rcvCThread,int rcvThread,int rcvBuffer,
        int rcvDestroyTime,int packetLen,int maxSndRoll,int maxRcvRoll,
        ResourceType sndResourceType,ResourceType rcvResourceType
    )
        throws InputException
    {
        try{
            this.open(
                null,sndAddr,sndPort,sndCThread,sndThread,sndBuffer,
                null,rcvAddr,rcvPort,rcvCThread,rcvThread,rcvBuffer,
                rcvDestroyTime,packetLen,maxSndRoll,maxRcvRoll,
                sndResourceType,rcvResourceType
            ) ;
        }catch( InputException in ){
            this.close() ;
            throw in ;
        }catch( Exception e ){
            this.close() ;
        }
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 通信処理をオープンします.
     * <BR>
     * @param scall 送信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param sndAddr 送信バインド先アドレスを設定します.<BR>
     *                [null]を設定した場合、デフォルト値となります.
     * @param sndPort 送信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param sndCThread 送信コアスレッドサイズを設定します.<BR>
     *                   設定可能な最小値は[1]です.<BR>
     *                   設定可能な最大値は[8]です.
     * @param sndThread 送信スレッドサイズを設定します.<BR>
     *                  設定可能な最小値は[1]です.<BR>
     *                  設定可能な最大値は[8]です.
     * @param sndBuffer 送信バッファ長を設定します.<BR>
     *                  設定単位は[バイト]です.
     * @param rcall 受信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param rcvAddr 受信バインド先アドレスを設定します.<BR>
     *                [null]を設定した場合、デフォルト値となります.
     * @param rcvPort 受信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param rcvCThread 受信コアスレッドサイズを設定します.<BR>
     *                   設定可能な最小値は[1]です.<BR>
     *                   設定可能な最大値は[8]です.
     * @param rcvThread 受信スレッドサイズを設定します.<BR>
     *                  設定可能な最小値は[1]です.<BR>
     *                  設定可能な最大値は[8]です.
     * @param rcvBuffer 受信バッファ長を設定します.<BR>
     *                  設定単位は[バイト]です.
     * @param rcvDestroyTime 受信パケット削除時間を設定します.<BR>
     *                       この値は0以上を指定する必要があります.<BR>
     *                       また、設定単位は[ミリ秒]です.
     * @param packetLen パケットサイズを設定します.<BR>
     *                  設定可能な最小値は[512]です.<BR>
     *                  設定可能な最大値は[8192]です.
     * @param maxSndRoll 送信ロール管理数を設定します.<BR>
     *                   設定可能な最小値は[32]です.これ以上小さな値を
     *                   設定しても意味を持ちません.<BR>
     *                   設定可能な最大数は[9999]です.<BR>
     * @param maxRcvRoll 受信ロール管理数を設定します.<BR>
     *                   設定可能な最小値は[32]です.これ以上小さな値を
     *                   設定しても意味を持ちません.<BR>
     *                   設定可能な最大数は[9999]です.<BR>
     * @param sndResourceType 送信リソースタイプを設定します.
     * @param rcvResourceType 受信リソースタイプを設定します.
     * @exception InputException 入力例外.
     */
    public UIO(
        UIOResultCallback scall,InetAddress sndAddr,int sndPort,int sndCThread,int sndThread,int sndBuffer,
        UIOResultCallback rcall,InetAddress rcvAddr,int rcvPort,int rcvCThread,int rcvThread,int rcvBuffer,
        int rcvDestroyTime,int packetLen,int maxSndRoll,int maxRcvRoll,
        ResourceType sndResourceType,ResourceType rcvResourceType
    )
        throws InputException
    {
        try{
            this.open(
                scall,sndAddr,sndPort,sndCThread,sndThread,sndBuffer,
                rcall,rcvAddr,rcvPort,rcvCThread,rcvThread,rcvBuffer,
                rcvDestroyTime,packetLen,maxSndRoll,maxRcvRoll,
                sndResourceType,rcvResourceType
            ) ;
        }catch( InputException in ){
            this.close() ;
            throw in ;
        }catch( Exception e ){
            this.close() ;
        }
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.close() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オープン処理.
     * <BR><BR>
     * 通信処理をオープンします.<BR>
     * また、この処理の場合送信ポートと受信ポートはデフォルト値が
     * 設定されます.<BR>
     * デフォルトの送信ポートは[22055]です.<BR>
     * デフォルトの受信ポートは[22155]です.
     */
    public final void open()
    {
        try{
            this.open(
                null,null,-1,UdpIODef.DEFAULT_CORE_THREAD_SIZE,UdpIODef.DEFAULT_THREAD_SIZE,UdpIODef.DEFAULT_UDP_BUFFER,
                null,null,-1,UdpIODef.DEFAULT_CORE_THREAD_SIZE,UdpIODef.DEFAULT_THREAD_SIZE,UdpIODef.DEFAULT_UDP_BUFFER,
                UdpIODef.RCV_PACKET_CLEAR_TIME,UdpIODef.DEFAULT_PACKET,-1,-1,null,null
            ) ;
        }catch( Exception e ){
            this.close() ;
        }
    }
    
    /**
     * オープン処理.
     * <BR><BR>
     * 通信処理をオープンします.<BR>
     * また、この処理の場合送信ポートと受信ポートはデフォルト値が
     * 設定されます.<BR>
     * デフォルトの送信ポートは[22055]です.<BR>
     * デフォルトの受信ポートは[22155]です.
     * <BR>
     * @param scall 送信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param rcall 受信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @exception InputException 入力例外.
     */
    public final void open( UIOResultCallback scall,UIOResultCallback rcall )
        throws InputException
    {
        try{
            this.open(
                scall,null,-1,UdpIODef.DEFAULT_CORE_THREAD_SIZE,UdpIODef.DEFAULT_THREAD_SIZE,UdpIODef.DEFAULT_UDP_BUFFER,
                rcall,null,-1,UdpIODef.DEFAULT_CORE_THREAD_SIZE,UdpIODef.DEFAULT_THREAD_SIZE,UdpIODef.DEFAULT_UDP_BUFFER,
                UdpIODef.RCV_PACKET_CLEAR_TIME,UdpIODef.DEFAULT_PACKET,-1,-1,null,null
            ) ;
        }catch( InputException in ){
            this.close() ;
            throw in ;
        }catch( Exception e ){
            this.close() ;
        }
    }
    
    /**
     * オープン処理.
     * <BR><BR>
     * 通信処理をオープンします.
     * <BR>
     * @param maxRcvRoll 受信ロール管理数を設定します.<BR>
     *                   設定可能な最小値は[32]です.これ以上小さな値を
     *                   設定しても意味を持ちません.<BR>
     *                   設定可能な最大数は[9999]です.
     * @exception InputException 入力例外.
     */
    public final void open( int maxRcvRoll )
        throws InputException
    {
        try{
            this.open(
                null,null,-1,UdpIODef.DEFAULT_CORE_THREAD_SIZE,UdpIODef.DEFAULT_THREAD_SIZE,UdpIODef.DEFAULT_UDP_BUFFER,
                null,null,-1,UdpIODef.DEFAULT_CORE_THREAD_SIZE,UdpIODef.DEFAULT_THREAD_SIZE,UdpIODef.DEFAULT_UDP_BUFFER,
                UdpIODef.RCV_PACKET_CLEAR_TIME,UdpIODef.DEFAULT_PACKET,-1,maxRcvRoll,null,null
            ) ;
        }catch( InputException in ){
            this.close() ;
            throw in ;
        }catch( Exception e ){
            this.close() ;
        }
    }
    
    /**
     * オープン処理.
     * <BR><BR>
     * 通信処理をオープンします.
     * <BR>
     * @param scall 送信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param rcall 受信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param maxRcvRoll 受信ロール管理数を設定します.<BR>
     *                   設定可能な最小値は[32]です.これ以上小さな値を
     *                   設定しても意味を持ちません.<BR>
     *                   設定可能な最大数は[9999]です.
     * @exception InputException 入力例外.
     */
    public final void open( UIOResultCallback scall,UIOResultCallback rcall,int maxRcvRoll )
        throws InputException
    {
        try{
            this.open(
                scall,null,-1,UdpIODef.DEFAULT_CORE_THREAD_SIZE,UdpIODef.DEFAULT_THREAD_SIZE,UdpIODef.DEFAULT_UDP_BUFFER,
                rcall,null,-1,UdpIODef.DEFAULT_CORE_THREAD_SIZE,UdpIODef.DEFAULT_THREAD_SIZE,UdpIODef.DEFAULT_UDP_BUFFER,
                UdpIODef.RCV_PACKET_CLEAR_TIME,UdpIODef.DEFAULT_PACKET,-1,maxRcvRoll,null,null
            ) ;
        }catch( InputException in ){
            this.close() ;
            throw in ;
        }catch( Exception e ){
            this.close() ;
        }
    }
    
    /**
     * オープン処理.
     * <BR><BR>
     * 通信処理をオープンします.
     * <BR>
     * @param sndPort 送信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param rcvPort 受信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param packetLen パケットサイズを設定します.<BR>
     *                  設定可能な最小値は[512]です.<BR>
     *                  設定可能な最大値は[8192]です.
     * @param maxRcvRoll 受信ロール管理数を設定します.<BR>
     *                   設定可能な最小値は[32]です.これ以上小さな値を
     *                   設定しても意味を持ちません.<BR>
     *                   設定可能な最大数は[9999]です.<BR>
     * @exception InputException 入力例外.
     */
    public final void open( int sndPort,int rcvPort,int packetLen,int maxRcvRoll )
        throws InputException
    {
        try{
            this.open(
                null,null,sndPort,UdpIODef.DEFAULT_CORE_THREAD_SIZE,UdpIODef.DEFAULT_THREAD_SIZE,UdpIODef.DEFAULT_UDP_BUFFER,
                null,null,rcvPort,UdpIODef.DEFAULT_CORE_THREAD_SIZE,UdpIODef.DEFAULT_THREAD_SIZE,UdpIODef.DEFAULT_UDP_BUFFER,
                UdpIODef.RCV_PACKET_CLEAR_TIME,packetLen,-1,maxRcvRoll,null,null
            ) ;
        }catch( InputException in ){
            this.close() ;
            throw in ;
        }catch( Exception e ){
            this.close() ;
        }
    }
    
    /**
     * オープン処理.
     * <BR><BR>
     * 通信処理をオープンします.
     * <BR>
     * @param scall 送信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param sndPort 送信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param rcall 受信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param rcvPort 受信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param packetLen パケットサイズを設定します.<BR>
     *                  設定可能な最小値は[512]です.<BR>
     *                  設定可能な最大値は[8192]です.
     * @param maxRcvRoll 受信ロール管理数を設定します.<BR>
     *                   設定可能な最小値は[32]です.これ以上小さな値を
     *                   設定しても意味を持ちません.<BR>
     *                   設定可能な最大数は[9999]です.<BR>
     * @exception InputException 入力例外.
     */
    public final void open( UIOResultCallback scall,int sndPort,UIOResultCallback rcall,int rcvPort,int packetLen,int maxRcvRoll )
        throws InputException
    {
        try{
            this.open(
                scall,null,sndPort,UdpIODef.DEFAULT_CORE_THREAD_SIZE,UdpIODef.DEFAULT_THREAD_SIZE,UdpIODef.DEFAULT_UDP_BUFFER,
                rcall,null,rcvPort,UdpIODef.DEFAULT_CORE_THREAD_SIZE,UdpIODef.DEFAULT_THREAD_SIZE,UdpIODef.DEFAULT_UDP_BUFFER,
                UdpIODef.RCV_PACKET_CLEAR_TIME,packetLen,-1,maxRcvRoll,null,null
            ) ;
        }catch( InputException in ){
            this.close() ;
            throw in ;
        }catch( Exception e ){
            this.close() ;
        }
    }
    
    /**
     * オープン処理.
     * <BR><BR>
     * 通信処理をオープンします.
     * <BR>
     * @param sndPort 送信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param rcvPort 受信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param packetLen パケットサイズを設定します.<BR>
     *                  設定可能な最小値は[512]です.<BR>
     *                  設定可能な最大値は[8192]です.
     * @param maxRcvRoll 受信ロール管理数を設定します.<BR>
     *                   設定可能な最小値は[32]です.これ以上小さな値を
     *                   設定しても意味を持ちません.<BR>
     *                   設定可能な最大数は[9999]です.<BR>
     * @param resourceType リソースタイプを設定します.
     * @exception InputException 入力例外.
     */
    public final void open( int sndPort,int rcvPort,int packetLen,int maxRcvRoll,ResourceType resourceType )
        throws InputException
    {
        try{
            this.open(
                null,null,sndPort,UdpIODef.DEFAULT_CORE_THREAD_SIZE,UdpIODef.DEFAULT_THREAD_SIZE,UdpIODef.DEFAULT_UDP_BUFFER,
                null,null,rcvPort,UdpIODef.DEFAULT_CORE_THREAD_SIZE,UdpIODef.DEFAULT_THREAD_SIZE,UdpIODef.DEFAULT_UDP_BUFFER,
                UdpIODef.RCV_PACKET_CLEAR_TIME,packetLen,-1,maxRcvRoll,resourceType,resourceType
            ) ;
        }catch( InputException in ){
            this.close() ;
            throw in ;
        }catch( Exception e ){
            this.close() ;
        }
    }
    
    /**
     * オープン処理.
     * <BR><BR>
     * 通信処理をオープンします.
     * <BR>
     * @param scall 送信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param sndPort 送信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param rcall 受信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param rcvPort 受信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param packetLen パケットサイズを設定します.<BR>
     *                  設定可能な最小値は[512]です.<BR>
     *                  設定可能な最大値は[8192]です.
     * @param maxRcvRoll 受信ロール管理数を設定します.<BR>
     *                   設定可能な最小値は[32]です.これ以上小さな値を
     *                   設定しても意味を持ちません.<BR>
     *                   設定可能な最大数は[9999]です.<BR>
     * @param resourceType リソースタイプを設定します.
     * @exception InputException 入力例外.
     */
    public final void open(
        UIOResultCallback scall,int sndPort,UIOResultCallback rcall,int rcvPort,
        int packetLen,int maxRcvRoll,ResourceType resourceType
    )
        throws InputException
    {
        try{
            this.open(
                scall,null,sndPort,UdpIODef.DEFAULT_CORE_THREAD_SIZE,UdpIODef.DEFAULT_THREAD_SIZE,UdpIODef.DEFAULT_UDP_BUFFER,
                rcall,null,rcvPort,UdpIODef.DEFAULT_CORE_THREAD_SIZE,UdpIODef.DEFAULT_THREAD_SIZE,UdpIODef.DEFAULT_UDP_BUFFER,
                UdpIODef.RCV_PACKET_CLEAR_TIME,packetLen,-1,maxRcvRoll,resourceType,resourceType
            ) ;
        }catch( InputException in ){
            this.close() ;
            throw in ;
        }catch( Exception e ){
            this.close() ;
        }
    }
    
    /**
     * オープン処理.
     * <BR><BR>
     * 通信処理をオープンします.
     * <BR>
     * @param sndPort 送信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param sndThread 送信スレッドサイズを設定します.<BR>
     *                  設定可能な最小値は[1]です.<BR>
     *                  設定可能な最大値は[8]です.
     * @param rcvPort 受信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param rcvThread 受信スレッドサイズを設定します.<BR>
     *                  設定可能な最小値は[1]です.<BR>
     *                  設定可能な最大値は[8]です.
     * @param packetLen パケットサイズを設定します.<BR>
     *                  設定可能な最小値は[512]です.<BR>
     *                  設定可能な最大値は[8192]です.
     * @param maxRcvRoll 受信ロール管理数を設定します.<BR>
     *                   設定可能な最小値は[32]です.これ以上小さな値を
     *                   設定しても意味を持ちません.<BR>
     *                   設定可能な最大数は[9999]です.<BR>
     * @exception InputException 入力例外.
     */
    public final void open( int sndPort,int sndThread,int rcvPort,int rcvThread,int packetLen,int maxRcvRoll )
        throws InputException
    {
        try{
            this.open(
                null,null,sndPort,( int )( ( double )sndThread / 2.0f + 0.5f ),sndThread,UdpIODef.DEFAULT_UDP_BUFFER,
                null,null,rcvPort,( int )( ( double )rcvThread / 2.0f + 0.5f ),rcvThread,UdpIODef.DEFAULT_UDP_BUFFER,
                UdpIODef.RCV_PACKET_CLEAR_TIME,packetLen,-1,maxRcvRoll,null,null
            ) ;
        }catch( InputException in ){
            this.close() ;
            throw in ;
        }catch( Exception e ){
            this.close() ;
        }
    }
    
    /**
     * オープン処理.
     * <BR><BR>
     * 通信処理をオープンします.
     * <BR>
     * @param scall 送信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param sndPort 送信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param sndThread 送信スレッドサイズを設定します.<BR>
     *                  設定可能な最小値は[1]です.<BR>
     *                  設定可能な最大値は[8]です.
     * @param rcall 受信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param rcvPort 受信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param rcvThread 受信スレッドサイズを設定します.<BR>
     *                  設定可能な最小値は[1]です.<BR>
     *                  設定可能な最大値は[8]です.
     * @param packetLen パケットサイズを設定します.<BR>
     *                  設定可能な最小値は[512]です.<BR>
     *                  設定可能な最大値は[8192]です.
     * @param maxRcvRoll 受信ロール管理数を設定します.<BR>
     *                   設定可能な最小値は[32]です.これ以上小さな値を
     *                   設定しても意味を持ちません.<BR>
     *                   設定可能な最大数は[9999]です.<BR>
     * @exception InputException 入力例外.
     */
    public final void open(
        UIOResultCallback scall,int sndPort,int sndThread,UIOResultCallback rcall,int rcvPort,int rcvThread,
        int packetLen,int maxRcvRoll
    )
        throws InputException
    {
        try{
            this.open(
                scall,null,sndPort,( int )( ( double )sndThread / 2.0f + 0.5f ),sndThread,UdpIODef.DEFAULT_UDP_BUFFER,
                rcall,null,rcvPort,( int )( ( double )rcvThread / 2.0f + 0.5f ),rcvThread,UdpIODef.DEFAULT_UDP_BUFFER,
                UdpIODef.RCV_PACKET_CLEAR_TIME,packetLen,-1,maxRcvRoll,null,null
            ) ;
        }catch( InputException in ){
            this.close() ;
            throw in ;
        }catch( Exception e ){
            this.close() ;
        }
    }
    
    /**
     * オープン処理.
     * <BR><BR>
     * 通信処理をオープンします.
     * <BR>
     * @param sndPort 送信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param sndThread 送信スレッドサイズを設定します.<BR>
     *                  設定可能な最小値は[1]です.<BR>
     *                  設定可能な最大値は[8]です.
     * @param rcvPort 受信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param rcvThread 受信スレッドサイズを設定します.<BR>
     *                  設定可能な最小値は[1]です.<BR>
     *                  設定可能な最大値は[8]です.
     * @param packetLen パケットサイズを設定します.<BR>
     *                  設定可能な最小値は[512]です.<BR>
     *                  設定可能な最大値は[8192]です.
     * @param maxRcvRoll 受信ロール管理数を設定します.<BR>
     *                   設定可能な最小値は[32]です.これ以上小さな値を
     *                   設定しても意味を持ちません.<BR>
     *                   設定可能な最大数は[9999]です.<BR>
     * @param resourceType リソースタイプを設定します.
     * @exception InputException 入力例外.
     */
    public final void open( int sndPort,int sndThread,int rcvPort,int rcvThread,int packetLen,int maxRcvRoll,ResourceType resourceType )
        throws InputException
    {
        try{
            this.open(
                null,null,sndPort,( int )( ( double )sndThread / 2.0f + 0.5f ),sndThread,UdpIODef.DEFAULT_UDP_BUFFER,
                null,null,rcvPort,( int )( ( double )rcvThread / 2.0f + 0.5f ),rcvThread,UdpIODef.DEFAULT_UDP_BUFFER,
                UdpIODef.RCV_PACKET_CLEAR_TIME,packetLen,-1,maxRcvRoll,resourceType,resourceType
            ) ;
        }catch( InputException in ){
            this.close() ;
            throw in ;
        }catch( Exception e ){
            this.close() ;
        }
    }
    
    /**
     * オープン処理.
     * <BR><BR>
     * 通信処理をオープンします.
     * <BR>
     * @param scall 送信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param sndPort 送信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param sndThread 送信スレッドサイズを設定します.<BR>
     *                  設定可能な最小値は[1]です.<BR>
     *                  設定可能な最大値は[8]です.
     * @param rcall 受信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param rcvPort 受信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param rcvThread 受信スレッドサイズを設定します.<BR>
     *                  設定可能な最小値は[1]です.<BR>
     *                  設定可能な最大値は[8]です.
     * @param packetLen パケットサイズを設定します.<BR>
     *                  設定可能な最小値は[512]です.<BR>
     *                  設定可能な最大値は[8192]です.
     * @param maxRcvRoll 受信ロール管理数を設定します.<BR>
     *                   設定可能な最小値は[32]です.これ以上小さな値を
     *                   設定しても意味を持ちません.<BR>
     *                   設定可能な最大数は[9999]です.<BR>
     * @param resourceType リソースタイプを設定します.
     * @exception InputException 入力例外.
     */
    public final void open(
        UIOResultCallback scall,int sndPort,int sndThread,UIOResultCallback rcall,int rcvPort,int rcvThread,
        int packetLen,int maxRcvRoll,ResourceType resourceType
    )
        throws InputException
    {
        try{
            this.open(
                scall,null,sndPort,( int )( ( double )sndThread / 2.0f + 0.5f ),sndThread,UdpIODef.DEFAULT_UDP_BUFFER,
                rcall,null,rcvPort,( int )( ( double )rcvThread / 2.0f + 0.5f ),rcvThread,UdpIODef.DEFAULT_UDP_BUFFER,
                UdpIODef.RCV_PACKET_CLEAR_TIME,packetLen,-1,maxRcvRoll,resourceType,resourceType
            ) ;
        }catch( InputException in ){
            this.close() ;
            throw in ;
        }catch( Exception e ){
            this.close() ;
        }
    }
    
    /**
     * オープン処理.
     * <BR><BR>
     * 通信処理をオープンします.
     * <BR>
     * @param sndAddr 送信バインド先アドレスを設定します.<BR>
     *                [null]を設定した場合、デフォルト値となります.
     * @param sndPort 送信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param sndCThread 送信コアスレッドサイズを設定します.<BR>
     *                   設定可能な最小値は[1]です.<BR>
     *                   設定可能な最大値は[8]です.
     * @param sndThread 送信スレッドサイズを設定します.<BR>
     *                  設定可能な最小値は[1]です.<BR>
     *                  設定可能な最大値は[8]です.
     * @param sndBuffer 送信バッファ長を設定します.<BR>
     *                  設定単位は[バイト]です.
     * @param rcvAddr 受信バインド先アドレスを設定します.<BR>
     *                [null]を設定した場合、デフォルト値となります.
     * @param rcvPort 受信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param rcvCThread 受信コアスレッドサイズを設定します.<BR>
     *                   設定可能な最小値は[1]です.<BR>
     *                   設定可能な最大値は[8]です.
     * @param rcvThread 受信スレッドサイズを設定します.<BR>
     *                  設定可能な最小値は[1]です.<BR>
     *                  設定可能な最大値は[8]です.
     * @param rcvBuffer 受信バッファ長を設定します.<BR>
     *                  設定単位は[バイト]です.
     * @param rcvDestroyTime 受信パケット削除時間を設定します.<BR>
     *                       この値は0以上を指定する必要があります.<BR>
     *                       また、設定単位は[ミリ秒]です.
     * @param packetLen パケットサイズを設定します.<BR>
     *                  設定可能な最小値は[512]です.<BR>
     *                  設定可能な最大値は[8192]です.
     * @param maxSndRoll 送信ロール管理数を設定します.<BR>
     *                   設定可能な最小値は[32]です.これ以上小さな値を
     *                   設定しても意味を持ちません.<BR>
     *                   設定可能な最大数は[9999]です.<BR>
     * @param maxRcvRoll 受信ロール管理数を設定します.<BR>
     *                   設定可能な最小値は[32]です.これ以上小さな値を
     *                   設定しても意味を持ちません.<BR>
     *                   設定可能な最大数は[9999]です.<BR>
     * @param sndResourceType 送信リソースタイプを設定します.
     * @param rcvResourceType 受信リソースタイプを設定します.
     * @exception InputException 入力例外.
     */
    public final void open(
        InetAddress sndAddr,int sndPort,int sndCThread,int sndThread,int sndBuffer,
        InetAddress rcvAddr,int rcvPort,int rcvCThread,int rcvThread,int rcvBuffer,
        int rcvDestroyTime,int packetLen,int maxSndRoll,int maxRcvRoll,
        ResourceType sndResourceType,ResourceType rcvResourceType
    )
        throws InputException
    {
        try{
            this.open(
                null,sndAddr,sndPort,sndCThread,sndThread,sndBuffer,
                null,rcvAddr,rcvPort,rcvCThread,rcvThread,rcvBuffer,
                rcvDestroyTime,packetLen,maxSndRoll,maxRcvRoll,
                sndResourceType,rcvResourceType
            ) ;
        }catch( InputException in ){
            this.close() ;
            throw in ;
        }catch( Exception e ){
            this.close() ;
        }
    }
    
    /**
     * オープン処理.
     * <BR><BR>
     * 通信処理をオープンします.
     * <BR>
     * @param scall 送信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param sndAddr 送信バインド先アドレスを設定します.<BR>
     *                [null]を設定した場合、デフォルト値となります.
     * @param sndPort 送信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param sndCThread 送信コアスレッドサイズを設定します.<BR>
     *                   設定可能な最小値は[1]です.<BR>
     *                   設定可能な最大値は[8]です.
     * @param sndThread 送信スレッドサイズを設定します.<BR>
     *                  設定可能な最小値は[1]です.<BR>
     *                  設定可能な最大値は[8]です.
     * @param sndBuffer 送信バッファ長を設定します.<BR>
     *                  設定単位は[バイト]です.
     * @param rcall 受信完了コールバックオブジェクトを設定します.<BR>
     *              [null]を設定することで、未設定になります.
     * @param rcvAddr 受信バインド先アドレスを設定します.<BR>
     *                [null]を設定した場合、デフォルト値となります.
     * @param rcvPort 受信バインド先ポート番号を設定します.<BR>
     *                [-1]を設定した場合、空いているポート番号が割り当てられます.
     * @param rcvCThread 受信コアスレッドサイズを設定します.<BR>
     *                   設定可能な最小値は[1]です.<BR>
     *                   設定可能な最大値は[8]です.
     * @param rcvThread 受信スレッドサイズを設定します.<BR>
     *                  設定可能な最小値は[1]です.<BR>
     *                  設定可能な最大値は[8]です.
     * @param rcvBuffer 受信バッファ長を設定します.<BR>
     *                  設定単位は[バイト]です.
     * @param rcvDestroyTime 受信パケット削除時間を設定します.<BR>
     *                       この値は0以上を指定する必要があります.<BR>
     *                       また、設定単位は[ミリ秒]です.
     * @param packetLen パケットサイズを設定します.<BR>
     *                  設定可能な最小値は[512]です.<BR>
     *                  設定可能な最大値は[8192]です.
     * @param maxSndRoll 送信ロール管理数を設定します.<BR>
     *                   設定可能な最小値は[32]です.これ以上小さな値を
     *                   設定しても意味を持ちません.<BR>
     *                   設定可能な最大数は[9999]です.<BR>
     * @param maxRcvRoll 受信ロール管理数を設定します.<BR>
     *                   設定可能な最小値は[32]です.これ以上小さな値を
     *                   設定しても意味を持ちません.<BR>
     *                   設定可能な最大数は[9999]です.<BR>
     * @param sndResourceType 送信リソースタイプを設定します.
     * @param rcvResourceType 受信リソースタイプを設定します.
     * @exception InputException 入力例外.
     */
    public final void open(
        UIOResultCallback scall,InetAddress sndAddr,int sndPort,int sndCThread,int sndThread,int sndBuffer,
        UIOResultCallback rcall,InetAddress rcvAddr,int rcvPort,int rcvCThread,int rcvThread,int rcvBuffer,
        int rcvDestroyTime,int packetLen,int maxSndRoll,int maxRcvRoll,
        ResourceType sndResourceType,ResourceType rcvResourceType
    )
        throws InputException
    {
        int i ;
        
        if(
            sndCThread < UdpIODef.MIN_THREAD_SIZE || sndCThread > UdpIODef.MAX_THREAD_SIZE ||
            sndThread < UdpIODef.MIN_THREAD_SIZE || sndThread > UdpIODef.MAX_THREAD_SIZE ||
            rcvCThread < UdpIODef.MIN_THREAD_SIZE || rcvCThread > UdpIODef.MAX_THREAD_SIZE ||
            rcvThread < UdpIODef.MIN_THREAD_SIZE || rcvThread > UdpIODef.MAX_THREAD_SIZE ||
            ( sndPort != -1 && ( sndPort < 0 || sndPort > 65535 ) ) ||
            ( rcvPort != -1 && ( rcvPort < 0 || rcvPort > 65535 ) ) ||
            packetLen < UdpIODef.MIN_PACKET || packetLen > UdpIODef.MAX_PACKET ||
            rcvDestroyTime <= 0 ||
            maxSndRoll > UdpIODef.MAX_ROLL_SIZE ||
            maxRcvRoll > UdpIODef.MAX_ROLL_SIZE
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.close() ;
        m_sync.create() ;
        m_sndSync.create() ;
        m_rcvSync.create() ;
        
        // リソースタイプが定義されていない場合、デフォルト設定.
        sndResourceType = ( sndResourceType == null ) ? UdpIODef.DEF_RESOURCE_TYPE : sndResourceType ;
        rcvResourceType = ( rcvResourceType == null ) ? UdpIODef.DEF_RESOURCE_TYPE : rcvResourceType ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                m_sendCallback = scall ;
                m_rcvCallback = rcall ;
                
                m_packetLength = packetLen ;
                m_destroyPacketTime = rcvDestroyTime ;
                
                ////////////////////////
                // 送信オブジェクト生成.
                ////////////////////////
                
                // 送信用管理ロールを生成.
                m_sndMan = new TelegramResourceRoll( true,sndResourceType,maxSndRoll ) ;
                
                // 送信UDPプロトコル.
                if( sndAddr == null ){
                    if( sndPort == -1 ){
                        m_sndPrt.openTo( UdpProtocol.SND_MODE,sndBuffer,UdpIODef.DEFAULT_SEND_PORT ) ;
                    }
                    else{
                        m_sndPrt.openTo( UdpProtocol.SND_MODE,sndBuffer,sndPort ) ;
                    }
                }
                else{
                    m_sndPrt.openTo( UdpProtocol.SND_MODE,sndBuffer,sndPort,sndAddr ) ;
                }
                
                // 送信テーブルを生成.
                m_sndResult.create( sndResourceType ) ;
                
                // 送信コアスレッド.
                m_sndResultThreads = new UdpSndResultThread[ sndCThread ] ;
                
                for( i = 0 ; i < sndCThread ; i ++ ){
                    m_sndResultThreads[ i ] = new UdpSndResultThread(
                        m_sndPrt,m_sndResult,m_sndSync
                    ) ;
                }
                
                // 送信スレッド.
                m_sndThreads = new UdpIOSndThread[ sndThread ] ;
                
                for( i = 0 ; i < sndThread ; i ++ ){
                    m_sndThreads[ i ] = new UdpIOSndThread() ;
                    m_sndThreads[ i ].create(
                        m_sendCallback,m_sndPrt,m_sndResult,m_sndMan,m_sndSync,m_sndSeq,packetLen
                    ) ;
                }
                
                
                
                ////////////////////////
                // 受信オブジェクト生成.
                ////////////////////////
                
                // 受信用管理ロールを生成.
                m_rcvMan = new TelegramResourceRoll( true,rcvResourceType,maxRcvRoll ) ;
                
                // 受信UDPプロトコル.
                if( rcvAddr == null ){
                    if( rcvPort == -1 ){
                        m_rcvPrt.openTo( UdpProtocol.RCV_MODE,rcvBuffer,UdpIODef.DEFAULT_RECAIVE_PORT ) ;
                    }
                    else{
                        m_rcvPrt.openTo( UdpProtocol.RCV_MODE,rcvBuffer,rcvPort ) ;
                    }
                }
                else{
                    m_rcvPrt.openTo( UdpProtocol.RCV_MODE,rcvBuffer,rcvPort,rcvAddr ) ;
                }
                
                // 受信テーブルを生成.
                m_rcvPacket.create( rcvResourceType ) ;
                
                // 受信コアスレッド.
                m_rcvPacketThreads = new UdpRcvConnectThread[ rcvCThread ] ;
                
                for( i = 0 ; i < rcvCThread ; i ++ ){
                    m_rcvPacketThreads[ i ] = new UdpRcvConnectThread(
                        m_rcvPrt,m_rcvPacket,m_rcvMan,m_rcvSync
                    ) ;
                }
                
                // 受信スレッド.
                m_rcvThreads = new UdpIORcvThread[ rcvThread ] ;
                
                for( i = 0 ; i < rcvThread ; i ++ ){
                    m_rcvThreads[ i ] = new UdpIORcvThread() ;
                    m_rcvThreads[ i ].create(
                        rcall,m_rcvPrt,m_rcvPacket,m_rcvMan,m_rcvSync,packetLen,rcvDestroyTime
                    ) ;
                }
                
            }
            
        }catch( Exception e ){
            this.close() ;
            throw new InputException( e ) ;
        }
        
    }
    
    /**
     * クローズ処理.
     * <BR><BR>
     * オープン状態のオブジェクトをクローズします.
     */
    public final void close()
    {
        int i ;
        int len ;
        
        m_sync.clear() ;
        m_sndSync.clear() ;
        m_rcvSync.clear() ;
        
        ////////////////////////
        // 受信オブジェクト破棄.
        ////////////////////////
        
        // 受信スレッドを破棄.
        if( m_rcvThreads != null ){
            len = m_rcvThreads.length ;
            for( i = 0 ; i < len ; i ++ ){
                if( m_rcvThreads[ i ] != null ){
                    m_rcvThreads[ i ].clear() ;
                    m_rcvThreads[ i ] = null ;
                }
            }
        }
        
        m_rcvThreads = null ;
        
        // 受信コアスレッドを破棄.
        if( m_rcvPacketThreads != null ){
            len = m_rcvPacketThreads.length ;
            for( i = 0 ; i < len ; i ++ ){
                if( m_rcvPacketThreads[ i ] != null ){
                    m_rcvPacketThreads[ i ].destroy() ;
                    m_rcvPacketThreads[ i ] = null ;
                }
            }
        }
        
        m_rcvPacketThreads = null ;
        
        m_rcvPrt.close() ;
        m_rcvPacket.clear() ;
        
        
        ////////////////////////
        // 送信オブジェクト破棄.
        ////////////////////////
        
        // 送信スレッドを破棄.
        if( m_sndThreads != null ){
            len = m_sndThreads.length ;
            for( i = 0 ; i < len ; i ++ ){
                if( m_sndThreads[ i ] != null ){
                    m_sndThreads[ i ].clear() ;
                    m_sndThreads[ i ] = null ;
                }
            }
        }
        
        m_sndThreads = null ;
        
        // 送信コアスレッドを破棄.
        if( m_sndResultThreads != null ){
            len = m_sndResultThreads.length ;
            for( i = 0 ; i < len ; i ++ ){
                if( m_sndResultThreads[ i ] != null ){
                    m_sndResultThreads[ i ].destroy() ;
                    m_sndResultThreads[ i ] = null ;
                }
            }
        }
        
        // 送信ロールを破棄.
        if( m_sndMan != null ){
            m_sndMan.destroy() ;
            m_sndMan = null ;
        }
        
        // 受信ロールを破棄.
        if( m_rcvMan != null ){
            m_rcvMan.destroy() ;
            m_rcvMan = null ;
        }
        
        m_rcvPacketThreads = null ;
        m_sendCallback = null ;
        m_rcvCallback = null ;
        
        m_sndPrt.close() ;
        m_sndResult.clear() ;
        m_sndSeq.clear() ;
        
        m_packetLength = -1 ;
        m_destroyPacketTime = -1 ;
        m_sendFlip = 0 ;
    }
    
    /**
     * 接続可能をチェックする確認電文を送信.
     * <BR><BR>
     * 接続可能をチェックする確認電文を送信します.
     * <BR>
     * @param addr 接続先のアドレスを設定します.
     * @param port 接続先のポート番号を設定します.
     * @return int 現在受信側が接続中の数が返されます.<BR>
     *              [-1]が返された場合、確認電文の受信に失敗しました.
     */
    public final int checkConnect( InetAddress addr,int port )
    {
        return this.checkConnect( addr,port,UdpIODef.MIN_RESEND_COUNT ) ;
    }
    
    /**
     * 接続可能をチェックする確認電文を送信.
     * <BR><BR>
     * 接続可能をチェックする確認電文を送信します.
     * <BR>
     * @param addr 接続先のアドレスを設定します.
     * @param port 接続先のポート番号を設定します.
     * @param max 最大ループ回数を設定します.
     * @return int 現在受信側が接続中の数が返されます.<BR>
     *              [-1]が返された場合、確認電文の受信に失敗しました.
     */
    public final int checkConnect( InetAddress addr,int port,int max )
    {
        int ret ;
        UdpIOSndThread std = null ;
        
        ret = -1 ;
        
        try{
            
            synchronized( m_sync.get() ){
                std = m_sndThreads[ 0 ] ;
            }
            
            if( std != null ){
                
                ret = std.checkConnect( addr,port,max ) ;
                
            }
            
        }catch( Exception e ){
            ret = -1 ;
        }finally{
            std = null ;
        }
        
        return ret ;
    }
    
    /**
     * 送信処理.
     * <BR><BR>
     * 電文送信処理を実施します.<BR>
     * また、この処理はデフォルトポート番号に送信します.
     * <BR>
     * @param addr 対象の送信アドレスを設定します.
     * @param binary 送信電文を設定します.
     * @return int 送信されたID情報が返されます.<BR>
     *             [-1]が返された場合送信に失敗しました.
     */
    public final int send( InetAddress addr,byte[] binary )
    {
        return this.send(
            addr,UdpIODef.DEFAULT_RECAIVE_PORT,binary
        ) ;
    }
    
    /**
     * 送信処理.
     * <BR><BR>
     * 電文送信処理を実施します.
     * <BR>
     * @param conn 送信対象のアドレス+ポート番号を設定します.
     * @param binary 送信電文を設定します.
     * @return int 送信されたID情報が返されます.<BR>
     *             [-1]が返された場合送信に失敗しました.
     */
    public final int send( ConnectAddress conn,byte[] binary )
    {
        return this.send(
            conn.getAddress(),conn.getPort(),binary
        ) ;
    }
    
    /**
     * 送信処理.
     * <BR><BR>
     * 電文送信処理を実施します.
     * <BR>
     * @param addr 送信対象のアドレスを設定します.
     * @param port 送信対象のポート番号を設定します.
     * @param binary 送信電文を設定します.
     * @return int 送信されたID情報が返されます.<BR>
     *             [-1]が返された場合送信に失敗しました.
     */
    public final int send( InetAddress addr,int port,byte[] binary )
    {
        if(
            addr == null || port < 0 || port > 65535 ||
            binary == null || binary.length <= 0
        ){
            return -1 ;
        }
        
        return this.sendTo( addr,port,binary ) ;
    }
    
    /**
     * 送信処理.
     * <BR><BR>
     * 電文送信処理を実施します.<BR>
     * また、この処理はデフォルトポート番号に送信します.
     * <BR>
     * @param addr 対象の送信アドレスを設定します.
     * @param binary 送信電文を設定します.
     * @return int 送信されたID情報が返されます.<BR>
     *             [-1]が返された場合送信に失敗しました.
     */
    public final int send( InetAddress addr,BinResource binary )
    {
        return this.send(
            addr,UdpIODef.DEFAULT_RECAIVE_PORT,binary
        ) ;
    }
    
    /**
     * 送信処理.
     * <BR><BR>
     * 電文送信処理を実施します.
     * <BR>
     * @param conn 送信対象のアドレス+ポート番号を設定します.
     * @param binary 送信電文を設定します.
     * @return int 送信されたID情報が返されます.<BR>
     *             [-1]が返された場合送信に失敗しました.
     */
    public final int send( ConnectAddress conn,BinResource binary )
    {
        return this.send(
            conn.getAddress(),conn.getPort(),binary
        ) ;
    }
    
    /**
     * 送信処理.
     * <BR><BR>
     * 電文送信処理を実施します.
     * <BR>
     * @param addr 送信対象のアドレスを設定します.
     * @param port 送信対象のポート番号を設定します.
     * @param binary 送信電文を設定します.
     * @return int 送信されたID情報が返されます.<BR>
     *             [-1]が返された場合送信に失敗しました.
     */
    public final int send( InetAddress addr,int port,BinResource binary )
    {
        if(
            addr == null || port < 0 || port > 65535 ||
            binary == null || binary.isUse() == false || binary.size() <= 0
        ){
            return -1 ;
        }
        
        return this.sendTo( addr,port,binary ) ;
    }
    
    /**
     * 受信処理.
     * <BR><BR>
     * 受信処理を実施します.
     * <BR>
     * @param out 受信元のアドレス/ポート番号が返されます.<BR>
     *            [null]を設定した場合、受信元のアドレス/ポート番号は取得されません.
     * @return byte[] 受信情報が返されます.<BR>
     *                [null]が返された場合、受信情報は存在しません.
     */
    public final byte[] receive( ConnectAddress out )
    {
        return this.receive( out,null ) ;
    }
    
    /**
     * 受信処理.
     * <BR><BR>
     * 受信処理を実施します.
     * <BR>
     * @param out 受信元のアドレス/ポート番号が返されます.<BR>
     *            [null]を設定した場合、受信元のアドレス/ポート番号は取得されません.
     * @param id 受信されたID情報を取得します.
     * @return byte[] 受信情報が返されます.<BR>
     *                [null]が返された場合、受信情報は存在しません.
     */
    public final byte[] receive( ConnectAddress out,int[] id )
    {
        BinResource res = null ;
        byte[] ret = null ;
        
        if( out != null ){
            out.clear() ;
        }
        if( id == null || id.length <= 0 ){
            id = null ;
        }
        
        res = this.receiveTo( id,out ) ;
        if( res != null && res.isUse() == true ){
            ret = res.getBinary() ;
            res.clear() ;
            res = null ;
        }
        
        return ret ;
    }
    
    /**
     * 受信処理.
     * <BR><BR>
     * 受信処理を実施します.
     * <BR>
     * @param out 受信元のアドレス/ポート番号が返されます.<BR>
     *            [null]を設定した場合、受信元のアドレス/ポート番号は取得されません.
     * @return BinResource 受信情報が返されます.<BR>
     *                     [null]が返された場合、受信情報は存在しません.
     */
    public final BinResource receiveTo( ConnectAddress out )
    {
        return this.receiveTo( out,null ) ;
    }
    
    /**
     * 受信処理.
     * <BR><BR>
     * 受信処理を実施します.
     * <BR>
     * @param out 受信元のアドレス/ポート番号が返されます.<BR>
     *            [null]を設定した場合、受信元のアドレス/ポート番号は取得されません.
     * @param id 受信されたID情報を取得します.
     * @return BinResource 受信情報が返されます.<BR>
     *                     [null]が返された場合、受信情報は存在しません.
     */
    public final BinResource receiveTo( ConnectAddress out,int[] id )
    {
        if( out != null ){
            out.clear() ;
        }
        if( id == null || id.length <= 0 ){
            id = null ;
        }
        
        return this.receiveTo( id,out ) ;
    }
    
    /**
     * 受信処理.
     * <BR><BR>
     * 受信処理を実施します.
     * <BR>
     * @param out 受信元のアドレス/ポート番号が返されます.<BR>
     *            [null]を設定した場合、受信元のアドレス/ポート番号は取得されません.
     * @param timeout 受信タイムアウト値を設定します.<BR>
     *                [0]以下を設定した場合、無限待ちになります.<BR>
     *                また、設定単位は[ミリ秒]です.
     * @return byte[] 受信情報が返されます.
     * @exception ConnectTimeoutException 受信タイムアウト例外.
     * @exception ExecutionException 実行例外.
     */
    public final byte[] receive( ConnectAddress out,int timeout )
        throws ConnectTimeoutException,ExecutionException
    {
        return this.receive( out,null,timeout ) ;
    }
    
    /**
     * 受信処理.
     * <BR><BR>
     * 受信処理を実施します.
     * <BR>
     * @param out 受信元のアドレス/ポート番号が返されます.<BR>
     *            [null]を設定した場合、受信元のアドレス/ポート番号は取得されません.
     * @param id 受信されたID情報を取得します.
     * @param timeout 受信タイムアウト値を設定します.<BR>
     *                [0]以下を設定した場合、無限待ちになります.<BR>
     *                また、設定単位は[ミリ秒]です.
     * @return byte[] 受信情報が返されます.
     * @exception ConnectTimeoutException 受信タイムアウト例外.
     * @exception ExecutionException 実行例外.
     */
    public final byte[] receive( ConnectAddress out,int[] id,int timeout )
        throws ConnectTimeoutException,ExecutionException
    {
        BinResource res = null ;
        byte[] ret = null ;
        
        if( out != null ){
            out.clear() ;
        }
        if( id == null || id.length <= 0 ){
            id = null ;
        }
        
        res = this.receiveTo( out,id,timeout ) ;
        if( res != null && res.isUse() == true ){
            ret = res.getBinary() ;
            res.clear() ;
            res = null ;
        }
        
        return ret ;
    }
    
    /**
     * 受信処理.
     * <BR><BR>
     * 受信処理を実施します.
     * <BR>
     * @param out 受信元のアドレス/ポート番号が返されます.<BR>
     *            [null]を設定した場合、受信元のアドレス/ポート番号は取得されません.
     * @param timeout 受信タイムアウト値を設定します.<BR>
     *                [0]以下を設定した場合、無限待ちになります.<BR>
     *                また、設定単位は[ミリ秒]です.
     * @return BinResource 受信情報が返されます.<BR>
     *                     [null]が返された場合、受信情報は存在しません.
     * @exception ConnectTimeoutException 受信タイムアウト例外.
     * @exception ExecutionException 実行例外.
     */
    public final BinResource receiveTo( ConnectAddress out,int timeout )
        throws ConnectTimeoutException,ExecutionException
    {
        return receiveTo( out,null,timeout ) ;
    }
    
    /**
     * 受信処理.
     * <BR><BR>
     * 受信処理を実施します.
     * <BR>
     * @param out 受信元のアドレス/ポート番号が返されます.<BR>
     *            [null]を設定した場合、受信元のアドレス/ポート番号は取得されません.
     * @param id 受信されたID情報を取得します.
     * @param timeout 受信タイムアウト値を設定します.<BR>
     *                [0]以下を設定した場合、無限待ちになります.<BR>
     *                また、設定単位は[ミリ秒]です.
     * @return BinResource 受信情報が返されます.<BR>
     *                     [null]が返された場合、受信情報は存在しません.
     * @exception ConnectTimeoutException 受信タイムアウト例外.
     * @exception ExecutionException 実行例外.
     */
    public final BinResource receiveTo( ConnectAddress out,int[] id,int timeout )
        throws ConnectTimeoutException,ExecutionException
    {
        long tm ;
        BinResource ret = null ;
        
        if( out != null ){
            out.clear() ;
        }
        
        tm = ( timeout <= 0 ) ? Long.MAX_VALUE : System.currentTimeMillis() + timeout ;
        
        if( id == null || id.length <= 0 ){
            id = null ;
        }
        
        try{
            
            for( ;; ){
                
                if( ( ret = this.receiveTo( id,out ) ) != null ){
                    break ;
                }
                else if( tm <= System.currentTimeMillis() ){
                    throw new ConnectTimeoutException(
                        "受信タイムアウト(" + timeout +
                        ")が発生しました"
                    ) ;
                }
                
                UtilCom.idleTime() ;
                
            }
            
        }catch( ConnectTimeoutException ct ){
            if( ret != null ){ ret.clear() ; }
            if( out != null ){ out.clear() ; }
            ret = null ;
            throw ct ;
        }catch( ExecutionException ee ){
            if( ret != null ){ ret.clear() ; }
            if( out != null ){ out.clear() ; }
            ret = null ;
            throw ee ;
        }catch( Exception e ){
            if( ret != null ){ ret.clear() ; }
            if( out != null ){ out.clear() ; }
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 送信UDPローカルアドレスを取得.
     * <BR><BR>
     * 送信UDPローカルアドレスを取得します.
     * <BR>
     * @return InetAddress 送信UDPローカルアドレスが返されます.
     */
    public final InetAddress getUDPSendLocalAddress()
    {
        InetAddress ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_sndPrt.getLocalAddress() ;
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 送信UDPローカルポートを取得.
     * <BR><BR>
     * 送信UDPローカルポート番号を取得します.
     * <BR>
     * @return int 送信UDPローカルポート番号が返されます.
     */
    public final int getUDPSendLocalPort()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_sndPrt.getLocalPort() ;
            }
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 送信コアスレッド数を取得.
     * <BR><BR>
     * 送信コアスレッド数を取得します.
     * <BR>
     * @return int 送信コアスレッド数が返されます.
     */
    public final int getSendCoreThread()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_sndResultThreads.length ;
            }
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 送信スレッド数を取得.
     * <BR><BR>
     * 送信スレッド数を取得します.
     * <BR>
     * @return int 送信スレッド数が返されます.
     */
    public final int getSendThread()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_sndThreads.length ;
            }
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 受信UDPローカルアドレスを取得.
     * <BR><BR>
     * 受信UDPローカルアドレスを取得します.
     * <BR>
     * @return InetAddress 受信UDPローカルアドレスが返されます.
     */
    public final InetAddress getUDPReceiveLocalAddress()
    {
        InetAddress ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_rcvPrt.getLocalAddress() ;
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 受信UDPローカルポートを取得.
     * <BR><BR>
     * 受信UDPローカルポート番号を取得します.
     * <BR>
     * @return int 受信UDPローカルポート番号が返されます.
     */
    public final int getUDPReceiveLocalPort()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_rcvPrt.getLocalPort() ;
            }
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 受信コアスレッド数を取得.
     * <BR><BR>
     * 受信コアスレッド数を取得します.
     * <BR>
     * @return int 受信コアスレッド数が返されます.
     */
    public final int getReceiveCoreThread()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_rcvPacketThreads.length ;
            }
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 受信スレッド数を取得.
     * <BR><BR>
     * 受信スレッド数を取得します.
     * <BR>
     * @return int 受信スレッド数が返されます.
     */
    public final int getReceiveThread()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_rcvThreads.length ;
            }
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 送受信パケット長を取得.
     * <BR><BR>
     * 送受信を行うパケット長を取得します.
     * <BR>
     * @return int 送受信パケット長が返されます.
     */
    public final int getPacketLength()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_packetLength ;
            }
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 受信パケット破棄時間を取得.
     * <BR><BR>
     * 受信パケット破棄時間を取得します.
     * <BR>
     * @return int 受信パケット破棄時間が返されます.
     */
    public final int getDestroyReceivePacketTime()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_destroyPacketTime ;
            }
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * オープンチェック.
     * <BR><BR>
     * オープン状態であるかチェックします.
     * <BR>
     * @return boolean オープン状態か返されます.<BR>
     *                 [true]が返された場合オープン状態です.<BR>
     *                 [false]が返された場合クローズ状態です.
     */
    public final boolean isOpen()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = true ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * 電文送信処理.
     */
    private final int sendTo( InetAddress addr,int port,Object binary )
    {
        int max ;
        int ret ;
        
        UdpIOSndThread snd = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                snd = m_sndThreads[ m_sendFlip ] ;
                max = m_sndThreads.length ;
                m_sendFlip = ( m_sendFlip+1 >= max ) ? 0 : m_sendFlip + 1 ;
                
            }
            
            // 送信ロールに電文セット.
            ret = snd.setData( addr,port,binary ) ;
            
        }catch( Exception e ){
            ret = -1 ;
        }finally{
            snd = null ;
        }
        
        return ret ;
    }
    
    /**
     * 受信処理を実行.
     */
    private final BinResource receiveTo( int[] id,ConnectAddress out )
    {
        
        TelegramValue val = null ;
        BinResource ret = null ;
        
        try{
            
            val = new TelegramValue() ;
            
            // 受信ロール管理から、受信データを取得.
            if( m_rcvMan.get( val ) == false ){
                // 存在しない場合は[null]を戻す.
                return null ;
            }
            
            // IDの引数が存在する場合.
            if( id != null ){
                id[ 0 ] = val.getID() ;
            }
            
            // 接続先情報を格納する引数が存在する場合.
            if( out != null ){
                out.create( val.getAddress(),val.getPort() ) ;
            }
            
            // 対象のバイナリリソースを取得.
            ret = ( BinResource )val.getBinary() ;
            
            // 対象要素が存在する場合.
            if( val != null ){
                val.clear() ;
            }
            val = null ;
            
        }catch( Exception e ){
            if( ret != null ){
                ret.clear() ;
            }
            if( out != null ){
                out.clear() ;
            }
            if( id != null ){
                id[ 0 ] = -1 ;
            }
            ret = null ;
        }finally{
            if( val != null ){
                val.clear() ;
            }
            val = null ;
        }
        
        return ret ;
        
    }
    
}

