/*
 * @(#)MgcValueImple.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.net.mgc ;

import java.net.InetAddress;

import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.net.ConnectAddress;
import com.JRcServer.commons.serialize.SerializeUtil;

/**
 * マルチキャストグループ要素内部オブジェクト.
 * <BR><BR>
 * 対象のマルチキャストグループ要素内部オブジェクトです.
 *  
 * @version 1.0.0 2005/07/29
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
class MgcValueImple implements MgcValue,MgcWriteValue
{
	
    static {
        serialVersionUID = SerializeUtil.serialVersionUID(
            MgcValueImple.class.getName()
        ) ;
    }
    
    /**
     * シリアライズUID.
     */
    private static final long serialVersionUID ;
    
    /**
     * サーバ名.
     */
    private String m_serverName = null ;
    
    /**
     * サーバID.
     */
    private int m_serverID = -1 ;
    
    /**
     * プロトコルタイプを設定.
     */
    private int m_protocolType = -1 ;
    
    /**
     * バインドアドレス.
     */
    private InetAddress m_addr = null ;
    
    /**
     * バインドポート.
     */
    private int m_port = -1 ;
    
    /**
     * 現在コネクションカウント.
     */
    private int m_nowConnectCnt = -1 ;
    
    /**
     * MgcServer標準時間.
     */
    private long m_defTime = 0L ;
    
    /**
     * 前回応答時間.
     */
    private transient int m_beforeResponse = Integer.MIN_VALUE ;
    
    /**
     * 更新時間.
     */
    private transient long m_updateTime = Long.MIN_VALUE ;
    
    /**
     * 応答なしカウント.
     */
    private int m_noResponseCount = 0 ;
    
    
    
    /**
     * コンタストラクタ.
     */
    private MgcValueImple()
    {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 条件を設定してオブジェクトを生成します.
     * <BR>
     * @param name 対象のサーバ名を設定します.
     * @param id 対象のサーバIDを設定します.
     * @param ptype 対象のプロトコルタイプを設定します.<BR>
     *              [MgcValue.PROTOCOL_TYPE_TCP]を設定した場合、TCP/IPを示します.<BR>
     *              [MgcValue.PROTOCOL_TYPE_UDP]を設定した場合、UDPを示します.<BR>
     *              [MgcValue.PROTOCOL_TYPE_MCAST]を設定した場合、Multicastを示します.
     * @param addr 通知対象のバインドアドレスを設定します.<BR>
     *             バインドアドレスを設定しない場合は[null]を設定してください.
     * @param port 通知対象のバインドポート番号を設定します.
     * @exception InputException 入力例外.
     */
    protected MgcValueImple( String name,int id,int ptype,InetAddress addr,int port )
        throws InputException
    {
        if(
            name == null || name.length() <= 0 ||
            id < 0 || port <= 0 || port >= 65535 ||
            (
                ptype != PROTOCOL_TYPE_TCP &&
                ptype != PROTOCOL_TYPE_UDP &&
                ptype != PROTOCOL_TYPE_MCAST
            )
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            addr = ( addr == null ) ? InetAddress.getLocalHost() : addr ;
            m_serverName = name ;
            m_serverID = id ;
            m_protocolType = ptype ;
            m_addr = addr ;
            m_port = port ;
            m_nowConnectCnt = 0 ;
            m_defTime = 0L ;
            m_beforeResponse = 0 ;
            m_updateTime = System.currentTimeMillis() ;
            m_noResponseCount = 0 ;
        }catch( Exception e ){
            this.destroy() ;
            throw new InputException( e ) ;
        }
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    protected final void destroy()
    {
        m_serverName = null ;
        m_serverID = -1 ;
        m_protocolType = -1 ;
        m_addr = null ;
        m_port = -1 ;
        m_nowConnectCnt = -1 ;
        m_defTime = 0L ;
        m_beforeResponse = Integer.MIN_VALUE ;
        m_updateTime = Long.MIN_VALUE ;
        m_noResponseCount = 0 ;
    }
    
    /**
     * 前回応答時間を更新.
     * <BR><BR>
     * 前回の応答時間を更新します.
     */
    public final void update()
    {
        synchronized( this ){
            if( m_serverName != null ){
                m_updateTime = System.currentTimeMillis() ;
                // アップデートが設定された場合
                // 応答なしカウントをZero設定.
                m_noResponseCount = 0 ;
            }
        }
    }
    
    /**
     * 接続カウントを設定.
     * <BR><BR>
     * 接続カウントを設定します.
     * <BR>
     * @param cnt 対象のカウントを設定します.
     */
    public final void setConnectCount( int cnt )
    {
        synchronized( this ){
            if( m_serverName != null ){
                m_nowConnectCnt = ( cnt <= 0 ) ? 0 : cnt ;
            }
        }
    }
    
    /**
     * 応答時間を設定.
     * <BR><BR>
     * 対象の応答時間を設定します.
     * <BR>
     * @param time 応答時間を設定します.
     */
    public final void setResponseTime( int time )
    {
        if( time >= 0 ){
            synchronized( this ){
                if( m_serverName != null ){
                    m_beforeResponse = time ;
                    // 応答時間が設定された場合
                    // 応答なしカウントをZero設定.
                    m_noResponseCount = 0 ;
                }
            }
        }
    }
    
    /**
     * MgcServer標準時間を設定.
     * <BR><BR>
     * MgcServer標準時間を設定します.<BR>
     * この値はグループ内で一番低い値が標準時間となります.
     * <BR>
     * @param time 対象の標準時間を設定します.
     */
    public final void setDefaultTime( long time )
    {
        long tmp ;
        
        if( time > 0 ){
            
            tmp = System.currentTimeMillis() - time ;
            
            synchronized( this ){
                if( m_serverName != null ){
                    m_defTime = ( m_defTime <= tmp ) ? tmp : m_defTime ;
                }
            }
            
        }
        
    }
    
    /**
     * 応答なしカウントを１インクリメント.
     * <BR><BR>
     * 応答なしカウントを１インクリメントします.
     */
    public final void addNoResponseCount()
    {
        synchronized( this ){
            // 応答なしカウントがセットされた場合、
            // アップデートタイムも更新.
            m_updateTime = System.currentTimeMillis() ;
            m_noResponseCount ++ ;
        }
    }
    
    /**
     * 設定されているサーバ名を取得.
     * <BR><BR>
     * 設定されているサーバ名を取得します.
     * <BR>
     * @return String 設定されているサーバ名が返されます.<BR>
     *                [null]が返された場合、このオブジェクトは無効です.
     */
    public final String getServerName()
    {
        String ret = null ;
        
        synchronized( this ){
            ret = m_serverName ;
        }
        
        return ret ;
    }
    
    /**
     * 設定されているサーバIDを取得.
     * <BR><BR>
     * 設定されているサーバIDを取得します.
     * <BR>
     * @return int 設定されているサーバIDが返されます.<BR>
     *             [-1]が返された場合、このオブジェクトは無効です.
     */
    public final int getServerID()
    {
        int ret ;
        
        synchronized( this ){
            ret = m_serverID ;
        }
        
        return ret ;
    }
    
    /**
     * 設定されているプロトコルタイプを取得.
     * <BR><BR>
     * 設定されているプロトコルタイプを取得します.
     * <BR>
     * @return int 設定されているプロトコルタイプが返されます.<BR>
     *             [MgcValue.PROTOCOL_TYPE_TCP]が返された場合、TCP/IPを示します.<BR>
     *             [MgcValue.PROTOCOL_TYPE_UDP]が返された場合、UDPを示します.<BR>
     *             [MgcValue.PROTOCOL_TYPE_MCAST]が返された場合、Multicastを示します.<BR>
     *             [-1]が返された場合、このオブジェクトは無効です.
     */
    public final int getProtocolType()
    {
        int ret ;
        
        synchronized( this ){
            ret = m_protocolType ;
        }
        
        return ret ;
    }
    
    /**
     * 設定されているバインドアドレスを取得.
     * <BR><BR>
     * 設定されているバインドアドレスを取得します.
     * <BR>
     * @return InetAddress 設定されているバインドアドレスが返されます.<BR>
     *                     [null]が返された場合、このオブジェクトは無効です.
     */
    public final InetAddress getBindAddress()
    {
        InetAddress ret = null ;
        
        synchronized( this ){
            ret = m_addr ;
        }
        
        return ret ;
    }
    
    /**
     * 設定されているバインドポートを取得.
     * <BR><BR>
     * 設定されているバインドポート番号を取得します.
     * <BR>
     * @return int 設定されているバインドポート番号が返されます.<BR>
     *             [-1]が返された場合、このオブジェクトは無効です.
     */
    public final int getBindPort()
    {
        int ret ;
        
        synchronized( this ){
            ret = m_port ;
        }
        
        return ret ;
    }
    
    /**
     * 設定されているバインド情報を取得.
     * <BR><BR>
     * 設定されているバインド情報を取得します.
     * <BR>
     * @param out 取得先のオブジェクトを設定します.
     * @return boolean 取得結果が返されます.<BR>
     *                 [true]が返された場合、正常に取得できました.<BR>
     *                 [false]が返された場合、取得出来ませんでした.
     */
    public final boolean getBind( ConnectAddress out )
    {
        boolean ret ;
        
        ret = false ;
        if( out != null ){
            
            try{
                synchronized( this ){
                    if( m_serverName != null ){
                        out.create( m_addr,m_port ) ;
                        ret = true ;
                    }
                    else{
                        out.clear() ;
                    }
                }
            }catch( Exception e ){
                out.clear() ;
                ret = false ;
            }
        }
        
        return ret ;
    }
    
    /**
     * 更新時間を取得.
     * <BR><BR>
     * 更新時間を取得します.
     * <BR>
     * @return long 更新時間返されます.<BR>
     *              [-1L]が返された場合、このオブジェクトは無効です.<BR>
     *              [Long.MIN_VALUE]が返された場合、この条件は設定されていません.
     */
    public final long getUpdateTime()
    {
        long ret ;
        
        synchronized( this ){
            if( m_serverName != null ){
                ret = m_updateTime ;
            }
            else{
                ret = -1L ;
            }
        }
        
        return ret ;
    }
    
    /**
     * 現在のコネクションカウントを取得.
     * <BR><BR>
     * 現在のコネクションカウントを取得します.
     * <BR>
     * @return int 現在のコネクションカウントが返されます.<BR>
     *             [-1]が返された場合、このオブジェクトは無効です.
     */
    public final int getConnectCount()
    {
        int ret ;
        
        synchronized( this ){
            if( m_serverName != null ){
                ret = m_nowConnectCnt ;
            }
            else{
                ret = -1 ;
            }
        }
        
        return ret ;
    }
    
    /**
     * 現在のMgc標準時間を取得.
     * <BR><BR>
     * 現在のMgc標準時間を取得します.
     * <BR>
     * @return long 現在のMgc標準時間が返されます.<BR>
     *              [-1L]が返された場合、このオブジェクトは無効です.
     */
    public final long getDefaultTime()
    {
        long ret ;
        
        synchronized( this ){
            if( m_serverName != null ){
                ret = System.currentTimeMillis() - m_defTime ;
            }
            else{
                ret = -1L ;
            }
        }
        
        return ret ;
    }
    
    /**
     * 前回応答時間を取得.
     * <BR><BR>
     * 前回の応答時間を取得します.
     * <BR>
     * @return int 前回の応答時間が返されます.<BR>
     *             [-1]が返された場合、このオブジェクトは無効です.<BR>
     *             [Integer.MIN_VALUE]が返された場合、この条件は設定されていません.
     */
    public final int getBeforeResponse()
    {
        int ret ;
        
        synchronized( this ){
            if( m_serverName != null ){
                ret = m_beforeResponse ;
            }
            else{
                ret = -1 ;
            }
        }
        
        return ret ;
    }
    
    /**
     * 応答なしカウントを取得.
     * <BR><BR>
     * 現在の応答なしカウントを取得します.
     * <BR>
     * @return int 現在の応答なしカウントが返されます.
     */
    public final int getNoResponseCount()
    {
        int ret ;
        
        synchronized( this ){
            ret = m_noResponseCount ;
        }
        
        return ret ;
    }
    
    /**
     * オブジェクトの順位を比較.
     * <BR><BR>
     * オブジェクトの順位を比較します.
     * <BR>
     * @param o 比較対照のオブジェクトを設定します.
     * @return int 比較結果が返されます.<BR>
     *             このオブジェクトが指定されたオブジェクトより
     *             小さい場合は負の整数、等しい場合はゼロ、大きい場合は正の整数.
     */
    public final int compareTo( Object o )
    {
        int cnt ;
        int bef ;
        int tcnt ;
        int tbef ;
        
        cnt = ( ( MgcValueImple )o ).getConnectCount() ;
        bef = ( ( MgcValueImple )o ).getBeforeResponse() ;
        
        synchronized( this ){
            tcnt = m_nowConnectCnt ;
            tbef = m_beforeResponse ;
        }
        
        // 現在のコネクションカウントが同じ場合.
        if( tcnt == cnt ){
            // 応答時間で計算.
            return tbef - bef ;
        }
        else{
            // コネクションカウントで計算.
            return tcnt - cnt ;
        }
    }
    
    /**
     * このオブジェクトが有効であるかチェック.
     * <BR><BR>
     * このオブジェクトが有効であるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、オブジェクトは有効です.<BR>
     *                 [false]が返された場合、オブジェクトは無効です.
     */
    public final boolean isUse()
    {
        boolean ret ;
        
        synchronized( this ){
            if( m_serverName != null ){
                ret = true ;
            }
            else{
                ret = false ;
            }
        }
        
        return ret ;
    }
    
    
}
