/*
 * @(#)MgcCommon.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.net.mgc ;

import java.net.InetAddress;

import com.JRcServer.commons.conv.ScIO;
import com.JRcServer.commons.def.BaseDef;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.net.NetDef;
import com.JRcServer.commons.util.ConvertParam;
import com.JRcServer.commons.util.UtilCom;

/**
 * マルチキャストグループ共通処理.
 * <BR><BR>
 * 対象のマルチキャストグループ共通処理です.
 *  
 * @version 1.0.0 2005/07/29
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
class MgcCommon
{
    
    /**
     * MGC-ServerEND時の送信回数.
     */
    public static final int SEND_MGC_SERVER_END_COUNT = 3 ;
    
    /**
     * MGG-Serverヘッダタイプ : 通常電文.
     */
    public static final int HEADER_TYPE_DEFAULT = 0 ;
    
    /**
     * MGG-Serverヘッダタイプ : 戻り要求電文.
     */
    public static final int HEADER_TYPE_RETURN = 1 ;
    
    /**
     * MGG-Serverヘッダタイプ : 終了電文.
     */
    public static final int HEADER_TYPE_END = 2 ;
    
    /**
     * 暗号ランダムコード最大値.
     */
    public static final int MAX_RAND_CODE = ScIO.TABLE_LENGTH ;
    
    
    
    /**
     * デフォルトキャラクタセット.
     */
    private static final String DEF_CHARSET = BaseDef.UTF8 ;
    
    /**
     * MGC-Clientリターン指定 : 最小値.
     */
    private static final int MIN_MGCCL_RETCNT = 0 ;
    
    /**
     * MGC-Clientリターン指定 : 最小値.
     */
    private static final int MAX_MGCCL_RETCNT = 3 ;
    
    
    
    /**
     * ヘッダ長.
     */
    private static final int MGC_HEADER_LENGTH = 6 ;
    
    /**
     * ヘッダタイプ : MGC共通.
     * [mgc@]
     */
    private static final byte[] MGC_COMMON_HEADER = {
        0x0000006d,0x00000067,0x00000063,0x00000040
    } ;
    
    /**
     * ヘッダタイプ : MGC-Serverチェック用.
     * [mgc@S]
     */
    private static final byte[] MGC_SERVER_CH_HEADER = {
        0x0000006d,0x00000067,0x00000063,0x00000040,0x00000053
    } ;
    
    /**
     * ヘッダタイプ : MGC-Server.
     * [mgc@SV]
     */
    private static final byte[] MGC_SERVER_HEADER = {
        0x0000006d,0x00000067,0x00000063,0x00000040,0x00000053,0x00000056
    } ;
    
    /**
     * ヘッダタイプ : MGC-ServerReturn.
     * [mgc@SR]
     */
    private static final byte[] MGC_SERVER_RET_HEADER = {
        0x0000006d,0x00000067,0x00000063,0x00000040,0x00000053,0x00000052
    } ;
    
    /**
     * ヘッダタイプ : MGC-ServerEND.
     * [mgc@SE]
     */
    private static final byte[] MGC_SERVER_END_HEADER = {
        0x0000006d,0x00000067,0x00000063,0x00000040,0x00000053,0x00000045
    } ;
    
    /**
     * ヘッダタイプ : MGC-Client.
     * [mgc@CL]
     */
    private static final byte[] MGC_CLIENT_HEADER = {
        0x0000006d,0x00000067,0x00000063,0x00000040,0x00000043,0x0000004c
    } ;
    
    
    
    /**
     * MGC-Server電文を生成.
     * <BR><BR>
     * MGC-Server通常電文を生成します.
     * <BR>
     * @param value 対象のオブジェクトを生成します.
     * @param table 対象の暗号表(256byte)を設定します.<BR>
     *              設定可能なサイズは[256]byteです.<BR>
     *              また、[null]を設定した場合、デフォルト条件になります.
     * @return byte[] 生成されたMGC-Server電文が返されます.
     * @exception InputException 入力例外.
     */
    public static final byte[] createMgcServer( MgcValue value,byte[] table )
        throws InputException
    {
        return MgcCommon.createMgcServer( 0,value,table ) ;
    }
    
    /**
     * MGC-Server電文を生成.
     * <BR><BR>
     * MGC-Server電文を生成します.
     * <BR>
     * @param mode MGC-Serverヘッダタイプを設定します.<BR>
     *             [HEADER_TYPE_DEFAULT]を設定した場合、通常電文になります.<BR>
     *             [HEADER_TYPE_RETURN]を設定した場合、戻り要求電文になります.<BR>
     *             [HEADER_TYPE_END]を設定した場合、終了電文になります.
     * @param value 対象のオブジェクトを生成します.
     * @param table 対象の暗号表(256byte)を設定します.<BR>
     *              設定可能なサイズは[256]byteです.<BR>
     *              また、[null]を設定した場合、デフォルト条件になります.
     * @return byte[] 生成されたMGC-Server電文が返されます.
     * @exception InputException 入力例外.
     */
    public static final byte[] createMgcServer( int mode,MgcValue value,byte[] table )
        throws InputException
    {
        int i ;
        int len ;
        int off ;
        int pnt ;
        int nameLen ;
        int addrLen ;
        int rndCd ;
        
        int serverID ;
        int protocolType ;
        int port ;
        int count ;
        
        String serverName = null ;
        InetAddress addr = null ;
        byte[] nameBin = null ;
        byte[] addrBin = null ;
        byte[] ret = null ;
        
        if(
            value == null || value.isUse() == false ||
            ( table != null && table.length != MAX_RAND_CODE )
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        // 対象ランダムコードをセット.
        rndCd = UtilCom.random( MAX_RAND_CODE ) ;
        
        try{
            
            // 対象オブジェクトから情報を取得.
            synchronized( value ){
                serverName = value.getServerName() ;
                serverID = value.getServerID() ;
                protocolType = value.getProtocolType() ;
                addr = value.getBindAddress() ;
                port = value.getBindPort() ;
                count = value.getConnectCount() ;
            }
            
            // サーバ名をバイナリに変換.
            nameBin = serverName.getBytes( DEF_CHARSET ) ;
            serverName = null ;
            
            // アドレス情報をバイナリに変換.
            addrBin = addr.getAddress() ;
            addr = null ;
            
            // サーバ名とアドレス情報が正しい場合.
            if(
                ( nameLen = nameBin.length ) > 0 &&
                ( addrLen = addrBin.length ) > 0
            )
            {
                
                // 電文ヘッダ長を取得.
                len = MGC_HEADER_LENGTH ;
                
                // 空の電文を生成.
                ret = new byte[
                    len +           // ヘッダ.
                    1 +             // 設定ランダムコード.
                    4 +             // サーバID.
                    2 + nameLen +   // サーバ名.
                    1 +             // プロトコルタイプ.
                    2 + addrLen +   // アドレス.
                    2 +             // ポート.
                    4 +             // コネクション数.
                    8               // サーバ時間.
                ] ;
                
                //////////////////
                // ヘッダをセット.
                //////////////////
                if( mode == HEADER_TYPE_DEFAULT ){
                    
                    // 通常電文に設定.
                    for( i = 0 ; i < len ; i ++ ){
                        ret[ i ] = MGC_SERVER_HEADER[ i ] ;
                    }
                    
                }
                else if( mode == HEADER_TYPE_RETURN ){
                    
                    // 戻り要求電文に設定.
                    for( i = 0 ; i < len ; i ++ ){
                        ret[ i ] = MGC_SERVER_RET_HEADER[ i ] ;
                    }
                    
                }
                else if( mode == HEADER_TYPE_END ){
                    
                    // 終了電文に設定.
                    for( i = 0 ; i < len ; i ++ ){
                        ret[ i ] = MGC_SERVER_END_HEADER[ i ] ;
                    }
                    
                }
                pnt = len ;
                
                // ランダムコードをセット.
                ret[ pnt ] = ( byte )rndCd ;
                pnt += 1 ;
                
                // 暗号オフセット値をセット.
                off = pnt ;
                
                // サーバIDをセット.
                ConvertParam.convertInt( ret,pnt,serverID ) ;
                pnt += 4 ;
                
                // サーバ名長をセット.
                ConvertParam.convertShort( ret,pnt,( short )nameLen ) ;
                pnt += 2 ;
                
                // サーバ名をセット.
                System.arraycopy( nameBin,0,ret,pnt,nameLen ) ;
                pnt += nameLen ;
                
                // プロトコルタイプをセット.
                ret[ pnt ] = ( byte )( protocolType & 0x000000ff ) ;
                pnt += 1 ;
                
                // アドレス長をセット.
                ConvertParam.convertShort( ret,pnt,( short )addrLen ) ;
                pnt += 2 ;
                
                // アドレスをセット.
                System.arraycopy( addrBin,0,ret,pnt,addrLen ) ;
                pnt += addrLen ;
                
                // ポート番号をセット.
                ConvertParam.convertShort( ret,pnt,( short )port ) ;
                pnt += 2 ;
                
                // 現在の接続数を取得.
                ConvertParam.convertInt( ret,pnt,count ) ;
                pnt += 4 ;
                
                // 現在のマシン時刻をセット.
                ConvertParam.convertLong( ret,pnt,System.currentTimeMillis() ) ;
                
                /////////////
                // 暗号処理.
                /////////////
                if( table != null ){
                    // 暗号表が存在する場合.
                    ScIO.input( ret,serverID,table,rndCd,off,ret.length-off ) ;
                }
                else{
                    // 暗号表が存在しない場合.
                    ScIO.input( ret,serverID,rndCd,off,ret.length-off ) ;
                }
                
            }
            
        }catch( Exception e ){
            ret = null ;
            throw new InputException( e ) ;
        }finally{
            serverName = null ;
            addr = null ;
            nameBin = null ;
            addrBin = null ;
        }
        
        return ret ;
    }
    
    /**
     * MGC-Server電文を取得.
     * <BR><BR>
     * MGC-Server電文を取得します.
     * <BR>
     * @param out 取得対象のオブジェクトを設定します.
     * @param bin 取得対象のバイナリ情報を設定します.
     * @param table 対象の暗号表(256byte)を設定します.<BR>
     *              設定可能なサイズは[256]byteです.<BR>
     *              また、[null]を設定した場合、デフォルト条件になります.
     * @param thisServerID チェック対象のサーバIDを設定します.<BR>
     *                     この情報は受信対象の条件が正しいかチェックする為の引数です.<BR>
     *                     値として、自マシンの[MgcValue.getServerID()]を設定します.
     * @exception InputException 入力例外.
     */
    public static final void getMgcServer( MgcServerValue out,byte[] bin,byte[] table,int thisServerID )
        throws InputException
    {
        int len ;
        int pnt ;
        int off ;
        int rndCd ;
        int id ;
        int protocolType ;
        int port ;
        int connCnt ;
        long time ;
        
        String name = null ;
        InetAddress addr = null ;
        byte[] tmp = null ;
        
        if( out == null || bin == null || bin.length <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            // ヘッダを読み飛ばす.
            pnt = MGC_SERVER_HEADER.length ;
            
            // ランダムコードを取得.
            rndCd = ( int )( bin[ pnt ] & 0x000000ff ) ;
            pnt += 1 ;
            
            // 暗号解析オフセット値を取得.
            off = pnt ;
            
            ////////////////
            // 暗号解析処理.
            ////////////////
            if( table != null ){
                // 暗号表が存在する場合.
                ScIO.output( bin,thisServerID,table,rndCd,off,bin.length-off ) ;
            }
            else{
                // 暗号表が存在しない場合.
                ScIO.output( bin,thisServerID,rndCd,off,bin.length-off ) ;
            }
            
            // サーバIDを取得.
            id = ConvertParam.convertInt( pnt,bin ) ;
            pnt += 4 ;
            
            // 解析されたServerIDと自マシンのサーバIDが一致しない場合.
            if( id != thisServerID ){
                throw new InputException(
                    "受信されたサーバID(" + id +
                    ")はこのマシンのサーバID(" + thisServerID +
                    ")と一致しないため、不正な電文です"
                ) ;
            }
            
            // サーバ名長を取得.
            len = ( int )ConvertParam.convertShort( pnt,bin ) ;
            pnt += 2 ;
            
            if( len <= 0 ){
                throw new InputException(
                    "サーバ電文:サーバ名長(" + len +
                    ")が不正です"
                ) ;
            }
            
            // サーバ名を取得.
            tmp = new byte[ len ] ;
            System.arraycopy( bin,pnt,tmp,0,len ) ;
            name = new String( tmp,DEF_CHARSET ) ;
            tmp = null ;
            pnt += len ;
            
            // プロトコルタイプを取得.
            protocolType = ( int )( bin[ pnt ] & 0x000000ff ) ;
            pnt += 1 ;
            
            if(
                protocolType != MgcValue.PROTOCOL_TYPE_TCP &&
                protocolType != MgcValue.PROTOCOL_TYPE_UDP &&
                protocolType != MgcValue.PROTOCOL_TYPE_MCAST
            )
            {
                throw new InputException(
                    "サーバ電文:プロトコルタイプ(" + protocolType +
                    ")が不正です"
                ) ;
            }
            
            // アドレス長を取得.
            len = ( int )ConvertParam.convertShort( pnt,bin ) ;
            pnt += 2 ;
            
            if( len <= 0 ){
                throw new InputException(
                    "サーバ電文:アドレス長(" + len +
                    ")が不正です"
                ) ;
            }
            
            // アドレスを取得.
            tmp = new byte[ len ] ;
            System.arraycopy( bin,pnt,tmp,0,len ) ;
            addr = NetDef.getByAddress( tmp ) ;
            tmp = null ;
            pnt += len ;
            
            // ポート番号を取得.
            port = ( int )ConvertParam.convertShort( pnt,bin ) ;
            pnt += 2 ;
            
            // 接続数を取得.
            connCnt = ConvertParam.convertInt( pnt,bin ) ;
            pnt += 4 ;
            
            // 送信元時間を取得.
            time = ConvertParam.convertLong( pnt,bin ) ;
            
            // 情報を引数に設定.
            out.create(
                name,id,protocolType,addr,port,connCnt,time
            ) ;
            
        }catch( InputException in ){
            if( out != null ){
                out.clear() ;
            }
            throw in ;
        }catch( Exception e ){
            if( out != null ){
                out.clear() ;
            }
            throw new InputException( e ) ;
        }finally{
            name = null ;
            addr = null ;
            tmp = null ;
        }
    }
    
    /**
     * MGC-Server電文から、電文送信時間を取得.
     * <BR><BR>
     * MGC-Server電文から、電文送信時間を取得します.
     * <BR>
     * @param bin 対象のバイナリを設定します.
     * @return long 電文送信時間が返されます.
     */
    public static final long getMgcServerByTime( byte[] bin )
    {
        int len ;
        long ret ;
        
        if( bin == null || ( len = bin.length ) <= 0 ){
            return -1L ;
        }
        
        try{
            
            // 対象のマシン時間を取得.
            ret = ConvertParam.convertLong( len-8,bin ) ;
            
        }catch( Exception e ){
            ret = -1L ;
        }
        
        return ret ;
    }
    
    /**
     * MGC-Client電文を生成.
     * <BR><BR>
     * MGC-Client電文を生成します.
     * <BR>
     * @param retCnt 受信回数を設定します.<BR>
     *               指定可能な最小値は[0]です.<BR>
     *               指定可能な最大値は[3]です.
     * @return byte[] 生成されたMGC-Client電文が返されます.
     * @exception InputException 入力例外.
     */
    public static final byte[] createMgcClient( int retCnt )
        throws InputException
    {
        int i ;
        int len ;
        
        byte[] ret = null ;
        
        retCnt = ( retCnt < MIN_MGCCL_RETCNT ) ? MIN_MGCCL_RETCNT : retCnt ;
        retCnt = ( retCnt > MAX_MGCCL_RETCNT ) ? MAX_MGCCL_RETCNT : retCnt ;
        
        try{
            
            // 空の電文を生成.
            len = MGC_CLIENT_HEADER.length ;
            ret = new byte[ len + 1 ] ;
            
            // ヘッダをセット.
            for( i = 0 ; i < len ; i ++ ){
                ret[ i ] = MGC_CLIENT_HEADER[ i ] ;
            }
            
            // 受信回数を設定.
            ret[ len ] = ( byte )( retCnt & 0x000000ff ) ;
            
        }catch( Exception e ){
            ret = null ;
            throw new InputException( e ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * MGC-Client電文から受信回数を取得.
     * <BR><BR>
     * MGC-Client電文から受信回数を取得します.<BR>
     * この情報はサーバ側から見れば「送信回数」になります.
     * <BR>
     * @param bin 対象のMGC-CLIENT電文を設定します.
     * @return int 受信回数が返されます.
     * @exception InputException 入力例外.
     */
    public static final int getMgcClient( byte[] bin )
        throws InputException
    {
        int ret ;
        
        if( bin == null || bin.length < 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            ret = ( int )( bin[ MGC_CLIENT_HEADER.length ] & 0x000000ff ) ;
            
        }catch( Exception e ){
            throw new InputException( e ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象電文がMGCであるかチェック.
     * <BR><BR>
     * 対象の電文がMGCであるかチェックします.
     * <BR>
     * @param bin チェック対象の電文情報を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、対象電文はMGCです.<BR>
     *                 [false]が返された場合、対象電文はMGCではありません.
     */
    public static final boolean isMgc( byte[] bin )
    {
        int i ;
        int len ;
        boolean ret ;
        
        if( bin == null || bin.length <= 0 ){
            return false ;
        }
        
        try{
            
            len = MGC_COMMON_HEADER.length ;
            for( i = 0,ret = true ; i < len ; i ++ ){
                if( bin[ i ] != MGC_COMMON_HEADER[ i ] ){
                    ret = false ;
                    break ;
                }
            }
            
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 対象電文がMGC-Serverであるかチェック.
     * <BR><BR>
     * 対象の電文がMGC-Serverであるかチェックします.
     * <BR>
     * @param bin チェック対象の電文情報を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、対象電文はMGC-Serverです.<BR>
     *                 [false]が返された場合、対象電文はMGC-Serverではありません.
     */
    public static final boolean isMgcServer( byte[] bin )
    {
        int i ;
        int len ;
        boolean ret ;
        
        if( bin == null || bin.length <= 0 ){
            return false ;
        }
        
        try{
            
            len = MGC_SERVER_CH_HEADER.length ;
            for( i = 0,ret = true ; i < len ; i ++ ){
                if( bin[ i ] != MGC_SERVER_CH_HEADER[ i ] ){
                    ret = false ;
                    break ;
                }
            }
            
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 対象電文がMGC-Server戻り要求電文であるかチェック.
     * <BR><BR>
     * 対象の電文がMGC-Server戻り要求電文であるかチェックします.
     * <BR>
     * @param bin チェック対象の電文情報を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、対象電文はMGC-Server戻り要求電文です.<BR>
     *                 [false]が返された場合、対象電文はMGC-Server戻り要求電文ではありません.
     */
    public static final boolean isMgcServerReturn( byte[] bin )
    {
        int i ;
        int len ;
        boolean ret ;
        
        if( bin == null || bin.length <= 0 ){
            return false ;
        }
        
        try{
            
            len = MGC_SERVER_RET_HEADER.length ;
            for( i = 0,ret = true ; i < len ; i ++ ){
                if( bin[ i ] != MGC_SERVER_RET_HEADER[ i ] ){
                    ret = false ;
                    break ;
                }
            }
            
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 対象電文がMGC-Server終了電文であるかチェック.
     * <BR><BR>
     * 対象の電文がMGC-Server終了電文であるかチェックします.
     * <BR>
     * @param bin チェック対象の電文情報を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、対象電文はMGC-Server終了電文です.<BR>
     *                 [false]が返された場合、対象電文はMGC-Server終了電文ではありません.
     */
    public static final boolean isMgcServerEnd( byte[] bin )
    {
        int i ;
        int len ;
        boolean ret ;
        
        if( bin == null || bin.length <= 0 ){
            return false ;
        }
        
        try{
            
            len = MGC_SERVER_END_HEADER.length ;
            for( i = 0,ret = true ; i < len ; i ++ ){
                if( bin[ i ] != MGC_SERVER_END_HEADER[ i ] ){
                    ret = false ;
                    break ;
                }
            }
            
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 対象電文がMGC-Clientであるかチェック.
     * <BR><BR>
     * 対象の電文がMGC-Clientであるかチェックします.
     * <BR>
     * @param bin チェック対象の電文情報を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、対象電文はMGC-Clientです.<BR>
     *                 [false]が返された場合、対象電文はMGC-Clientではありません.
     */
    public static final boolean isMgcClient( byte[] bin )
    {
        int i ;
        int len ;
        boolean ret ;
        
        if( bin == null || bin.length <= 0 ){
            return false ;
        }
        
        try{
            
            len = MGC_CLIENT_HEADER.length ;
            for( i = 0,ret = true ; i < len ; i ++ ){
                if( bin[ i ] != MGC_CLIENT_HEADER[ i ] ){
                    ret = false ;
                    break ;
                }
            }
            
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
}

