/*
 * @(#)MgcClient.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.net.mgc ;

import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.net.BaseMultiCast;
import com.JRcServer.commons.net.MultiCastUtil;
import com.JRcServer.commons.net.MultiCastV4;
import com.JRcServer.commons.thread.Synchronized;

/**
 * MGC-Clientオブジェクト.
 * <BR><BR>
 * このオブジェクトはMGC-Server通信の補助的存在を担います.<BR>
 * 役割として、同一カテゴリの存在を把握する機能をもっており、接続状況や
 * 切断情報を管理します.<BR>
 * そのため実際の電文交換を行う場合には、別途プロトコルを設置する必要があります.

 *  
 * @version 1.0.0 2005/08/03
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
public class MgcClient
{
    
    /**
     * マルチキャストオブジェクト.
     */
    private BaseMultiCast m_mcast = null ;
    
    /**
     * コネクション用グループID.
     */
    private int m_groupID = -1 ;
    
    /**
     * 接続管理情報.
     */
    private final MgcManagerImple m_man = new MgcManagerImple() ;
    
    /**
     * Mgcクライアント受信スレッド.
     */
    private MgcCLRcvThread m_rcvThread = null ;
    
    /**
     * Mgcクライアント管理スレッド.
     */
    private MgcCLThread m_thread = null ;
    
    /**
     * 通信テーブル.
     */
    private final MgcTable m_connTable = new MgcTable() ;
    
    
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    
    /**
     * コンストラクタ.
     */
    public MgcClient()
    {
        m_sync.clear() ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.close() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * クライアントオープン.
     * <BR><BR>
     * 対象のクライアントをオープンします.
     * <BR>
     * @param mcast このオブジェクトが利用するマルチキャストを設定します.
     * @param id 対象のグループIDを設定します.
     * @exception InputException 入力例外.
     */
    public final void open( BaseMultiCast mcast,int id )
        throws InputException
    {
        this.open( mcast,id,null,-1 ) ;
    }
    
    /**
     * クライアントオープン.
     * <BR><BR>
     * 対象のクライアントをオープンします.
     * <BR>
     * @param mcast このオブジェクトが利用するマルチキャストを設定します.
     * @param id 対象のグループIDを設定します.
     * @param time 各接続状況の管理時間を設定します.<BR>
     *             この値は[30000]以下を設定しても意味を持ちません.<BR>
     *             また、設定単位はミリ秒です.
     * @exception InputException 入力例外.
     */
    public final void open( BaseMultiCast mcast,int id,int time )
        throws InputException
    {
        this.open( mcast,id,null,time ) ;
    }
    
    /**
     * クライアントオープン.
     * <BR><BR>
     * 対象のクライアントをオープンします.
     * <BR>
     * @param mcast このオブジェクトが利用するマルチキャストを設定します.
     * @param id 対象のグループIDを設定します.
     * @param table 通信に対する暗号表を設定します.<BR>
     *              [null]を設定した場合、デフォルトの暗号表を利用します.
     * @exception InputException 入力例外.
     */
    public final void open( BaseMultiCast mcast,int id,byte[] table )
        throws InputException
    {
        this.open( mcast,id,table,-1 ) ;
    }
    
    /**
     * クライアントオープン.
     * <BR><BR>
     * 対象のクライアントをオープンします.
     * <BR>
     * @param mcast このオブジェクトが利用するマルチキャストを設定します.
     * @param id 対象のグループIDを設定します.
     * @param table 通信に対する暗号表を設定します.<BR>
     *              [null]を設定した場合、デフォルトの暗号表を利用します.
     * @param time 各接続状況の管理時間を設定します.<BR>
     *             この値は[30000]以下を設定しても意味を持ちません.<BR>
     *             また、設定単位はミリ秒です.
     * @exception InputException 入力例外.
     */
    public final void open( BaseMultiCast mcast,int id,byte[] table,int time )
        throws InputException
    {
        
        if(
            mcast == null || mcast.isOpen() == false ||
            id < 0 || id > MultiCastUtil.MULTICAST_END_V4 ||
            ( table != null && table.length != MgcCommon.MAX_RAND_CODE )
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.close() ;
        m_sync.create() ;
        
        try{
            
            m_groupID = id ;
            ( ( MultiCastV4 )mcast ).joinGroup( m_groupID ) ;
            
            synchronized( m_sync.get() ){
                
                m_mcast = mcast ;
                
                m_thread = new MgcCLThread( id,m_man,m_connTable,table,time ) ;
                m_rcvThread = new MgcCLRcvThread( mcast,m_connTable ) ;
                
            }
            
        }catch( InputException in ){
            this.close() ;
            throw in ;
        }catch( Exception e ){
            this.close() ;
            throw new InputException( e ) ;
        }
        
    }
    
    /**
     * サーバクローズ.
     * <BR><BR>
     * サーバをクローズします.
     */
    public final void close()
    {
        try{
            synchronized( m_sync.get() ){
                
                try{
                    m_rcvThread.destroy() ;
                }catch( Exception ee ){
                }
                
                try{
                    m_thread.destroy() ;
                }catch( Exception ee ){
                }
                
                try{
                    m_mcast.leaveGroup() ;
                }catch( Exception ee ){
                }
                
                m_man.clear() ;
                m_connTable.clear() ;
                
            }
        }catch( Exception e ){
        }
        
        m_sync.clear() ;
        m_mcast = null ;
        m_groupID = -1 ;
        m_rcvThread = null ;
        m_thread = null ;
        
    }
    
    /**
     * サーバ通知を送信.
     * <BR><BR>
     * サーバ通知を送信します.<BR>
     * この処理を実行することで、各サーバから現在のステータスが受信されます.
     */
    public final void sendNotice()
    {
        this.sendNotice( 1 ) ;
    }
    
    /**
     * サーバ通知を送信.
     * <BR><BR>
     * サーバ通知を送信します.<BR>
     * この処理を実行することで、各サーバから現在のステータスが受信されます.
     * <BR>
     * @param cnt 受信回数を設定します.<BR>
     *            設定可能な最大値は[3]です.<BR>
     *            通常は[1]を設定します.
     */
    public final void sendNotice( int cnt )
    {
        byte[] snd = null ;
        
        try{
            synchronized( m_sync.get() ){
                snd = MgcCommon.createMgcClient( cnt ) ;
                m_mcast.send( snd ) ;
            }
        }catch( Exception e ){
        }finally{
            snd = null ;
        }
    }
    
    /**
     * グループIDを取得.
     * <BR><BR>
     * 設定されているグループIDを取得します.
     * <BR>
     * @return int 設定されているグループIDが返されます.
     *             [-1]が返された場合、このオブジェクトはクローズされています.
     */
    public final int getGroupID()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_groupID ;
            }
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * マネージャオブジェクトを取得.
     * <BR><BR>
     * マネージャオブジェクトを取得します.
     * <BR>
     * @return MgcManager マネージャオブジェクトが返されます.
     *                    [null]が返された場合、このオブジェクトはクローズされています.
     */
    public final MgcManager getManager()
    {
        MgcManager ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_man ;
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 設定されている暗号表を取得.
     * <BR><BR>
     * 設定されている暗号表を取得します.
     * <BR>
     * @return byte[] 設定されている暗号表が返されます.
     *                [null]が返された場合、デフォルト値が設定されているか
     *                オブジェクトがクローズされています.
     */
    public final byte[] getCodeTable()
    {
        byte[] ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_thread.getCodeTable() ;
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 接続状況監視時間を取得.
     * <BR><BR>
     * 設定されている接続状況を監視する時間を取得します.
     * <BR>
     * @return int 接続状況監視時間が返されます.<BR>
     *             [-1]が返された場合、オブジェクトがクローズされています.
     */
    public final int getCheckConnectTime()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_thread.getCheckConnectTime() ;
            }
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * オープンチェック.
     * <BR><BR>
     * このオブジェクトがオープンされているかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、オープンされています.<BR>
     *                 [false]が返された場合、クローズされています.
     */
    public final boolean isOpen()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                if(
                    m_thread.isUse() == true &&
                    m_rcvThread.isUse() == true
                )
                {
                    ret = true ;
                }
                else{
                    ret = false ;
                }
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
}

