/*
 * @(#)MgcCLRcvThread.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.net.mgc ;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.JRcServer.commons.exception.ExecutionException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.net.BaseMultiCast;
import com.JRcServer.commons.net.ConnectAddress;
import com.JRcServer.commons.net.ConnectTimeoutException;
import com.JRcServer.commons.thread.ExecutionThread;
import com.JRcServer.commons.thread.LoopThread;
import com.JRcServer.commons.thread.Synchronized;
import com.JRcServer.commons.util.UtilCom;

/**
 * MGC-Client受信スレッド.
 *  
 * @version 1.0.0 2005/07/29
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
class MgcCLRcvThread extends ExecutionThread
{
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( MgcCLRcvThread.class ) ;
    
    /**
     * デフォルト受信時間.
     */
    private static final int DEF_RCVTIME = 250 ;
    
    /**
     * 電文送信回数 : 最大値.
     */
    private static final int MAX_SEND = 3 ;
    
    
    
    /**
     * マルチキャストオブジェクト.
     */
    private BaseMultiCast m_mcast = null ;
    
    /**
     * 受信アドレス格納オブジェクト.
     */
    private ConnectAddress m_addr = null ;
    
    /**
     * クライアントテーブル.
     */
    private MgcTable m_clTable = null ;
    
    
    
    /**
     * ループスレッド.
     */
    private final LoopThread m_thread = new LoopThread() ;
    
    /**
     * 同期処理.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    
    /**
     * コンストラクタ.
     */
    private MgcCLRcvThread(){}
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象の条件を設定してオブジェクトを生成します.
     * <BR>
     * @param mcast マルチキャストオブジェクトを設定します.
     * @param clTbl 対象のMgcクライアントテーブルを設定します.
     * @exception InputException 入力例外.
     */
    public MgcCLRcvThread( BaseMultiCast mcast,MgcTable clTbl )
        throws InputException
    {
        if(
            mcast == null || mcast.isOpen() == false || mcast.isGroup() == false ||
            clTbl == null
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        m_sync.create() ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                // 基本条件設定.
                m_mcast = mcast ;
                m_clTable = clTbl ;
                
                m_addr = new ConnectAddress() ;
                
                // スレッドを生成.
                m_thread.create( this ) ;
                m_thread.startThread() ;
                
            }
            
        }catch( Exception e ){
            this.destroy() ;
            throw new InputException( e ) ;
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクト情報を破棄します.
     */
    public final void destroy()
    {
        try{
            synchronized( m_sync.get() ){
                
                m_thread.clear() ;
                
                m_mcast = null ;
                m_clTable = null ;
                m_addr = null ;
                
            }
        }catch( Exception e ){
        }
        
        m_sync.clear() ;
        
        m_mcast = null ;
        m_clTable = null ;
        m_addr = null ;
        
    }
    
    /**
     * オブジェクト利用可能チェック.
     * <BR><BR>
     * オブジェクト情報が利用可能かチェックします.
     * <BR>
     * @return boolean オブジェクトが利用可能かチェックします.
     */
    public final boolean isUse()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_thread.isThread() ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * 実行初期化処理をサポートします.
     * <BR><BR>
     * 実行初期化処理をサポートします.<BR>
     * この処理は、スレッド処理が開始された時に呼び出されます.
     * <BR>
     * @param obj 実行開始時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void init( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行終了化処理をサポートします.
     * <BR><BR>
     * 実行終了化処理をサポートします.<BR>
     * この処理は、スレッド処理が終了された時に呼び出されます.
     * <BR>
     * @param obj 実行終了時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void exit( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * ストップ処理をサポートします。
     * <BR><BR>
     * ストップ処理をサポートします。<BR>
     * この処理は、スレッドでのストップ処理に対して呼び出し実行されます.
     * <BR>
     * @param obj ストップ時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void stop( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行処理をサポートします。
     * <BR><BR>
     * 実行処理をサポートします。<BR>
     * この処理は、スレッドでの実行処理に対して呼び出し実行されます.
     * <BR>
     * @param obj 実行時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void execution( Object obj )
        throws ExecutionException
    {
        int i ;
        int cnt ;
        long befTime ;
        
        BaseMultiCast mcast = null ;
        MgcTable clTbl = null ;
        ConnectAddress addr = null ;
        byte[] rcv = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                mcast = m_mcast ;
                clTbl = m_clTable ;
                
                addr = m_addr ;
                
            }
            
            ///////////////////////////
            // マルチキャストから受信.
            ///////////////////////////
            try{
                // 受信処理.
                rcv = mcast.receive( addr,DEF_RCVTIME ) ;
                
            }catch( ConnectTimeoutException ct ){
                rcv = null ;
            }
            
            ////////////////////////////
            // 受信情報が存在する場合.
            ////////////////////////////
            if( rcv != null ){
                
                //<><><><><><><><><><><><><><><><><><><><><><><><><><>
                // 受信された電文タイプが「サーバ送信」の場合.
                //<><><><><><><><><><><><><><><><><><><><><><><><><><>
                if( MgcCommon.isMgcServer( rcv ) == true ){
                    
                    // クライアントテーブルに登録.
                    clTbl.set( rcv ) ;
                    
                }
                
                rcv = null ;
                
            }
            else{
                
                // 一定期間待機.
                UtilCom.idleTime() ;
                
            }
            
        }catch( NullPointerException nul ){
            throw new ExecutionException(
                nul,ExecutionException.LEVEL_STOP
            ) ;
        }catch( ExecutionException ee ){
            throw ee ;
        }catch( InputException in ){
            LOG.error( "エラーが発生しました",in ) ;
        }catch( Exception e ){
        }finally{
            mcast = null ;
            clTbl = null ;
            addr = null ;
            rcv = null ;
        }
        
    }
    
}

