/*
 * @(#)UdpProtocol.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.net;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.SocketException;
import java.net.UnknownHostException;

import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.thread.Synchronized;

/**
 * UDPプロトコルオブジェクト.
 * <BR><BR>
 * UDPプロトコルで処理するオブジェクトです.
 *
 * @version 1.00, 2004/10/19
 * @author  Masahito Suzuki
 * @since   JRcCommons 1.00
 */
public class UdpProtocol implements BaseUdpProtocol
{
    
    /**
     * UDPプロトコルオブジェクト.
     */
    private DatagramSocket m_udp = null ;
    
    /**
     * 受信用バイナリ情報.
     */
    private final byte[] m_binary = new byte[ BaseUdpProtocol.BUF_LENGTH ] ;
    
    /**
     * 最終処理時間.
     */
    private long m_lastTime = 0L ;
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    /**
     * 送信同期オブジェクト.
     */
    private final Synchronized m_sendSync = new Synchronized() ;
    
    /**
     * 受信同期オブジェクト.
     */
    private final Synchronized m_recvSync = new Synchronized() ;
    
    
    /**
     * コンストラクタ.
     */
    public UdpProtocol()
    {
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.close() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オープン処理.
     * <BR><BR>
     * ポート番号をデフォルトポート番号で指定して、オープンします.
     * <BR>
     * @param bufLen 送受信バッファ長を設定します.
     * @exception NotBindException バインド失敗.
     */
    public final void open( int bufLen )
        throws NotBindException
    {
        this.openTo( UdpProtocol.ALL_MODE,bufLen ) ;
    }
    
    /**
     * オープン処理.
     * <BR><BR>
     * 対象のポート番号をオープンします.
     * <BR>
     * @param bufLen 送受信バッファ長を設定します.
     * @param port オープン対象のポート番号を設定します.
     * @exception InputException 入力例外.
     * @exception NotBindException バインド失敗.
     */
    public final void open( int bufLen,int port )
        throws InputException,NotBindException
    {
        this.openTo( UdpProtocol.ALL_MODE,bufLen,port ) ;
    }
    
    /**
     * オープン処理.
     * <BR><BR>
     * 対象の自IPアドレス上にポート番号でオープンします.<BR>
     * この処理の場合、対象のアドレスはDHCPから割り振られたアドレスで無い事が
     * 条件となります.
     * <BR>
     * @param bufLen 送受信バッファ長を設定します.
     * @param port オープン対象のポート番号を設定します.
     * @param addr オープン対象の自IPアドレスを設定します.
     * @exception InputException 入力例外.
     * @exception NotBindException バインド失敗.
     */
    public final void open( int bufLen,int port,InetAddress addr )
        throws InputException,NotBindException
    {
        this.openTo( UdpProtocol.ALL_MODE,bufLen,port,addr ) ;
    }
    
    /**
     * オープン処理.
     * <BR><BR>
     * ポート番号をデフォルトポート番号で指定して、オープンします.
     * <BR>
     * @param bufMode バッファモードを設定します.<BR>
     *                [UdpProtocol#NOT_MODE]を指定した場合、デフォルト値の
     *                バッファ長を設定します.<BR>
     *                [UdpProtocol#SND_MODE]を指定した場合、送信側バッファのみ
     *                バッファ値を設定します.<BR>
     *                [UdpProtocol#RCV_MODE]を指定した場合、受信側バッファのみ
     *                バッファ値を設定します.<BR>
     *                [UdpProtocol#ALL_MODE]を指定した場合、送信/受信側バッファに
     *                バッファ値を設定します.<BR>
     * @param bufLen 送受信バッファ長を設定します.
     * @exception NotBindException バインド失敗.
     */
    public final void openTo( int bufMode,int bufLen )
        throws NotBindException
    {
        
        this.close() ;
        
        m_sync.create() ;
        m_sendSync.create() ;
        m_recvSync.create() ;
        
        try{
            
            //this.checkUDP( addr ) ;
            
            synchronized( m_sync.get() ){
                
                m_udp = new DatagramSocket() ;
                
                if( this.createUDP( bufMode,bufLen ) == false ){
                    this.close() ;
                }
                
            }
        }catch( NullPointerException nul ){
            this.close() ;
        //}catch( UnknownHostException uh ){
        //  this.close() ;
        //  throw new NotBindException( uh ) ;
        }catch( SocketException so ){
            this.close() ;
            throw new NotBindException( so ) ;
        //}catch( NotBindException nb ){
        //  this.close() ;
        //  throw nb ;
        }finally{
            //addr = null ;
        }
    }
    
    /**
     * オープン処理.
     * <BR><BR>
     * 対象のポート番号をオープンします.
     * <BR>
     * @param bufMode バッファモードを設定します.<BR>
     *                [UdpProtocol#NOT_MODE]を指定した場合、デフォルト値の
     *                バッファ長を設定します.<BR>
     *                [UdpProtocol#SND_MODE]を指定した場合、送信側バッファのみ
     *                バッファ値を設定します.<BR>
     *                [UdpProtocol#RCV_MODE]を指定した場合、受信側バッファのみ
     *                バッファ値を設定します.<BR>
     *                [UdpProtocol#ALL_MODE]を指定した場合、送信/受信側バッファに
     *                バッファ値を設定します.<BR>
     * @param bufLen 送受信バッファ長を設定します.
     * @param port オープン対象のポート番号を設定します.
     * @exception InputException 入力例外.
     * @exception NotBindException バインド失敗.
     */
    public final void openTo( int bufMode,int bufLen,int port )
        throws InputException,NotBindException
    {
        
        if( port <= NetDef.PORT_MIN || port >= NetDef.PORT_MAX ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.close() ;
        
        m_sync.create() ;
        m_sendSync.create() ;
        m_recvSync.create() ;
        
        try{
            
            //this.checkUDP( addr ) ;
            
            synchronized( m_sync.get() ){
                
                m_udp = new DatagramSocket( port ) ;
                
                if( this.createUDP( bufMode,bufLen ) == false ){
                    this.close() ;
                }
                
            }
        }catch( NullPointerException nul ){
            this.close() ;
        //}catch( UnknownHostException uh ){
        //  this.close() ;
        //  throw new NotBindException( uh ) ;
        }catch( SocketException so ){
            this.close() ;
            throw new NotBindException( so ) ;
        //}catch( NotBindException nb ){
        //  this.close() ;
        //  throw nb ;
        }finally{
            //addr = null ;
        }
    }
    
    /**
     * オープン処理.
     * <BR><BR>
     * 対象の自IPアドレス上にポート番号でオープンします.<BR>
     * この処理の場合、対象のアドレスはDHCPから割り振られたアドレスで無い事が
     * 条件となります.
     * <BR>
     * @param bufMode バッファモードを設定します.<BR>
     *                [UdpProtocol#NOT_MODE]を指定した場合、デフォルト値の
     *                バッファ長を設定します.<BR>
     *                [UdpProtocol#SND_MODE]を指定した場合、送信側バッファのみ
     *                バッファ値を設定します.<BR>
     *                [UdpProtocol#RCV_MODE]を指定した場合、受信側バッファのみ
     *                バッファ値を設定します.<BR>
     *                [UdpProtocol#ALL_MODE]を指定した場合、送信/受信側バッファに
     *                バッファ値を設定します.<BR>
     * @param bufLen 送受信バッファ長を設定します.
     * @param port オープン対象のポート番号を設定します.
     * @param addr オープン対象の自IPアドレスを設定します.
     * @exception InputException 入力例外.
     * @exception NotBindException バインド失敗.
     */
    public final void openTo( int bufMode,int bufLen,int port,InetAddress addr )
        throws InputException,NotBindException
    {
        if( addr == null || port <= NetDef.PORT_MIN || port >= NetDef.PORT_MAX ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.close() ;
        
        m_sync.create() ;
        m_sendSync.create() ;
        m_recvSync.create() ;
        
        try{
            
            //this.checkUDP( addr ) ;
            
            synchronized( m_sync.get() ){
                
                m_udp = new DatagramSocket( port,addr ) ;
                
                if( this.createUDP( bufMode,bufLen ) == false ){
                    this.close() ;
                }
                
            }
        }catch( NullPointerException nul ){
            this.close() ;
        }catch( SocketException so ){
            this.close() ;
            throw new NotBindException( so ) ;
        //}catch( NotBindException nb ){
        //  this.close() ;
        //  throw nb ;
        }
    }
    
    /**
     * クローズ処理.
     * <BR><BR>
     * クローズ処理を行います.
     */
    public final void close()
    {
        
        m_sync.clear() ;
        m_sendSync.clear() ;
        m_recvSync.clear() ;
        m_lastTime = 0L ;
        
        try{
            m_udp.close() ;
        }catch( Exception t ){
        }finally{
            m_udp = null ;
        }
        
    }
    
    /**
     * データ送信.
     * <BR><BR>
     * 対象のデータを送信します.
     * <BR>
     * @param binary 送信対象のデータを設定します.
     * @param addr 送信対象のIPアドレス/ポート番号を指定します.
     * @exception InputException 入力例外.
     * @exception UndefineBindException 未バインド例外.
     */
    public final void send( byte[] binary,ConnectAddress addr )
        throws InputException,UndefineBindException
    {
        if( addr == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.send( binary,addr.getAddress(),addr.getPort() ) ;
    }
    
    /**
     * データ送信.
     * <BR><BR>
     * 対象のデータを送信します.
     * <BR>
     * @param binary 送信対象のデータを設定します.
     * @param addr 送信対象のIPアドレスを指定します.
     * @param port 送信対象のポート番号を設定します.
     * @exception InputException 入力例外.
     * @exception UndefineBindException 未バインド例外.
     */
    public final void send( byte[] binary,InetAddress addr,int port )
        throws InputException,UndefineBindException
    {
        DatagramPacket packet = null ;
        
        if(
            binary == null || addr == null ||
            port < NetDef.PORT_MIN || port >= NetDef.PORT_MAX
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            synchronized( m_sendSync.get() ){
                packet = new DatagramPacket( binary,0,binary.length,addr,port ) ;
                m_udp.send( packet ) ;
                m_lastTime = System.currentTimeMillis() ;
            }
            
        }catch( NullPointerException nul ){
            throw new UndefineBindException( nul ) ;
        }catch( UnknownHostException uk ){
            throw new InputException( uk ) ;
        }catch( IOException io ){
            throw new UndefineBindException( io ) ;
        }finally{
            packet = null ;
        }
    }
    
    /**
     * データ受信.
     * <BR><BR>
     * データを受信します.<BR>
     * データが存在しない場合[null]が返されます.
     * <BR>
     * @param addr 受信先のIPアドレスとポート番号が格納された
     *             内容が返されます.
     * @return byte[] 受信されたバイナリ情報が返されます.<BR>
     *                受信対象の情報が存在しない場合[null]が返されます.
     * @exception UndefineBindException バインド未定義例外.
     * @exception ConnectTimeoutException タイムアウト例外.
     */
    public final byte[] receive( ConnectAddress addr )
        throws UndefineBindException,ConnectTimeoutException
    {
        return this.receive( addr,0 ) ;
    }
    
    /**
     * データ受信.
     * <BR><BR>
     * データを受信します.<BR>
     * データが存在しない場合[null]が返されます.
     * <BR>
     * @param addr 受信先のIPアドレスとポート番号が格納された
     *             内容が返されます.
     * @param timeout 受信タイムアウト値を設定します.
     * @return byte[] 受信されたバイナリ情報が返されます.<BR>
     *                受信対象の情報が存在しない場合[null]が返されます.
     * @exception UndefineBindException バインド未定義例外.
     * @exception ConnectTimeoutException タイムアウト例外.
     */
    public final byte[] receive( ConnectAddress addr,int timeout )
        throws UndefineBindException,ConnectTimeoutException
    {
        int len ;
        
        byte[] ret = null ;
        DatagramPacket packet = null ;
        
        try{
            
            synchronized( m_recvSync.get() ){
                
                packet = new DatagramPacket( m_binary,BaseUdpProtocol.BUF_LENGTH ) ;
                m_udp.setSoTimeout( ( timeout < 0 ) ? 0 : timeout ) ;
                m_udp.receive( packet ) ;
                
                len = packet.getLength() ;
                ret = new byte[ len ] ;
                System.arraycopy( m_binary,0,ret,0,len ) ;
                
                if( addr != null ){
                    addr.create( packet.getAddress(),packet.getPort() ) ;
                }
                m_lastTime = System.currentTimeMillis() ;
                
            }
            
        }catch( NullPointerException nul ){
            ret = null ;
        }catch( InterruptedIOException ii ){
            ret = null ;
            throw new ConnectTimeoutException( ii ) ;
        }catch( InputException in ){
            ret = null ;
        }catch( IOException io ){
            throw new UndefineBindException( io ) ;
        }finally{
            packet = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * データ受信.
     * <BR><BR>
     * データを受信します.<BR>
     * データが存在しない場合[null]が返されます.
     * <BR>
     * @param out 受信されたバイナリ情報が設定されます.
     * @param addr 受信先のIPアドレスとポート番号が格納された
     *             内容が返されます.
     * @return int 受信されたバイナリ情報長が返されます.
     * @exception InputException 入力例外
     * @exception UndefineBindException バインド未定義例外.
     * @exception ConnectTimeoutException タイムアウト例外.
     */
    public final int receive( byte[] out,ConnectAddress addr )
        throws InputException,UndefineBindException,ConnectTimeoutException
    {
        return this.receive( out,addr,0 ) ;
    }
    
    /**
     * データ受信.
     * <BR><BR>
     * データを受信します.<BR>
     * データが存在しない場合[null]が返されます.
     * <BR>
     * @param out 受信されたバイナリ情報が設定されます.
     * @param addr 受信先のIPアドレスとポート番号が格納された
     *             内容が返されます.
     * @param timeout 受信タイムアウト値を設定します.
     * @return int 受信されたバイナリ情報長が返されます.
     * @exception InputException 入力例外
     * @exception UndefineBindException バインド未定義例外.
     * @exception ConnectTimeoutException タイムアウト例外.
     */
    public final int receive( byte[] out,ConnectAddress addr,int timeout )
        throws InputException,UndefineBindException,ConnectTimeoutException
    {
        int len ;
        int ret ;
        
        DatagramPacket packet = null ;
        
        if( out == null || out.length <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            synchronized( m_recvSync.get() ){
                
                packet = new DatagramPacket( m_binary,BaseUdpProtocol.BUF_LENGTH ) ;
                m_udp.setSoTimeout( ( timeout < 0 ) ? 0 : timeout ) ;
                m_udp.receive( packet ) ;
                
                ret = packet.getLength() ;
                len = out.length ;
                len = ( len <= ret ) ? len : ret ;
                System.arraycopy( m_binary,0,out,0,len ) ;
                
                if( addr != null ){
                    addr.create( packet.getAddress(),packet.getPort() ) ;
                }
                m_lastTime = System.currentTimeMillis() ;
                
            }
            
        }catch( NullPointerException nul ){
            ret = -1 ;
        }catch( InterruptedIOException ii ){
            ret = -1 ;
            throw new ConnectTimeoutException( ii ) ;
        }catch( IOException io ){
            throw new UndefineBindException( io ) ;
        }finally{
            packet = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * ローカルアドレスを取得.
     * <BR><BR>
     * 対象のローカルアドレスを取得します.
     * <BR>
     * @param addr 対象のローカルアドレスが返されます.
     */
    public final void getLocal( ConnectAddress addr )
    {
        try{
            if( addr != null ){
                synchronized( m_sync.get() ){
                    addr.create(
                        m_udp.getLocalAddress(),
                        m_udp.getLocalPort()
                    ) ;
                }
            }
        }catch( Exception t ){
            if( addr != null ){
                
                try{
                    addr.clear() ;
                    addr.create( NetDef.NOT_ADDR,NetDef.PORT_MIN ) ;
                }catch( Exception tt ){
                }
                
            }
        }
    }
    
    /**
     * ローカルアドレスを取得.
     * <BR><BR>
     * 対象のローカルアドレスを取得します.
     * <BR>
     * @return ConnectAddress 対象のローカルアドレスが返されます.
     */
    public final ConnectAddress getLocal()
    {
        ConnectAddress ret = new ConnectAddress() ;
        
        try{
            synchronized( m_sync.get() ){
                ret = new ConnectAddress(
                    m_udp.getLocalAddress(),
                    m_udp.getLocalPort()
                ) ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * ローカルアドレス情報を取得.
     * <BR><BR>
     * ローカルアドレス情報を取得します.
     * <BR>
     * @return InetAddress ローカルアドレス情報が返されます.
     */
    public final InetAddress getLocalAddress()
    {
        InetAddress ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_udp.getLocalAddress() ;
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * ローカルポート番号を取得.
     * <BR><BR>
     * ローカルポート番号を取得します.
     * <BR>
     * @return int ローカルポート番号が返されます.
     */
    public final int getLocalPort()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_udp.getLocalPort() ;
            }
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 設定バッファ長を取得.
     * <BR><BR>
     * 設定されているバッファ長を取得します.
     * <BR>
     * @return int 設定バッファ長が返されます.
     */
    public final int getBuffer()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_udp.getSendBufferSize() ;
            }
        }catch( Exception t ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * オープンチェック.
     * <BR><BR>
     * オープンされているかチェックします.
     * <BR>
     * @return boolean オープン状態が返されます.<BR>
     *                 [true]が返された場合、ソケットはオープンされています.<BR>
     *                 [false]が返された場合、ソケットはオープンされていません.
     */
    public final boolean isOpen()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                m_udp.getSendBufferSize() ;
                ret = ( m_udp == null ) ? false : true ;
            }
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * 生成チェック.
     */
    /*private final void checkUDP( InetAddress addr )
        throws NotBindException
    {
        if(
            addr == null &&
            NetDef.LOCAL_HOST.equalsIgnoreCase( addr.getHostAddress() ) == false &&
            NowAddress.getInstance().isUse( addr ) == false
        )
        {
            if( addr != null ){
                throw new NotBindException( "指定アドレス(" + addr + ")は存在しません" ) ;
            }
            else{
                throw new NotBindException( "不正なアドレスが設定されています" ) ;
            }
        }
    }*/
    
    /**
     * 初期処理共通.
     */
    private final boolean createUDP( int bufMode,int bufLen )
    {
        boolean ret ;
        
        bufLen = ( bufLen <= NetDef.DEF_BUFLEN ) ? NetDef.DEF_BUFLEN : bufLen ;
        
        try{
             
            if( ( bufMode & BaseUdpProtocol.SND_MODE ) != 0 ){
                m_udp.setSendBufferSize(
                    ( int )( bufLen * NetDef.NETWORK_SEND_BUFFER )
                ) ;
            }
            else{
                m_udp.setSendBufferSize(
                    ( int )( NetDef.DEF_BUFLEN * NetDef.NETWORK_SEND_BUFFER )
                ) ;
            }
            
            if( ( bufMode & BaseUdpProtocol.RCV_MODE ) != 0 ){
                m_udp.setReceiveBufferSize(
                    ( int )( bufLen * NetDef.NETWORK_RECEIVE_BUFFER )
                ) ;
            }
            else{
                m_udp.setReceiveBufferSize(
                    ( int )( NetDef.DEF_BUFLEN * NetDef.NETWORK_RECEIVE_BUFFER )
                ) ;
            }
            
            m_udp.setSoTimeout( 0 ) ;
            
            ret = true ;
            
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
        
    }
    
}

