/*
 * @(#)TelegramResourceRoll.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.net;

import java.net.InetAddress;

import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.resource.BinResource;
import com.JRcServer.commons.resource.Resource;
import com.JRcServer.commons.resource.ResourceType;
import com.JRcServer.commons.util.Roll;

/**
 * 電文リソースロール.
 * <BR><BR>
 * 電文リソース用のロール管理を行うオブジェクトです.
 *
 * @version     1.00, 2005/05/28
 * @author      Masahito Suzuki
 * @since   JRcCommons 1.00
 */
public class TelegramResourceRoll implements Roll
{
    /**
     * デフォルト管理数.
     */
    private static final int DEF_SIZE = 32 ;
    
    
    
    /**
     * 送信ロール配列.
     */
    private TelegramResourceValue[] m_roll = null ;
    
    /**
     * リソースタイプ.
     */
    private ResourceType m_resType = null ;
    
    /**
     * ロール管理最大数.
     */
    private int m_max = 0 ;
    
    /**
     * 現在位置.
     */
    private int m_now = 0 ;
    
    /**
     * 格納数.
     */
    private int m_nowLen = 0 ;
    
    /**
     * 生成モード.
     */
    private boolean m_mode = false ;
    
    /**
     * 追加予約フラグ.
     */
    private int m_addResCnt = 0 ;
    
    /**
     * コンストラクタ.
     */
    private TelegramResourceRoll(){}
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * ロール管理数を設定して生成します.<BR>
     * また、このコンストラクタで生成されるキャッシュモードは
     * キャッシュ非再作成モードです.
     * <BR>
     * @param resType 対象のリソースタイプを設定します.
     * @param max ロール管理数を設定します.<BR>
     *            設定値は[255]以下を設定した場合、その値となります.
     * @exception InputException 入力例外.
     */
    public TelegramResourceRoll( ResourceType resType,int max )
        throws InputException
    {
        this( false,resType,max ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * ロール管理数を設定して生成します.<BR>
     * また、キャッシュモードの選択が可能です.<BR>
     * キャッシュモードは、(1)キャッシュ再作成モードと(2)キャッシュ非再作成モードの
     * ２つのモードが存在します.<BR>
     * これは、渡されるキャッシュサイズに設定するバイナリサイズの上限が一定の場合は、(2)を
     * 利用したほうが効率がよくなります.<BR>
     * その反対にキャッシュサイズに設定するバイナリサイズの上限が一定では内場合は、(1)を
     * 利用したほうが効率よくなります.<BR>
     * また設定した値を受け取るオブジェクト(TelegramValue)はキャッシュモードによって
     * 戻り値のオブジェクト型が変わります.<BR>
     * (1)の場合[TelegramValue.getBinary()]において、戻り値は(BinResource)となります.<BR>
     * (2)の場合[TelegramValue.getBinary()]において、戻り値は(byte[])となります.<BR>
     * このため、キャッシュモードを選択する場合は利用条件に合わせて定義してください.
     * <BR>
     * @param mode キャッシュモードを設定します.<BR>
     *             [true]を設定した場合、キャッシュ再作成モードとします.<BR>
     *             [false]を設定した場合、キャッシュ非再作成モードとします.<BR>
     * @param resType 対象のリソースタイプを設定します.
     * @param max ロール管理数を設定します.<BR>
     *            設定値は[32]以下を設定した場合、その値となります.
     * @exception InputException 入力例外.
     */
    public TelegramResourceRoll( boolean mode,ResourceType resType,int max )
        throws InputException
    {
        int i ;
        TelegramResourceValue[] roll = null ;
        
        if( resType == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        m_mode = mode ;
        max = ( max <= DEF_SIZE ) ? DEF_SIZE : max ;
        
        roll = new TelegramResourceValue[ max ] ;
        
        // キャッシュ再作成モード.
        if( mode == true ){
            for( i = 0 ; i < max ; i ++ ){
                roll[ i ] = new TelegramResourceValue() ;
                roll[ i ].RES = null ;
                roll[ i ].ADDR = null ;
                roll[ i ].PORT = -1 ;
            }
        }
        // キャッシュ非再作成モード.
        else{
            for( i = 0 ; i < max ; i ++ ){
                roll[ i ] = new TelegramResourceValue() ;
                roll[ i ].RES = Resource.createBinResource( resType,1 ) ;
                roll[ i ].ADDR = null ;
                roll[ i ].PORT = -1 ;
            }
        }
        
        m_roll = roll ;
        m_resType = resType ;
        m_max = max ;
        m_now = 0 ;
        m_nowLen = 0 ;
        m_addResCnt = 0 ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        this.destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public final void destroy()
    {
        int i ;
        int len ;
        
        TelegramResourceValue[] roll = null ;
        
        len = m_max ;
        
        if( len > 0 ){
            
            roll = m_roll ;
            
            for( i = 0 ; i < len ; i ++ ){
                if( roll[ i ] != null ){
                    roll[ i ].clear() ;
                }
                roll[ i ] = null ;
            }
            
        }
        
        m_roll = null ;
        m_resType = null ;
        m_max = 0 ;
        m_now = 0 ;
        m_nowLen = 0 ;
        m_addResCnt = 0 ;
    }
    
    /**
     * 追加予約をONにセット.
     * <BR><BR>
     * 追加予約をONにセットします.
     */
    public final void addReservationByON()
    {
        synchronized( this ){
            m_addResCnt ++ ;
        }
    }
    
    /**
     * 追加予約をOFFにセット.
     * <BR><BR>
     * 追加予約をOFFにセットします.
     */
    public final void addReservationByOFF()
    {
        synchronized( this ){
            m_addResCnt = ( m_addResCnt <= 0 ) ? 0 : m_addResCnt - 1 ;
        }
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象情報を追加します.<BR>
     * この処理は最後のロールに情報を追加します.
     * <BR>
     * @param value 対象の情報を設定します.
     * @param addr 対象のIPアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @return boolean 設定の合否が返されます.<BR>
     *                 [true]が返された場合、正しく設定されました.<BR>
     *                 [false]が返された場合、空き情報が存在しないことから、
     *                 正しく設定できませんでした.
     * @exception InputException 入力例外.
     */
    public final boolean add( byte[] value,InetAddress addr,int port )
        throws InputException
    {
        return this.addTo( false,-1,value,addr,port,0 ) ;
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象情報を追加します.<BR>
     * この処理は最後のロールに情報を追加します.
     * <BR>
     * @param value 対象の情報を設定します.
     * @param addr 対象のIPアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @param resendCnt 再送カウントを設定します.
     * @return boolean 設定の合否が返されます.<BR>
     *                 [true]が返された場合、正しく設定されました.<BR>
     *                 [false]が返された場合、空き情報が存在しないことから、
     *                 正しく設定できませんでした.
     * @exception InputException 入力例外.
     */
    public final boolean add( byte[] value,InetAddress addr,int port,int resendCnt )
        throws InputException
    {
        return this.addTo( false,-1,value,addr,port,resendCnt ) ;
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象情報を追加します.<BR>
     * この処理は最後のロールに情報を追加します.
     * <BR>
     * @param id 対象のID情報を設定します.
     * @param value 対象の情報を設定します.
     * @param addr 対象のIPアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @return boolean 設定の合否が返されます.<BR>
     *                 [true]が返された場合、正しく設定されました.<BR>
     *                 [false]が返された場合、空き情報が存在しないことから、
     *                 正しく設定できませんでした.
     * @exception InputException 入力例外.
     */
    public final boolean add( int id,byte[] value,InetAddress addr,int port )
        throws InputException
    {
        return this.addTo( false,id,value,addr,port,0 ) ;
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象情報を追加します.<BR>
     * この処理は最後のロールに情報を追加します.
     * <BR>
     * @param id 対象のID情報を設定します.
     * @param value 対象の情報を設定します.
     * @param addr 対象のIPアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @param resendCnt 再送カウントを設定します.
     * @return boolean 設定の合否が返されます.<BR>
     *                 [true]が返された場合、正しく設定されました.<BR>
     *                 [false]が返された場合、空き情報が存在しないことから、
     *                 正しく設定できませんでした.
     * @exception InputException 入力例外.
     */
    public final boolean add( int id,byte[] value,InetAddress addr,int port,int resendCnt )
        throws InputException
    {
        return this.addTo( false,id,value,addr,port,resendCnt ) ;
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象情報を追加します.<BR>
     * この処理は最後のロールに情報を追加します.<BR>
     * またこのメソッドはキャッシュモードが、キャッシュ再作成モードでないと
     * 処理できません.
     * <BR>
     * @param value 対象の情報を設定します.
     * @param addr 対象のIPアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @return boolean 設定の合否が返されます.<BR>
     *                 [true]が返された場合、正しく設定されました.<BR>
     *                 [false]が返された場合、空き情報が存在しないことから、
     *                 正しく設定できませんでした.
     * @exception InputException 入力例外.
     */
    public final boolean add( BinResource value,InetAddress addr,int port )
        throws InputException
    {
        return this.addTo( false,-1,value,addr,port,0 ) ;
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象情報を追加します.<BR>
     * この処理は最後のロールに情報を追加します.<BR>
     * またこのメソッドはキャッシュモードが、キャッシュ再作成モードでないと
     * 処理できません.
     * <BR>
     * @param value 対象の情報を設定します.
     * @param addr 対象のIPアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @param resendCnt 再送カウントを設定します.
     * @return boolean 設定の合否が返されます.<BR>
     *                 [true]が返された場合、正しく設定されました.<BR>
     *                 [false]が返された場合、空き情報が存在しないことから、
     *                 正しく設定できませんでした.
     * @exception InputException 入力例外.
     */
    public final boolean add( BinResource value,InetAddress addr,int port,int resendCnt )
        throws InputException
    {
        return this.addTo( false,-1,value,addr,port,resendCnt ) ;
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象情報を追加します.<BR>
     * この処理は最後のロールに情報を追加します.<BR>
     * またこのメソッドはキャッシュモードが、キャッシュ再作成モードでないと
     * 処理できません.
     * <BR>
     * @param id 対象のID情報を設定します.
     * @param value 対象の情報を設定します.
     * @param addr 対象のIPアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @return boolean 設定の合否が返されます.<BR>
     *                 [true]が返された場合、正しく設定されました.<BR>
     *                 [false]が返された場合、空き情報が存在しないことから、
     *                 正しく設定できませんでした.
     * @exception InputException 入力例外.
     */
    public final boolean add( int id,BinResource value,InetAddress addr,int port )
        throws InputException
    {
        return this.addTo( false,id,value,addr,port,0 ) ;
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象情報を追加します.<BR>
     * この処理は最後のロールに情報を追加します.<BR>
     * またこのメソッドはキャッシュモードが、キャッシュ再作成モードでないと
     * 処理できません.
     * <BR>
     * @param id 対象のID情報を設定します.
     * @param value 対象の情報を設定します.
     * @param addr 対象のIPアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @param resendCnt 再送カウントを設定します.
     * @return boolean 設定の合否が返されます.<BR>
     *                 [true]が返された場合、正しく設定されました.<BR>
     *                 [false]が返された場合、空き情報が存在しないことから、
     *                 正しく設定できませんでした.
     * @exception InputException 入力例外.
     */
    public final boolean add( int id,BinResource value,InetAddress addr,int port,int resendCnt )
        throws InputException
    {
        return this.addTo( false,id,value,addr,port,resendCnt ) ;
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象情報を追加します.<BR>
     * この処理は最初のロールに情報を追加します.
     * <BR>
     * @param value 対象の情報を設定します.
     * @param addr 対象のIPアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @return boolean 設定の合否が返されます.<BR>
     *                 [true]が返された場合、正しく設定されました.<BR>
     *                 [false]が返された場合、空き情報が存在しないことから、
     *                 正しく設定できませんでした.
     * @exception InputException 入力例外.
     */
    public final boolean addHead( byte[] value,InetAddress addr,int port )
        throws InputException
    {
        return this.addTo( true,-1,value,addr,port,0 ) ;
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象情報を追加します.<BR>
     * この処理は最初のロールに情報を追加します.
     * <BR>
     * @param value 対象の情報を設定します.
     * @param addr 対象のIPアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @param resendCnt 再送カウントを設定します.
     * @return boolean 設定の合否が返されます.<BR>
     *                 [true]が返された場合、正しく設定されました.<BR>
     *                 [false]が返された場合、空き情報が存在しないことから、
     *                 正しく設定できませんでした.
     * @exception InputException 入力例外.
     */
    public final boolean addHead( byte[] value,InetAddress addr,int port,int resendCnt )
        throws InputException
    {
        return this.addTo( true,-1,value,addr,port,resendCnt ) ;
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象情報を追加します.<BR>
     * この処理は最初のロールに情報を追加します.
     * <BR>
     * @param id 対象のID情報を設定します.
     * @param value 対象の情報を設定します.
     * @param addr 対象のIPアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @return boolean 設定の合否が返されます.<BR>
     *                 [true]が返された場合、正しく設定されました.<BR>
     *                 [false]が返された場合、空き情報が存在しないことから、
     *                 正しく設定できませんでした.
     * @exception InputException 入力例外.
     */
    public final boolean addHead( int id,byte[] value,InetAddress addr,int port )
        throws InputException
    {
        return this.addTo( true,id,value,addr,port,0 ) ;
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象情報を追加します.<BR>
     * この処理は最初のロールに情報を追加します.
     * <BR>
     * @param id 対象のID情報を設定します.
     * @param value 対象の情報を設定します.
     * @param addr 対象のIPアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @param resendCnt 再送カウントを設定します.
     * @return boolean 設定の合否が返されます.<BR>
     *                 [true]が返された場合、正しく設定されました.<BR>
     *                 [false]が返された場合、空き情報が存在しないことから、
     *                 正しく設定できませんでした.
     * @exception InputException 入力例外.
     */
    public final boolean addHead( int id,byte[] value,InetAddress addr,int port,int resendCnt )
        throws InputException
    {
        return this.addTo( true,id,value,addr,port,resendCnt ) ;
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象情報を追加します.<BR>
     * この処理は最初のロールに情報を追加します.<BR>
     * またこのメソッドはキャッシュモードが、キャッシュ再作成モードでないと
     * 処理できません.
     * <BR>
     * @param value 対象の情報を設定します.
     * @param addr 対象のIPアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @return boolean 設定の合否が返されます.<BR>
     *                 [true]が返された場合、正しく設定されました.<BR>
     *                 [false]が返された場合、空き情報が存在しないことから、
     *                 正しく設定できませんでした.
     * @exception InputException 入力例外.
     */
    public final boolean addHead( BinResource value,InetAddress addr,int port )
        throws InputException
    {
        return this.addTo( true,-1,value,addr,port,0 ) ;
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象情報を追加します.<BR>
     * この処理は最初のロールに情報を追加します.<BR>
     * またこのメソッドはキャッシュモードが、キャッシュ再作成モードでないと
     * 処理できません.
     * <BR>
     * @param value 対象の情報を設定します.
     * @param addr 対象のIPアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @param resendCnt 再送カウントを設定します.
     * @return boolean 設定の合否が返されます.<BR>
     *                 [true]が返された場合、正しく設定されました.<BR>
     *                 [false]が返された場合、空き情報が存在しないことから、
     *                 正しく設定できませんでした.
     * @exception InputException 入力例外.
     */
    public final boolean addHead( BinResource value,InetAddress addr,int port,int resendCnt )
        throws InputException
    {
        return this.addTo( true,-1,value,addr,port,resendCnt ) ;
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象情報を追加します.<BR>
     * この処理は最初のロールに情報を追加します.<BR>
     * またこのメソッドはキャッシュモードが、キャッシュ再作成モードでないと
     * 処理できません.
     * <BR>
     * @param id 対象のID情報を設定します.
     * @param value 対象の情報を設定します.
     * @param addr 対象のIPアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @return boolean 設定の合否が返されます.<BR>
     *                 [true]が返された場合、正しく設定されました.<BR>
     *                 [false]が返された場合、空き情報が存在しないことから、
     *                 正しく設定できませんでした.
     * @exception InputException 入力例外.
     */
    public final boolean addHead( int id,BinResource value,InetAddress addr,int port )
        throws InputException
    {
        return this.addTo( true,id,value,addr,port,0 ) ;
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象情報を追加します.<BR>
     * この処理は最初のロールに情報を追加します.<BR>
     * またこのメソッドはキャッシュモードが、キャッシュ再作成モードでないと
     * 処理できません.
     * <BR>
     * @param id 対象のID情報を設定します.
     * @param value 対象の情報を設定します.
     * @param addr 対象のIPアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @param resendCnt 再送カウントを設定します.
     * @return boolean 設定の合否が返されます.<BR>
     *                 [true]が返された場合、正しく設定されました.<BR>
     *                 [false]が返された場合、空き情報が存在しないことから、
     *                 正しく設定できませんでした.
     * @exception InputException 入力例外.
     */
    public final boolean addHead( int id,BinResource value,InetAddress addr,int port,int resendCnt )
        throws InputException
    {
        return this.addTo( true,id,value,addr,port,resendCnt ) ;
    }
    
    /**
     * 情報取得.
     * <BR><BR>
     * 対象の情報を取得します.<BR>
     * この処理はキャッシュモードによって、受け取られるオブジェクト型が
     * 変わります.<BR>
     * キャッシュ再作成モード(true)の場合、受け取り先のメソッド[TelegramValue.getBinary()]
     * で受け取る戻り値のオブジェクト型は[BinResource]です.<BR>
     * キャッシュ非再作成モード(false)の場合は、受け取る戻り値は[byte[]]です.
     * <BR>
     * @param out 格納先の電文要素オブジェクトを設定します.
     * @return boolean 取得結果が返されます.<BR>
     *                 [true]が返された場合取得されました.<BR>
     *                 [false]が返された場合取得されませんでした.
     */
    public final boolean get( TelegramValue out )
    {
        int pnt ;
        boolean ret ;
        
        TelegramResourceValue[] roll = null ;
        
        ret = false ;
        
        synchronized( this ){
            
            if( m_nowLen <= 0 || out == null ){
                ret = false ;
            }
            else{
                
                roll = m_roll ;
                
                if( ( pnt = m_now - m_nowLen ) < 0 ){
                    pnt = ( m_max + pnt ) ;
                }
                
                // 受け取り先オブジェクトをクリア.
                out.clear() ;
                
                // キャッシュが再作成モードの場合.
                if( m_mode == true ){
                    out.setBinary( roll[ pnt ].RES ) ;
                    roll[ pnt ].RES = null ;
                }
                // キャッシュが非再作成モードの場合.
                else{
                    out.setBinary( roll[ pnt ].RES.getBinary() ) ;
                    roll[ pnt ].RES.reset() ;
                }
                
                out.setAddress( roll[ pnt ].ADDR ) ;
                out.setPort( roll[ pnt ].PORT ) ;
                out.setID( roll[ pnt ].ID ) ;
                out.setResendCount( roll[ pnt ].RESEND_CNT ) ;
                out.setUse() ;
                
                roll[ pnt ].ADDR = null ;
                roll[ pnt ].PORT = -1 ;
                roll[ pnt ].ID = -1 ;
                
                m_nowLen -- ;
                
                ret = true ;
                
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * 対象の管理サイズを取得.
     * <BR><BR>
     * 対象の管理サイズが返されます.
     * <BR>
     * @return int 管理サイズが返されます.
     */
    public final int getMax()
    {
        int ret ;
        
        synchronized( this ){
            ret = m_max ;
        }
        
        return ret ;
    }
    
    /**
     * 現在の格納数を取得.
     * <BR><BR>
     * 現在の格納数を取得します.
     * <BR>
     * @return int 現在の格納数が返されます.
     */
    public final int getSize()
    {
        int ret ;
        
        synchronized( this ){
            ret = m_nowLen ;
        }
        
        return ret ;
    }
    
    /**
     * キャッシュモードを取得.
     * <BR><BR>
     * キャッシュモードを取得します.
     * <BR>
     * @return boolean キャッシュモードを取得します.<BR>
     *                 [true]が返された場合はキャッシュ再作成モードです.<BR>
     *                 [false]が返された場合はキャッシュ非再作成モードです.
     */
    public final boolean isMode()
    {
        boolean ret ;
        
        synchronized( this ){
            ret = m_mode ;
        }
        
        return ret ;
    }
    
    /**
     * データが追加できるかチェック.
     * <BR><BR>
     * データが追加できるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、設定可能です.<BR>
     *                 [false]が返された場合、ロールは満杯のため設定はできません.
     */
    public final boolean isAdd()
    {
        boolean ret ;
        
        synchronized( this ){
            if( ( m_nowLen + m_addResCnt ) >= m_max ){
                ret = false ;
            }
            ret = true ;
        }
        
        return ret ;
    }
    
    /**
     * 追加予約が行われているかチェック.
     * <BR><BR>
     * 追加予約が行われているかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                [true]が返された場合、追加予約されています.<BR>
     *                [false]が返された場合追加予約はされていません.
     */
    public final boolean isAddReservation()
    {
        boolean ret ;
        
        synchronized( this ){
            ret = ( m_addResCnt > 0 ) ? true : false ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * 条件を追加.
     */
    private final boolean addTo( boolean hlMode,int id,byte[] value,InetAddress addr,int port,int resendCnt )
        throws InputException
    {
        int next ;
        boolean ret ;
        
        TelegramResourceValue[] roll = null ;
        
        if( value == null || value.length <= 0 || addr == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        ret = false ;
        
        synchronized( this ){
            
            if( ( m_nowLen + m_addResCnt ) >= m_max ){
                ret = false ;
            }
            else{
                
                // 追加ポイントを取得.
                next = this.getAddPoint( hlMode ) ;
                
                // ロールオブジェクトを取得.
                roll = m_roll ;
                
                // キャッシュ再作成モードの場合.
                if( m_mode == true ){
                    if( roll[ next ].RES != null ){
                        roll[ next ].RES.clear() ;
                        roll[ next ].RES = null ;
                    }
                    roll[ next ].RES = Resource.createBinResource( m_resType,value ) ;
                }
                // キャッシュ非再作成モードの場合.
                else{
                    
                    // 前回のデータが存在しないか、前回のデータ長と
                    // 今回設定するデータ長*係数より大きい場合.
                    if(
                        roll[ next ].RES == null ||
                        roll[ next ].RES.getAllSize() > Resource.getRenewRollValueCode( value.length )
                    )
                    {
                        // 前回のデータが存在する場合.
                        if( roll[ next ].RES != null ){
                            roll[ next ].RES.clear() ;
                            roll[ next ].RES = null ;
                        }
                        // 新たに生成.
                        roll[ next ].RES= Resource.createBinResource( m_resType,value ) ;
                    }
                    // 再利用可能な場合.
                    else{
                        roll[ next ].RES.reset() ;
                        roll[ next ].RES.setBinary( 0,value ) ;
                    }
                    
                }
                
                // 共通追加処理.
                roll[ next ].ADDR = addr ;
                roll[ next ].PORT = port ;
                roll[ next ].ID = id ;
                roll[ next ].RESEND_CNT = ( resendCnt <= 0 ) ?
                    0 : resendCnt ;
                
                ret = true ;
            }
            
        }
        
        return ret ;
    }
    
    /**
     * 条件を追加.
     */
    private final boolean addTo( boolean hlMode,int id,BinResource value,InetAddress addr,int port,int resendCnt )
        throws InputException
    {
        int next ;
        boolean ret ;
        TelegramResourceValue[] roll = null ;
        
        if( value == null || value.isUse() == false || value.size() <= 0 || addr == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        ret = false ;
        
        synchronized( this ){
            
            // モードがキャッシュ非作成モードの場合.
            // データ格納領域が存在しない場合.
            if( m_mode == false && ( m_nowLen + m_addResCnt ) >= m_max ){
                ret = false ;
            }
            else{
                
                // 追加ポイントを取得.
                next = this.getAddPoint( hlMode ) ;
                
                // ロールオブジェクトを取得.
                roll = m_roll ;
                
                // 情報をセット.
                if( roll[ next ].RES != null ){
                    roll[ next ].RES.clear() ;
                    roll[ next ].RES = null ;
                }
                roll[ next ].RES = value ;
                roll[ next ].ADDR = addr ;
                roll[ next ].PORT = port ;
                roll[ next ].ID = id ;
                roll[ next ].RESEND_CNT = ( resendCnt <= 0 ) ?
                    0 : resendCnt ;
                
                ret = true ;
            }
            
        }
        
        return ret ;
    }
    
    /**
     * 追加位置を取得.
     */
    private final int getAddPoint( boolean hlMode )
    {
        int ret ;
        
        ret = m_now ;
        
        // 先頭に追加する場合.
        if( hlMode == true ){
            
            // データが存在する場合.
            if( m_nowLen > 0 ){
                if( ( ret = m_now - m_nowLen ) < 0 ){
                    ret = ( m_max + ret ) ;
                }
                ret -- ;
                if( ret < 0 ){
                    ret = m_max - 1 ;
                }
            }
            // データが存在しない場合.
            else{
                if( ret >= m_max ){
                    ret = 0 ;
                }
                m_now = ret + 1 ;
            }
            
            m_nowLen ++ ;
            
        }
        // 最後に追加する場合.
        else{
            
            if( ret >= m_max ){
                ret = 0 ;
            }
            
            m_nowLen ++ ;
            m_now = ret + 1 ;
            
        }
        
        return ret ;
    }
}

/**
 * 要素情報.
 */
class TelegramResourceValue
{
    BinResource RES = null ;
    InetAddress ADDR = null ;
    int PORT = -1 ;
    int ID = -1 ;
    int RESEND_CNT = -1 ;
    
    protected final void finalize() throws Exception
    {
        this.clear() ;
    }
    
    public final void clear()
    {
        if( RES != null ){
            RES.clear() ;
        }
        RES = null ;
        ADDR = null ;
        PORT = -1 ;
        ID = -1 ;
        RESEND_CNT = -1 ;
    }
}

