/*
 * @(#)NowAddressCore.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.net;

import java.net.InetAddress;

import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.exception.UnsupportException;
import com.JRcServer.commons.util.CharTable;

/**
 * 現在のIPアドレス情報取得.
 * <BR><BR>
 * 現在のIPアドレス情報を取得します.
 *  
 * @version 1.0.0 2004/09/03
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
public class NowAddress
{
    
    /**
     * IPアドレス取得スレッド.
     */
    private final NowAddressThread m_thread = new NowAddressThread() ;
    
    /**
     * シングルトン.
     */
    private static final NowAddress SNGL = new NowAddress() ;
    
    /**
     * 生成フラグ.
     */
    private volatile boolean m_createFlg = false ;
    
    /**
     * コンストラクタ.
     */
    private NowAddress()
    {
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            m_thread.clear() ;
            m_createFlg = false ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクト情報を取得.
     * <BR><BR>
     * オブジェクト情報を取得します.
     * <BR>
     * @return NowAddress 自オブジェクト情報が返されます.
     */
    public static final NowAddress getInstance()
    {
        SNGL.create() ;
        return SNGL ;
    }
    
    /**
     * 即時実行.
     * <BR><BR>
     * コマンド処理をインターバルに関係なく即時実行します.
     */
    public final void flush()
    {
        m_thread.flush() ;
    }
    
    /**
     * インターバル時間の設定.
     * <BR><BR>
     * チェックタイミングのインターバル時間を設定します.
     * <BR>
     * @param interval 取得間隔のタイミング値を設定します.
     *                 設定可能な最小値は[5000]です.<BR>
     *                 設定可能な最大値は[300000]です.<BR>
     *                 また、この設定単位はミリ秒です.
     * @exception InputException 入力例外.
     */
    public final void setInterval( int interval )
        throws InputException
    {
        try{
            m_thread.setInterval( interval ) ;
        }catch( InputException in ){
            throw in ;
        }
    }
    
    /**
     * 設定インターバル時間を取得.
     * <BR><BR>
     * 設定されているインターバル時間を取得します.
     * <BR>
     * @return int 設定されているインターバル時間が返されます.<BR>
     *             また、設定されている単位はミリ秒です.
     */
    public final int getInterval()
    {
        return m_thread.getInterval() ;
    }
    
    /**
     * ローカルIPアドレス情報を取得.
     * <BR><BR>
     * ローカルIPアドレス情報を取得します.
     * <BR>
     * @return String 対象ローカルIPアドレス情報が返されます.
     */
    public final String getLocalAddress()
    {
        return( String )m_thread.getNow( NowAddressCore.CODE_NOW_IPADDR ) ;
    }
    
    /**
     * ローカルホスト情報を取得.
     * <BR><BR>
     * ローカルホスト情報を取得します.
     * <BR>
     * @return String 対象ローカルホスト情報が返されます.
     */
    public final String getLocalHost()
    {
        return ( String )m_thread.getNow( NowAddressCore.CODE_NOW_DOMAIN ) ;
    }
    
    /**
     * ローカルIPアドレス群を取得.
     * <BR><BR>
     * ローカルIPアドレス群を取得します.
     * <BR>
     * @return String[] 対象ローカルIPアドレス群が返されます.
     */
    public final String[] getAllLocalIpAddress()
    {
        return ( String[] )m_thread.getNow( NowAddressCore.CODE_ALL_IPADDR ) ;
    }
    
    /**
     * 指定IPアドレスに対するホスト名を取得.
     * <BR><BR>
     * 指定されたIPアドレスに対するホスト名を取得します.
     * <BR>
     * @param addr 取得対象のIPアドレスを設定します.
     * @return String 対象のホスト名が返されます.
     */
    public final String getHostName( String addr )
    {
        String ret = null ;
        CharTable tbl = null ;
        
        if( addr == null ){
            return null ;
        }
        
        tbl = ( CharTable )m_thread.getNow( NowAddressCore.CODE_ALL_DOMAIN ) ;
        if( tbl == null ){
            return null ;
        }
        
        ret = ( String )tbl.get( addr ) ;
        tbl = null ;
        
        return ret ;
    }
    
    /**
     * ローカルIPアドレスに対する媒体名を取得.
     * <BR><BR>
     * ローカルIPアドレスに対する媒体名を取得します.
     * <BR>
     * @return String 対象の媒体名が返されます.
     * @exception UnsupportException サポート外例外.
     */
    public final String getLocalAdapter()
        throws UnsupportException
    {
        
        if( NetConfig.getInstance().useObject() == false ){
            throw new UnsupportException(
                "媒体名取得はこのOSではサポートされていません"
            ) ;
        }
        
        return NetConfig.getInstance().getAddressByAdapter(
            this.getLocalAddress()
        ) ;
    }
    
    /**
     * ローカルIPアドレス群に対する媒体名を取得.
     * <BR><BR>
     * ローカルIPアドレス群に対する媒体名を取得します.
     * <BR>
     * @return String[] 対象の媒体名が返されます.
     * @exception UnsupportException サポート外例外.
     */
    public final String[] getAllLocalAdapter()
        throws UnsupportException
    {
        int i ;
        int len ;
        
        String[] addrs = null ;
        String[] ret = null ;
        
        if( NetConfig.getInstance().useObject() == false ){
            throw new UnsupportException(
                "媒体名取得はこのOSではサポートされていません"
            ) ;
        }
        
        addrs = this.getAllLocalIpAddress() ;
        
        if( addrs != null ){
            
            len = addrs.length ;
            ret = new String[ len ] ;
            
            for( i = 0 ; i < len ; i ++ ){
                
                ret[ i ] = NetConfig.getInstance().getAddressByAdapter(
                    addrs[ i ]
                ) ;
                
                addrs[ i ] = null ;
                
            }
            
            addrs = null ;
            
        }
        else{
            
            ret = null ;
            
        }
        
        return ret ;
    }
    
    /**
     * 指定アドレス情報が存在するかチェック.
     * <BR><BR>
     * 指定アドレス情報が存在するかチェックします.
     * <BR>
     * @param addr チェック対象のアドレスを設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、利用可能です.
     *                 [false]が返された場合、利用不可能です.
     */
    public final boolean isUse( InetAddress addr )
    {
        boolean ret ;
        
        try{
            ret = this.isUse( addr.getHostAddress() ) ;
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 指定アドレス情報が存在するかチェック.
     * <BR><BR>
     * 指定アドレス情報が存在するかチェックします.
     * <BR>
     * @param addr チェック対象のアドレスを設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、利用可能です.
     *                 [false]が返された場合、利用不可能です.
     */
    public final boolean isUse( String addr )
    {
        int i ;
        int len ;
        boolean ret ;
        
        String ipAddr = null ;
        String[] code = null ;
        
        ret = false ;
        
        try{
            
            code = this.getAllLocalIpAddress() ;
            len = code.length ;
            
            for( i = 0 ; i < len ; i ++ ){
                if( addr.equalsIgnoreCase( code[ i ] ) == true ){
                    ret = true ;
                    break ;
                }
            }
            
        }catch( Exception t ){
            ret = false ;
        }finally{
            ipAddr = null ;
            code = null ;
        }
        
        return ret ;
    }
    
    
    /**
     * スレッド条件の生成.
     * <BR><BR>
     * スレッド条件を生成します.
     */
    private synchronized final void create()
    {
        
        if( m_createFlg == false ){
            m_thread.create() ;
            m_createFlg = true ;
        }
        
    }
    
}

