/*
 * @(#)NetAdapterByBsd.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.net;

import com.JRcServer.commons.exception.AccessException;
import com.JRcServer.commons.util.UtilCom;

/**
 * ネットワークアドレス取得BSD系UNIX版.
 * <BR><BR>
 * BSD版のネットワークアドレス取得処理です.
 *  
 * @version 1.0.0 2004/10/13
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
class NetAdapterByBsd extends NetAdapterBase
{
    
    /**
     * IPアドレス取得コマンド.
     */
    public static final String IPCONFIG_COMMAND = "ifconfig -a";
    
    /**
     * 媒体開始文字.
     */
    private static final String ADAPTER_NAME_1 = "flags" ;
    private static final String ADAPTER_NAME_2 = "=" ;
    
    /**
     * IPアドレス区切り値.
     */
    private static final char DIV_IP_CODE = '.' ;
    
    /**
     * macアドレス区切り値.
     */
    private static final char DIV_MAC_CODE = ':' ;
    
    /**
     * IPアドレス.
     */
    private static final String IP_ADDR = "inet" ;
    
    /**
     * サブネットマスク.
     */
    private static final String SUB_NET = "netmask" ;
    
    /**
     * macアドレス.
     */
    private static final String MAC_ADDR = "ether" ;
    
    /**
     * サブネットマスクヘッダ.
     */
    private static final String SUB_NET_HEADER = "0x" ;
    
    
    
    /**
     * コンストラクタ.
     */
    public NetAdapterByBsd()
    {
    }
    
    /**
     * ネットワーク情報取得処理.
     * <BR><BR>
     * 対象のネットワーク情報を取得します.
     * <BR>
     * @exception AccessException アクセス例外.
     */
    public final void flush() throws AccessException
    {
        int i,j ;
        int len ;
        int lenJ ;
        int pnt ;
        int off ;
        int st,et ;
        int localCheck ;
        
        char code ;
        
        String[] cmdOut = null ;
        String cmd = null ;
        StringBuffer buf = null ;
        String name = null ;
        String value = null ;
        
        try{
            
            this.clear() ;
            cmdOut = super.runExec( NetAdapterByBsd.IPCONFIG_COMMAND ) ;
            
            len = cmdOut.length ;
            for( i = 0,localCheck = 0 ; i < len ; i ++ ){
                
                cmd = cmdOut[ i ].toLowerCase() ;
                cmdOut[ i ] = null ;
                
                // adapter名を調べる.
                if(
                    ( off = cmd.indexOf( NetAdapterByBsd.ADAPTER_NAME_1 ) ) != -1 &&
                    cmd.indexOf( NetAdapterByBsd.ADAPTER_NAME_2,off ) != -1
                )
                {
                    pnt = off ;
                    
                    name = UtilCom.trimPlus(
                        cmd.substring( 0,cmd.indexOf( NetAdapterByBsd.DIV_MAC_CODE ) )
                    ) ;
                    
                }
                
                // IPアドレスの場合.
                if( ( pnt = cmd.indexOf( NetAdapterByBsd.IP_ADDR ) ) != -1 ){
                    
                    pnt = pnt + NetAdapterByBsd.IP_ADDR.length() ;
                    lenJ = cmd.length() ;
                    
                    for( j = pnt,st = -1,et = -1 ; j < lenJ ; j ++ ){
                        
                        code = cmd.charAt( j ) ;
                        
                        if( st == -1 ){
                            if( code >= '0' && code <= '9' ){
                                st = j ;
                            }
                        }
                        else if(
                            code == NetAdapterByBsd.DIV_IP_CODE ||
                            ( code >= '0' && code <= '9' )
                        )
                        {
                            et = j ;
                        }
                        else{
                            break ;
                        }
                        
                    }
                    
                    value = UtilCom.trimPlus( cmd.substring( st,et+1 ) ) ;
                    
                    // 対象IPアドレスがローカルアドレスを示す場合.
                    if( value.equals( NetDef.LOCAL_HOST ) == true ){
                        localCheck ++ ;
                    }
                    
                    super.addCategory( name,NetAdapter.CATEGORY_IPADDR,value ) ;
                }
                
                // サブネットマスクの場合.
                if( ( off = cmd.indexOf( NetAdapterByBsd.SUB_NET ) ) != -1 ){
                    
                    if( ( pnt = cmd.indexOf( NetAdapterByBsd.SUB_NET_HEADER,off ) ) != -1 ){
                        pnt = pnt + NetAdapterByBsd.SUB_NET_HEADER.length() ;
                    }
                    else{
                        pnt = off + NetAdapterByBsd.SUB_NET.length() ;
                    }
                    
                    lenJ = cmd.length() ;
                    
                    for( j = pnt,st = -1,et = -1 ; j < lenJ ; j ++ ){
                        
                        code = cmd.charAt( j ) ;
                        
                        if( st == -1 ){
                            if(
                                ( code >= '0' && code <= '9' ) ||
                                ( code >= 'a' && code <= 'f' )
                            )
                            {
                                st = j ;
                            }
                        }
                        else if(
                            ( code >= '0' && code <= '9' ) ||
                            ( code >= 'a' && code <= 'f' )
                        )
                        {
                            et = j ;
                        }
                        else{
                            break ;
                        }
                        
                    }
                    
                    value = UtilCom.trimPlus(
                        UtilCom.changeString(
                            cmd.substring( st,et+1 ),
                            NetAdapterByBsd.SUB_NET,
                            " "
                        )
                    ) ;
                    
                    super.addCategory(
                        name,
                        NetAdapter.CATEGORY_SUBNET,
                        NetAdapterByBsd.convertStringByIpAddress( value )
                    ) ;
                }
                
                // MACアドレスの場合.
                if( ( pnt = cmd.indexOf( NetAdapterByBsd.MAC_ADDR ) ) != -1 ){
                    pnt = pnt + NetAdapterByBsd.MAC_ADDR.length() ;
                    lenJ = cmd.length() ;
                    
                    for( j = pnt,st = -1,et = -1 ; j < lenJ ; j ++ ){
                        
                        code = cmd.charAt( j ) ;
                        
                        if( st == -1 ){
                            if(
                                ( code >= '0' && code <= '9' ) ||
                                ( code >= 'a' && code <= 'f' )
                            )
                            {
                                st = j ;
                            }
                        }
                        else if(
                            code == NetAdapterByBsd.DIV_MAC_CODE ||
                            ( code >= '0' && code <= '9' ) ||
                            ( code >= 'a' && code <= 'f' )
                        )
                        {
                            et = j ;
                        }
                        else{
                            break ;
                        }
                        
                    }
                    
                    value = UtilCom.trimPlus( cmd.substring( st,et+1 ) ).toUpperCase() ;
                    super.addCategory( name,NetAdapter.CATEGORY_MACADDR,value ) ;
                }
                
            }
            
            // ローカルアドレスが存在しない場合.
            if( localCheck == 0 ){
                // アドレスを追加.
                super.addCategory(
                    NetAdapterBase.DEFAULT_ETHER_NAME,
                    NetAdapter.CATEGORY_IPADDR,
                    NetDef.LOCAL_HOST
                ) ;
                // ホスト名を追加.
                super.addCategory(
                    NetAdapterBase.DEFAULT_ETHER_NAME,
                    NetAdapter.CATEGORY_SUBNET,
                    NetAdapterBase.LO_HOST_SUB
                ) ;
            }
            
        }catch( AccessException ac ){
            this.clear() ;
            throw ac ;
        }catch( Exception t ){
            this.clear() ;
        }finally{
            cmdOut = null ;
            cmd = null ;
            buf = null ;
            name = null ;
            value = null ;
        }
        
    }
    
    /**
     * サブネットマスクをIPアドレス形式に変換.
     */
    private static final String convertStringByIpAddress( String code )
    {
        int i ;
        int len ;
        
        byte[] bin = null ;
        StringBuffer buf = null ;
        String ret = null ;
        
        buf = new StringBuffer() ;
        
        try{
            bin = UtilCom.convertString16ToBinary( code.toUpperCase() ) ;
            
            len = bin.length ;
            for( i = 0 ; i < len ; i ++ ){
                if( i != 0 ){
                    buf.append( NetAdapterByBsd.DIV_IP_CODE ) ;
                }
                buf.append( ( int )( bin[ i ] & 0x000000ff ) ) ;
            }
            
            ret = buf.toString() ;
            
        }catch( Exception t ){
            ret = "0.0.0.0" ;
        }finally{
            bin = null ;
            buf = null ;
        }
        
        return ret ;
        
    }
    
}

