/*
 * @(#)MultiCastV4.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.net;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.DatagramPacket;
import java.net.InetAddress;
import java.net.MulticastSocket;
import java.net.SocketException;
import java.net.UnknownHostException;

import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.thread.Synchronized;

/**
 * マルチキャストIPV4版.
 * <BR><BR>
 * マルチキャストUDPプロトコルIPV4版のオブジェクトです.
 *
 * @version 1.00, 2005/02/07
 * @author  Masahito Suzuki
 * @since   JRcCommons 1.00
 */
public class MultiCastV4 implements BaseMultiCast
{
    
    /**
     * マルチキャストソケット.
     */
    private MulticastSocket m_socket = null ;
    
    /**
     * 格納ID.
     */
    private int m_id = 0 ;
    
    /**
     * オープンアドレス.
     */
    private InetAddress m_addr = null ;
    
    /**
     * オープンポート番号.
     */
    private int m_port = 0 ;
    
    /**
     * デフォルトTTL(TimeToLive).
     */
    private byte m_defTTL = 0 ;
    
    /**
     * 受信用バッファ.
     */
    private final byte[] m_buffer = new byte[ BaseMultiCast.BUFFER ] ;
    
    /**
     * 同期処理.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    /**
     * 受信用同期処理.
     */
    private final Synchronized m_recvSync = new Synchronized() ;
    
    /**
     * コンストラクタ.
     */
    public MultiCastV4()
    {
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.close() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オープン処理.
     * <BR><BR>
     * ポート番号をデフォルトポート番号で指定して、オープンします.
     * <BR>
     * @exception NotBindException バインド失敗.
     */
    public final void open() throws NotBindException
    {
        try{
            this.open(
                BaseMultiCast.DEF_PORT,
                BaseMultiCast.DEF_BUFFER,
                BaseMultiCast.DEF_TTL
            ) ;
        }catch( Exception e ){
            this.close() ;
        }
    }
    
    /**
     * オープン処理.
     * <BR><BR>
     * ポート番号をデフォルトポート番号で指定して、オープンします.
     * <BR>
     * @param buf 送受信バッファ長を設定します.
     * @exception InputException 入力例外.
     * @exception NotBindException バインド失敗.
     */
    public final void open( int buf )
        throws InputException,NotBindException
    {
        try{
            this.open(
                BaseMultiCast.DEF_PORT,
                buf,
                BaseMultiCast.DEF_TTL
            ) ;
        }catch( Exception e ){
            this.close() ;
        }
    }
    
    /**
     * オープン処理.
     * <BR><BR>
     * ポート番号を指定して、オープンします.
     * <BR>
     * @param port オープンポート番号を設定します.
     * @param buf 送受信バッファ長を設定します.
     * @exception InputException 入力例外.
     * @exception NotBindException バインド失敗.
     */
    public final void open( int port,int buf )
        throws InputException,NotBindException
    {
        try{
            this.open(
                port,
                buf,
                BaseMultiCast.DEF_TTL
            ) ;
        }catch( Exception e ){
            this.close() ;
        }
    }
    
    /**
     * オープン処理.
     * <BR><BR>
     * ポート番号を指定して、オープンします.
     * <BR>
     * @param port オープンポート番号を設定します.
     * @param buf 送受信バッファ長を設定します.
     * @param defTTL デフォルトのTTL(TimeToLive)を設定します.
     * @exception InputException 入力例外.
     * @exception NotBindException バインド失敗.
     */
    public final void open( int port,int buf,byte defTTL )
        throws InputException,NotBindException
    {
        if( port < 0 || port > 65535 || buf <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        defTTL = ( defTTL <= BaseMultiCast.DEF_TTL ) ?
            BaseMultiCast.DEF_TTL : defTTL ;
        
        try{
            
            this.close() ;
            m_sync.create() ;
            m_recvSync.create() ;
            
            synchronized( m_sync.get() ){
                
                m_port = port ;
                m_defTTL = defTTL ;
                m_socket = new MulticastSocket( port ) ;
                m_socket.setSendBufferSize(
                    ( int )( buf * NetDef.NETWORK_SEND_BUFFER )
                ) ;
                m_socket.setReceiveBufferSize(
                    ( int )( buf * NetDef.NETWORK_RECEIVE_BUFFER )
                ) ;
                m_socket.setTimeToLive( defTTL ) ;
            }
            
        }catch( NullPointerException nul ){
            this.close() ;
        }catch( SocketException so ){
            this.close() ;
            throw new NotBindException( so ) ;
        }catch( IOException io ){
            this.close() ;
            throw new NotBindException( io ) ;
        }
        
    }
    
    /**
     * クローズ処理.
     * <BR><BR>
     * クローズ処理を行います.
     */
    public final void close()
    {
        m_sync.clear() ;
        m_recvSync.clear() ;
        
        try{
            m_socket.leaveGroup( m_addr ) ;
        }catch( Exception e ){
        }
        try{
            m_socket.close() ;
        }catch( Exception e ){
        }
        
        m_socket = null ;
        m_addr = null ;
        m_id = 0 ;
        m_port = 0 ;
        m_defTTL = 0 ;
    }
    
    /**
     * 利用インターフェイスを設定.
     * <BR><BR>
     * マルチキャストを行う利用インターフェイスを設定します.<BR>
     * また、ここで言うインターフェイスは接続先のネットワークを意味する
     * ものであり、例えばLANが２つあった時の利用先を指定する場合に
     * 利用します.
     * <BR>
     * @param inf 対象のインターフェイス先を設定します.
     * @exception InputException 入力例外.
     * @exception NotInterfaceException インターフェイス非存在例外.
     */
    public final void setInterface( InetAddress inf )
        throws InputException,NotInterfaceException
    {
        if( inf == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            synchronized( m_sync.get() ){
                m_socket.setInterface( inf ) ;
            }
        }catch( NullPointerException ne ){
        }catch( SocketException se ){
            throw new NotInterfaceException( se ) ;
        }
    }
    
    /**
     * グループに参加.
     * <BR><BR>
     * 対象IDのグループに参加します.
     * <BR>
     * @param id グループ参加対象のIDを指定します.
     * @exception InputException 入力例外.
     */
    public final void joinGroup( int id )
        throws InputException
    {
        this.joinGroup( id,-1 ) ;
    }
    
    /**
     * グループに参加.
     * <BR><BR>
     * 対象IDのグループに参加します.
     * <BR>
     * @param id グループ参加対象のIDを指定します.
     * @param ttl 対象グループの有効期限を設定します.
     * @exception InputException 入力例外.
     */
    public final void joinGroup( int id,int ttl )
        throws InputException
    {
        if( id <= 0 || ( ttl != -1 && ttl <= 0 ) || ttl > 255 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            synchronized( m_sync.get() ){
                
                this.leaveGroup() ;
                
                m_id = id ;
                m_addr = MultiCastUtil.getIDByMulitCast_V4( id ) ;
                m_socket.joinGroup( m_addr ) ;
                
                if( ttl != -1 ){
                    m_socket.setTimeToLive( ttl ) ;
                }
                
            }
        }catch( NullPointerException nul ){
            this.leaveGroup() ;
        }catch( InputException in ){
            this.leaveGroup() ;
            throw in ;
        }catch( Exception e ){
            this.leaveGroup() ;
            throw new InputException( e ) ;
        }
        
    }
    
    /**
     * 現在参加しているグループから離脱.
     * <BR><BR>
     * 現在参加しているグループから離脱します.
     */
    public final void leaveGroup()
    {
        try{
            synchronized( m_sync.get() ){
                try{
                    m_socket.leaveGroup( m_addr ) ;
                }catch( Exception e ){
                }finally{
                    m_addr = null ;
                    m_id = 0 ;
                }
            }
        }catch( Exception e ){
        }
    }
    
    /**
     * データ送信.
     * <BR><BR>
     * 対象のデータを送信します.
     * <BR>
     * @param binary 送信対象のデータを設定します.
     * @exception InputException 入力例外.
     * @exception UndefineBindException 未バインド例外.
     */
    public final void send( byte[] binary )
        throws InputException,UndefineBindException
    {
        DatagramPacket pac = null ;
        
        if( binary == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            pac = new DatagramPacket(
                binary,binary.length,m_addr,m_port
            ) ;
            
            synchronized( m_sync.get() ){
                m_socket.send( pac ) ;
            }
            
        }catch( NullPointerException nul ){
            throw new UndefineBindException( nul ) ;
        }catch( UnknownHostException uk ){
            throw new InputException( uk ) ;
        }catch( IOException io ){
            throw new UndefineBindException( io ) ;
        }finally{
            pac = null ;
        }
        
    }
    
    /**
     * データ送信.
     * <BR><BR>
     * 対象のデータを送信します.
     * <BR>
     * @param binary 送信対象のデータを設定します.
     * @param ttl マルチキャストパケット有効期間を設定します.
     * @exception InputException 入力例外.
     * @exception UndefineBindException 未バインド例外.
     */
    public final void send( byte[] binary,byte ttl )
        throws InputException,UndefineBindException
    {
        DatagramPacket pac = null ;
        
        if( binary == null || ttl <= 0 || ttl > 255 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            pac = new DatagramPacket(
                binary,binary.length,m_addr,m_port
            ) ;
            
            synchronized( m_sync.get() ){
                m_socket.send( pac,ttl ) ;
            }
            
        }catch( NullPointerException nul ){
            throw new UndefineBindException( nul ) ;
        }catch( UnknownHostException uk ){
            throw new InputException( uk ) ;
        }catch( IOException io ){
            throw new UndefineBindException( io ) ;
        }finally{
            pac = null ;
        }
    }
    
    /**
     * データ送信.
     * <BR><BR>
     * 対象のデータを送信します.
     * <BR>
     * @param id 送信対象のグループIDを指定します.
     * @param port 送信対象のグループポート番号を指定します.
     * @param binary 送信対象のデータを設定します.
     * @exception InputException 入力例外.
     * @exception UndefineBindException 未バインド例外.
     */
    public final void send( int id,int port,byte[] binary )
        throws InputException,UndefineBindException
    {
        DatagramPacket pac = null ;
        
        if( id <= 0 || port < 0 || port > 65535 || binary == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            pac = new DatagramPacket(
                binary,binary.length,
                MultiCastUtil.getIDByMulitCast_V4( id ),port
            ) ;
            
            synchronized( m_sync.get() ){
                m_socket.send( pac ) ;
            }
            
        }catch( NullPointerException nul ){
            throw new UndefineBindException( nul ) ;
        }catch( UnknownHostException uk ){
            throw new InputException( uk ) ;
        }catch( IOException io ){
            throw new UndefineBindException( io ) ;
        }finally{
            pac = null ;
        }
        
    }
    
    /**
     * データ送信.
     * <BR><BR>
     * 対象のデータを送信します.
     * <BR>
     * @param id 送信対象のグループIDを指定します.
     * @param port 送信対象のグループポート番号を指定します.
     * @param binary 送信対象のデータを設定します.
     * @param ttl マルチキャストパケット有効期間を設定します.
     * @exception InputException 入力例外.
     * @exception UndefineBindException 未バインド例外.
     */
    public final void send( int id,int port,byte[] binary,byte ttl )
        throws InputException,UndefineBindException
    {
        DatagramPacket pac = null ;
        
        if(
            id <= 0 || port < 0 || port > 65535 || binary == null ||
            ttl <= 0 || ttl > 255
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            pac = new DatagramPacket(
                binary,binary.length,
                MultiCastUtil.getIDByMulitCast_V4( id ),port
            ) ;
            
            synchronized( m_sync.get() ){
                m_socket.send( pac,ttl ) ;
            }
            
        }catch( NullPointerException nul ){
            throw new UndefineBindException( nul ) ;
        }catch( UnknownHostException uk ){
            throw new InputException( uk ) ;
        }catch( IOException io ){
            throw new UndefineBindException( io ) ;
        }finally{
            pac = null ;
        }
        
    }
        
    /**
     * データ受信.
     * <BR><BR>
     * データを受信します.<BR>
     * データが存在しない場合[null]が返されます.
     * <BR>
     * @param addr 受信先のIPアドレスとポート番号が格納された
     *             内容が返されます.
     * @return byte[] 受信されたバイナリ情報が返されます.<BR>
     *                受信対象の情報が存在しない場合[null]が返されます.
     * @exception InputException 入力例外
     * @exception UndefineBindException バインド未定義例外.
     * @exception ConnectTimeoutException タイムアウト例外.
     */
    public final byte[] receive( ConnectAddress addr )
        throws InputException,UndefineBindException,ConnectTimeoutException
    {
        return this.receive( addr,0 ) ;
    }
    
    /**
     * データ受信.
     * <BR><BR>
     * データを受信します.<BR>
     * データが存在しない場合[null]が返されます.
     * <BR>
     * @param addr 受信先のIPアドレスとポート番号が格納された
     *             内容が返されます.
     * @param timeout 受信タイムアウト値を設定します.
     * @return byte[] 受信されたバイナリ情報が返されます.<BR>
     *                受信対象の情報が存在しない場合[null]が返されます.
     * @exception InputException 入力例外
     * @exception UndefineBindException バインド未定義例外.
     * @exception ConnectTimeoutException タイムアウト例外.
     */
    public final byte[] receive( ConnectAddress addr,int timeout )
        throws InputException,UndefineBindException,ConnectTimeoutException
    {
        int len ;
        
        byte[] ret = null ;
        DatagramPacket packet = null ;
        
        if( addr == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            synchronized( m_recvSync.get() ){
                
                packet = new DatagramPacket( m_buffer,BaseMultiCast.BUFFER ) ;
                m_socket.setSoTimeout( ( timeout < 0 ) ? 0 : timeout ) ;
                m_socket.receive( packet ) ;
                
                len = packet.getLength() ;
                ret = new byte[ len ] ;
                System.arraycopy( m_buffer,0,ret,0,len ) ;
                
                addr.create( packet.getAddress(),packet.getPort() ) ;
                
            }
            
        }catch( NullPointerException nul ){
            ret = null ;
        }catch( InterruptedIOException ii ){
            ret = null ;
            throw new ConnectTimeoutException( ii ) ;
        }catch( IOException io ){
            throw new UndefineBindException( io ) ;
        }finally{
            packet = null ;
        }
        
        return ret ;
    }
    
    /**
     * データ受信.
     * <BR><BR>
     * データを受信します.
     * <BR>
     * @param out 受信されたバイナリ情報が設定されます.
     * @param addr 受信先のIPアドレスとポート番号が格納された
     *             内容が返されます.
     * @return int 受信されたバイナリ情報長が返されます.
     * @exception InputException 入力例外
     * @exception UndefineBindException バインド未定義例外.
     * @exception ConnectTimeoutException タイムアウト例外.
     */
    public final int receive( byte[] out,ConnectAddress addr )
        throws InputException,UndefineBindException,ConnectTimeoutException
    {
        return this.receive( out,addr,0 ) ;
    }
    
    /**
     * データ受信.
     * <BR><BR>
     * データを受信します.
     * <BR>
     * @param out 受信されたバイナリ情報が設定されます.
     * @param addr 受信先のIPアドレスとポート番号が格納された
     *             内容が返されます.
     * @param timeout 受信タイムアウト値を設定します.
     * @return int 受信されたバイナリ情報長が返されます.
     * @exception InputException 入力例外
     * @exception UndefineBindException バインド未定義例外.
     * @exception ConnectTimeoutException タイムアウト例外.
     */
    public final int receive( byte[] out,ConnectAddress addr,int timeout )
        throws InputException,UndefineBindException,ConnectTimeoutException
    {
        int len ;
        int ret ;
        
        DatagramPacket packet = null ;
        
        if( out == null || out.length <= 0 || addr == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            synchronized( m_recvSync.get() ){
                
                packet = new DatagramPacket( m_buffer,BaseMultiCast.BUFFER ) ;
                m_socket.setSoTimeout( ( timeout < 0 ) ? 0 : timeout ) ;
                m_socket.receive( packet ) ;
                
                ret = packet.getLength() ;
                len = out.length ;
                len = ( len <= ret ) ? len : ret ;
                System.arraycopy( m_buffer,0,out,0,len ) ;
                
                addr.create( packet.getAddress(),packet.getPort() ) ;
                
            }
            
        }catch( NullPointerException nul ){
            ret = -1 ;
        }catch( InterruptedIOException ii ){
            ret = -1 ;
            throw new ConnectTimeoutException( ii ) ;
        }catch( IOException io ){
            throw new UndefineBindException( io ) ;
        }finally{
            packet = null ;
        }
        
        return ret ;
    }
    
    /**
     * 有効期限を設定.
     * <BR><BR>
     * 有効期限を設定します.
     * <BR>
     * @return ttl 現在参加しているグループに対する有効期限を設定します.
     * @exception InputException 入力例外.
     */
    public final void setTTL( int ttl ) throws InputException
    {
        if( ttl < 0 || ttl > 255 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            synchronized( m_sync.get() ){
                m_socket.setTimeToLive( ttl ) ;
            }
        }catch( Exception e ){
            throw new InputException( e ) ;
        }
    }
    
    /**
     * 利用インターフェイスを取得.
     * <BR><BR>
     * マルチキャストを行う利用インターフェイスを取得します.
     * <BR>
     * @return InetAddress 対象のインターフェイス先が返されます.
     */
    public final InetAddress getInterface()
    {
        InetAddress ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_socket.getInterface() ;
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * ローカルアドレスを取得.
     * <BR><BR>
     * 対象のローカルアドレスを取得します.
     * <BR>
     * @param addr 対象のローカルアドレスが返されます.
     */
    public final void getLocal( ConnectAddress addr )
    {
        try{
            if( addr != null ){
                synchronized( m_sync.get() ){
                    addr.create(
                        m_socket.getLocalAddress(),
                        m_socket.getLocalPort()
                    ) ;
                }
            }
        }catch( Exception t ){
            if( addr != null ){
                
                try{
                    addr.clear() ;
                    addr.create( NetDef.NOT_ADDR,NetDef.PORT_MIN ) ;
                }catch( Exception tt ){
                }
                
            }
        }
    }
    
    /**
     * ローカルアドレスを取得.
     * <BR><BR>
     * 対象のローカルアドレスを取得します.
     * <BR>
     * @return ConnectAddress 対象のローカルアドレスが返されます.
     */
    public final ConnectAddress getLocal()
    {
        ConnectAddress ret = new ConnectAddress() ;
        
        try{
            synchronized( m_sync.get() ){
                ret = new ConnectAddress(
                    m_socket.getLocalAddress(),
                    m_socket.getLocalPort()
                ) ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * ローカルアドレス情報を取得.
     * <BR><BR>
     * ローカルアドレス情報を取得します.
     * <BR>
     * @return InetAddress ローカルアドレス情報が返されます.
     */
    public final InetAddress getLocalAddress()
    {
        InetAddress ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_socket.getLocalAddress() ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * ローカルポート番号を取得.
     * <BR><BR>
     * ローカルポート番号を取得します.
     * <BR>
     * @return int ローカルポート番号が返されます.
     */
    public int getLocalPort()
    {
        int ret  ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_socket.getLocalPort() ;
            }
        }catch( Exception t ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 設定バッファ長を取得.
     * <BR><BR>
     * 設定されているバッファ長を取得します.
     * <BR>
     * @return int 設定バッファ長が返されます.
     */
    public final int getBuffer()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_socket.getSendBufferSize() ;
            }
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * グループIDを取得.
     * <BR><BR>
     * 参加しているグループIDが返されます.
     * <BR>
     * @return int 現在参加しているグループIDが返されます.<BR>
     *             現在グループに参加して無い場合[-1]が返されます.
     */
    public final int getGroupID()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_id ;
            }
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 設定されている有効期限を取得.
     * <BR><BR>
     * 設定されている有効期限が返されます.
     * <BR>
     * @return int 現在参加しているグループに対する有効期限が返されます.
     */
    public final int getTTL()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_socket.getTimeToLive() ;
            }
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * グループ参加チェック.
     * <BR><BR>
     * 現在グループに参加しているかチェックします.
     * <BR>
     * @return boolean グループ参加状態が返されます.<BR>
     *                 [true]が返された場合グループに参加しています.<BR>
     *                 [false]が返された場合グループに参加していません.
     */
    public final boolean isGroup()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = ( m_id != 0L ) ? true : false ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * オープンチェック.
     * <BR><BR>
     * オープンされているかチェックします.
     * <BR>
     * @return boolean オープン状態が返されます.<BR>
     *                 [true]が返された場合、ソケットはオープンされています.<BR>
     *                 [false]が返された場合、ソケットはオープンされていません.
     */
    public final boolean isOpen()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = ( m_socket != null ) ? true : false ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
}

