/*
 * @(#)ConnectAddress.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.net;

import java.net.InetAddress;
import java.net.UnknownHostException;

import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.thread.Synchronized;

/**
 * 接続アドレスオブジェクト.
 * <BR><BR>
 * 接続アドレス情報を管理するオブジェクトです.
 *
 * @version 1.0.0 2003/12/10
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
public class ConnectAddress
{
    
    /**
     * 非存在IPアドレス.
     */
    private static final String NOT_IPADDR = "NO-ADDRESS" ;
    
    /**
     * 非存在ポート番号.
     */
    private static final String NOT_PORT = "0" ;
    
    /**
     * フォーマット区切り文字開始.
     */
    private static final String ST_SETCD = " (" ;
    
    /**
     * フォーマット区切り文字終了.
     */
    private static final String ED_SETCD = ")" ;
    
    
    
    /**
     * アドレス情報.
     */
    private InetAddress m_address = null ;
    
    /**
     * 対象ポート番号.
     */
    private int m_port = -1 ;
    
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    
    
    /**
     * コンストラクタ.
     */
    public ConnectAddress()
    {
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 情報を生成します.
     * <BR>
     * @param address 設定対象のInetAddressオブジェクトを設定します.
     * @param port ポート番号情報を設定します.
     * @exception InputException 入力例外.
     */
    public ConnectAddress( InetAddress address,int port )
        throws InputException
    {
        try{
            this.create( address,port ) ;
        }catch(InputException in){
            throw in ;
        }
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 情報を生成します.
     * <BR>
     * @param host 設定対象のホスト名を設定します.
     * @param port ポート番号情報を設定します.
     * @exception InputException 入力例外.
     */
    public ConnectAddress( String host,int port )
        throws InputException
    {
        try{
            this.create( host,port ) ;
        }catch(InputException in){
            throw in ;
        }
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 情報を生成します.
     * <BR>
     * @param obj 設定対象のConnectAddressオブジェクトを設定します.
     * @exception InputException 入力例外.
     */
    public ConnectAddress( ConnectAddress obj )
        throws InputException
    {
        try{
            this.create( obj ) ;
        }catch(InputException in){
            throw in ;
        }
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 情報を生成します.
     * <BR>
     * @param address 設定対象のInetAddressオブジェクトを設定します.
     * @param port ポート番号情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void create( InetAddress address,int port )
        throws InputException
    {
        if( address == null || port < NetDef.PORT_MIN || port > NetDef.PORT_MAX ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        try{
            
            this.clear() ;
            m_sync.create() ;
            synchronized( m_sync.get() ){
                
                m_address = InetAddress.getByName( address.getHostAddress() ) ;
                m_port = port ;
                
            }
            
        }catch(NullPointerException nul){
        }catch(IllegalArgumentException il){
            throw new InputException(il) ;
        }catch( UnknownHostException uh ){
            throw new InputException( uh ) ;
        }
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 情報を生成します.
     * <BR>
     * @param host 設定対象のホスト名を設定します.
     * @param port ポート番号情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void create( String host,int port )
        throws InputException
    {
        if( host == null || port < NetDef.PORT_MIN || port > NetDef.PORT_MAX ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        try{
            
            this.clear() ;
            m_sync.create() ;
            synchronized( m_sync.get() ){
                
                m_address = InetAddress.getByName( host ) ;
                m_port = port ;
                
            }
        }catch(NullPointerException nul){
        }catch(IllegalArgumentException il){
            throw new InputException(il) ;
        }catch( UnknownHostException uh ){
            throw new InputException( uh ) ;
        }
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 情報を生成します.
     * <BR>
     * @param obj 設定対象のConnectAddressオブジェクトを設定します.
     * @exception InputException 入力例外.
     */
    public final void create( ConnectAddress obj )
        throws InputException
    {
        try{
            this.create( obj.getAddress(),obj.getPort() ) ;
        }catch(NullPointerException nul){
        }catch(InputException in){
            throw in ;
        }
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 格納情報をクリアします.
     */
    public final void clear()
    {
        m_sync.clear() ;
        m_address = null ;
        m_port = -1 ;
    }
    
    /**
     * InetAddressオブジェクトの取得.
     * <BR><BR>
     * 設定されているInetAddressオブジェクトが返されます.
     * <BR>
     * @return InetAddress 設定されているInetAddressオブジェクトが返されます.<BR>
     *                     情報が設定されていない場合[null]が返されます.
     */
    public final InetAddress getAddress()
    {
        InetAddress ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = InetAddress.getByName( m_address.getHostAddress() ) ;
            }
        }catch(Exception t){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 設定ポート番号の取得.
     * <BR><BR>
     * 設定されているポート番号を取得します.
     * <BR>
     * @return int 設定ポート情報が返されます.<BR>
     *             情報が設定されていない場合[-1]が返されます.
     */
    public final int getPort()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_port ;
            }
        }catch(Exception t){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 文字列情報の取得.
     * <BR><BR>
     * 格納されている情報を文字列で取得します.
     * <BR>
     * @return String 格納されている情報が返されます.
     */
    public final String toString()
    {
        String ret = null ;
        StringBuffer buf = null ;
        
        buf = new StringBuffer() ;
        
        try{
            synchronized( m_sync.get() ){
                buf.append( m_address.getHostAddress() ) ;
                buf.append( ConnectAddress.ST_SETCD ) ;
                buf.append( m_port ) ;
                buf.append( ConnectAddress.ED_SETCD ) ;
            }
        }catch( Exception t ){
            buf.append( ConnectAddress.NOT_IPADDR ) ;
            buf.append( ConnectAddress.ST_SETCD ) ;
            buf.append( ConnectAddress.NOT_PORT ) ;
            buf.append( ConnectAddress.ED_SETCD ) ;
        }finally{
            ret = buf.toString() ;
            buf = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象内容が正しく設定されているかチェック.
     * <BR><BR>
     * 対象内容が正しく設定されているかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、正しく設定されています.<BR>
     *                 [false]が返された場合、正しく設定されていません.
     */
    public final boolean isUse()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = ( m_address == null || m_port < 0 ) ?
                    true : false ;
            }
        }catch(Exception t){
            ret = false ;
        }
        
        return ret ;
    }
    
}
