/*
 * @(#)BaseTcpProtocol.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.net;

import java.io.ByteArrayOutputStream;
import java.net.InetAddress;

import com.JRcServer.commons.exception.InputException;

/**
 * TCP/IPプロトコルインターフェイス.
 * <BR><BR>
 * 対象のTCP/IP処理を行うインターフェイスです.
 *
 * @version 1.00, 2004/10/05
 * @author  Masahito Suzuki
 * @since   JRcCommons 1.00
 */
public interface BaseTcpProtocol
{
    
    /**
     * 接続処理.
     * <BR><BR>
     * 接続先を設定します.
     * <BR>
     * @param addr 接続先のアドレスを設定します.
     * @param port 接続先のポート番号を設定します.
     * @exception InputException 入力例外.
     * @exception NotConnectException 未コネクション例外.
     */
    public void connect( InetAddress addr,int port )
        throws InputException,NotConnectException ;
    
    /**
     * 接続処理.
     * <BR><BR>
     * 接続先を設定します.
     * <BR>
     * @param addr 接続先のアドレスを設定します.
     * @param port 接続先のポート番号を設定します.
     * @param loAddr バインド先のローカルアドレスを設定します.
     * @param loPort バインド先のローカルポートを設定します.
     * @exception InputException 入力例外.
     * @exception NotBindException バインド例外.
     * @exception NotConnectException 未コネクション例外.
     */
    public void connect( InetAddress addr,int port,InetAddress loAddr,int loPort )
        throws InputException,NotBindException,NotConnectException ;
    
    /**
     * 接続処理.
     * <BR><BR>
     * 接続先を設定します.
     * <BR>
     * @param addr 接続先のアドレスを設定します.
     * @param port 接続先のポート番号を設定します.
     * @param bufLen 送受信バッファ長を設定します.
     * @exception InputException 入力例外.
     * @exception NotConnectException 未コネクション例外.
     */
    public void connect( InetAddress addr,int port,int bufLen )
        throws InputException,NotConnectException ;
    
    /**
     * 接続処理.
     * <BR><BR>
     * 接続先を設定します.
     * <BR>
     * @param addr 接続先のアドレスを設定します.
     * @param port 接続先のポート番号を設定します.
     * @param loAddr バインド先のローカルアドレスを設定します.
     * @param loPort バインド先のローカルポートを設定します.
     * @param bufLen 送受信バッファ長を設定します.
     * @exception InputException 入力例外.
     * @exception NotBindException バインド例外.
     * @exception NotConnectException 未コネクション例外.
     */
    public void connect( InetAddress addr,int port,InetAddress loAddr,int loPort,int bufLen )
        throws InputException,NotBindException,NotConnectException ;
    
    /**
     * コネクション破棄.
     * <BR><BR>
     * コネクションを破棄します.
     */
    public void disconnect() ;
    
    /**
     * 送信処理.
     * <BR><BR>
     * 送信処理を設定します.
     * <BR>
     * @param message 送信対象のメッセージを設定します.
     * @exception InputException 入力例外.
     * @exception UndefineBindException 未バインド例外.
     */
    public void send( byte[] message )
        throws InputException,UndefineBindException ;
    
    /**
     * 送信処理.
     * <BR><BR>
     * 送信処理を設定します.
     * <BR>
     * @param message 送信対象のメッセージを設定します.
     * @param length 対象メッセージのデータ長を設定します.
     * @exception InputException 入力例外.
     * @exception UndefineBindException 未バインド例外.
     */
    public void send( byte[] message,int length )
        throws InputException,UndefineBindException ;
    
    /**
     * 送信処理.
     * <BR><BR>
     * 送信処理を設定します.
     * <BR>
     * @param message 送信対象のメッセージを設定します.
     * @param offset 対象メッセージのオフセット値を設定します.
     * @param length 対象メッセージのデータ長を設定します.
     * @exception InputException 入力例外.
     * @exception UndefineBindException 未バインド例外.
     */
    public void send( byte[] message,int offset,int length )
        throws InputException,UndefineBindException ;
    
    /**
     * 受信処理.
     * <BR><BR>
     * 受信処理を行います.
     * <BR>
     * @param timeout 受信タイムアウト値を設定します.
     * @return byte[] 受信されたバイナリ情報が返されます.
     * @exception UndefineBindException 未バインド例外.
     * @exception ConnectTimeoutException タイムアウト例外.
     */
    public byte[] receive( int timeout )
        throws UndefineBindException,ConnectTimeoutException ;
    
    /**
     * 受信処理.
     * <BR><BR>
     * 受信処理を行います.
     * <BR>
     * @param out 受信されたバイナリ情報が返されます.
     * @param timeout 受信タイムアウト値を設定します.
     * @exception InputException 入力例外.
     * @exception UndefineBindException 未バインド例外.
     * @exception ConnectTimeoutException タイムアウト例外.
     */
    public void receive( ByteArrayOutputStream out,int timeout )
        throws InputException,UndefineBindException,ConnectTimeoutException ;
    
    /**
     * ローカルアドレスを取得.
     * <BR><BR>
     * 対象のローカルアドレスを取得します.
     * <BR>
     * @param addr 対象のローカルアドレスが返されます.
     */
    public void getLocal( ConnectAddress addr ) ;
    
    /**
     * 接続先アドレスを取得.
     * <BR><BR>
     * 対象の接続先アドレスを取得します.
     * <BR>
     * @param addr 対象の接続アドレスが返されます.
     */
    public void getConnect( ConnectAddress addr ) ;
    
    /**
     * 送受信バッファ長を取得.
     * <BR><BR>
     * 設定されている送受信バッファ長が返されます.
     * <BR>
     * @return int 対象の送受信バッファ長が返されます.
     */
    public int getBuffer() ;
    
    /**
     * 最終処理時間を取得.
     * <BR><BR>
     * 最終処理時間を取得します.
     * <BR>
     * @return long 最終処理時間が返されます.
     */
    public long getLastTime() ;
    
    /**
     * 接続チェック.
     * <BR><BR>
     * 接続されているかチェックします.
     * <BR>
     * @return boolean 接続状況が返されます.<BR>
     *                 [true]が返された場合、接続されています.
     *                 [false]が返された場合、接続されていません.
     */
    public boolean isConnect() ;
    
}

