/*
 * @(#)EFileOutputStream.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.io.env;

import java.io.File;
import java.io.FileDescriptor;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;

import com.JRcServer.commons.io.IODef;

/**
 * 環境変数対応版[java.io.FileOutputStream]オブジェクト.
 * <BR><BR>
 * [java.io.OutputStream]の書き込み対象名に環境変数が含まれている場合、
 * それらを解析して、書き込み対象に変換します.
 *  
 * @version 1.0.0 2005/05/02
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
public class EnvFileOutputStream extends OutputStream
{
    
    /**
     * ファイル情報取得オブジェクト.
     */
    private FileOutputStream m_out = null ;
    
    /**
     * コンストラクタ.
     */
    private EnvFileOutputStream(){}
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 書き込みファイルをオープンします.
     * <BR>
     * @param file 対象のファイルオブジェクトを設定します.
     * @exception IOException ファイル名が存在しない場合発生.
     * @exception SecurityException セキュリティマネージャが存在し、
     *                              checkRead メソッドがファイルへの書き込み
     *                              アクセスを拒否する場合発生.
     */
    public EnvFileOutputStream( File file )
        throws IOException
    {
        m_out = new FileOutputStream( file ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 書き込みファイルをオープンします.
     * <BR>
     * @param fdObj 対象のファイルディスクリプタを設定します.
     * @exception SecurityException セキュリティマネージャが存在し、
     *                              checkRead メソッドがファイルへの書き込み
     *                              アクセスを拒否する場合発生.
     */
    public EnvFileOutputStream( FileDescriptor fdObj )
    {
        m_out = new FileOutputStream( fdObj ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 書き込みファイルをオープンします.
     * <BR>
     * @param name 対象のファイル名を設定します.<BR>
     *             このファイル名に対して環境変数名を含めて設定する事が
     *             可能です.<BR>
     * @exception IOException ファイル名が存在しない場合発生.
     * @exception SecurityException セキュリティマネージャが存在し、
     *                              checkRead メソッドがファイルへの書き込み
     *                              アクセスを拒否する場合発生.
     */
    public EnvFileOutputStream( String name )
        throws IOException
    {
        m_out = new FileOutputStream( IODef.convertEnvName( name ) ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 書き込みファイルをオープンします.
     * <BR>
     * @param name 対象のファイル名を設定します.<BR>
     *             このファイル名に対して環境変数名を含めて設定する事が
     *             可能です.<BR>
     * @param append [true]の場合、バイトはファイルの先頭ではなく最後に書き込まれます.
     * @exception IOException ファイル名が存在しない場合発生.
     * @exception SecurityException セキュリティマネージャが存在し、
     *                              checkRead メソッドがファイルへの書き込み
     *                              アクセスを拒否する場合発生.
     */
    public EnvFileOutputStream( String name,boolean append )
        throws IOException
    {
        m_out = new FileOutputStream( IODef.convertEnvName( name ),append ) ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.close() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報クローズ.
     * <BR><BR>
     * 情報をクローズします.
     * <BR>
     * @exception IOException 入出力エラーが発生した場合.
     */
    public final void close()
        throws IOException
    {
        if( m_out != null ){
            m_out.close() ;
        }
    }
    
    /**
     * 情報を強制的に書き込み.
     * <BR><BR>
     * 情報を強制的に書き込みます.
     * <BR>
     * @exception IOException 入出力エラーが発生した場合.
     */
    public final void flush() throws IOException
    {
        if( m_out != null ){
            m_out.flush() ;
        }
    }
    
    /**
     * ストリームに関連したファイル記述子を取得.
     * <BR><BR>
     * ストリームに関連したファイル記述子を取得します.
     * <BR>
     * @return FileDescriptor この FileOutputStream オブジェクトによって使用されている
     *                        ファイルシステムのファイルへの接続を表す FileDescriptor オブジェクト.
     * @exception IOException 入出力エラーが発生した場合.
     */
    public final FileDescriptor getFD()
        throws IOException
    {
        return ( m_out != null ) ? m_out.getFD() : null ;
    }
    
    /**
     * 情報を書き込み.
     * <BR><BR>
     * 情報を書き込みます.
     * <BR>
     * @param b 書き込み対象の情報を設定します.<BR>
     *          また書き込み有効な条件は下位8ビットです.
     * @exception IOException IO例外.
     */
    public final void write( int b ) throws IOException
    {
        if( m_out != null ){
            m_out.write( b ) ;
        }
    }
    
    /**
     * 情報を書き込み.
     * <BR><BR>
     * 情報を書き込みます.
     * <BR>
     * @param b 書き込み対象の情報を設定します.
     * @exception IOException IO例外.
     */
    public final void write( byte[] b )
        throws IOException
    {
        if( m_out != null ){
            m_out.write( b ) ;
        }
    }
    
    /**
     * 情報を書き込み.
     * <BR><BR>
     * 情報を書き込みます.
     * <BR>
     * @param binary 書き込み対象の情報を設定します.
     * @param offset 書き込み対象のオフセット値を設定します.
     * @param length 書き込み対象のデータ長を設定します.
     * @exception IOException IO例外.
     */
    public final void write( byte[] binary,int offset,int length )
        throws IOException
    {
        if( m_out != null ){
            m_out.write( binary,offset,length ) ;
        }
    }
}

