/*
 * @(#)EnvFileInputStream.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.io.env;

import java.io.File;
import java.io.FileDescriptor;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;

import com.JRcServer.commons.io.IODef;

/**
 * 環境変数対応版[java.io.FileInputStream]オブジェクト.
 * <BR><BR>
 * [java.io.FileInputStream]の読み込み対象名に環境変数が含まれている場合、
 * それらを解析して、読み込み対象に変換します.
 *  
 * @version 1.0.0 2005/05/02
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
public class EnvFileInputStream extends InputStream
{
    
    /**
     * ファイル情報取得オブジェクト.
     */
    private FileInputStream m_in = null ;
    
    /**
     * コンストラクタ.
     */
    private EnvFileInputStream(){}
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 読み込みファイルをオープンします.
     * <BR>
     * @param file 対象のファイルオブジェクトを設定します.
     * @exception IOException ファイル名が存在しない場合発生.
     * @exception SecurityException セキュリティマネージャが存在し、
     *                              checkRead メソッドがファイルへの読み込み
     *                              アクセスを拒否する場合発生.
     */
    public EnvFileInputStream( File file )
        throws IOException
    {
        m_in = new FileInputStream( file ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 読み込みファイルをオープンします.
     * <BR>
     * @param fdObj 対象のファイルディスクリプタを設定します.
     * @exception SecurityException セキュリティマネージャが存在し、
     *                              checkRead メソッドがファイルへの読み込み
     *                              アクセスを拒否する場合発生.
     */
    public EnvFileInputStream( FileDescriptor fdObj )
    {
        m_in = new FileInputStream( fdObj ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 読み込みファイルをオープンします.
     * <BR>
     * @param name 対象のファイル名を設定します.<BR>
     *             このファイル名に対して環境変数名を含めて設定する事が
     *             可能です.<BR>
     * @exception IOException ファイル名が存在しない場合発生.
     * @exception SecurityException セキュリティマネージャが存在し、
     *                              checkRead メソッドがファイルへの読み込み
     *                              アクセスを拒否する場合発生.
     */
    public EnvFileInputStream( String name )
        throws IOException
    {
        m_in = new FileInputStream( IODef.convertEnvName( name ) ) ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.close() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報クローズ.
     * <BR><BR>
     * 情報をクローズします.
     * <BR>
     * @exception IOException 入出力エラーが発生した場合.
     */
    public final void close()
        throws IOException
    {
        if( m_in != null ){
            m_in.close() ;
        }
        m_in = null ;
    }
    
    /**
     * ストリームに関連したファイル記述子を取得.
     * <BR><BR>
     * ストリームに関連したファイル記述子を取得します.
     * <BR>
     * @return FileDescriptor この FileOutputStream オブジェクトによって使用されている
     *                        ファイルシステムのファイルへの接続を表す FileDescriptor オブジェクト.
     * @exception IOException 入出力エラーが発生した場合.
     */
    public final FileDescriptor getFD()
        throws IOException
    {
        return ( m_in != null ) ? m_in.getFD() : null ;
    }
    
    /**
     * １バイトのデータを読み込みます.
     * <BR><BR>
     * １バイトのデータを読み込みます.
     * <BR>
     * @return int １バイトのデータが返されます.
     * @exception IOException IO例外.
     */
    public final int read() throws IOException
    {
        return ( m_in != null ) ? m_in.read() : -1 ;
    }
    
    /**
     * 指定バイト情報を読み込み.
     * <BR><BR>
     * 指定バイト情報を読み込みます.
     * <BR>
     * @param binary 読み込まれた情報が返されるバイナリを設定します.<BR>
     *               [binary == null]の場合NullPointerExceptionが発生します.
     * @return int 読み込まれたバイト数が返されます.<BR>
     *             読み込み先が存在しない場合[-1]が返されます.
     * @exception IOException IO例外.
     */
    public final int read( byte[] binary ) throws IOException
    {
        return ( m_in != null ) ? m_in.read( binary ) : -1 ;
    }
    
    /**
     * 指定バイト情報を読み込み.
     * <BR><BR>
     * 指定バイト情報を読み込みます.
     * <BR>
     * @param binary 読み込まれた情報が返されるバイナリを設定します.<BR>
     *               [binary == null]の場合NullPointerExceptionが発生します.
     * @param offset 読み込み格納先の位置を設定します.
     * @param length 読み込み情報長を設定します.
     * @return int 読み込まれたバイト数が返されます.<BR>
     *             読み込み先が存在しない場合[-1]が返されます.
     * @exception IOException IO例外.
     */
    public final int read( byte[] binary,int offset,int length )
        throws IOException
    {
        return ( m_in != null ) ? m_in.read( binary,offset,length ) : -1 ;
    }
    
    /**
     * スキップ処理.
     * <BR><BR>
     * スキップ処理を行います.
     * <BR>
     * @param n スキップ長を設定します.
     * @return long 実際にスキップされたデータ長が返されます.
     */
    public final long skip( long n ) throws IOException
    {
        return ( m_in != null ) ? m_in.skip( n ) : -1L ;
    }
    
    /**
     * 読み込みが可能なデータ長を取得.
     * <BR><BR>
     * 読み込みが可能なデータ長を取得します.
     * <BR>
     * @return int 読み込みが可能なデータ長が返されます.
     * @exception IOException IO例外.
     */
    public final int available() throws IOException
    {
        return ( m_in != null ) ? m_in.available() : -1 ;
    }
    
    /**
     * マーク処理.
     * <BR><BR>
     * マーク処理を行います.<BR>
     * またマークした位置に戻す場合は
     * [EFileInputStream.reset()]を利用します.
     * <BR>
     * @param offset 現在の位置からのオフセット値を設定します.
     */
    public final void mark( int offset )
    {
        if( m_in != null ){
            m_in.mark( offset ) ;
        }
    }
    
    /**
     * マーク処理に対するリセット処理.
     * <BR><BR>
     * リセット処理を行います.
     * <BR>
     * @exception IOException IO例外.
     */
    public final void reset() throws IOException
    {
        if( m_in != null ){
            m_in.reset() ;
        }
    }
    
    /**
     * マーク処理サポートチェック.
     * <BR><BR>
     * マーク処理がサポートされているかチェックします.
     * <BR>
     * @return boolean マーク処理がサポートされているか返されます.<BR>
     *                 [true]が返された場合サポートされています.
     *                 [false]が返された場合サポートされていません.
     */
    public final boolean markSupported()
    {
        return ( m_in != null ) ? m_in.markSupported() : false ;
    }
    
}

