/*
 * @(#)UMountMonitor.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.io;

import com.JRcServer.commons.exception.ExecutionException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.exception.SettingException;
import com.JRcServer.commons.thread.ExecutionThread;
import com.JRcServer.commons.thread.LoopThread;
import com.JRcServer.commons.thread.Synchronized;
import com.JRcServer.commons.util.UtilCom;
import com.JRcServer.commons.util.array.ObjectArray;

/**
 * アンマウント監視処理.
 * <BR><BR>
 * アンマウント監視処理を実施します.<BR>
 * このオブジェクトはインターフェイス[com.JRcServer.commons.io.SwitchMount]を
 * 監視するためのオブジェクトです.<BR>
 * また、監視に対する処理対象は指定時間に対するアクセスが無い場合の
 * アンマウント処理を行うものです.
 *
 * @version 1.00, 2004/11/09
 * @author  Masahito Suzuki
 * @since   JRcCommons 1.00
 */
public class UMountMonitor extends ExecutionThread
{
    
    /**
     * アンマウント実施タイミング : デフォルト.
     * 15秒.
     */
    private static final int DEF_UNMOUNT = 15000 ;
    
    /**
     * アンマウント実施タイミング : 最小値.
     * 1秒.
     */
    private static final int MIN_UNMOUNT = 1000 ;
    
    /**
     * アンマウント実施タイミング : 最大値.
     * 60分.
     */
    private static final int MAX_UNMOUNT = 3600000 ;
    
    
    
    /**
     * アンマウント実施時間.
     */
    private int m_timer = 0 ;
    
    /**
     * ループスレッド.
     */
    private final LoopThread m_thread = new LoopThread() ;
    
    /**
     * アンマウントオブジェクト管理オブジェクト.
     */
    private final ObjectArray m_man = new ObjectArray() ;
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    
    
    /**
     * コンストラクタ.
     */
    public UMountMonitor()
    {
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * アンマウント監視モニターを生成します.
     */
    public final void create()
    {
        try{
            this.create( UMountMonitor.DEF_UNMOUNT ) ;
        }catch( Exception t ){
            this.clear() ;
        }
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * アンマウント監視モニターを生成します.
     * <BR>
     * @param time 設定対象のタイム値を設定します.<BR>
     *             設定可能な最小値は[1000]です.<BR>
     *             設定可能な最大値は[3600000]です.<BR>
     *             また、設定単位はミリ秒です.
     * @exception InputException 入力例外.
     */
    public final void create( int time )
        throws InputException
    {
        if(
            time < UMountMonitor.MIN_UNMOUNT ||
            time > UMountMonitor.MAX_UNMOUNT
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.clear() ;
        m_sync.create() ;
        
        try{
            synchronized( m_sync.get() ){
                
                m_timer = time ;
                m_thread.create( this ) ;
                m_thread.startThread() ;
                
            }
            
        }catch( NullPointerException nul ){
            this.clear() ;
        }catch( InputException in ){
            this.clear() ;
        }catch( SettingException st ){
            this.clear() ;
        }
        
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public final void clear()
    {
        int i ;
        int len ;
        
        ObjectArray ary = null ;
        
        m_sync.clear() ;
        m_thread.clear() ;
        m_timer = 0 ;
        
        ary = m_man ;
        len = ary.size() ;
        for( i = 0 ; i < len ; i ++ ){
            ( ( SwitchMount )ary.get( i ) ).clear() ;
        }
        ary.clear() ;
        ary = null ;
    }
    
    /**
     * アンマウント用オブジェクトを設定.
     * <BR><BR>
     * アンマウント用オブジェクトを設定します.
     * <BR>
     * @param val 設定対象のオブジェクトを設定します.
     * @exception InputException 入力例外.
     */
    public final void setObject( SwitchMount val )
        throws InputException
    {
        int i ;
        int len ;
        boolean flg ;
        
        if( val == null || val.isCreate() == false ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            synchronized( m_sync.get() ){
                len = m_man.size() ;
                for( i = 0,flg = true ; i < len ; i ++ ){
                    if( m_man.get( i ) == val ){
                        flg = false ;
                        break ;
                    }
                }
                if( flg == true ){
                    m_man.add( val ) ;
                }
            }
        }catch( Exception t ){
        }
    }
    
    /**
     * アンマウントタイム値を設定.
     * <BR><BR>
     * アンマウントタイム値を設定します.
     * <BR>
     * @param time 設定対象のタイム値を設定します.<BR>
     *             設定可能な最小値は[1000]です.<BR>
     *             設定可能な最大値は[3600000]です.<BR>
     *             また、設定単位はミリ秒です.
     * @exception InputException 入力例外.
     */
    public final void setTimer( int time )
        throws InputException
    {
        if(
            time < UMountMonitor.MIN_UNMOUNT ||
            time > UMountMonitor.MAX_UNMOUNT
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            synchronized( m_sync.get() ){
                m_timer = time ;
            }
        }catch( Exception t ){
        }
        
    }
    
    /**
     * アンマウントタイム値を取得.
     * <BR><BR>
     * アンマウントタイム値を取得します.
     * <BR>
     * @return int 設定されているアンマウント値が返されます.
     */
    public final int getTimer()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_timer ;
            }
        }catch( Exception t ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 動作チェック.
     * <BR><BR>
     * このオブジェクトのスレッドが動作しているかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、スレッドは動作中です.<BR>
     *                 [false]が返された場合、スレッドは停止中です.
     */
    public final boolean isThread()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_thread.isThread() ;
            }
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 情報生成フラグ.
     * <BR><BR>
     * オブジェクト情報が生成されているかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.
     */
    public final boolean isCreate()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = true ;
            }
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 実行初期化処理をサポートします.
     * <BR><BR>
     * 実行初期化処理をサポートします.<BR>
     * この処理は、スレッド処理が開始された時に呼び出されます.
     * <BR>
     * @param obj 実行開始時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void init( Object obj )
        throws ExecutionException
    {
    }
    
    /**
     * 実行終了化処理をサポートします.
     * <BR><BR>
     * 実行終了化処理をサポートします.<BR>
     * この処理は、スレッド処理が終了された時に呼び出されます.
     * <BR>
     * @param obj 実行終了時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void exit( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * ストップ処理をサポートします。
     * <BR><BR>
     * ストップ処理をサポートします。<BR>
     * この処理は、スレッドでのストップ処理に対して呼び出し実行されます.
     * <BR>
     * @param obj ストップ時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void stop( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行処理をサポートします。
     * <BR><BR>
     * 実行処理をサポートします。<BR>
     * この処理は、スレッドでの実行処理に対して呼び出し実行されます.
     * <BR>
     * @param obj 実行時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void execution( Object obj )
        throws ExecutionException
    {
        int i ;
        int len ;
        
        long umntTime ;
        long nowTime ;
        
        ObjectArray ary = null ;
        SwitchMount val = null ;
        
        try{
            
            UtilCom.idleTime() ;
            
            synchronized( m_sync.get() ){
                ary = m_man ;
                len = ary.size() ;
                umntTime = ( long )( m_timer & 0x00000000ffffffffL ) ;
            }
            
            for( i = len-1 ; i >= 0 ; i -- ){
                
                val = ( SwitchMount )ary.get( i ) ;
                
                // オブジェクトが既にクローズされている場合.
                if( val.isCreate() == false ){
                    
                    // オブジェクトから破棄.
                    ary.remove( i ) ;
                    
                }
                // 更新時間がタイムアウトの場合.
                else if(
                    val.isState() == true &&
                    val.getUpdateTime() + umntTime < System.currentTimeMillis()
                )
                {
                    
                    // アンマウント.
                    val.unMount() ;
                    
                }
                
                val = null ;
            }
            
        }catch( NullPointerException nul ){
            throw new ExecutionException( nul,ExecutionException.LEVEL_STOP ) ;
        }catch( ExecutionException ee ){
            throw ee ;
        }catch( Exception t ){
        }finally{
            ary = null ;
            val = null ;
        }
    }
    
}

