/*
 * @(#)RandomOutputStream.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.io;

import java.io.IOException;
import java.io.OutputStream;

import com.JRcServer.commons.exception.BaseException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.thread.Synchronized;

/**
 * ランダム出力ストリーム.
 * <BR><BR>
 * [com.JRcServer.commons.io.RandomIO]をOutputStreamで
 * 利用可能にしたオブジェクトです.
 *
 * @version 1.00, 2004/06/30
 * @author  Masahito Suzuki
 * @since   JRcCommons 1.00
 */
class RandomOutputStream extends OutputStream
{
    
    /**
     * 書き込みバッファ長.
     */
    private static final int WRITE_BUFFER = 1024 ;
    
    /**
     * 書き込みバッファマスク.
     */
    private static final int WRITE_MASK = ~ ( WRITE_BUFFER - 1 ) ;
    
    /**
     * 書き込みバッファシフト値.
     */
    private static final int WRITE_SHIFT = 10 ;
    
    
    /**
     * ランダムアクセスファイル.
     */
    private RandomIO m_fp = null ;
    
    /**
     * 書き込み位置.
     */
    private long m_seek = 0L ;
    
    /**
     * 読み込みバッファ長.
     */
    private final byte[] m_buf = new byte[ RandomOutputStream.WRITE_BUFFER ] ;
    
    /**
     * 同期オブジェクト.
     */
    private Synchronized m_sync = null ;
    
    
    /**
     * コンストラクタ.
     */
    private RandomOutputStream()
    {
        super() ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象のオブジェクトを設定します.
     * <BR>
     * @param fp 書き込み対象のランダムアクセスオブジェクトを設定します.
     * @exception InputException 入力例外.
     */
    public RandomOutputStream( RandomIO fp )
        throws InputException
    {
        super() ;
        
        if( fp == null || fp.isOpen() == false ){
            throw new InputException( "引数は不正です" ) ;
        }
        this.close() ;
        
        m_fp = fp ;
        m_sync = fp.getSynchronized() ;
    }
    
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.close() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報クローズ.
     * <BR><BR>
     * 情報をクローズします.
     */
    public final void close()
    {
        m_fp = null ;
        m_sync = null ;
        m_seek = 0L ;
    }
    
    /**
     * 情報を強制的に書き込み.
     * <BR><BR>
     * 情報を強制的に書き込みます.
     * <BR>
     * throws IOException IO例外.
     */
    public final void flush() throws IOException
    {
        
    }
    
    /**
     * 情報を書き込み.
     * <BR><BR>
     * 情報を書き込みます.
     * <BR>
     * @param b 書き込み対象の情報を設定します.<BR>
     *          また書き込み有効な条件は下位8ビットです.
     * @exception IOException IO例外.
     */
    public final void write( int b ) throws IOException
    {
        byte[] buf = null ;
        
        try{
            synchronized( m_sync.get() ){
                
                buf = m_buf ;
                
                buf[ 0 ] = ( byte )( b & 0x000000ff ) ;
                m_fp.write( buf,m_seek,0,1 ) ;
                m_seek ++ ;
                
            }
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報を書き込み.
     * <BR><BR>
     * 情報を書き込みます.
     * <BR>
     * @param b 書き込み対象の情報を設定します.
     * @exception IOException IO例外.
     */
    public final void write( byte[] b )
        throws IOException
    {
        this.write( b,0,b.length ) ;
    }
    
    /**
     * 情報を書き込み.
     * <BR><BR>
     * 情報を書き込みます.
     * <BR>
     * @param binary 書き込み対象の情報を設定します.
     * @param offset 書き込み対象のオフセット値を設定します.
     * @param length 書き込み対象のデータ長を設定します.
     * @exception IOException IO例外.
     */
    public final void write( byte[] binary,int offset,int length )
        throws IOException
    {
        
        if(
            (
                offset | length | ( offset + length ) |
                ( binary.length - ( offset + length ) )
            ) < 0
        )
        {
            throw new IndexOutOfBoundsException( "引数は不正です" ) ;
        }
        else if ( length == 0 ){
            return ;
        }
        
        try{
            synchronized( m_sync.get() ){
                m_fp.write( binary,m_seek,offset,length ) ;
                m_seek += length ;
            }
        }catch( NullPointerException nul ){
        }catch( BaseException be ){
            throw new IOException( be.getMessage() ) ;
        }catch( Exception e ){
            throw new IOException( e.getMessage() ) ;
        }
    }
    
    /**
     * 読み込み位置を設定.
     * <BR><BR>
     * 読み込み対象のシークポイントを設定します.
     * <BR>
     * @param seek 読み込み位置を設定します.
     */
    public final void setSeek( long seek )
    {
        if( seek < 0L ){
            return ;
        }
        
        try{
            synchronized( m_sync.get() ){
                m_seek = seek ;
            }
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 現在の読み込み位置を取得.
     * <BR><BR>
     * 現在の読み込み位置を取得します.
     * <BR>
     * @return long 読み込み位置が返されます.
     */
    public final long getSeek()
    {
        long ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_seek ;
            }
        }catch( Exception t ){
            ret = -1L ;
        }
        
        return ret ;
    }
    
    /**
     * ファイルサイズを取得.
     * <BR><BR>
     * ファイルサイズを取得します.
     * <BR>
     * @return long 対象のファイルサイズを取得します.
     */
    public final long getLength()
    {
        long ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_fp.getLength() ;
            }
        }catch( Exception t ){
            ret = 0L ;
        }
        
        return ret ;
    }
    
    /**
     * 同期オブジェクトを取得.
     * <BR><BR>
     * 同期オブジェクトを取得します.
     * <BR>
     * @return Synchronized 同期オブジェクトが返されます.
     */
    public final Synchronized getSynchronized()
    {
        return m_sync ;
    }
    
    
}

