/*
 * @(#)RandomInputStream.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.io;

import java.io.IOException;
import java.io.InputStream;

import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.thread.Synchronized;
import com.JRcServer.commons.util.array.LongArray;

/**
 * ランダム入力ストリーム.
 * <BR><BR>
 * [com.JRcServer.commons.io.RandomIO]をInputStreamで利用可能にしたオブジェクトです.
 *  
 * @version 1.0.0 2003/11/10
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
class RandomInputStream extends InputStream
{
    
    /**
     * 読み込みバッファ長.
     */
    private static final int READ_BUFFER = 1 ;
    
    /**
     * ランダムアクセスファイル.
     */
    private RandomIO m_fp = null ;
    
    /**
     * 読み込み位置.
     */
    private long m_seek = 0L ;
    
    /**
     * 読み込み終了長.
     */
    private long m_length = 0L ;
    
    /**
     * マークポイント.
     */
    private final LongArray m_mark = new LongArray() ;
    
    /**
     * 読み込みバッファ長.
     */
    private final byte[] m_buf = new byte[ RandomInputStream.READ_BUFFER ] ;
    
    /**
     * 同期オブジェクト.
     */
    private Synchronized m_sync = null ;
    
    
    /**
     * コンストラクタ.
     */
    private RandomInputStream()
    {
        super() ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象のオブジェクトを設定します.
     * <BR>
     * @param fp 読み込み対象のランダムアクセスオブジェクトを設定します.
     * @exception InputException 入力例外.
     */
    public RandomInputStream( RandomIO fp )
        throws InputException
    {
        super() ;
        
        if( fp == null || fp.isOpen() == false ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.close() ;
        
        m_sync = fp.getSynchronized() ;
        m_fp = fp ;
        m_length = fp.getLength() ;
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象のオブジェクトを設定します.
     * <BR>
     * @param fp 読み込み対象のランダムアクセスオブジェクトを設定します.
     * @exception InputException 入力例外.
     */
    public RandomInputStream( RandomIO fp,long offset,long length )
        throws InputException
    {
        super() ;
        
        if( fp == null || fp.isOpen() == false ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.close() ;
        
        m_sync = fp.getSynchronized() ;
        m_fp = fp ;
        m_seek = offset ;
        m_length = length ;
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.close() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報クローズ.
     * <BR><BR>
     * 情報をクローズします.
     */
    public final void close()
    {
        m_fp = null ;
        m_sync = null ;
        m_seek = 0L ;
        m_length = 0L ;
        m_mark.clear() ;
    }
    
    /**
     * １バイトのデータを読み込みます.
     * <BR><BR>
     * １バイトのデータを読み込みます.
     * <BR>
     * @return int １バイトのデータが返されます.
     * @exception IOException IO例外.
     */
    public final int read() throws IOException
    {
        int len ;
        int ret ;
        byte[] buf = null ;
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_length <= m_seek ){
                    ret = -1 ;
                }
                else{
                    
                    buf = m_buf ;
                    len = m_fp.read( buf,m_seek,0,1 ) ;
                    
                    if( len < 0 ){
                        ret = -1 ;
                    }
                    else{
                        ret = ( int )( buf[ 0 ] & 0x000000ff ) ;
                        m_seek ++ ;
                    }
                    
                }
            }
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 指定バイト情報を読み込み.
     * <BR><BR>
     * 指定バイト情報を読み込みます.
     * <BR>
     * @param binary 読み込まれた情報が返されるバイナリを設定します.<BR>
     *               [binary == null]の場合NullPointerExceptionが発生します.
     * @return int 読み込まれたバイト数が返されます.<BR>
     *             読み込み先が存在しない場合[-1]が返されます.
     * @exception IOException IO例外.
     */
    public final int read( byte[] binary ) throws IOException
    {
        return this.read( binary,0,binary.length ) ;
    }
    
    /**
     * 指定バイト情報を読み込み.
     * <BR><BR>
     * 指定バイト情報を読み込みます.
     * <BR>
     * @param binary 読み込まれた情報が返されるバイナリを設定します.<BR>
     *               [binary == null]の場合NullPointerExceptionが発生します.
     * @param offset 読み込み格納先の位置を設定します.
     * @param length 読み込み情報長を設定します.
     * @return int 読み込まれたバイト数が返されます.<BR>
     *             読み込み先が存在しない場合[-1]が返されます.
     * @exception IOException IO例外.
     */
    public final int read( byte[] binary,int offset,int length )
        throws IOException
    {
        int ret ;
        
        if(
            (
                offset | length | ( offset + length ) |
                ( binary.length - ( offset + length ) )
            ) < 0
        )
        {
            throw new IndexOutOfBoundsException( "引数は不正です" ) ;
        }
        else if ( length == 0 ){
            return 0 ;
        }
        
        try{
            synchronized( m_sync.get() ){
                if( m_length <= m_seek + length ){
                    ret = ( m_length <= m_seek ) ?
                        -1 :
                        m_fp.read( binary,m_seek,offset,( int )( m_length - m_seek ) ) ;
                }
                else{
                    ret = m_fp.read( binary,m_seek,offset,length ) ;
                }
                
                m_seek += ( ret <= 0 ) ? 0 : ret ;
            }
            
        }catch( Exception e ){
            throw new IOException( e.getMessage() ) ;
        }
        
        return ret ;
    }
    
    /**
     * スキップ処理.
     * <BR><BR>
     * スキップ処理を行います.
     * <BR>
     * @param n スキップ長を設定します.
     * @return long 実際にスキップされたデータ長が返されます.
     */
    public final long skip( long n ) throws IOException
    {
        long seek ;
        long length ;
        long ret ;
        
        ret = 0L ;
        
        try{
                
            if( n > 0L ){
                
                synchronized( m_sync.get() ){
                    
                    seek = m_seek ;
                    length = m_length ;
                    
                    if( length <= ( n + seek ) ){
                        
                        ret = length - seek ;
                        m_seek = length ;
                        
                    }
                    else{
                        
                        ret = n ;
                        m_seek += n ;
                        
                    }
                    
                }
            }
            
        }catch( Exception t ){
            ret = 0L ;
        }
        
        return ret ;
        
    }
    
    /**
     * 読み込みが可能なデータ長を取得.
     * <BR><BR>
     * 読み込みが可能なデータ長を取得します.
     * <BR>
     * @return int 読み込みが可能なデータ長が返されます.
     * @exception IOException IO例外.
     */
    public final int available() throws IOException
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = ( int )( m_length - m_seek ) ;
            }
        }catch( Exception t ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * マーク処理.
     * <BR><BR>
     * マーク処理を行います.<BR>
     * またマークした位置に戻す場合は
     * [RandomInputStream.reset()]を利用します.
     * <BR>
     * @param offset 現在の位置からのオフセット値を設定します.
     */
    public final void mark( int offset )
    {
        long seek ;
        
        try{
            
            synchronized( m_sync.get() ){
                seek = m_seek + offset ;
                
                if( seek >= 0 && seek < m_length ){
                    m_mark.add( seek ) ;
                }
            }
            
        }catch( Exception t ){
        }
    }
    
    /**
     * マーク処理に対するリセット処理.
     * <BR><BR>
     * リセット処理を行います.
     * <BR>
     * @exception IOException IO例外.
     */
    public final void reset() throws IOException
    {
        try{
            synchronized( m_sync.get() ){
                
                if( m_mark.size() == 0 ){
                    throw new IOException(
                        "マークした位置情報は存在しません"
                    ) ;
                }
                
                m_seek = m_mark.remove( 0 ) ;
                
            }
        }catch( Exception t ){
        }
    }
    
    /**
     * マーク処理サポートチェック.
     * <BR><BR>
     * マーク処理がサポートされているかチェックします.
     * <BR>
     * @return boolean マーク処理がサポートされているか返されます.<BR>
     *                 [true]が返された場合サポートされています.
     *                 [false]が返された場合サポートされていません.
     */
    public final boolean markSupported()
    {
        return true ;
    }
    
    /**
     * 現在の読み込み位置を取得.
     * <BR><BR>
     * 現在の読み込み位置を取得します.
     * <BR>
     * @return long 読み込み位置が返されます.
     */
    public final long getSeek()
    {
        long ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_seek ;
            }
        }catch( Exception t ){
            ret = -1L ;
        }
        
        return ret ;
    }
    
    /**
     * ファイルサイズを取得.
     * <BR><BR>
     * ファイルサイズを取得します.
     * <BR>
     * @return long 対象のファイルサイズを取得します.
     */
    public final long getLength()
    {
        long ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_length ;
            }
        }catch( Exception t ){
            ret = 0L ;
        }
        
        return ret ;
    }
    
    /**
     * 同期オブジェクトを取得.
     * <BR><BR>
     * 同期オブジェクトを取得します.
     * <BR>
     * @return Synchronized 同期オブジェクトが返されます.
     */
    public final Synchronized getSynchronized()
    {
        return m_sync ;
    }
    
    /**
     * マークされている情報数を取得.
     * <BR><BR>
     * 現在マークされている情報数を取得します.
     * <BR>
     * @return int マークされている情報数が返されます.
     */
    public final int getMarkSize()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_mark.size() ;
            }
        }catch( Exception t ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
}

