/*
 * @(#)RandomIO.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.io;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.util.Arrays;

import com.JRcServer.commons.exception.AccessException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.io.env.EnvRandomAccessFile;
import com.JRcServer.commons.thread.Synchronized;

/**
 * ランダムなファイルアクセスを制御します.
 * <BR><BR>
 * ランダムなファイルアクセスを制御するオブジェクトです.
 * 
 * @version 1.0.0 2004/10/26
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
public class RandomIO
{
    /**
     * setLength生成時のバッファサイズ.
     */
    private static final int ONE_BUFFER_LENGTH = 0x00010000 ;
    
    /**
     * setLength生成時のバッファマスク.
     */
    private static final long ONE_BUFFER_MASK = 0x000000000000ffffL ;
    
    /**
     * setLength生成時のバッファ整備幅.
     */
    private static final long ONE_BUFFER_SHIFT = 16L ;
    
    /**
     * ファイルポインタ.
     */
    private EnvRandomAccessFile m_fp = null ;
    
    /**
     * オープンファイル名.
     */
    private String m_name = null ;
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    /**
     * 生成用バッファ.
     */
    private final byte[] m_buf = new byte[ ONE_BUFFER_LENGTH ] ;
    
    /**
     * コンストラクタ.
     */
    public RandomIO()
    {
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * ファイル名を指定してオープンします.
     * <BR>
     * @param fileName オープン対象のファイル名を指定します.<BR>
     *                 ファイルプロトコルは[Local]条件のみ指定可能です.
     * @param newFlg ファイルオープンモードを指定します.<BR>
     *        [true]を指定した場合、ファイルが存在しない場合、新規作成を行い、<BR>
     *         ファイルが存在する場合は、１度削除して新規に作成します.<BR>
     *        [false]を指定した場合、ファイルが存在しない場合、新規作成を行い、<BR>
     *         ファイルが存在する場合は、そのファイルを利用継続してオープンします.<BR>
     * @exception FileAccessException ファイルアクセス例外.
     * @exception InputException 入力例外.
     */
    public RandomIO( String fileName,boolean newFlg )
        throws FileAccessException,InputException
    {
        try{
            this.open( fileName,newFlg ) ;
        }catch( FileAccessException fa ){
            throw fa ;
        }catch( InputException in ){
            throw in ;
        }
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.close() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * ファイルをオープンします.
     * <BR><BR>
     * ファイル名を指定してオープンします.
     * <BR>
     * @param fileName オープン対象のファイル名を指定します.y
     * @param newFlg ファイルオープンモードを指定します.<BR>
     *        [true]を指定した場合、ファイルが存在しない場合、新規作成を行い、<BR>
     *         ファイルが存在する場合は、１度削除して新規に作成します.<BR>
     *        [false]を指定した場合、ファイルが存在しない場合、新規作成を行い、<BR>
     *         ファイルが存在する場合は、そのファイルを利用継続してオープンします.<BR>
     * @exception FileAccessException ファイルアクセス例外.
     * @exception InputException 入力例外.
     */
    public final void open( String fileName,boolean newFlg )
        throws FileAccessException,InputException
    {
        int protocol ;
        long len ;
        
        if( fileName == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            this.close() ;
            m_sync.create() ;
            
            synchronized( m_sync.get() ){
                
                if( newFlg == true ){
                    if( IOCom.isFileExists( fileName ) == true ){
                        IOCom.deleteFile( fileName ) ;
                    }
                }
                
                m_fp = new EnvRandomAccessFile( fileName,"rw" ) ;
                len = m_fp.length() ;
                
                if( newFlg == false && len > 0 ){
                    m_fp.seek( len ) ;
                }
                
                m_name = fileName ;
            }
            
        }catch( NullPointerException t ){
            this.close() ;
        }catch( IllegalArgumentException ia ){
            this.close() ;
            throw new FileAccessException( ia ) ;
        }catch( FileNotFoundException fno ){
            this.close() ;
            throw new FileAccessException( fno ) ;
        }catch( AccessException ac ){
            this.close() ;
            throw new FileAccessException( ac ) ;
        }catch( IOException io ){
            this.close() ;
            throw new FileAccessException( io ) ;
        }
    }
    
    /**
     * ファイルをクローズします.
     * <BR><BR>
     * オープンしたファイルをクローズします.
     */
    public final void close()
    {
        try{
            
            synchronized( m_sync.get() ){
                
                try{
                    m_fp.close() ;
                }catch( Exception e ){
                }
                
                m_fp = null ;
                m_name = null ;
                
            }
            
        }catch( Exception io ){
            
            m_fp = null ;
            m_name = null ;
            
        }
        
        m_sync.clear() ;
    }
    
    /**
     * ファイルの読み込み処理を行います.
     * <BR><BR>
     * 現在のファイルポイントからファイル内容を取得します.
     * <BR>
     * @return byte[] 読み込まれたバイナリ情報が返されます.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public final byte[] read()
        throws InputException,FileAccessException
    {
        byte[] ret = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                ret = this.read( ( int )m_fp.length() ) ;
            }
            
        }catch( NullPointerException nul ){
            ret = null ;
        }catch( InputException in ){
            ret = null ;
            throw in ;
        }catch( FileAccessException fa ){
            ret = null ;
            throw fa ;
        }catch( IOException io ){
            ret = null ;
            throw new FileAccessException( io ) ;
        }
        
        return ret ;
    }
    
    /**
     * ファイルの読み込み処理を行います.
     * <BR><BR>
     * 指定ファイルポイントからファイル内容を取得します.
     * <BR>
     * @param size 読み込み情報サイズを指定します.
     * @return byte[] 読み込まれたバイナリ情報が返されます.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public final byte[] read( int size )
        throws InputException,FileAccessException
    {
        int len ;
        int max ;
        int pnt ;
        
        byte[] tmp = null ;
        byte[] ret = null ;
        
        if( size <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            synchronized( m_sync.get() ){
                
                tmp = new byte[ size ] ;
                len = m_fp.read( tmp ) ;
                
                if( len != size ){
                    if( len <= 0 ){
                        ret = null ;
                    }
                    else{
                        ret = new byte[ len ] ;
                        System.arraycopy( tmp,0,ret,0,len ) ;
                    }
                }
                else{
                    ret = tmp ;
                }
            }
            
        }catch( NullPointerException nul ){
            ret = null ;
        }catch( IOException io ){
            ret = null ;
            throw new FileAccessException( io ) ;
        }finally{
            tmp = null ;
        }
        
        return ret ;
    }
    
    /**
     * ファイルの読み込み処理を行います.
     * <BR><BR>
     * 指定ファイルポイントからファイル内容を取得します.
     * <BR>
     * @param out 読み込まれた情報が格納されます.
     * @return int 読み込まれたバイナリ情報数が返されます.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public final int read( byte[] out )
        throws InputException,FileAccessException
    {
        int ret ;
        int outLen ;
        int max ;
        int pnt ;
        
        byte[] tmp = null ;
        
        if( out == null || ( outLen = out.length ) <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            synchronized( m_sync.get() ){
                
                ret = m_fp.read( out ) ;
                
            }
            
        }catch( NullPointerException nul ){
            ret = 0 ;
        }catch( IOException io ){
            throw new FileAccessException( io ) ;
        }finally{
            tmp = null ;
        }
        
        return ret ;
    }
    
    /**
     * ファイルの読み込み処理を行います.
     * <BR><BR>
     * 指定ファイルポイントからファイル内容を取得します.
     * <BR>
     * @param point 読み込みファイルポイントを指定します.
     * @param size 読み込み情報サイズを指定します.
     * @return byte[] 読み込まれたバイナリ情報が返されます.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public final byte[] read( long point,int size )
        throws InputException,FileAccessException
    {
        int len ;
        int max ;
        int pnt ;
        
        byte[] tmp = null ;
        byte[] ret = null ;
        
        if( point < 0L ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            synchronized( m_sync.get() ){
                
                tmp = new byte[ size ] ;
                
                m_fp.seek( point ) ;
                len = m_fp.read( tmp ) ;
                
                if( len != size ){
                    if( len <= 0 ){
                        ret = null ;
                    }
                    else{
                        ret = new byte[ len ] ;
                        System.arraycopy( tmp,0,ret,0,len ) ;
                    }
                }
                else{
                    ret = tmp ;
                }
                
            }
            
        }catch( NullPointerException nul ){
            ret = null ;
        }catch( IOException io ){
            ret = null ;
            throw new FileAccessException( io ) ;
        }finally{
            tmp = null ;
        }
        
        return ret ;
    }
    
    /**
     * ファイルの読み込み処理を行います.
     * <BR><BR>
     * 指定ファイルポイントからファイル内容を取得します.
     * <BR>
     * @param out 読み込まれた情報が格納されます.
     * @param point 読み込みファイルポイントを指定します.
     * @return int 読み込まれたバイナリ情報数が返されます.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public final int read( byte[] out,long point )
        throws InputException,FileAccessException
    {
        int ret ;
        int outLen ;
        int max ;
        int pnt ;
        
        byte[] tmp = null ;
        
        if( out == null || ( outLen = out.length ) <= 0 || point < 0L ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            synchronized( m_sync.get() ){
                
                m_fp.seek( point ) ;
                ret = m_fp.read( out ) ;
                
            }
            
        }catch( NullPointerException nul ){
            ret = 0 ;
        }catch( IOException io ){
            throw new FileAccessException( io ) ;
        }finally{
            tmp = null ;
        }
        
        return ret ;
    }
    
    /**
     * ファイルの読み込み処理を行います.
     * <BR><BR>
     * 指定ファイルポイントからファイル内容を取得します.
     * <BR>
     * @param out 読み込まれた情報が格納されます.
     * @param point 読み込みファイルポイントを指定します.
     * @param offset 読み込みオフセット値を設定します.
     * @return int 読み込まれたバイナリ情報数が返されます.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public final int read( byte[] out,long point,int offset )
        throws InputException,FileAccessException
    {
        int len ;
        int ret ;
        int outLen ;
        int max ;
        int pnt ;
        
        byte[] tmp = null ;
        
        if( out == null || ( outLen = out.length ) <= 0 || point < 0L || offset < 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            synchronized( m_sync.get() ){
                
                len = outLen - offset ;
                m_fp.seek( point ) ;
                ret = m_fp.read( out,offset,len ) ;
                
            }
            
        }catch( NullPointerException nul ){
            ret = 0 ;
        }catch( IOException io ){
            throw new FileAccessException( io ) ;
        }finally{
            tmp = null ;
        }
        
        return ret ;
    }
    
    /**
     * ファイルの読み込み処理を行います.
     * <BR><BR>
     * 指定ファイルポイントからファイル内容を取得します.
     * <BR>
     * @param out 読み込まれた情報が格納されます.
     * @param point 読み込みファイルポイントを指定します.
     * @param offset 読み込みオフセット値を設定します.
     * @param length 読み込みファイルサイズを設定します.
     * @return int 読み込まれたバイナリ情報数が返されます.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public final int read( byte[] out,long point,int offset,int length )
        throws InputException,FileAccessException
    {
        int ret ;
        int outLen ;
        int max ;
        int pnt ;
        
        byte[] tmp = null ;
        
        if(
            out == null || ( outLen = out.length ) <= 0 ||
            point < 0L || offset < 0 || length <= 0
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            synchronized( m_sync.get() ){
                
                m_fp.seek( point ) ;
                ret = m_fp.read( out,offset,length ) ;
                
            }
            
        }catch( NullPointerException nul ){
            ret = 0 ;
        }catch( IOException io ){
            throw new FileAccessException( io ) ;
        }
        
        return ret ;
    }
    
    /**
     * ファイルの書き込み処理を行います.
     * <BR><BR>
     * 指定されたバイナリ情報をファイルに出力します.
     * <BR>
     * @param binary ファイルに書き込むバイナリ情報を指定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public final void write( byte[] binary )
        throws InputException,FileAccessException
    {
        if( binary == null || binary.length <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            synchronized( m_sync.get() ){
                m_fp.write( binary ) ;
            }
            
        }catch( NullPointerException nul ){
        }catch( IOException io ){
            throw new FileAccessException( io ) ;
        }
    }
    
    /**
     * ファイルの書き込み処理を行います.
     * <BR><BR>
     * 指定されたバイナリ情報をファイルに出力します.
     * <BR>
     * @param binary ファイルに書き込むバイナリ情報を指定します.
     * @param offset 書き込み元のバイナリ情報のオフセット値を指定します.
     * @param size 書き込み元のバイナリ情報サイズを指定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public final void write( byte[] binary,int offset,int size )
        throws InputException,FileAccessException
    {
        if(
            binary == null || binary.length <= 0 ||
            offset < 0 || size <= 0
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            synchronized( m_sync.get() ){
                m_fp.write( binary,offset,size ) ;
            }
            
        }catch( NullPointerException nul ){
        }catch( IOException io ){
            throw new FileAccessException( io ) ;
        }
    }
    
    /**
     * ファイルの書き込み処理を行います.
     * <BR><BR>
     * 指定されたバイナリ情報をファイルに出力します.
     * <BR>
     * @param binary ファイルに書き込むバイナリ情報を指定します.
     * @param point 書き込みファイルポイントを指定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public final void write( byte[] binary,long point )
        throws InputException,FileAccessException
    {
        if( binary == null || binary.length <= 0 || point < 0L ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            synchronized( m_sync.get() ){
                m_fp.seek( point ) ;
                m_fp.write( binary ) ;
            }
            
        }catch( NullPointerException nul ){
        }catch( IOException io ){
            throw new FileAccessException( io ) ;
        }
    }
    
    /**
     * ファイルの書き込み処理を行います.
     * <BR><BR>
     * 指定されたバイナリ情報をファイルに出力します.
     * <BR>
     * @param binary ファイルに書き込むバイナリ情報を指定します.
     * @param point 書き込みファイルポイントを指定します.
     * @param offset 書き込み元のバイナリ情報のオフセット値を指定します.
     * @param size 書き込み元のバイナリ情報サイズを指定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public final void write( byte[] binary,long point,int offset,int size )
        throws InputException,FileAccessException
    {
        if(
            binary == null || binary.length <= 0 ||
            offset < 0 || size <= 0 || point < 0L
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            synchronized( m_sync.get() ){
                m_fp.seek( point ) ;
                m_fp.write( binary,offset,size ) ;
            }
            
        }catch( NullPointerException nul ){
        }catch( IOException io ){
            throw new FileAccessException( io ) ;
        }
    }
    
    /**
     * ファイルポインタ位置の設定.
     * <BR><BR>
     * ファイル書き込みポインタ位置の設定を行います.
     * <BR>
     * @param point ファイルポインタ位置を設定します.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public final void setPoint( long point )
        throws FileAccessException
    {
        try{
            synchronized( m_sync.get() ){
                m_fp.seek( point ) ;
            }
        }catch( NullPointerException nul ){
        }catch( IOException io ){
            throw new FileAccessException( io ) ;
        }
    }
    
    /**
     * ファイルサイズの設定.
     * <BR><BR>
     * 現在ファイルサイズの設定を行います.
     * <BR>
     * @param length 現在オープンされているファイルサイズを設定します.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public final void setLength( long length )
        throws FileAccessException
    {
        long i ;
        long len ;
        int etc ;
        long max ;
        long nowLen ;
        long befSeek ;
        
        byte[] bin = null ;
        RandomAccessFile fp = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                fp = m_fp ;
                bin = m_buf ;
                
                if( ( nowLen = fp.length() ) < length ){
                    
                    befSeek = fp.getFilePointer() ;
                    fp.seek( nowLen ) ;
                    
                    max = length - nowLen ;
                    max = ( long )(
                        (
                            ( ( max & (~ONE_BUFFER_MASK) ) >> ONE_BUFFER_SHIFT ) +
                            ( ( ( max & ONE_BUFFER_MASK ) != 0 ) ? 1L : 0L )
                        ) * ONE_BUFFER_LENGTH
                    ) ;
                    
                    Arrays.fill( bin,( byte )0x00 ) ;
                    
                    len = ( long )( ( max & (~ONE_BUFFER_MASK) ) >> ONE_BUFFER_SHIFT ) ;
                    
                    for( i = 0 ; i < len ; i ++ ){
                        fp.write( bin,0,ONE_BUFFER_LENGTH ) ;
                    }
                    
                    m_fp.seek( befSeek ) ;
                    
                }
                
            }
            
        }catch( NullPointerException nul ){
        }catch( IOException io ){
            throw new FileAccessException( io ) ;
        }finally{
            bin = null ;
            fp = null ;
        }
    }
    
    
    
    /**
     * 現在ファイルポインタ位置の取得.
     * <BR><BR>
     * 現在設定されているファイルポインタ位置を取得します.
     * <BR>
     * @return long 現在のファイルポインタ位置が返されます.
     */
    public final long getPoint()
    {
        long ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_fp.getFilePointer() ;
            }
        }catch( Exception t ){
            ret = 0L ;
        }
        
        return ret ;
    }
    
    /**
     * ファイルサイズの取得.
     * <BR><BR>
     * 現在ファイルサイズを取得します.
     * <BR>
     * @return long 現在オープンされているファイルサイズが返されます.
     */
    public final long getLength()
    {
        long ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_fp.length() ;
            }
        }catch( Exception t ){
            ret = 0L ;
        }
        
        return ret ;
    }
    
    /**
     * 現在オープンされているファイル名の取得.
     * <BR><BR>
     * 現在オープンされているファイル名を取得します.
     * <BR>
     * @return String オープン対象のファイル名が格納されます.<BR>
     *                ファイルがオープンされていない場合[null]が返されます.
     */
    public final String getFileName()
    {
        String ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_name ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 読み込み用オブジェクトを取得.
     * <BR><BR>
     * 読み込み用オブジェクトを取得します.
     * <BR>
     * @param seek 読み込み開始位置を設定します.
     * @return InputStream 読み込み用オブジェクトが返されます.
     */
    public final InputStream getInputStream( long seek )
    {
        RandomInputStream ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = new RandomInputStream( this,seek,this.getLength() ) ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 読み込み用オブジェクトを取得.
     * <BR><BR>
     * 読み込み用オブジェクトを取得します.
     * <BR>
     * @param seek 読み込み開始位置を設定します.
     * @param length 読み込み終了長を設定します.
     * @return InputStream 読み込み用オブジェクトが返されます.
     */
    public final InputStream getInputStream( long seek,long length )
    {
        RandomInputStream ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = new RandomInputStream( this,seek,length ) ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 書き込み用オブジェクトを取得.
     * <BR><BR>
     * 書き込み用オブジェクトを取得します.
     * <BR>
     * @param seek 書き込み開始位置を設定します.
     * @return OutputStream 書き込み用オブジェクトが返されます.
     */
    public final OutputStream getOutputStream( long seek )
    {
        RandomOutputStream ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = new RandomOutputStream( this ) ;
                ret.setSeek( seek ) ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 同期オブジェクトを取得.
     * <BR><BR>
     * 同期オブジェクトを取得します.
     * <BR>
     * @return Synchronized 同期オブジェクトが返されます.
     */
    public final Synchronized getSynchronized()
    {
        return m_sync ;
    }
    
    /**
     * ファイルオープンチェック.
     * <BR><BR>
     * ファイルオープンチェックを行います.
     * <BR>
     * @return boolean ファイルオープンチェックが返されます.<BR>
     *                 [true]が返された場合、ファイルはオープン済みです.<BR>
     *                 [false]が返された場合、ファイルはオープンされていません.
     */
    public final boolean isOpen()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = ( m_fp != null ) ? true : false ;
            }
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
}

