/*
 * @(#)IOCom.java
 *
 * Copyright (c) 2003 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.io;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.CharArrayWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.net.URL;
import java.net.URLConnection;
import java.util.ArrayList;

import com.JRcServer.commons.def.BaseDef;
import com.JRcServer.commons.exception.AccessException;
import com.JRcServer.commons.exception.ExistException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.exception.NotExistException;
import com.JRcServer.commons.net.ftp.FTPClient;
import com.JRcServer.commons.sys.AutoUserPasswd;
import com.JRcServer.commons.util.MemoryIndex;
import com.JRcServer.commons.util.UtilCom;
import com.JRcServer.commons.util.array.ObjectArray;

/**
 * ファイルI/O汎用処理.
 * <BR><BR>
 * ファイル関連のI/O汎用処理を提供します.
 * <BR>
 *
 * @version 1.00, 2003/10/27
 * @author  Masahito Suzuki
 * @since   JRcCommons 1.00
 */
public class IOCom
{
    
    /**
     * プロトコル名 : FTP.
     */
    public static final String PROTOCOL_FTP = IODef.PROTOCOL_FTP ;
    
    /**
     * プロトコル名 : HTTP.
     */
    public static final String PROTOCOL_HTTP = IODef.PROTOCOL_HTTP ;
    
    /**
     * プロトコル名 : HTTPS.
     */
    public static final String PROTOCOL_HTTPS = IODef.PROTOCOL_HTTPS ;
    
    
    /**
     * プロトコルタイプ : 通常のローカルファイル.
     */
    public static final int TYPE_PROTOCOL_LOCAL = IODef.TYPE_PROTOCOL_LOCAL ;
    
    /**
     * プロトコルタイプ : FTP.
     */
    public static final int TYPE_PROTOCOL_FTP = IODef.TYPE_PROTOCOL_FTP ;
    
    /**
     * プロトコルタイプ : HTTP.
     */
    public static final int TYPE_PROTOCOL_HTTP = IODef.TYPE_PROTOCOL_HTTP ;
    
    /**
     * プロトコルタイプ : HTTPS.
     */
    public static final int TYPE_PROTOCOL_HTTPS = IODef.TYPE_PROTOCOL_HTTPS ;
    
    /**
     * デフォルトFTPポート番号.
     */
    public static final int FTP_PORT = IODef.FTP_PORT ;
    
    /**
     * デフォルトHTTPポート番号.
     */
    public static final int HTTP_PORT = IODef.HTTP_PORT ;
    
    /**
     * デフォルトHTTPSポート番号.
     */
    public static final int HTTPS_PORT = IODef.HTTPS_PORT ;
    
    /**
     * JAR拡張子長.
     */
    public static final int JAR_LENGTH = ".jar".length() ;
    
    /**
     * ZIP拡張子長.
     */
    public static final int ZIP_LENGTH = ".zip".length() ;
    
    
    
    /**
     * ファイル名＋ディレクトリ名最大長.
     */
    private static final int MAX_FNAME_LENGTH = 260 ;
    
    /**
     * １度に読み込むファイルサイズ.
     */
    private static final int LOAD_FILE_LENGTH = 512 ;
    
    /**
     * EOFコード.
     */
    private static final int EOF_CODE = IODef.EOF ;
    
    /**
     * ダミーファイル名長 : 最大値.
     */
    private static final int MAX_DUMMY_NAME_LENGTH = ( IOCom.MAX_FNAME_LENGTH / 2 ) ;
    
    
    /**
     * ファイル同一名解決-1.
     */
    private static final String FILE_EQUAL_1 = "-copy[" ;
    
    /**
     * ファイル同一名解決-2.
     */
    private static final String FILE_EQUAL_2 = "]-" ;
    
    
    
    /**
     * ファイル操作処理用同期.
     */
    private static final Object FILE_SYNC = new Object() ;
    
    /**
     * コンストラクタ.
     */
    private IOCom()
    {}
    
    /**
     * 対象のディレクトリを生成.
     * <BR><BR>
     * 対象のディレクトリ情報を生成します.<BR>
     * このメソッドでの生成方法として、fullpath,dirName の２つの
     * 引数を用いて、処理を行います.<BR>
     * たとえば、「fullpath = "c:\\test"」で、「dirName = "1"」の場合、
     * 引数 fullpath は、ディレクトリとして存在する必要があります.<BR>
     * 生成結果として、ディレクトリ 「c;\\test\\1」が生成されます.
     *
     * <BR>
     * @param fullpath 生成を行う先頭ディレクトリ名を
     *                 フルパスで設定します.
     * @param dirName 生成対象のディレクトリ名を設定します.
     * @exception InputException 入力例外.
     * @exception ExistException 存在例外.
     * @exception NotExistException 非存在例外.
     * @exception DirectoryAccessException ディレクトリアクセス例外.
     */
    public static final void mkdir( String fullpath,String dirName )
        throws
            InputException,ExistException,
            NotExistException,DirectoryAccessException
    {
        File fp = null ;
        StringBuffer buf = null ;
        String full = null ;
        String all = null ;
        
        if( fullpath == null || dirName == null ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        try{
            
            dirName = IODef.convertEnvName( dirName ) ;
            
            if( IOCom.isDirExists( fullpath ) == false ){
                
                throw new NotExistException( 
                    "フルパスディレクトリ(" + fullpath +
                    ")は存在しません"
                 ) ;
                
            }
            
            full = IOCom.getFullPath( fullpath ) ;
            
            buf = new StringBuffer() ;
            buf.append( full ) ;
            buf.append( BaseDef.FILE_SEPARATOR ) ;
            buf.append( dirName ) ;
            
            all = buf.toString() ;
            
            buf = null ;
            
            if( IOCom.isDirExists( all ) == true ){
                
                throw new ExistException( 
                    "生成対象のディレクトリ( fullpath : [" + full +
                    "] : new Dir :[" + dirName + 
                    "] )はすでに存在します"
                 ) ;
                
            }
            
            fp = new File( all ) ;
            
            if( fp.mkdir() == false ){
                throw new DirectoryAccessException( 
                    "ディレクトリ(" + all +
                    ")の作成に失敗しました"
                ) ;
            }
            
        }catch( ExistException ee ){
            throw ee ;
        }catch( NotExistException ne ){
            throw ne ;
        }catch( DirectoryAccessException de ){
            throw de ;
        }finally{
            
            fp = null ;
            full = null ;
            all = null ;
            buf = null ;
            
        }
        
    }
    
    /**
     * 対象のディレクトリを生成.
     * <BR><BR>
     * 対象のディレクトリ情報を生成します.
     * <BR>
     * @param dirName 生成対象のディレクトリ名を設定します.
     * @return String 結果のファイル名が返されます.
     * @exception InputException 入力例外.
     * @exception ExistException 存在例外.
     * @exception DirectoryAccessException ディレクトリアクセス例外.
     */
    public static final String mkdirs( String dirName )
        throws InputException,ExistException,DirectoryAccessException
    {
        File fp = null ;
        StringBuffer buf = null ;
        String tmp = null ;
        String ret = null ;
        
        try{
            
            if( dirName == null ){
                throw new InputException( "引数は不正です" ) ;
            }
            
            dirName = IODef.convertEnvName( dirName ) ;
            
            if( IOCom.isDirExists( dirName ) == true ){
                
                fp = new File( dirName ) ;
                tmp = fp.getCanonicalPath() ;
                buf = new StringBuffer() ;
                
                buf.append( tmp ) ;
                buf.append( BaseDef.FILE_SEPARATOR ) ;
                ret = buf.toString() ;
                
            }
            else{
                
                fp = new File( dirName ) ;
                tmp = fp.getCanonicalPath() ;
                buf = new StringBuffer() ;
                
                buf.append( tmp ) ;
                buf.append( BaseDef.FILE_SEPARATOR ) ;
                ret = buf.toString() ;
                
                if( fp.mkdirs() == false ){
                    throw new DirectoryAccessException( 
                        "ディレクトリ(" + dirName +
                        ")の作成に失敗しました"
                    ) ;
                }
                
            }
            
        }catch( InputException in ){
            throw in ;
        }catch( DirectoryAccessException de ){
            throw de ;
        }catch( Exception t ){
            throw new DirectoryAccessException( t ) ;
        }finally{
            fp = null ;
            buf = null ;
            tmp = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象のキーワードに対する紐付け情報を取得.
     * <BR><BR>
     * 対象のキーワードに対する紐付け情報を取得します.<BR>
     * たとえば、対象ファイル名内に以下の内容が格納されていると
     * します.<BR>
     * <PRE>
     * keyword = "12345"
     * </PRE>
     * そして、この[keyword]の紐付け情報を取得したい場合、
     * <PRE>
     * String val = IOCom.getValue( "ファイル名","keyword" ) ;
     * </PRE>
     * と指定することにより、戻り値のStringに紐付け情報が格納されます.<BR>
     * <BR>
     * @param fileName 対象のファイル名を設定します.
     * @param key 対象のキーワード情報を設定します.
     * @return String キーワードに対する紐付け情報が返されます.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public static final String getValue( String fileName,String key )
        throws InputException,FileAccessException
    {
        int i ;
        int st ;
        int len ;
        int mode ;
        int type ;
        char cote ;
        
        char[] oneLine = null ;
        
        BufferedReader in = null ;
        String tmp = null ;
        String wordLine = null ;
        StringBuffer buf = null ;
        String ret = null ;
        
        if( fileName == null || key == null ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        key.toLowerCase()  ;
        
        try{
            
            fileName = IODef.convertEnvName( fileName ) ;
            type = IODef.getNameByProtocolType( fileName ) ;
            
            if( type == IOCom.TYPE_PROTOCOL_LOCAL ){
                
                in = new BufferedReader( 
                    new InputStreamReader(  
                        new FileInputStream( fileName )
                    )
                ) ;
                
            }
            else{
                
                throw new InputException(
                    "対象のファイル指定(" + fileName +
                    ")は不正です"
                ) ;
                
            }
            
            for( ; ( tmp = in.readLine() ) != null ; ){
                
                wordLine = tmp ;
                tmp.toLowerCase() ;
                
                if( ( st = tmp.indexOf( key ) ) != -1 ){
                    
                    len = tmp.length() ;
                    
                    oneLine = new char[ len ] ;
                    wordLine.getChars( 0,len,oneLine,0 ) ;
                    buf = new StringBuffer() ;
                    
                    st += key.length() ;
                    mode = 0 ;
                    cote = 0 ;
                    ret = null ;
                    
                    for( i = st ; i < len ; i ++ ){
                        
                        if( mode == 0 && oneLine[ i ] == '=' ){
                            ret = null ;
                            mode = 1 ;
                        }
                        else if( mode == 1 ){
                            if( cote == 0 &&
                                ( oneLine[ i ] == '\"' || oneLine[ i ] == '\'' )
                            )
                            {
                                ret = null ;
                                cote = oneLine[ i ] ;
                                mode = 2 ;
                            }
                            else if(
                                oneLine[ i ] != ' ' &&
                                oneLine[ i ] != '\t' &&
                                oneLine[ i ] != '　' &&
                                oneLine[ i ] != '<' &&
                                oneLine[ i ] != '>' &&
                                oneLine[ i ] != '[' &&
                                oneLine[ i ] != ']' &&
                                oneLine[ i ] != '(' &&
                                oneLine[ i ] != ')'
                            )
                            {
                                i -- ;
                                ret = null ;
                                cote = 0 ;
                                mode = 2 ;
                            }
                            
                        }else if( mode == 2 ){
                            
                            if( cote != 0 && oneLine[ i ] == cote ){
                                cote = 0 ;
                                ret = UtilCom.trimPlus( buf.toString() ) ;
                                buf = null ;
                                break ;
                            }
                            else if(
                                cote == 0 &&
                                (
                                    oneLine[ i ] == ' ' ||
                                    oneLine[ i ] == '\t' ||
                                    oneLine[ i ] == '　' ||
                                    oneLine[ i ] == '\"' ||
                                    oneLine[ i ] == '\'' ||
                                    oneLine[ i ] == '<' ||
                                    oneLine[ i ] == '>' ||
                                    oneLine[ i ] == '[' ||
                                    oneLine[ i ] == ']' ||
                                    oneLine[ i ] == '(' ||
                                    oneLine[ i ] == ')' ||
                                    oneLine[ i ] == '\r' ||
                                    oneLine[ i ] == '\n'
                                )
                            )
                            {
                                cote = 0 ;
                                ret = UtilCom.trimPlus( buf.toString() ) ;
                                buf = null ;
                                break ;
                            }
                            else{
                                buf.append( oneLine[ i ] ) ;
                            }
                        }
                        
                    }
                    
                    if( mode == 0 || cote != 0 ){
                        
                        buf = null ;
                        oneLine = null ;
                        continue ;
                        
                    }
                    else if( mode == 1 ){
                        ret = "" ;
                    }
                    else if( mode == 2 && ret == null && buf != null ){
                        ret = UtilCom.trimPlus( buf.toString() ) ;
                    }
                    
                    break ;
                }
                
            }
            
        }catch( IOException io ){
            throw new FileAccessException( io ) ;
        }finally{
            
            try{
                in.close() ;
            }catch( Exception t ){
            }
            
            in = null ;
            tmp = null ;
            wordLine = null ;
            buf = null ;
            oneLine = null ;
            
        }
        
        return ret ;
    }
    
    /**
     * 指定ファイルから、行単位で情報を格納.
     * <BR><BR>
     * 指定ファイルから、行単位で情報を格納します.
     * また、この情報はファイル情報をメモリに蓄えるため、
     * 巨大なファイルを開くことはあまりお勧めできません.<BR>
     * <BR>
     * @param fileName 読み込み対象のファイル名を指定します.
     * @return MemoryIndex 読み込まれた情報が返されます.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public static final MemoryIndex readFileToLine( String fileName )
        throws InputException,FileAccessException
    {
        MemoryIndex ret = null ;
        
        try{
            
            ret = new MemoryIndex() ;
            IOCom.readFileToLine( ret,fileName,null ) ;
            
        }catch( InputException in ){
            
            try{
                ret.clear() ;
            }catch( Exception t ){
            }
            ret = null ;
            
            throw in ;
        }catch( FileAccessException fa ){
            
            try{
                ret.clear() ;
            }catch( Exception t ){
            }
            ret = null ;
            
            throw fa ;
        }
        
        return ret ;
    }
    
    /**
     * 指定ファイルから、行単位で情報を格納.
     * <BR><BR>
     * 指定ファイルから、行単位で情報を格納します.
     * また、この情報はファイル情報をメモリに蓄えるため、
     * 巨大なファイルを開くことはあまりお勧めできません.<BR>
     * <BR>
     * @param fileName 読み込み対象のファイル名を指定します.
     * @param encode 読み込みエンコード名を設定します.<BR>
     *               デフォルトエンコーディングで読み込む場合[null]を
     *               指定します.
     * @return MemoryIndex 読み込まれた情報が返されます.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public static final MemoryIndex readFileToLine( String fileName,String encode )
        throws InputException,FileAccessException
    {
        MemoryIndex ret = null ;
        
        try{
            
            ret = new MemoryIndex() ;
            IOCom.readFileToLine( ret,fileName,encode ) ;
            
        }catch( InputException in ){
            
            try{
                ret.clear() ;
            }catch( Exception t ){
            }
            ret = null ;
            
            throw in ;
        }catch( FileAccessException fa ){
            
            try{
                ret.clear() ;
            }catch( Exception t ){
            }
            ret = null ;
            
            throw fa ;
        }
        
        return ret ;
        
    }
    
    /**
     * 指定ファイルから、行単位で情報を格納.
     * <BR><BR>
     * 指定ファイルから、行単位で情報を格納します.
     * また、この情報はファイル情報をメモリに蓄えるため、
     * 巨大なファイルを開くことはあまりお勧めできません.<BR>
     * <BR>
     * @param out 取得された情報が格納されます.
     * @param fileName 読み込み対象のファイル名を指定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public static final void readFileToLine( MemoryIndex out,String fileName )
        throws InputException,FileAccessException
    {
        try{
            
            IOCom.readFileToLine( out,fileName,null ) ;
            
        }catch( InputException in ){
            throw in ;
        }catch( FileAccessException fa ){
            throw fa ;
        }
    }
    
    /**
     * 指定ファイルから、行単位で情報を格納.
     * <BR><BR>
     * 指定ファイルから、行単位で情報を格納します.
     * また、この情報はファイル情報をメモリに蓄えるため、
     * 巨大なファイルを開くことはあまりお勧めできません.<BR>
     * <BR>
     * @param out 取得された情報が格納されます.
     * @param fileName 読み込み対象のファイル名を指定します.
     * @param encode 読み込みエンコード名を設定します.<BR>
     *               デフォルトエンコーディングで読み込む場合[null]を
     *               指定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public static final void readFileToLine( MemoryIndex out,String fileName,String encode )
        throws InputException,FileAccessException
    {
        int type ;
        
        BufferedReader in = null ;
        String tmp = null ;
        
        if( fileName == null || out == null ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        out.clear() ;
        
        try{
            
            fileName = IODef.convertEnvName( fileName ) ;
            type = IODef.getNameByProtocolType( fileName ) ;
            
            if( type == IOCom.TYPE_PROTOCOL_LOCAL ){
                
                if( encode == null ){
                    
                    in = new BufferedReader( 
                        new InputStreamReader( 
                            new FileInputStream( fileName )
                         )
                     ) ;
                    
                }else{
                    
                    in = new BufferedReader( 
                        new InputStreamReader( 
                            new FileInputStream( fileName ),
                            encode
                         )
                     ) ;
                    
                }
                
            }
            else{
                
                throw new InputException(
                    "対象のファイル指定(" + fileName +
                    ")は不正です"
                ) ;
                
            }
            
            for( ; ( tmp = in.readLine() ) != null ; ){
                out.addIndex( tmp ) ;
            }
            
        }catch( IOException io ){
            throw new FileAccessException( io ) ;
        }finally{
            
            try{
                in.close() ;
            }catch( Exception t ){
            }
            
            in = null ;
            tmp = null ;
        }
        
    }
    
    /**
     * 指定ファイルに行単位で、情報を書き込む.
     * <BR><BR>
     * 指定ファイルに対して、行単位で情報を書き込みます.
     * <BR>
     * @param in 書き込み対象の情報が格納されます.
     * @param fileName 書き込み対象のファイル名を指定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public static final void writeFileToLine( ObjectArray in,String fileName )
        throws InputException,FileAccessException
    {
        try{
            
            IOCom.writeFileToLine( in,fileName,null ) ;
            
        }catch( InputException inp ){
            throw inp ;
        }catch( FileAccessException fa ){
            throw fa ;
        }
    }
    
    /**
     * 指定ファイルに行単位で、情報を書き込む.
     * <BR><BR>
     * 指定ファイルに対して、行単位で情報を書き込みます.
     * <BR>
     * @param in 書き込み対象の情報が格納されます.
     * @param fileName 書き込み対象のファイル名を指定します.
     * @param encode 書き込みエンコード名を設定します.<BR>
     *               デフォルトエンコーディングで書き込む場合[null]を
     *               指定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public static final void writeFileToLine( ObjectArray in,String fileName,String encode )
        throws InputException,FileAccessException
    {
        int i ;
        int len ;
        
        BufferedWriter out = null ;
        
        if( fileName == null || in == null ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        try{
            
            fileName = IODef.convertEnvName( fileName ) ;
            
            if( encode == null ){
                
                out = new BufferedWriter(
                    new OutputStreamWriter(
                        new FileOutputStream( fileName )
                    )
                ) ;
                
            }else{
                
                out = new BufferedWriter(
                    new OutputStreamWriter(
                        new FileOutputStream( fileName ),
                        encode
                    )
                ) ;
                
            }
            
            len = in.size() ;
            for( i = 0 ; i < len ; i ++ ){
                
                out.write( ( String )in.get( i ) ) ;
                out.newLine() ;
                
            }
            out.flush() ;
            
        }catch( IOException io ){
            throw new FileAccessException( io ) ;
        }finally{
            
            try{
                out.flush() ;
            }catch( Exception t ){
            }
            try{
                out.close() ;
            }catch( Exception t ){
            }
            
            out = null ;
        }
        
    }
    
    /**
     * 対象のダミーファイルを生成.
     * <BR><BR>
     * 対象のダミーファイルを生成します.
     * また、生成したダミーファイル名は、指定ディレクトリに対して
     * 一意な情報として生成されます.
     * <BR>
     * @param path 生成対象のフルパス名を設定します.
     * @param length 生成されるダミーファイル名長を生成します.
     * @return String 生成されたダミーファイル名が返されます.
     */
    public static final String getDummyFile( String path,int length )
    {
        
        StringBuffer buf = null ;
        String fullPath = null ;
        String rndName = null ;
        String ret = null ;
        
        fullPath = IOCom.getFullPath( path ) ;
        
        for( ;; ){
            
            buf = new StringBuffer() ;
            
            rndName = UtilCom.randomString( length,true,true,true ) ;
            
            buf.append( fullPath ) ;
            buf.append( rndName ) ;
            
            ret = null ;
            ret = buf.toString() ;
            buf = null ;
            
            if( IOCom.isFileExists( ret ) == false ){
                break ;
            }
            
        }
        
        
        fullPath = null ;
        rndName = null ;
        buf = null ;
        
        return ret ;
    }
    
    /**
     * ファイル名に対するフルパス名の取得.
     * <BR><BR>
     * 指定ファイル名に対するフルパス情報を取得します.<BR>
     * このメソッドで取得した情報には、ファイル名は含みません.
     * <BR>
     * たとえば以下の情報の場合は、<BR>
     * [c:\program\test.java] の場合 [c:\program\]<BR>
     * のみが取得対象となります.
     * <BR>
     * @param fileName ファイル名を設定します.
     * @return String フルパス情報が返されます.
     */
    public static final String getFullPath( String fileName )
    {
        int i ;
        int len ;
        
        ArrayList path = null ;
        StringBuffer buf = null ;
        String fullPathName = null ;
        File file = null ;
        
        String ret = null ;
        
        if( fileName == null ){
            return null ;
        }
        
        try{
            
            fileName = IODef.convertEnvName( fileName ) ;
            file = new File( fileName ) ;
            fullPathName = file.getCanonicalPath() ;
            
            if( file.isDirectory() == false ){
                
                path = UtilCom.pauseString( fullPathName,"/\\" ) ;
                len = path.size() ;
                
                buf = new StringBuffer() ;
                
                buf.append( BaseDef.FILE_SEPARATOR ) ;
                buf.append( path.get( 0 ) ) ;
                buf.append( BaseDef.FILE_SEPARATOR ) ;
                
                for( i = 1 ; i < len ; i ++ ){
                    
                    if( IOCom.isDirExists( buf.toString() ) == false ){
                        break ;
                    }
                    
                    ret = buf.toString() ;
                    
                    buf.append( path.get( i ) ) ;
                    buf.append( BaseDef.FILE_SEPARATOR ) ;
                    
                }
                
            }else{
                
                buf = new StringBuffer() ;
                buf.append( fullPathName ) ;
                buf.append( BaseDef.FILE_SEPARATOR ) ;
                ret = buf.toString() ;
                
            }
            
            file = new File( ret ) ;
            ret = file.getCanonicalPath() ;
            ret += BaseDef.FILE_SEPARATOR ;
            
        }catch( IOException io ){
            ret = null ;
        }catch( InputException in ){
            ret = null ;
        }finally{
            
            try{
                path.clear() ;
            }catch( Exception t ){
            }
            
            buf = null ;
            fullPathName = null ;
            file = null ;
            
            path = null ;
        }
        
        return ret ;
    }
    
    /**
     * ファイル名に対するファイル名を取得.
     * <BR><BR>
     * ファイル名に対するファイル名を取得します.
     * このメソッドは、以下の情報<BR>
     * [c:\program\test.java] の場合 [test.java]<BR>
     * のみが取得対象となります.
     * <BR>
     * @param fileName ファイル名を設定します.
     * @return String ファイル名が返されます.
     */
    public static final String getFileName( String fileName )
    {
        int i ;
        int len ;
        
        ArrayList path = null ;
        String fullPathName = null ;
        File file = null ;
        
        String ret = null ;
        
        if( fileName == null ){
            return null ;
        }
        
        
        try{
            
            fileName = IODef.convertEnvName( fileName ) ;
            file = new File( fileName ) ;
            
            fullPathName = file.getCanonicalPath() ;
            
            if( file.isDirectory() == false ){
                
                path = UtilCom.pauseString( fullPathName,"\\/" ) ;
                    
                len = path.size() - 1 ;
                
                ret = ( String )path.get( len );
                
            }else{
                
                ret = BaseDef.FILE_SEPARATOR ;
                
            }
            
        }catch( IOException io ){
        
            ret = null ;
            
        }catch( InputException in ){
            
            ret = null ;
            
        }finally{
            
            fullPathName = null ;
            file = null ;
            if( path != null ){
                path.clear() ;
            }
            path = null ;
        }
        
        return ret ;
    }
    
    /**
     * ファイル名に対するフルパス+ファイル名を取得.
     * <BR><BR>
     * ファイル名に対するフルパス+ファイル名を取得します.
     * このメソッドは、以下の情報<BR>
     * [c:\program\test.java] の場合 [c:\program\test.java]<BR>
     * となります.
     * <BR>
     * @param fileName フルパス+ファイル名を設定します.
     * @return String フルパス+ファイル名が返されます.
     */
    public static final String getName( String fileName )
    {
        int i ;
        int len ;
        
        ArrayList path = null ;
        String fullPathName = null ;
        File file = null ;
        StringBuffer buf = null ;
        
        String ret = null ;
        
        if( fileName == null ){
            return null ;
        }
        
        
        try{
            
            fileName = IODef.convertEnvName( fileName ) ;
            file = new File( fileName ) ;
            
            fullPathName = file.getCanonicalPath() ;
            
            path = UtilCom.pauseString( fullPathName,"\\/" ) ;
            if( path != null && ( len = path.size() ) > 0 ) {
                
                buf = new StringBuffer() ;
                for( i = 0 ; i < len ; i ++ ) {
                    
                    buf.append( BaseDef.FILE_SEPARATOR ) ;
                    buf.append( path.get( i ) ) ;
                    
                }
                
                file = new File( buf.toString() ) ;
                buf = null ;
                ret = file.getCanonicalPath() ;
                
            }
            else{
                
                ret = fullPathName ;
                
            }
            
        }catch( Exception io ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 指定ファイルのコピー.
     * <BR><BR>
     * 指定ファイルをコピーします.
     * <BR>
     * @param srcName コピー元のファイル名を指定します.
     * @param destName コピー先のファイル名を指定します.
     * @return String コピー先のファイル名が返されます.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public static final String copyFile( String srcName,String destName )
        throws InputException,FileAccessException
    {
        int i ;
        int len ;
        
        byte[] bin = null ;
        
        String srcFullName = null ;
        String destFullName = null ;
        
        BufferedInputStream in = null ;
        BufferedOutputStream out = null ;
        
        String name = null ;
        String fullPath = null ;
        StringBuffer buf = null ;
        
        String ret = null ;
        
        if( srcName == null || destName == null ){
            throw new InputException( "指定された引数は不正です" ) ;
        }
        
        try{
            
            fullPath = IOCom.getFullPath( srcName ) ;
            name = IOCom.getFileName( srcName ) ;
            
            buf = new StringBuffer() ;
            buf.append( fullPath ) ;
            buf.append( name ) ;
            
            srcFullName = buf.toString() ;
            
            if( IOCom.isFileExists( srcFullName ) == false ){
                throw new InputException( "コピー元のファイル名[" + srcFullName + "]は存在しません" ) ;
            }
            
            fullPath = IOCom.getFullPath( destName ) ;
            name = IOCom.getFileName( destName ) ;
            
            buf = new StringBuffer() ;
            buf.append( fullPath ) ;
            buf.append( name ) ;
            
            destFullName = buf.toString() ;
            
            synchronized( IOCom.FILE_SYNC ){
                
                if( IOCom.isFileExists( destFullName ) == true ){
                    
                    for( i = 1 ; ; i ++ ){
                        
                        buf = null ;
                        buf = new StringBuffer() ;
                        
                        destFullName = null ;
                        
                        buf.append( fullPath ) ;
                        buf.append( FILE_EQUAL_1 ) ;
                        buf.append( i ) ;
                        buf.append( FILE_EQUAL_2 ) ;
                        buf.append( name ) ;
                        
                        destFullName = buf.toString() ;
                        
                        if( IOCom.isFileExists( destFullName ) == false ){
                            break ;
                        }
                        
                    }
                }
                
                in = new BufferedInputStream(
                    new FileInputStream(
                        srcFullName
                    )
                ) ;
                
                out = new BufferedOutputStream(
                     new FileOutputStream(
                        destFullName
                    )
                ) ;
                
            }
            
            bin = new byte[ IOCom.LOAD_FILE_LENGTH ] ;
            for( ;; ){
                
                if(
                    ( len = in.read( bin,0,IOCom.LOAD_FILE_LENGTH ) ) == IOCom.EOF_CODE
                )
                {
                    break ;
                }
                
                out.write( bin,0,len ) ;
                
            }
            
            ret = destFullName ;
            
        }catch( InputException inp ){
            throw inp ;
        }catch( NullPointerException nul ){
            throw new InputException( nul ) ;
        }catch( IOException io ){
            throw new FileAccessException( io ) ;
        }finally{
            
            try{
                in.close() ;
            }catch( Exception t ){
            }
            
            try{
                out.flush() ;
            }catch( Exception t ){
            }
            
            try{
                out.close() ;
            }catch( Exception t ){
            }
            
            bin = null ;
            
            srcFullName = null ;
            destFullName = null ;
            
            in = null ;
            out = null ;
            
            name = null ;
            fullPath = null ;
            
        }
        
        return ret ;
    }
    
    /**
     * 指定ファイルの移動.
     * <BR><BR>
     * 指定ファイルを移動します.
     * <BR>
     * @param srcName 移動元のファイル名を指定します.
     * @param destName 移動先のファイル名を指定します.
     * @return String 移動先のファイル名が返されます.
     * @exception InputException 入力例外.
     */
    public static final String moveFile( String srcName,String destName )
        throws InputException
    {
        int i ;
        
        String fullPath = null ;
        String name = null ;
        String renameName = null ;
        StringBuffer buf = null ;
        
        String ret = null ;
        
        File src = null ;
        File dest = null ;
        File dummy = null ;
        
        if( srcName == null || destName == null ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        try{
            
            fullPath = IOCom.getFullPath( srcName ) ;
            name = IOCom.getFileName( srcName ) ;
            
            buf = new StringBuffer() ;
            buf.append( fullPath ) ;
            buf.append( name ) ;
            
            renameName = buf.toString() ;
            
            src = new File( renameName ) ;
            dummy = new File( renameName ) ;
            
            if( src.exists() == false ){
                throw new InputException( "変更元のファイル名[" + renameName + "]は存在しません" ) ;
            }
            
            fullPath = IOCom.getFullPath( destName ) ;
            name = IOCom.getFileName( destName ) ;
            
            buf = new StringBuffer() ;
            buf.append( fullPath ) ;
            buf.append( name ) ;
            
            renameName = buf.toString() ;
            dest = new File( renameName ) ;
            
            src.renameTo( dest ) ;
            
            synchronized( IOCom.FILE_SYNC ){
                
                if( dummy.exists() == true ){
                    
                    for( i = 1 ; ; i ++ ){
                        
                        buf = null ;
                        buf = new StringBuffer() ;
                        
                        renameName = null ;
                        dest = null ;
                        
                        buf.append( fullPath ) ;
                        buf.append( FILE_EQUAL_1 ) ;
                        buf.append( i ) ;
                        buf.append( FILE_EQUAL_2 ) ;
                        buf.append( name ) ;
                        
                        renameName = buf.toString() ;
                        
                        dest = new File( renameName ) ;
                        
                        src.renameTo( dest ) ;
                        
                        if( dummy.exists() == false ){
                            break ;
                        }
                    }
                }
                
            }
            
            ret = renameName ;
            
        }catch( InputException in ){
            
            ret = null ;
            throw in ;
            
        }catch( NullPointerException nul ){
            
            ret = null ;
            throw new InputException( nul ) ;
            
        }finally{
            
            fullPath = null ;
            name = null ;
            renameName = null ;
            src = null ;
            dest = null ;
            dummy = null ;
            buf = null ;
            
        }
        
        return ret ;
    }
    
    /**
     * 指定ファイル名の変更.
     * <BR><BR>
     * 指定ファイル名を変更します.
     * <BR>
     * @param srcName 変更元のファイル名を指定します.
     * @param destName 変更先のファイル名を指定します.
     * @return String 変更されたファイル名が返されます.
     * @exception InputException 入力例外.
     */
    public static final String renameFile( String srcName,String destName )
        throws InputException
    {
        int i ;
        
        String fullPath = null ;
        String name = null ;
        String renameName = null ;
        StringBuffer buf = null ;
        
        String ret = null ;
        
        File src = null ;
        File dest = null ;
        File dummy = null ;
        
        if( srcName == null || destName == null ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        try{
            
            fullPath = IOCom.getFullPath( srcName ) ;
            name = IOCom.getFileName( srcName ) ;
            
            buf = new StringBuffer() ;
            buf.append( fullPath ) ;
            buf.append( name ) ;
            
            renameName = buf.toString() ;
            
            src = new File( renameName ) ;
            dummy = new File( renameName ) ;
            
            if( src.exists() == false ){
                throw new InputException( "変更元のファイル名[" + renameName + "]は存在しません" ) ;
            }
            
            name = IOCom.getFileName( destName ) ;
            
            buf = new StringBuffer() ;
            buf.append( fullPath ) ;
            buf.append( name ) ;
            
            renameName = buf.toString() ;
            dest = new File( renameName ) ;
            
            src.renameTo( dest ) ;
            
            synchronized( IOCom.FILE_SYNC ){
                
                if( dummy.exists() == true ){
                    
                    for( i = 1 ; ; i ++ ){
                        
                        buf = null ;
                        buf = new StringBuffer() ;
                        
                        renameName = null ;
                        dest = null ;
                        
                        buf.append( fullPath ) ;
                        buf.append( FILE_EQUAL_1 ) ;
                        buf.append( i ) ;
                        buf.append( FILE_EQUAL_2 ) ;
                        buf.append( name ) ;
                        
                        renameName = buf.toString() ;
                        
                        dest = new File( renameName ) ;
                        
                        src.renameTo( dest ) ;
                        
                        if( dummy.exists() == false ){
                            break ;
                        }
                        
                    }
                    
                }
                
            }
            
            ret = renameName ;
            
        }catch( InputException in ){
            
            ret = null ;
            throw in ;
            
        }catch( NullPointerException nul ){
            
            ret = null ;
            throw new InputException( nul ) ;
            
        }finally{
            
            fullPath = null ;
            name = null ;
            renameName = null ;
            src = null ;
            dest = null ;
            dummy = null ;
            buf = null ;
            
        }
        
        return ret ;
    }
    
    /**
     * 指定ファイルの削除.
     * <BR><BR>
     * 指定ファイルの削除処理を行います.
     * <BR>
     * @param fileName 削除対象のファイル名を指定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public static final void deleteFile( String fileName )
        throws InputException,FileAccessException
    {
        if( fileName == null ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        fileName = IODef.convertEnvName( fileName ) ;
        File file = new File( fileName ) ;
        
        if( IOCom.isWrite( fileName ) == true ){
            if( file.delete() == false ){
                
                file = null ;
                throw new FileAccessException(
                    "ファイル[" + fileName + "]の削除に失敗しました"
                ) ;
                
            }
        }
        
        file = null ;
    }
    
    /**
     * 指定ディレクトリ内のディレクトリ／ファイルリストを取得.
     * <BR><BR>
     * 指定ディレクトリ内のディレクトリ／ファイルリストを取得します.
     * <BR>
     * @param dirName ファイルリスト取得対象のディレクトリ名を指定します.
     * @return String[] 指定ディレクトリ名内のファイルリストを取得します.<BR>
     *                  [null]が返された場合、情報は存在しません.
     * @exception InputException 入力例外.
     */
    public static final String[] getList( String dirName )
        throws InputException
    {
        int i ;
        int len ;
        int bufLen ;
        
        File fp = null ;
        StringBuffer buf = null ;
        String[] ret = null ;
        
        if( dirName == null ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        try{
            
            if( IOCom.isDirExists( dirName ) == false ){
                
                throw new InputException( "指定ディレクトリ名[" + dirName +
                    "]はディレクトリ名ではありません" ) ;
                
            }
            
            dirName = IOCom.getFullPath( dirName ) ;
            fp = new File( dirName ) ;
            
            ret = fp.list() ;
            if( ret != null ){
                
                bufLen = dirName.length() ;
                len = ret.length ;
                
                for( i = 0 ; i < len ; i ++ ){
                    
                    buf = new StringBuffer( bufLen ) ;
                    buf.append( dirName ) ;
                    buf.append( ret[ i ] ) ;
                    ret[ i ] = buf.toString() ;
                    
                    buf = null ;
                }
                
            }
            
        }catch( InputException in ){
            throw in ;
        }finally{
            fp = null ;
            buf = null ;
        }
        
        return ret ;
    }
    
    /**
     * 指定ファイル名のファイルサイズを取得.
     * <BR><BR>
     * 指定ファイル名のファイルサイズを取得します.<BR>
     * <BR>
     * @param fileName ファイルサイズ取得対象のファイル名を指定します.
     * @return long ファイルサイズが返されます.
     */
    public static final long getFileLength( String fileName )
    {
        int type ;
        long ret ;
        
        File file = null ;
        
        if( fileName == null ){
            return 0L ;
        }
        
        try{
            
            fileName = IODef.convertEnvName( fileName ) ;
            type = IODef.getNameByProtocolType( fileName ) ;
            
            if( type == IOCom.TYPE_PROTOCOL_LOCAL ){
                
                file = new File( fileName ) ;
                ret = ( file.exists() == true ) ? 
                    file.length() : 0L ;
                
            }
            else{
                ret = 0L ;
            }
            
        }catch( Exception t ){
            ret = 0L ;
        }finally{
            file = null ;
        }
        
        return ret ;
    }
    
    /**
     * 指定ファイル名のファイルサイズを取得.
     * <BR><BR>
     * 指定ファイル名のファイルサイズを取得します.<BR>
     * また、このメソッドは[URL(FTP,HTTP,HTTPS)]とに対応しています.
     * <BR>
     * @param mode プロトコル指定かローカル指定であるかを設定します.<BR>
     *             [true]を設定した場合プロトコル指定となります.
     *             [false]を設定した場合ローカル指定となります.
     * @param name 取得対象の情報名を指定します.
     * @param user mode=trueでnameがFTP指定の場合のユーザ名を設定します.
     * @param passwd mode=trueでnameがFTP指定の場合のパスワードを設定します.
     * @return long ファイルサイズが返されます.
     */
    public static final long getFileLength( boolean mode,String name,String user,String passwd )
        throws AccessException
    {
        int port ;
        long ret ;
        
        URL url = null ;
        URLConnection conn = null ;
        String protocol = null ;
        String file = null ;
        FTPClient ftp = null ;
        
        ret = 0L ;
        
        try{
            
            // 指定対象がURL名の場合.
            if( mode == true ){
                
                url = new URL( IODef.convertEnvName( name ) ) ;
                protocol = url.getProtocol().toLowerCase() ;
                file = UtilCom.trimPlus( url.getFile() ) ;
                
                // 対象プロトコルがFTPの場合.
                if( protocol.indexOf( IOCom.PROTOCOL_FTP ) != -1 ){
                    
                    ftp = new FTPClient() ;
                    ftp.setLoginList( false ) ;
                    //ftp.setDebugMode( true ) ;
                    
                    // ポート番号が指定されている場合.
                    if( ( port = url.getPort() ) != -1 ){
                        
                        // ユーザ名が指定されていない場合.
                        if( user == null ){
                            
                            // 指定ポートでFTPをオープン.
                            ftp.open(
                                false,
                                url.getHost(),
                                port
                            ) ;
                            
                        }
                        // ユーザ名が指定されている場合.
                        else{
                            
                            // 指定ポートでFTPをオープン.
                            ftp.open(
                                false,
                                url.getHost(),
                                port,
                                user,passwd
                            ) ;
                            
                        }
                        
                    }
                    // ポート番号が指定されていない場合.
                    else{
                        
                        // ユーザ名が指定されていない場合.
                        if( user == null ){
                            
                            // デフォルトポートでFTPをオープン.
                            ftp.open(
                                false,
                                url.getHost()
                            ) ;
                            
                        }
                        // ユーザ名が指定されている場合.
                        else{
                            
                            // デフォルトポートでFTPをオープン.
                            ftp.open(
                                false,
                                url.getHost(),
                                user,passwd
                            ) ;
                            
                        }
                        
                    }
                    
                    // 指定ファイル名の先頭に"/"が存在する場合.
                    file = ( file.indexOf( "/" ) != 0 ) ? file : file.substring( 1 ) ;
                    
                    ret = ftp.getSize( file ) ;
                    
                }
                // 対象プロトコルがHTTPの場合.
                else if(
                    protocol.indexOf( IOCom.PROTOCOL_HTTP ) != -1 ||
                    protocol.indexOf( IOCom.PROTOCOL_HTTPS ) != -1
                )
                {
                    
                    // ユーザ名/パスワードを指定している場合.
                    if( user != null && user.length() > 0 ){
                        
                        synchronized( AutoUserPasswd.getInstance().getSynchronized().get() ){
                            
                            // ユーザ名/パスワードを一時設定.
                            AutoUserPasswd.getInstance().setTmpUserPasswd(
                                user,passwd,protocol
                            ) ;
                            
                            // コネクション.
                            conn = url.openConnection() ;
                            conn.connect() ;
                            
                        }
                    }
                    // ユーザ名/パスワードが指定されていない場合.
                    else{
                        
                        conn = url.openConnection() ;
                        conn.connect() ;
                        
                    }
                    
                    ret = conn.getContentLength() ;
                    
                }
                // 不明なプロトコルの場合.
                else{
                    
                    // ローカルファイルとして取得.
                    ret = IOCom.getFileLength( name ) ;
                    
                }
                
            }
            // 指定対象がローカルファイル名の場合.
            else{
                
                ret = IOCom.getFileLength( name ) ;
                
            }
            
        }catch( AccessException ac ){
            throw ac ;
        }catch( Exception t ){
            throw new AccessException( t ) ;
        }finally{
            if( ftp != null ){
                ftp.close() ;
            }
            url = null ;
            conn = null ;
            protocol = null ;
            file = null ;
            ftp = null ;
        }
        
        return ret ;
    }
    
    /**
     * 指定情報( file or directory )の最終更新時間を取得.
     * <BR><BR>
     * 指定情報の最終更新時間を取得します.<BR>
     * <BR>
     * @param name 更新時間取得対象の情報名を指定します.
     * @return long 更新時間が返されます.<BR>
     *              指定情報名が存在しない場合[0L]が返されます.
     */
    public static final long getLastTime( String name )
    {
        int type ;
        long ret ;
        
        File file = null ;
        
        if( name == null ){
            return 0L ;
        }
        
        try{
            
            name = IODef.convertEnvName( name ) ;
            type = IODef.getNameByProtocolType( name ) ;
            
            if( type == IOCom.TYPE_PROTOCOL_LOCAL ){
                
                file = new File( name ) ;
                ret = ( file.exists() == true ) ? 
                    file.lastModified() : 0L ;
                
            }
            else{
                ret = 0L ;
            }
            
        }catch( Exception t ){
            ret = 0L ;
            file = null ;
        }
        
        return ret ;
    }
    
    /**
     * 指定情報( file or directory )の最終更新時間を取得.
     * <BR><BR>
     * 指定情報の最終更新時間を取得します.<BR>
     * また、このメソッドは[URL(FTP,HTTP,HTTPS)]に対応しています.
     * <BR>
     * @param mode プロトコル指定かローカル指定であるかを設定します.<BR>
     *             [true]を設定した場合プロトコル指定となります.
     *             [false]を設定した場合ローカル指定となります.
     * @param name 更新時間取得対象の情報名を指定します.
     * @param user mode=trueでnameがFTP指定の場合のユーザ名を設定します.
     * @param passwd mode=trueでnameがFTP指定の場合のパスワードを設定します.
     * @return long 更新時間が返されます.<BR>
     *              指定情報名が存在しない場合[0L]が返されます.
     */
    public static final long getLastTime( boolean mode,String name,String user,String passwd )
        throws AccessException
    {
        int port ;
        long ret ;
        
        URL url = null ;
        URLConnection conn = null ;
        String protocol = null ;
        String file = null ;
        FTPClient ftp = null ;
        
        ret = 0L ;
        
        try{
            
            // 指定対象がURL名の場合.
            if( mode == true ){
                
                url = new URL( IODef.convertEnvName( name ) ) ;
                protocol = url.getProtocol().toLowerCase() ;
                file = UtilCom.trimPlus( url.getFile() ) ;
                
                // 対象プロトコルがFTPの場合.
                if( protocol.indexOf( IOCom.PROTOCOL_FTP ) != -1 ){
                    
                    ftp = new FTPClient() ;
                    ftp.setLoginList( false ) ;
                    //ftp.setDebugMode( true ) ;
                    
                    // ポート番号が指定されている場合.
                    if( ( port = url.getPort() ) != -1 ){
                        
                        // ユーザ名が指定されていない場合.
                        if( user == null ){
                            
                            // 指定ポートでFTPをオープン.
                            ftp.open(
                                false,
                                url.getHost(),
                                port
                            ) ;
                            
                        }
                        // ユーザ名が指定されている場合.
                        else{
                            
                            // 指定ポートでFTPをオープン.
                            ftp.open(
                                false,
                                url.getHost(),
                                port,
                                user,passwd
                            ) ;
                            
                        }
                        
                    }
                    // ポート番号が指定されていない場合.
                    else{
                        
                        // ユーザ名が指定されていない場合.
                        if( user == null ){
                            
                            // デフォルトポートでFTPをオープン.
                            ftp.open(
                                false,
                                url.getHost()
                            ) ;
                            
                        }
                        // ユーザ名が指定されている場合.
                        else{
                            
                            // デフォルトポートでFTPをオープン.
                            ftp.open(
                                false,
                                url.getHost(),
                                user,passwd
                            ) ;
                            
                        }
                        
                    }
                    
                    // 指定ファイル名の先頭に"/"が存在する場合.
                    file = ( file.indexOf( "/" ) != 0 ) ? file : file.substring( 1 ) ;
                    
                    // ファイル時間を取得.
                    ret = ftp.getDate( file ) ;
                    
                }
                // 対象プロトコルがHTTPの場合.
                else if(
                    protocol.indexOf( IOCom.PROTOCOL_HTTP ) != -1 ||
                    protocol.indexOf( IOCom.PROTOCOL_HTTPS ) != -1
                )
                {
                    
                    // ユーザ名/パスワードを指定している場合.
                    if( user != null && user.length() > 0 ){
                        
                        synchronized( AutoUserPasswd.getInstance().getSynchronized().get() ){
                            
                            // ユーザ名/パスワードを一時設定.
                            AutoUserPasswd.getInstance().setTmpUserPasswd(
                                user,passwd,protocol
                            ) ;
                            
                            // コネクション.
                            conn = url.openConnection() ;
                            conn.connect() ;
                            
                        }
                    }
                    // ユーザ名/パスワードが指定されていない場合.
                    else{
                        
                        conn = url.openConnection() ;
                        conn.connect() ;
                        
                    }
                    
                    ret = conn.getLastModified() ;
                    
                }
                // 不明なプロトコルの場合.
                else{
                    
                    // ローカルファイルとして取得.
                    ret = IOCom.getLastTime( name ) ;
                    
                }
                
            }
            // 指定対象がローカルファイル名の場合.
            else{
                
                ret = IOCom.getLastTime( name ) ;
                
            }
            
        }catch( AccessException ac ){
            throw ac ;
        }catch( Exception t ){
            throw new AccessException( t ) ;
        }finally{
            if( ftp != null ){
                ftp.close() ;
            }
            url = null ;
            conn = null ;
            protocol = null ;
            file = null ;
            ftp = null ;
        }
        
        return ret ;
    }
    
    /**
     * 指定ファイル名に対象のバイナリ情報を出力.
     * <BR><BR>
     * 指定されたファイル名に対してバイナリ情報を出力します.
     * <BR>
     * @param name 出力対象のファイル名を設定します.
     * @param binary 出力対象のバイナリ情報を設定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public static final void setFile( String name,byte[] binary )
        throws InputException,FileAccessException
    {
        IOCom.setFile( name,true,binary ) ;
    }
    
    /**
     * 指定ファイル名に対象のバイナリ情報を出力.
     * <BR><BR>
     * 指定されたファイル名に対してバイナリ情報を出力します.
     * <BR>
     * @param name 出力対象のファイル名を設定します.
     * @param mode 書き込みモードを設定します.<BR>
     *             [true]を設定した場合、既に同一のファイルが存在しても上書きします.<BR>
     *             [false]を設定した場合、既に同一のファイルが存在した場合後に書き込みます.
     * @param binary 出力対象のバイナリ情報を設定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public static final void setFile( String name,boolean mode,byte[] binary )
        throws InputException,FileAccessException
    {
        BufferedOutputStream buf = null ;
        
        if( name == null || binary == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            name = IODef.convertEnvName( name ) ;
            
            if( mode == true ){
                buf = new BufferedOutputStream(
                    new FileOutputStream(
                        name
                    )
                ) ;
            }
            else{
                buf = new BufferedOutputStream(
                    new FileOutputStream(
                        name,true
                    )
                ) ;
            }
            
            buf.write( binary ) ;
            
        }catch( IOException io ){
            throw new FileAccessException( io ) ;
        }finally{
            
            try{
                buf.close() ;
            }catch( Exception t ){
            }
            
            buf = null ;
            
        }
        
    }
    
    /**
     * 指定ファイル名に対象の文字列情報を出力.
     * <BR><BR>
     * 指定されたファイル名に対して文字列情報を出力します.
     * <BR>
     * @param name 出力対象のファイル名を設定します.
     * @param value 出力対象の文字列情報を設定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public static final void setFileByString( String name,String value )
        throws InputException,FileAccessException
    {
        IOCom.setFileByString( name,true,value,BaseDef.THIS_CHARSET ) ;
    }
    
    /**
     * 指定ファイル名に対象の文字列情報を出力.
     * <BR><BR>
     * 指定されたファイル名に対して文字列情報を出力します.
     * <BR>
     * @param name 出力対象のファイル名を設定します.
     * @param mode 書き込みモードを設定します.<BR>
     *             [true]を設定した場合、既に同一のファイルが存在しても上書きします.<BR>
     *             [false]を設定した場合、既に同一のファイルが存在した場合後に書き込みます.
     * @param value 出力対象の文字列情報を設定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public static final void setFileByString( String name,boolean mode,String value )
        throws InputException,FileAccessException
    {
        IOCom.setFileByString( name,mode,value,BaseDef.THIS_CHARSET ) ;
    }
    
    /**
     * 指定ファイル名に対象の文字列情報を出力.
     * <BR><BR>
     * 指定されたファイル名に対して文字列情報を出力します.
     * <BR>
     * @param name 出力対象のファイル名を設定します.
     * @param mode 書き込みモードを設定します.<BR>
     *             [true]を設定した場合、既に同一のファイルが存在しても上書きします.<BR>
     *             [false]を設定した場合、既に同一のファイルが存在した場合後に書き込みます.
     * @param value 出力対象の文字列情報を設定します.
     * @param charset 出力対象のキャラクターセットを設定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public static final void setFileByString( String name,boolean mode,String value,String charset )
        throws InputException,FileAccessException
    {
        BufferedWriter buf = null ;
        
        if( name == null || value == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            name = IODef.convertEnvName( name ) ;
            
            if( mode == true ){
                buf = new BufferedWriter(
                    new OutputStreamWriter(
                        new FileOutputStream(
                            name
                        ),charset
                    )
                ) ;
            }
            else{
                buf = new BufferedWriter(
                    new OutputStreamWriter(
                        new FileOutputStream(
                            name,true
                        ),charset
                    )
                ) ;
            }
            
            buf.write( value ) ;
            
        }catch( IOException io ){
            throw new FileAccessException( io ) ;
        }finally{
            
            try{
                buf.close() ;
            }catch( Exception t ){
            }
            
            buf = null ;
            
        }
        
    }
    
    /**
     * 指定ファイルのInputStreamを取得.
     * <BR><BR>
     * 指定されたファイルのInputStreamを取得します.
     * <BR>
     * @param name 取得対象のファイル名を設定します.
     * @return InputStream 指定ファイル内容が返されます.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public static final InputStream getInputStream( String name )
        throws InputException,FileAccessException
    {
        int type ;
        
        InputStream ret = null ;
        
        try{
            
            name = IODef.convertEnvName( name ) ;
            type = IODef.getNameByProtocolType( name ) ;
            
            if( type == IOCom.TYPE_PROTOCOL_LOCAL ){
                
                if( name == null || IOCom.isRead( name ) == false ){
                    if( name == null ){
                        throw new InputException( "引数は不正です" ) ;
                    }
                    throw new InputException(
                        "対象のファイル名(" + name +
                        ")は読み込み権限が無いかファイルが存在しません"
                    ) ;
                }
                
                ret = new FileInputStream(
                    name
                ) ;
                
            }
            else{
                ret = null ;
            }
            
        }catch( IOException ie ){
            throw new FileAccessException( ie ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * 指定ファイルのInputStreamを取得.
     * <BR><BR>
     * 指定されたファイルのInputStreamを取得します.
     * <BR>
     * @param name 取得対象のファイル名を設定します.
     * @param offset 取得対象のオフセット値を設定します.
     * @param length 取得対象のデータ長を設定します.
     * @return InputStream 指定ファイル内容が返されます.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public static final InputStream getInputStream( String name,int offset,int length )
        throws InputException,FileAccessException
    {
        int type ;
        
        InputStream ret = null ;
        
        try{
            
            type = IODef.getNameByProtocolType( name ) ;
            
            if( type == IOCom.TYPE_PROTOCOL_LOCAL ){
                
                if( name == null || IOCom.isRead( name ) == false ){
                    if( name == null ){
                        throw new InputException( "引数は不正です" ) ;
                    }
                    throw new InputException(
                        "対象のファイル名(" + name +
                        ")は読み込み権限が無いかファイルが存在しません"
                    ) ;
                }
                
                name = IODef.convertEnvName( name ) ;
                ret = new RandomIO( name,false ).getInputStream( offset,length ) ;
                
            }
            else{
                ret = null ;
            }
            
        }catch( AccessException ac ){
            throw new FileAccessException( ac ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * 指定ファイル内容をバイナリ情報として取得.
     * <BR><BR>
     * 指定されたファイルの内容をバイナリ情報として取得します.<BR>
     * <BR>
     * @param name 取得対象のファイル名を設定します.
     * @return byte[] 指定ファイル内容が返されます.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public static final byte[] getFile( String name )
        throws InputException,FileAccessException
    {
        int type ;
        long len ;
        
        BufferedInputStream buf = null ;
        byte[] ret = null ;
        
        try{
            
            name = IODef.convertEnvName( name ) ;
            type = IODef.getNameByProtocolType( name ) ;
            
            if( type == IOCom.TYPE_PROTOCOL_LOCAL ){
                
                if( name == null || IOCom.isRead( name ) == false ){
                    if( name == null ){
                        throw new InputException( "引数は不正です" ) ;
                    }
                    throw new InputException(
                        "対象のファイル名(" + name +
                        ")は読み込み権限が無いかファイルが存在しません"
                    ) ;
                }
                
                buf = new BufferedInputStream(
                    new FileInputStream(
                        name
                    )
                ) ;
                
                len = IOCom.getFileLength( name ) ;
                
                if( len > 0x000000007fffffffL ){
                    throw new FileAccessException(
                        "指定されたファイル名(" + name +
                        ")はファイルサイズが大きいため取得できません"
                    ) ;
                }
                
                ret = new byte[ ( int )( len & 0x000000007fffffffL ) ] ;
                buf.read( ret ) ;
                
            }
            else{
                ret = null ;
            }
            
        }catch( IOException ie ){
            throw new FileAccessException( ie ) ;
        }catch( AccessException ac ){
            throw new FileAccessException( ac ) ;
        }finally{
            
            try{
                buf.close() ;
            }catch( Exception t ){
            }
            
            buf = null ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * 指定ファイル内容を文字列情報として取得.
     * <BR><BR>
     * 指定されたファイルの内容を文字列情報として取得します.<BR>
     * <BR>
     * @param name 取得対象のファイル名を設定します.
     * @return String 取得された情報が返されます.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public static final String getFileByString( String name )
        throws InputException,FileAccessException
    {
        return IOCom.getFileByString( name,BaseDef.THIS_CHARSET ) ;
    }
    
    /**
     * 指定ファイル内容を文字列情報として取得.
     * <BR><BR>
     * 指定されたファイルの内容を文字列情報として取得します.
     * <BR>
     * @param name 取得対象のファイル名を設定します.
     * @param charset 取得対象のキャラクターセットを設定します.
     * @return String 取得された情報が返されます.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public static final String getFileByString( String name,String charset )
        throws InputException,FileAccessException
    {
        int len ;
        int type ;
        char[] tmp = null ;
        
        CharArrayWriter ca = null ;
        BufferedReader buf = null ;
        String ret = null ;
        
        try{
            
            name = IODef.convertEnvName( name ) ;
            type = IODef.getNameByProtocolType( name ) ;
            
            if( type == IOCom.TYPE_PROTOCOL_LOCAL ){
                
                if( name == null || IOCom.isRead( name ) == false ){
                    if( name == null ){
                        throw new InputException( "引数は不正です" ) ;
                    }
                    throw new InputException(
                        "対象のファイル名(" + name +
                        ")は読み込み権限が無いかファイルが存在しません"
                    ) ;
                }
                
                tmp = new char[ IOCom.LOAD_FILE_LENGTH ] ;
                ca = new CharArrayWriter() ;
                buf = new BufferedReader(
                    new InputStreamReader(
                        new FileInputStream(
                            name
                        ),charset
                    )
                ) ;
                
                while( ( len = buf.read( tmp,0,IOCom.LOAD_FILE_LENGTH ) ) > 0 ){
                    ca.write( tmp,0,len ) ;
                }
                
                ret = ca.toString() ;
                ca.close() ;
                ca = null ;
                
            }
            else{
                ret = null ;
            }
            
        }catch( IOException ie ){
            throw new FileAccessException( ie ) ;
        }finally{
            
            try{
                buf.close() ;
            }catch( Exception t ){
            }
            
            try{
                ca.close() ;
            }catch( Exception t ){
            }
            
            buf = null ;
            ca = null ;
            tmp = null ;
            
        }
        
        return ret ;
    }
    
    /**
     * 指定名からプロトコルを取得.
     * <BR><BR>
     * 指定名からプロトコルを取得します.
     * <BR>
     * @param name 対象名を設定します.
     * @return int プロトコルタイプが返されます.<BR>
     *             [IOCom.TYPE_PROTOCOL_LOCAL]が返された場合
     *             プロトコル名でないか不明です.<BR>
     *             [IOCom.TYPE_PROTOCOL_FTP]が返された場合
     *             プロトコル名はFTPです.<BR>
     *             [IOCom.TYPE_PROTOCOL_HTTP]が返された場合
     *             プロトコル名はHTTPです.<BR>
     *             [IOCom.TYPE_PROTOCOL_HTTPS]が返された場合
     *             プロトコル名はHTTPSです.
     * @exception InputException 引数例外.
     */
    public static final int getProtocol( String name )
        throws InputException
    {
        String chk = null ;
        
        if( name == null || name.length() <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        return IODef.getNameByProtocolType( IODef.convertEnvName( name ) ) ;
    }
    
    /**
     * クラスパス内から、指定したファイル名のフルパス名を取得.
     * <BR><BR>
     * クラスパスがディレクトリで指定されている中から、指定している
     * ファイル名が存在する場合、そのファイル名をフルパスで返します.
     * <BR>
     * @param name 検索対象のファイル名を取得します.
     * @return String 取得されたファイル名のフルパス名が返されます.<BR>
     *                指定ファイル名が、クラスパス内に存在しない場合は[null]が
     *                返されます.
     */
    public static final String getSearchFileName( String name )
    {
        int i ;
        int len ;
        
        ArrayList lst = null ;
        String tmp = null ;
        String ret = null ;
        
        try{
            
            lst = UtilCom.getClassPath() ;
            if( lst != null ){
                len = lst.size() ;
                for( i = 0 ; i < len ; i ++ ){
                    
                    tmp = ( String )lst.get( i ) ;
                    if( IOCom.isDirExists( tmp ) == true ){
                        
                        ret = IOCom.searchClasspathToFileName( tmp,name ) ;
                        tmp = null ;
                        
                        if( ret != null ){
                            break ;
                        }
                        else{
                            ret = null ;
                        }
                    }
                    else if(
                        IOCom.isFileExists( tmp ) == true &&
                        tmp.endsWith( name ) == true &&
                        tmp.lastIndexOf( name ) == ( tmp.length() - name.length() )
                    )
                    {
                        ret = tmp ;
                        break ;
                    }
                }
            }
            
        }catch( Exception e ){
            ret = null ;
        }finally{
            if( lst != null ){
                lst.clear() ;
            }
            
            lst = null ;
            tmp = null ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * 指定名がURL名であるかチェック.
     * <BR><BR>
     * 指定名がURL名であるかチェックします.
     * <BR>
     * @param name チェック対象の名前を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合URL名です.<BR>
     *                 [false]が返された場合URL名ではありません.
     */
    public static final boolean isURL( String name )
    {
        boolean ret ;
        
        String tmp = null ;
        
        ret = false ;
        
        try{
            if( name != null ){
                
                name = IODef.convertEnvName( name ) ;
                tmp = UtilCom.trimPlus( name ).toLowerCase() ;
                
                // HTTP及びFTPのプロトコル指定の場合.
                if(
                    tmp.indexOf( IOCom.PROTOCOL_FTP ) == 0 ||
                    tmp.indexOf( IOCom.PROTOCOL_HTTP ) == 0 ||
                    tmp.indexOf( IOCom.PROTOCOL_HTTPS ) == 0
                )
                {
                    ret = true ;
                }
                
            }
        }catch( Exception t ){
            ret = false ;
        }finally{
            tmp = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 指定名がJARファイルであるかチェック.
     * <BR><BR>
     * 指定名がJARファイルであるかチェックします.
     * <BR>
     * @param name チェック対象の名前を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合JARファイルです.<BR>
     *                 [false]が返された場合JARファイルではありません.
     */
    public static final boolean isJar( String name )
    {
        int len ;
        int jarLen ;
        int zipLen ;
        boolean ret ;
        
        String tmp = null ;
        
        ret = false ;
        
        try{
            if( name != null ){
                
                name = IODef.convertEnvName( name ) ;
                tmp = UtilCom.trimPlus( name ) ;
                tmp = name.toLowerCase() ;
                
                len = tmp.length() ;
                jarLen = len - IOCom.JAR_LENGTH ;
                zipLen = len - IOCom.ZIP_LENGTH ;
                
                if(
                    tmp.lastIndexOf( ".jar" ) == jarLen ||
                    tmp.lastIndexOf( ".zip" ) == zipLen
                )
                {
                    ret = true ;
                }
                
            }
        }catch( Exception t ){
            ret = false ;
        }finally{
            tmp = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 指定ファイル名が存在するかチェック.
     * <BR><BR>
     * 指定ファイル名が存在するかチェックします.<BR>
     * <BR>
     * @param name ファイル名を指定します.
     * @return boolean ファイル存在フラグが返されます.<BR>
     *                 [true]が返された場合、指定したファイル名は存在します.<BR>
     *                 [false]が返された場合、指定したファイル名は存在しません.
     */
    public static final boolean isFileExists( String name )
    {
        int type ;
        boolean ret ;
        
        File file = null ;
        
        if( name == null || name.length() <= 0 ){
            return false ;
        }
        
        try{
            
            name = IODef.convertEnvName( name ) ;
            type = IODef.getNameByProtocolType( name ) ;
            
            if( type == IOCom.TYPE_PROTOCOL_LOCAL ){
                
                file = new File( name ) ;
                ret = ( file.exists() == true ) ?
                    (
                        ( file.isDirectory() == false ) ?
                            file.canRead() : false
                    ) 
                    : false ;
                
            }
            else{
                ret = false ;
            }
            
        }catch( Exception e ){
            ret = false ;
        }finally{
            file = null ;
        }
        
        return ret ;
    }
    
    /**
     * 指定ディレクトリ名が存在するかチェック.
     * <BR><BR>
     * 指定ディレクトリ名が存在するかチェックします.
     * <BR>
     * @param dirName ディレクトリ名を指定します.
     * @return boolean ディレクトリ存在フラグが返されます.<BR>
     *                 [true]が返された場合、指定したディレクトリ名は存在します.<BR>
     *                 [false]が返された場合、指定したディレクトリ名は存在しません.
     */
    public static final boolean isDirExists( String dirName )
    {
        boolean ret ;
        File check = null ;
        
        if( dirName == null ){
            return false ;
        }
        
        dirName = IODef.convertEnvName( dirName ) ;
        check = new File( dirName ) ;
        ret = check.isDirectory() ;
        check = null ;
        
        return ret ;
    }
    
    /**
     * 指定ファイル名が読み取り可能かチェック.
     * <BR><BR>
     * 指定ファイル名が読み取り可能かチェックします.<BR>
     * <BR>
     * @param name 読み取り可能チェック対象のディレクトリ／ファイル名を
     *             指定します.
     * @return boolean 読み取り可能か否かが返されます.<BR>
     *                 [true]が返された場合、読み取り可能です.<BR>
     *                 [false]が返された場合、読み取り不可能か存在しません.
     */
    public static final boolean isRead( String name )
    {
        int type ;
        boolean ret ;
        
        File file = null ;
        
        if( name == null || name.length() <= 0 ){
            return false ;
        }
        
        try{
            
            name = IODef.convertEnvName( name ) ;
            type = IODef.getNameByProtocolType( name ) ;
            
            if( type == IOCom.TYPE_PROTOCOL_LOCAL ){
                
                file = new File( name ) ;
                ret = file.canRead() ;
                
            }
            else{
                ret = false ;
            }
            
        }catch( Exception e ){
            ret = false ;
        }finally{
            file = null ;
        }
        
        return ret ;
    }
    
    /**
     * 指定ファイル名が書き込み可能かチェック.
     * <BR><BR>
     * 指定ファイル名が書き込み可能かチェックします.
     * <BR>
     * @param name 書き込み可能チェック対象のディレクトリ／ファイル名を
     *             指定します.
     * @return boolean 書き込み可能か否かが返されます.<BR>
     *                 [true]が返された場合、書き込み可能です.<BR>
     *                 [false]が返された場合、書き込み不可能か存在しません.
     */
    public static final boolean isWrite( String name )
    {
        boolean ret ;
        File check = null ;
        
        if( name == null ){
            return false ;
        }
        
        name = IODef.convertEnvName( name ) ;
        check = new File( name ) ;
        ret = check.canWrite() ;
        check = null ;
        
        return ret ;
    }
    
    
    
    /**
     * 対象ディレクトリ内から、ファイル名を検索.
     */
    private static final String searchClasspathToFileName( String dir,String name )
    {
        int i ;
        int len ;
        
        String[] lists = null ;
        String tmp = null ;
        String ret = null ;
        
        dir = (
            dir.endsWith( "\\" ) == false && dir.endsWith( "/" ) == false
        ) ? dir + BaseDef.FILE_SEPARATOR : dir ;
        
        try{
            
            ret = dir + name ;
            if( IOCom.isFileExists( ret ) == false ){
                
                ret = null ;
                
                lists = IOCom.getList( dir ) ;
                if( lists != null ){
                    len = lists.length ;
                    for( i = 0 ; i < len ; i ++ ){
                        
                        tmp = lists[ i ] ;
                        lists[ i ] = null ;
                        
                        if( IOCom.isDirExists( tmp ) == true ){
                            ret = IOCom.searchClasspathToFileName( tmp,name ) ;
                            tmp = null ;
                            if( ret != null ){
                                break ;
                            }
                        }
                        
                    }
                    
                }
                
            }
            
        }catch( Exception e ){
            ret = null ;
        }finally{
            tmp = null ;
            lists = null ;
        }
        
        return ret ;
    }
    
}
