/*
 * @(#)BufferWriter.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.io;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.OutputStreamWriter;

import com.JRcServer.commons.def.BaseDef;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.io.env.EnvFileOutputStream;
import com.JRcServer.commons.thread.Synchronized;


/**
 * 指定情報を出力します.
 * <BR><BR>
 * 指定された情報に対して、出力します.
 *
 * @version 1.0.0 2003/11/03
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
public class BufferWriter implements BaseWriter
{
    
    /**
     * ファイルポインタ.
     */
    private BufferedWriter m_fp = null ;
    
    /**
     * オープンファイル名.
     */
    private String m_fileName = null ;
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    
    
    /**
     * コンストラクタ.
     */
    public BufferWriter()
    {
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.close() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 出力対象ファイルオープン.
     * <BR><BR>
     * 出力対象のファイルをオープンします.
     * <BR>
     * @param fileName オープン対象のファイル名を指定します.
     * @param newFlg 新規に書き込みを行うか設定します。<BR>
     *               [true]を指定した場合、指定ファイル内の情報をクリアします。<BR>
     *               [false]を指定した場合、すでにファイル情報が存在する場合追加書き込みきます。
     * @exception FileAccessException ファイルIO例外
     * @exception InputException 入力例外
     */
    public final void open( String fileName,boolean newFlg )
        throws FileAccessException,InputException
    {
        
        try{
            this.open( fileName,null,newFlg ) ;
        }catch( InputException in ){
            throw in ;
        }catch( FileAccessException fa ){
            throw fa ;
        }
        
    }
    
    /**
     * 出力対象ファイルオープン.
     * <BR><BR>
     * 出力対象のファイルをオープンします.
     * <BR>
     * @param fileName オープン対象のファイル名を指定します.
     * @param charset 出力対象のエンコード名を設定します.
     * @param newFlg 新規に書き込みを行うか設定します。<BR>
     *               [true]を指定した場合、指定ファイル内の情報をクリアします。<BR>
     *               [false]を指定した場合、すでにファイル情報が存在する場合追加書き込みきます。
     * @exception FileAccessException ファイルIO例外
     * @exception InputException 入力例外
     */
    public final void open( String fileName,String charset,boolean newFlg )
        throws FileAccessException,InputException
    {
        if( fileName == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            this.close() ;
        }catch( Exception t ){
        }
        
        try{
            
            m_sync.create() ;
            
            if( charset == null ){
                synchronized( m_sync.get() ){
                    
                    m_fp = new BufferedWriter(
                        new OutputStreamWriter(
                            new EnvFileOutputStream(
                                fileName,( ( newFlg == false ) ? true : false )
                            )
                        )
                    ) ;
                    
                    m_fileName = fileName ;
                }
            }else{
                synchronized( m_sync.get() ){
                    
                    m_fp = new BufferedWriter(
                        new OutputStreamWriter(
                            new EnvFileOutputStream(
                                fileName,( ( newFlg == false ) ? true : false )
                            ),charset
                        )
                    ) ;
                    
                    m_fileName = fileName ;
                }
            }
            
        }catch( IOException io ){
            this.close() ;
            throw new FileAccessException( io ) ;
        }catch( Exception t ){
            this.close() ;
        }
        
    }
    
    /**
     * ファイルクローズ.
     * <BR><BR>
     * オープンしているファイル情報をクローズします.
     * また、バッファに対して情報が存在する場合、その情報を
     * 書き込んでクローズします.
     */
    public final void close()
    {
        
        try{
            synchronized( m_sync.get() ){
                m_fp.close() ;
            }
        }catch( Exception t ){
        }finally{
            try{m_fp.close();}catch( Exception ee ){}
            m_fp = null ;
            m_fileName = null ;
            m_sync.clear() ;
        }
        
        
    }
    
    /**
     * バッファ情報を出力.
     * <BR><BR>
     * バッファ情報をファイルに出力します。
     * <BR>
     * @exception FileAccessException ファイルIO例外
     */
    public final void flush() throws FileAccessException
    {
        
        try{
            synchronized( m_sync.get() ){
                m_fp.flush() ;
            }
        }catch( IOException io ){
            throw new FileAccessException( io ) ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 文字情報を出力.
     * <BR><BR>
     * 文字情報を出力します。
     * <BR>
     * @param stream 文字情報を設定します。
     * @exception InputException 入力例外
     * @exception FileAccessException ファイルアクセス例外.
     */
    public final void print( String stream )
        throws InputException,FileAccessException
    {
        
        if( stream == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            synchronized( m_sync.get() ){
                m_fp.write( stream ) ;
            }
        }catch( IOException io ){
            throw new FileAccessException( io ) ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * １行の文字情報を出力.
     * <BR><BR>
     * １行の文字情報を出力します。
     * <BR>
     * @exception FileAccessException ファイルアクセス例外.
     */
    public final void println()
        throws FileAccessException
    {
        try{
            synchronized( m_sync.get() ){
                m_fp.newLine() ;
            }
        }catch( IOException io ){
            throw new FileAccessException( io ) ;
        }catch( Exception t ){
        }
    }
    
    /**
     * １行の文字情報を出力.
     * <BR><BR>
     * １行の文字情報を出力します。
     * <BR>
     * @param stream 文字情報を設定します。
     * @exception InputException 入力例外
     * @exception FileAccessException ファイルアクセス例外.
     */
    public final void println( String stream )
        throws InputException,FileAccessException
    {
        
        StringBuffer buf = null ;
        if( stream == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        buf = new StringBuffer( stream.length() + 2 ) ;
        buf.append( stream ) ;
        buf.append( BaseDef.ENTER_SEPARATOR ) ;
        
        try{
            synchronized( m_sync.get() ){
                m_fp.write( buf.toString() ) ;
            }
        }catch( IOException io ){
            throw new FileAccessException( io ) ;
        }catch( Exception t ){
        }finally{
            buf = null ;
        }
        
    }
    
    /**
     * 書き込み容量の取得.
     * <BR><BR>
     * 現在のファイル書き込み容量を取得します。
     * また、現在バッファに格納されている情報数も数に含まれます.
     * <BR>
     * @return long ファイル書き込み容量を取得します。<BR>
     *              ファイルがオープンされていない場合[-1]が返されます.
     */
    public final long length()
    {
        long ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = ( m_fileName == null ) ? -1L : IOCom.getFileLength( m_fileName ) ;
            }
        }catch( Exception t ){
            ret = -1L ;
        }
        
        return ret ;
    }
    
    /**
     * オープンファイル名の取得.
     * <BR><BR>
     * 現在オープンされているファイル名の取得を行います。
     * <BR>
     * @return String 現在オープンされているファイル名を取得します。<BR>
     *   オープンされていない場合[null]が格納されます。
     */
    public final String getFileName()
    {
        String ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_fileName ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 現在状況を取得します。
     * <BR><BR>
     * 現在のファイルオープン状況を取得します。
     * <BR>
     * @return boolean 現在のファイルオープン状況が格納されています。<BR>
     *                 [true]が返された場合、ファイルはオープンされています。<BR>
     *                 [false]が返された場合、ファイルはオープンされていません。
     */
    public final boolean isOpen()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = ( m_fileName == null ) ? false : true ;
            }
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
}
