/*
 * @( # )ArrayIO.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.io;

import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.thread.Synchronized;

/**
 * ArrayIOオブジェクト.
 * <BR><BR>
 * 配列のような感覚で、ファイルのIOを行える
 * オブジェクトをサポートします.
 * <BR>
 *
 * @version   1.00, 2004/10/26
 * @author    Masahito Suzuki
 * @since   JRcCommons 1.00
 */
public class ArrayIO
{
    
    /**
     * ファイルデータコピー１サイズ.
     */
    private static final int PAUSE_LENGTH = 512 ;
    
    /**
     * ランダムファイルIO.
     */
    private RandomIO m_io = null ;
    
    /**
     * エレメント用データ.
     */
    private final byte[] m_element = new byte[ 1 ] ;
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    
    /**
     * コンストラクタ.
     */
    public ArrayIO()
    {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * ArrayIOオブジェクトの生成を行います.
     * 指定ファイル名のファイルが既に存在する場合、
     * その内容を引き継ぎます.
     * <BR>
     * @param fileName オープン対象のファイル名を指定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public ArrayIO( String fileName )
        throws InputException,FileAccessException
    {
        try{
            this.create( fileName ) ;
        }catch( InputException in ){
            throw in ;
        }catch( FileAccessException fa ){
            throw fa ;
        }
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * ArrayIOオブジェクトの生成を行います.
     * <BR>
     * @param fileName オープン対象のファイル名を指定します.
     * @param length 配列長最大値を設定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public ArrayIO( String fileName,int length )
        throws InputException,FileAccessException
    {
        try{
            this.create( fileName,length ) ;
        }catch( InputException in ){
            throw in ;
        }catch( FileAccessException fa ){
            throw fa ;
        }
    }
    
    /**
     * 情報生成
     * <BR><BR>
     * ArrayIOオブジェクトの生成を行います.
     * 指定ファイル名のファイルが既に存在する場合、
     * その内容を引き継ぎます.
     * <BR>
     * @param fileName オープン対象のファイル名を指定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public final void create( String fileName )
        throws InputException,FileAccessException
    {
        try{
            
            this.clear() ;
            m_sync.create() ;
            
            synchronized( m_sync.get() ){
                m_io = new RandomIO( fileName,false ) ;
            }
            
        }catch( NullPointerException nul ){
            this.clear() ;
        }catch( InputException in ){
            this.clear() ;
            throw in ;
        }catch( FileAccessException fa ){
            this.clear() ;
            throw fa ;
        }
    }
    
    /**
     * 情報生成
     * <BR><BR>
     * ArrayIOオブジェクトの生成を行います.
     * 指定ファイル名のファイルが既に存在する場合、
     * その内容を破棄してオープンします.
     * <BR>
     * @param fileName オープン対象のファイル名を指定します.
     * @param length 配列長最大値を設定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public final void create( String fileName,long length )
        throws InputException,FileAccessException
    {
        try{
            
            this.clear() ;
            m_sync.create() ;
            
            synchronized( m_sync.get() ){
                
                m_io = new RandomIO( fileName,true ) ;
                m_io.setLength( length ) ;
                
            }
            
        }catch( NullPointerException nul ){
            this.clear() ;
        }catch( InputException in ){
            this.clear() ;
            throw in ;
        }catch( FileAccessException fa ){
            this.clear() ;
            throw fa ;
        }
    }
    
    /**
     * ファイルクローズ.
     * <BR><BR>
     * オープンされているファイルのクローズ処理を行います.
     */
    public final void clear()
    {
        if( m_io != null ){
            m_io.close() ;
        }
        m_io = null ;
        m_sync.clear() ;
    }
    
    /**
     * 指定要素情報の取得.
     * <BR><BR>
     * 指定要素情報を取得します.
     * <BR>
     * @param no 取得対象の要素番号を設定します.
     * @return byte 取得要素情報が格納されます.
     * @exception InputException 入力例外.
     */
    public final byte getElement( int no ) throws InputException
    {
        byte ret ;
        byte[] tmp = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                tmp = m_element ;
                m_io.read( tmp,no ) ;
                ret = tmp[ 0 ] ;
            }
            
        }catch( NullPointerException nul ){
            ret = 0 ;
        }catch( InputException in ){
            throw in ;
        }catch( FileAccessException fa ){
            throw new InputException( fa ) ;
        }finally{
            tmp = null ;
        }
        
        return ret ;
    }
    
    /**
     * 指定要素情報の設定.
     * <BR><BR>
     * 指定要素情報を設定します.
     * <BR>
     * @param info 設定対象のバイト情報を設定します.
     * @param no 設定位置を設定します.
     * @exception InputException 入力例外.
     */
    public final void setElement( byte info,int no ) throws InputException
    {
        byte[] tmp = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                tmp = m_element ;
                tmp[0] = info ;
                m_io.write( tmp,no ) ;
            }
            
        }catch( NullPointerException nul ){
        }catch( InputException in ){
            throw in ;
        }catch( FileAccessException fa ){
            throw new InputException( fa ) ;
        }finally{
            tmp = null ;
        }
    }
    
    /**
     * 指定領域情報の取得.
     * <BR><BR>
     * 指定された領域の情報を取得します.
     * <BR>
     * @param point 取得開始位置を指定します.
     * @param size 取得サイズを指定します.
     * @return byte[] 取得バイナリ情報が格納されます.
     * @exception InputException 入力例外.
     */
    public final byte[] get( int point,int size ) throws InputException
    {
        byte[] ret = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                ret = new byte[ size ] ;
                m_io.read( ret,point ) ;
            }
            
        }catch( NullPointerException nul ){
            ret = null ;
        }catch( ArrayIndexOutOfBoundsException ai ){
            ret = null ;
            throw new InputException( ai ) ;
        }catch( InputException in ){
            ret = null ;
            throw in ;
        }catch( FileAccessException fa ){
            ret = null ;
            throw new InputException( fa ) ;
        }
        
        return ret ;
    }
    
    /**
     * 指定領域情報の取得.
     * <BR><BR>
     * 指定された領域の情報を取得します.
     * <BR>
     * @param binary 取得データを格納するバイナリ情報を設定します.
     * @param point 取得開始位置を指定します.
     * @return int 取得バイナリ情報数が格納されます.
     * @exception InputException 入力例外.
     */
    public final int get( byte[] binary,int point ) throws InputException
    {
        int ret ;
        
        try{
            
            synchronized( m_sync.get() ){
                ret = m_io.read( binary,point ) ;
            }
            
        }catch( NullPointerException nul ){
            ret = 0 ;
        }catch( ArrayIndexOutOfBoundsException ai ){
            throw new InputException( ai ) ;
        }catch( InputException in ){
            throw in ;
        }catch( FileAccessException fa ){
            throw new InputException( fa ) ;
        }
        return ret ;
    }
    
    /**
     * 指定領域情報の取得.
     * <BR><BR>
     * 指定された領域の情報を取得します.
     * <BR>
     * @param binary 取得データを格納するバイナリ情報を設定します.
     * @param point 取得開始位置を指定します.
     * @param length 取得データサイズを設定します.
     * @return int 取得バイナリ情報数が格納されます.
     * @exception InputException 入力例外.
     */
    public final int get( byte[] binary,int point,int length )
        throws InputException
    {
        int ret ;
        
        try{
            
            synchronized( m_sync.get() ){
                ret = m_io.read( binary,point,length ) ;
            }
            
        }catch( NullPointerException nul ){
            ret = 0 ;
        }catch( ArrayIndexOutOfBoundsException ai ){
            throw new InputException( ai ) ;
        }catch( InputException in ){
            throw in ;
        }catch( FileAccessException fa ){
            throw new InputException( fa ) ;
        }
        return ret ;
    }
    
    /**
     * 指定領域情報のセット.
     * <BR><BR>
     * 指定された領域に指定情報をセットします.
     * <BR>
     * @param binary セット対象の情報を指定します.
     * @param point 設定開始位置を指定します.
     * @exception InputException 入力例外.
     */
    public final void set( byte[] binary,int point )
        throws InputException
    {
        
        try{
            
            synchronized( m_sync.get() ){
                m_io.write( binary,point ) ;
            }
            
        }catch( NullPointerException nul ){
        }catch( InputException in ){
            throw in ;
        }catch( FileAccessException fa ){
            throw new InputException( fa ) ;
        }
    }
    
    /**
     * 指定領域情報のセット.
     * <BR><BR>
     * 指定された領域に指定情報をセットします.
     * <BR>
     * @param binary セット対象の情報を指定します.
     * @param point 設定開始位置を指定します.
     * @param length セット対象情報書き込みサイズを設定します.
     * @exception InputException 入力例外.
     */
    public final void set( byte[] binary,int point,int length )
        throws InputException
    {
        try{
            
            synchronized( m_sync.get() ){
                m_io.write( binary,point,0,length ) ;
            }
            
        }catch( NullPointerException nul ){
        }catch( InputException in ){
            throw in ;
        }catch( FileAccessException fa ){
            throw new InputException( fa ) ;
        }
    }
    
    /**
     * 指定内容と指定領域開始位置の内容を比較.
     * <BR><BR>
     * 指定された内容と、ファイル内容を比較します.
     * <BR>
     * @param binary 比較対照のバイナリ情報を設定します.
     * @param point 調べる開始ポイントを設定します.
     * @return boolean 比較結果が返されます.<BR>
     *                 [true]が返された場合、情報は一致しています.<BR>
     *                 [false]が返された場合、情報は一致してません.
     * @exception InputException 入力例外.
     */
    public final boolean equals( byte[] binary,int point )
        throws InputException
    {
        int i ;
        int len ;
        boolean ret ;
        
        byte[] tmp = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                len = binary.length ;
                tmp = this.get( point,len ) ;
            }
            
            for( i = 0,ret = true ; i < len ; i ++ ){
                if( binary[i] != tmp[i] ){
                    ret = false ;
                    break ;
                }
            }
        }catch( NullPointerException nul ){
            ret = false ;
        }catch( InputException in ){
            throw in ;
        }finally{
            tmp = null ;
        }
        return ret ;
    }
    
    /**
     * 領域の取得.
     * <BR><BR>
     * 領域値を取得します.
     * <BR>
     * @return long 設定されている領域値が返されます.
     */
    public final long getLength()
    {
        long ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_io.getLength() ;
            }
        }catch( Exception t ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 領域の変更.
     * <BR><BR>
     * 領域の変更処理を行います.
     * <BR>
     * @param newLength 新しい領域サイズを指定します.
     */
    public final void setLength( long newLength )
    {
        try{
            synchronized( m_sync.get() ){
                m_io.setLength( newLength ) ;
            }
        }catch( Exception t ){
        }
    }
    
    /**
     * オープンファイル名を取得.
     * <BR><BR>
     * 現在オープンされているファイル名を設定します.
     * <BR>
     * @return String ファイル名が返されます.
     */
    public final String getFileName()
    {
        String ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_io.getFileName() ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * ファイルオープンチェック.
     * <BR><BR>
     * ファイルオープンチェックを行います.
     * <BR>
     * @return boolean ファイルオープンチェックが返されます.<BR>
     *                 [true]が返された場合、ファイルはオープン済みです.<BR>
     *                 [false]が返された場合、ファイルはオープンされていません.
     */
    public final boolean isOpen()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_io.isOpen() ;
            }
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 指定領域のコピー処理.
     * <BR><BR>
     * 指定された領域をコピーします.
     * <BR>
     * @param src コピー元の[ArrayIO]オブジェクトを指定します.
     * @param srcPos コピー元のコピー開始位置を指定します.
     * @param dest コピー先の[ArrayIO]オブジェクトを指定します.
     * @param destPos コピー先のコピー開始位置を指定します.
     * @param length コピーするサイズを指定します.
     * @exception InputException 入力例外.
     */
    public static final void arraycopy( ArrayIO src,int srcPos,ArrayIO dest,int destPos,int length )
        throws InputException
    {
        int i ;
        int len ;
        int etc ;
        int next ;
        
        int pauseLength ;
        int getLength ;
        
        byte[] tmp = null ;
        
        try{
            
            pauseLength = ArrayIO.PAUSE_LENGTH ;
            
            tmp = new byte[pauseLength] ;
            
            len = length / pauseLength ;
            etc = length % pauseLength ;
            
            for( i = 0,next = 0 ; i < len ; i ++ ){
                getLength = src.get( tmp,srcPos + next ) ;
                dest.set( tmp,destPos + next,getLength ) ;
                next += getLength ;
            }
            if( etc != 0 ){
                getLength = src.get( tmp,srcPos + next,etc ) ;
                dest.set( tmp,destPos + next,getLength ) ;
            }
        }catch( InputException in ){
            throw in ;
        }finally{
            tmp = null ;
        }
    }
    
    /**
     * 指定領域のコピー処理.
     * <BR><BR>
     * 指定された領域をコピーします.
     * <BR>
     * @param src コピー元の[byte[]]バイナリを指定します.
     * @param srcPos コピー元のコピー開始位置を指定します.
     * @param dest コピー先の[ArrayIO]オブジェクトを指定します.
     * @param destPos コピー先のコピー開始位置を指定します.
     * @param length コピーするサイズを指定します.
     * @exception InputException 入力例外.
     */
    public static final void arraycopy( byte[] src,int srcPos,ArrayIO dest,int destPos,int length )
        throws InputException
    {
        byte[] tmp = null ;
        
        try{
            tmp = new byte[length] ;
            System.arraycopy( src,srcPos,tmp,0,length ) ;
            dest.set( tmp,destPos ) ;
        }catch( IndexOutOfBoundsException ind ){
            throw new InputException( ind ) ;
        }catch( NullPointerException nul ){
            throw new InputException( nul ) ;
        }catch( InputException in ){
            throw in ;
        }finally{
            tmp = null ;
        }
    }
    
    /**
     * 指定領域のコピー処理.
     * <BR><BR>
     * 指定された領域をコピーします.
     * <BR>
     * @param src コピー元の[ArrayIO]オブジェクトを指定します.
     * @param srcPos コピー元のコピー開始位置を指定します.
     * @param dest コピー先の[byte[]]バイナリを指定します.
     * @param destPos コピー先のコピー開始位置を指定します.
     * @param length コピーするサイズを指定します.
     * @exception InputException 入力例外.
     */
    public static final void arraycopy( ArrayIO src,int srcPos,byte[] dest,int destPos,int length )
        throws InputException
    {
        byte[] tmp = null ;
        
        try{
            tmp = src.get( srcPos,length ) ;
            System.arraycopy( tmp,0,dest,destPos,length ) ;
        }catch( IndexOutOfBoundsException ind ){
            throw new InputException( ind ) ;
        }catch( NullPointerException nul ){
            throw new InputException( nul ) ;
        }catch( InputException in ){
            throw in ;
        }finally{
            tmp = null ;
        }
    }
    
    
}
