/*
 * @(#)BaseException.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.exception;

import java.io.ByteArrayOutputStream;
import java.io.PrintStream;
import java.io.PrintWriter;

import com.JRcServer.commons.def.BaseDef;
import com.JRcServer.commons.util.UtilCom;

/**
 * 共通例外処理.
 * <BR><BR>
 * 共通例外用オブジェクトです.
 *  
 * @version 2003/10/26
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
public class BaseException extends Exception
{
    /**
     * デフォルトバッファ長.
     */
    private static final int DEFAULT_BUF_LEN = 512 ;
    
    /**
     * 区切り囲み腺.
     */
    private static final String CAT_CD = "//" ;
    
    /**
     * 区切り線.
     */
    private static final String CAT_LINE = " ::::::::::::::::: " ;
    
    /**
     * トレース情報区切り線.
     */
    private static final String TRACE_LINE =
        "######################" ;
    
    
    
    /**
     * 例外時のStackTrace書き込み処理( デフォルトはOn )
     */
    private static boolean m_stackTraceFlg = true ;
    
    
    
    /**
     * スタックトレースを格納します。
     */
    protected String m_stackTrace = null ;
    
    /**
     * スタックトレーススコープ数.
     */
    protected int m_traceScope = 0 ;
    
    /**
     * エラー文字情報の設定.
     */
    protected String m_message = null ;
    
    
    
    /**
     * コンタストラクタ.
     */
    public BaseException()
    {
        super() ;
    }
    
    /**
     * コンタストラクタ.
     * <BR><BR>
     * オブジェクト生成と同時に発生原因のExceptionを設定します.
     * <BR>
     * @param e スローエラー対象の例外を設定します。
     */
    public BaseException( Throwable e )
    {
        super( e.getLocalizedMessage() ) ;
        this.saveStackTrace( e ) ;
        m_message = e.getLocalizedMessage() ;
    }
    
    /**
     * コンタストラクタ.
     * <BR><BR>
     * オブジェクト生成と同時に発生原因のExceptionを設定します.
     * <BR>
     * @param e スローエラー対象の例外を設定します。
     */
    public BaseException( BaseException e )
    {
        super( e.getLocalizedMessage() ) ;
        this.saveStackTrace( e ) ;
        m_message = e.getLocalizedMessage() ;
    }
    
    /**
     * コンタストラクタ.
     * <BR><BR>
     * オブジェクト生成と同時にメッセージを設定します.
     * <BR>
     * @param message メッセージ情報を設定します.
     */
    public BaseException( String message )
    {
        super( message ) ;
        this.saveStackTrace( true,this ) ;
        m_message = message ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Throwable 例外処理が返されます.
     */
    protected final void finalize() throws Throwable
    {
        m_stackTrace = null ;
        m_message = null ;
        m_traceScope = 0 ;
    }
    
    
    /**
     * スタックトレースの出力.
     */
    public final void printStackTrace()
    {
        System.out.println( this.readStackTrace() ) ;
    }
    
    /**
     * スタックトレースの出力.
     * <BR><BR>
     * スタックトレースを出力します.
     * <BR>
     * @param s トレース出力先のオブジェクトを設定します.
     */
    public final void printStackTrace( PrintStream s )
    {
        if( s == null ){
            return ;
        }
        else if( this instanceof BaseException ){
            s.print( this.readStackTrace() ) ;
        }
        super.printStackTrace( s ) ;
    }
    
    /**
     * スタックトレースの出力.
     * <BR><BR>
     * スタックトレースを出力します.
     * <BR>
     * @param s トレース出力先のオブジェクトを設定します.
     */
    public final void printStackTrace( PrintWriter s )
    {
        if( s == null ){
            return ;
        }
        else if( this instanceof BaseException ){
            s.print( this.readStackTrace() ) ;
        }
        super.printStackTrace( s ) ;
    }

    
    /**
     * スタックトレース情報を取得します。
     * <BR><BR>
     * スタックトレース情報を取得します。
     * <BR>
     * @return String スタックトレース内容が格納されます。
     */
    public final String readStackTrace()
    {
        
        String traceData = null ;
        StringBuffer buf = null ;
        String ret = null ;
        
        buf = new StringBuffer( BaseException.DEFAULT_BUF_LEN ) ;
        
        if( m_stackTrace == null ){
            
            buf.append( BaseDef.ENTER_SEPARATOR ) ;
            
            buf.append( BaseException.TRACE_LINE ) ;
            buf.append( " ( trace - scope : 1 ) - start " ) ;
            buf.append( BaseException.TRACE_LINE ) ;
            buf.append( BaseDef.ENTER_SEPARATOR ) ;
            
            // スレッド名を表記.
            buf.append( Thread.currentThread().getName() ) ;
            buf.append( BaseDef.ENTER_SEPARATOR ) ;
            
            if(
                m_message != null &&
                "null".equals( ( m_message = m_message.trim() ).toLowerCase() ) == false
            ){
                buf.append( "message(" ) ;
                buf.append( m_message ) ;
                buf.append( ")" ) ;
                buf.append( BaseDef.ENTER_SEPARATOR ) ;
            }
            
            if(
                ( traceData = BaseException.getStackTraceEx( this ) ) != null &&
                "null".equals( traceData ) == false
            ) {
                buf.append( traceData ) ;
            }
            buf.append( BaseException.TRACE_LINE ) ;
            buf.append( " ( trace - scope     ) - end   " ) ;
            buf.append( BaseException.TRACE_LINE ) ;
            buf.append( BaseDef.ENTER_SEPARATOR ) ;
            
        }else{
            
            buf.append( BaseDef.ENTER_SEPARATOR ) ;
            
            buf.append( BaseException.TRACE_LINE ) ;
            buf.append( " ( trace - scope : " ) ;
            buf.append( m_traceScope ) ;
            buf.append( " ) - start " ) ;
            buf.append( BaseException.TRACE_LINE ) ;
            buf.append( BaseDef.ENTER_SEPARATOR ) ;
            
            // スレッド名を表記.
            buf.append( Thread.currentThread().getName() ) ;
            buf.append( BaseDef.ENTER_SEPARATOR ) ;
            
            if(
                m_message != null &&
                "null".equals( ( m_message = m_message.trim() ).toLowerCase() ) == false
            ){
                buf.append( "message(" ) ;
                buf.append( m_message ) ;
                buf.append( ")" ) ;
                buf.append( BaseDef.ENTER_SEPARATOR ) ;
            }
            
            if( m_stackTrace != null && "null".equals( m_stackTrace ) == false ) {
                buf.append( m_stackTrace ) ;
            }
            buf.append( BaseException.TRACE_LINE ) ;
            buf.append( " ( trace - scope     ) - end   " ) ;
            buf.append( BaseException.TRACE_LINE ) ;
            buf.append( BaseDef.ENTER_SEPARATOR ) ;
            
        }
        
        ret = buf.toString() ;
        buf = null ;
        
        return ret ;
        
    }
    
    /**
     * スタックトレース情報の表示有無を設定.
     * <BR><BR>
     * スタックトレース情報の表示有無を設定します.
     * <BR>
     * @param flg スタックトレース情報の表示の有無を設定します。
     */
    public static final void setStackTracFlg( boolean flg )
    {
        m_stackTraceFlg = flg ;
    }
    
    /**
     * メッセージ情報の追加.
     * <BR><BR>
     * 例外オブジェクトに格納されているメッセ－ジに対して、
     * メッセージ情報を追加します.
     * <BR>
     * @param msg 後に追加されるメッセージ情報.
     */
    public final void addMessage( String msg )
    {
        StringBuffer buf = null ;
        
        if( m_message != null ){
            
            buf = new StringBuffer() ;
            buf.append( m_message ) ;
            buf.append( " : " ) ;
            buf.append( msg ) ;
            m_message = buf.toString() ;
            
        }
        else{
            m_message = new String( msg ) ;
        }
        
        buf = null ;
    }
    
    /**
     * スタックトレース情報の表示有無を取得.
     * <BR><BR>
     * スタックトレース情報の表示有無を取得します.
     * <BR>
     * @return スタックトレース情報の表示の有無を取得します.
     */
    public static final boolean getStackTraceFlg()
    {
        return m_stackTraceFlg ;
    }
    
    /**
     * メッセージ情報の取得.
     * <BR><BR>
     * 格納対象のメッセージ情報を取得します.
     * <BR>
     * @return String 取得対象のメッセージが格納されます。
     */
    public final String getMessage()
    {
        return m_message ;
    }
    
    /**
     * Throwable 内スタックトレース情報取得.
     * <BR><BR>
     * 指定したThrowableオブジェクトのスタックトレース内容を取得します.
     * <BR>
     * @param obj 対象のThrowable オブジェクトを指定します.
     * @return String 取得対象のスタックトレースが返されます.<BR>
     *                [null]が返された場合、何らかのエラーにより、
     *                取得できなかったことを示します.
     */
    public static final String getStackTrace( Throwable obj )
    {
        String ret = null ;
        PrintWriter fp = null ;
        ByteArrayOutputStream bin = null ;
        
        try{
            
            bin = new ByteArrayOutputStream() ;
            fp = new PrintWriter( bin ) ;
            obj.printStackTrace( ( PrintWriter )fp ) ;
            fp.flush() ;
            
            ret = new String( bin.toByteArray() ) ;
            
        }catch( Exception t ){
            ret = null ;
        }finally{
            try{ fp.close() ; }catch( Exception ee ){}
            fp = null ;
            bin = null ;
        }
        
        return ret ;
    }
    
    /**
     * Throwable 内スタックトレース情報取得.
     * <BR><BR>
     * 指定したThrowableオブジェクトのスタックトレース内容を取得します.
     * <BR>
     * @param obj 対象のThrowable オブジェクトを指定します.
     * @return String 取得対象のスタックトレースが返されます.<BR>
     *                [null]が返された場合、何らかのエラーにより、
     *                取得できなかったことを示します.
     */
    public static final String getStackTraceEx( Throwable obj )
    {
        PrintWriter fp = null ;
        ByteArrayOutputStream b = null ;
        
        StringBuffer buf = null ;
        String trace = null ;
        String ret = null ;
        
        try{
            
            buf = new StringBuffer() ;
            
            if( obj instanceof BaseException ){
                trace = ( ( BaseException )obj ).m_stackTrace ;
            }
            else{
                b= new ByteArrayOutputStream() ;
                fp = new PrintWriter( b ) ;
                obj.printStackTrace( ( PrintWriter )fp ) ;
                fp.flush() ;
                trace = new String( b.toByteArray() ) ;
                b.close() ;
            }
            
            buf.append( BaseDef.ENTER_SEPARATOR ) ;
            buf.append( BaseException.CAT_LINE ) ;
            buf.append( BaseException.getDateTime() ) ;
            buf.append( BaseException.CAT_LINE ) ;
            buf.append( BaseDef.ENTER_SEPARATOR ) ;
            if( trace != null ) {
                buf.append( trace ) ;
            }
            
            ret = buf.toString() ;
            
        }catch( Exception t ){
            ret = null ;
        }finally{
            
            try{
                fp.close() ;
            }catch( Exception t1 ){
            }
            
            try{
                b.close() ;
            }catch( Exception t1 ){
            }
            
            buf = null ;
            fp = null ;
            b = null ;
        }
        
        return ret ;
    }
    
    /**
     * スタックトレースを記憶.
     * <BR><BR>
     * 対象の例外オブジェクトで発生しているスタックトレース情報を
     * 当例外オブジェクトへ記憶します.
     * <BR>
     * @param e 対象の例外オブジェクトを設定します.
     */
    public final void saveStackTrace( Throwable e )
    {
        this.saveStackTrace( false,e ) ;
    }
    
    /**
     * スタックトレースを記憶.
     * <BR><BR>
     * 対象の例外オブジェクトで発生しているスタックトレース情報を
     * 当例外オブジェクトへ記憶します.
     * <BR>
     * @param mode 実行モードを設定します.
     * @param e 対象の例外オブジェクトを設定します.
     */
    public final void saveStackTrace( boolean mode,Throwable e )
    {
        String traceData = null ;
        PrintWriter fp = null ;
        ByteArrayOutputStream b = null ;
        
        StringBuffer buf = null ;
        
        if( mode == false && ( e instanceof BaseException ) == true ){
            this.saveBaseStackTrace( ( BaseException )e ) ;
        }
        else{
            try{
                
                buf = new StringBuffer() ;
                
                b= new ByteArrayOutputStream() ;
                fp = new PrintWriter( b ) ;
                
                e.printStackTrace( ( PrintWriter )fp ) ;
                fp.flush() ;
                
                traceData = new String( b.toByteArray() ) ;
                b.close() ;
                b = null ;
                fp = null ;
                
                // 現在のスタックトレースを取得( 時間入り )
                buf.append( BaseDef.ENTER_SEPARATOR ) ;
                buf.append( BaseException.CAT_LINE ) ;
                buf.append( BaseException.getDateTime() ) ;
                buf.append( BaseException.CAT_LINE ) ;
                buf.append( BaseDef.ENTER_SEPARATOR ) ;
                if(
                    traceData != null &&
                    "null".equals( traceData ) == false
                ) {
                    buf.append( traceData ) ;
                }
                
                m_stackTrace = buf.toString() ;
                m_traceScope = 1 ;
                
            }catch( Exception t ){
            }finally{
                
                if( fp != null ) {
                    try{
                        fp.close() ;
                    }catch( Exception t1 ){
                    }
                }
                
                if( b != null ) {
                    try{
                        b.close() ;
                    }catch( Exception t1 ){
                    }
                }
                
                buf = null ;
                fp = null ;
                b = null ;
            }
        }
        
    }
    
    /**
     * スタックトレースを記憶.
     * <BR><BR>
     * 対象の例外オブジェクトで発生しているスタックトレース情報を
     * 当例外オブジェクトへ記憶します.
     * <BR>
     * @param com 対象の例外オブジェクトを設定します.
     */
    public final void saveBaseStackTrace( BaseException com )
    {
        
        String traceData = null ;
        StringBuffer buf = null ;
        PrintWriter fp = null ;
        ByteArrayOutputStream b = null ;
        
        try{
            
            buf = new StringBuffer( BaseException.DEFAULT_BUF_LEN ) ;
            
            // 以前のスタックトレースが存在する場合、
            // その情報を先に追加.
            if( com.m_stackTrace != null ){
                buf.append( com.m_stackTrace ) ;
                m_traceScope = com.m_traceScope + 1 ;
            }
            // 存在しない場合、格納スタックトレースを設定.
            else{
                buf.append( BaseException.getStackTraceEx( com ) ) ;
                m_traceScope = 2 ;
            }
            
            b= new ByteArrayOutputStream() ;
            fp = new PrintWriter( b ) ;
            
            com.printStackTrace( ( PrintWriter )fp ) ;
            fp.flush() ;
            
            traceData = new String( b.toByteArray() ) ;
            b.close() ;
            b = null ;
            fp = null ;
            
            // 現在のスタックとレースを取得( 時間入り )
            buf.append( BaseDef.ENTER_SEPARATOR ) ;
            buf.append( BaseException.CAT_LINE ) ;
            buf.append( BaseException.getDateTime() ) ;
            buf.append( BaseException.CAT_LINE ) ;
            buf.append( BaseDef.ENTER_SEPARATOR ) ;
            
            if(
                traceData != null &&
                "null".equals( traceData ) == false
            ) {
                buf.append( traceData ) ;
            }
            
            m_stackTrace = buf.toString() ;
            
        }catch( Exception t ){
            t.printStackTrace() ;
        }finally{
            
            if( fp != null ) {
                try{
                    fp.close() ;
                }catch( Exception t1 ){
                }
            }
            
            if( b != null ) {
                try{
                    b.close() ;
                }catch( Exception t1 ){
                }
            }
            
            buf = null ;
            fp = null ;
            b = null ;
            
        }
    }
    
    
    
    /**
     * 日付情報の設定.
     */
    private static final String getDateTime()
    {
        int i ;
        int len ;
        
        String arr[] = null ;
        String ret = null ;
        StringBuffer buf = null ;
        
        arr = UtilCom.getNowTimestampByString() ;
        buf = new StringBuffer() ;
        
        buf.append( BaseException.CAT_CD ) ;
        len = arr.length-1 ;
        
        for( i = 0 ; i < len ; i ++ ){
            buf.append( arr[ i ] ) ;
            buf.append( ":" ) ;
            arr[ i ] = null ;
        }
        
        buf.append( BaseException.CAT_CD ) ;
        ret = buf.toString() ;
        
        buf = null ;
        arr = null ;
        
        return ret ;
    }
    

}
