/*
 * @(#)ScIO.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.conv;

import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.resource.BinResource;
import com.JRcServer.commons.resource.BufferedBinResource;
import com.JRcServer.commons.resource.Resource;
import com.JRcServer.commons.resource.ResourceType;
import com.JRcServer.commons.util.SearchInt;
import com.JRcServer.commons.util.UtilCom;

/**
 * ScIO(Simple-Code-Input/Output).
 * <BR><BR>
 * 単純暗号入出力処理.<BR>
 * 極めて単純な暗号処理をサポートします.<BR>
 * 利用想定として、マルチキャスト(IPV4)による通信で
 * 利用するように考えています.
 *
 * @version 1.00, 2005/08/02
 * @author  Masahito Suzuki
 * @since   JRcCommons 1.00
 */
public class ScIO
{
    
    /**
     * 基本暗号表.
     */
    private static final byte[] DEF_C256 =
    {
        ( byte )0x007d,( byte )0x005c,( byte )0x0009,( byte )0x00b9,
        ( byte )0x00eb,( byte )0x0095,( byte )0x00c6,( byte )0x0052,
        ( byte )0x0028,( byte )0x0005,( byte )0x0088,( byte )0x0037,
        ( byte )0x00a3,( byte )0x00d8,( byte )0x00de,( byte )0x00bb,
        ( byte )0x0015,( byte )0x000b,( byte )0x00e9,( byte )0x00db,
        ( byte )0x0061,( byte )0x00b1,( byte )0x0041,( byte )0x00b7,
        ( byte )0x00ff,( byte )0x009b,( byte )0x0043,( byte )0x004c,
        ( byte )0x0067,( byte )0x00e6,( byte )0x0047,( byte )0x0079,
        ( byte )0x0071,( byte )0x0082,( byte )0x003f,( byte )0x00df,
        ( byte )0x00a0,( byte )0x00ca,( byte )0x00cb,( byte )0x00c3,
        ( byte )0x0033,( byte )0x00b8,( byte )0x002e,( byte )0x00e0,
        ( byte )0x0070,( byte )0x007c,( byte )0x0031,( byte )0x008e,
        ( byte )0x0090,( byte )0x007a,( byte )0x00f6,( byte )0x00ed,
        ( byte )0x00be,( byte )0x0030,( byte )0x000a,( byte )0x0068,
        ( byte )0x00e5,( byte )0x006b,( byte )0x004d,( byte )0x0055,
        ( byte )0x0016,( byte )0x0099,( byte )0x0027,( byte )0x00a1,
        ( byte )0x001e,( byte )0x00b6,( byte )0x0010,( byte )0x0076,
        ( byte )0x0029,( byte )0x00d5,( byte )0x00d3,( byte )0x00e4,
        ( byte )0x004b,( byte )0x006e,( byte )0x0020,( byte )0x00fe,
        ( byte )0x0007,( byte )0x0004,( byte )0x00ab,( byte )0x0059,
        ( byte )0x008b,( byte )0x0042,( byte )0x003c,( byte )0x005b,
        ( byte )0x0063,( byte )0x0065,( byte )0x00d0,( byte )0x00d9,
        ( byte )0x0035,( byte )0x00c0,( byte )0x00a8,( byte )0x0094,
        ( byte )0x0008,( byte )0x00fd,( byte )0x00f1,( byte )0x00cc,
        ( byte )0x0017,( byte )0x0018,( byte )0x00ee,( byte )0x00e1,
        ( byte )0x001c,( byte )0x001f,( byte )0x0032,( byte )0x00dd,
        ( byte )0x0060,( byte )0x00fb,( byte )0x001b,( byte )0x0050,
        ( byte )0x00ec,( byte )0x0062,( byte )0x00f9,( byte )0x0066,
        ( byte )0x00cd,( byte )0x00a7,( byte )0x0093,( byte )0x0038,
        ( byte )0x0078,( byte )0x0045,( byte )0x002d,( byte )0x0091,
        ( byte )0x0083,( byte )0x00f8,( byte )0x00da,( byte )0x0048,
        ( byte )0x00f4,( byte )0x00e3,( byte )0x004a,( byte )0x000c,
        ( byte )0x00e8,( byte )0x0044,( byte )0x00c9,( byte )0x0075,
        ( byte )0x00dc,( byte )0x002f,( byte )0x007e,( byte )0x0025,
        ( byte )0x00e2,( byte )0x000d,( byte )0x00e7,( byte )0x0064,
        ( byte )0x00f2,( byte )0x0057,( byte )0x009f,( byte )0x001a,
        ( byte )0x003e,( byte )0x008f,( byte )0x0021,( byte )0x0084,
        ( byte )0x0001,( byte )0x0023,( byte )0x005d,( byte )0x00d7,
        ( byte )0x000e,( byte )0x008c,( byte )0x00ce,( byte )0x0013,
        ( byte )0x00ad,( byte )0x0054,( byte )0x0040,( byte )0x00b2,
        ( byte )0x00f7,( byte )0x00b4,( byte )0x0012,( byte )0x0058,
        ( byte )0x0081,( byte )0x00af,( byte )0x0049,( byte )0x0024,
        ( byte )0x00f3,( byte )0x0019,( byte )0x0006,( byte )0x0014,
        ( byte )0x003b,( byte )0x006f,( byte )0x0053,( byte )0x00a5,
        ( byte )0x00ef,( byte )0x0098,( byte )0x0069,( byte )0x000f,
        ( byte )0x00fa,( byte )0x00a6,( byte )0x00c2,( byte )0x00a4,
        ( byte )0x00c1,( byte )0x0097,( byte )0x0056,( byte )0x00a9,
        ( byte )0x007b,( byte )0x0022,( byte )0x00c5,( byte )0x0072,
        ( byte )0x0002,( byte )0x0080,( byte )0x00aa,( byte )0x00c4,
        ( byte )0x0077,( byte )0x0096,( byte )0x004f,( byte )0x0003,
        ( byte )0x0034,( byte )0x003a,( byte )0x0089,( byte )0x00ea,
        ( byte )0x00bf,( byte )0x009d,( byte )0x0026,( byte )0x0011,
        ( byte )0x005f,( byte )0x006d,( byte )0x00b5,( byte )0x0073,
        ( byte )0x005e,( byte )0x00d1,( byte )0x007f,( byte )0x001d,
        ( byte )0x00b3,( byte )0x002b,( byte )0x009a,( byte )0x008d,
        ( byte )0x00a2,( byte )0x00fc,( byte )0x0039,( byte )0x0085,
        ( byte )0x00b0,( byte )0x0051,( byte )0x0046,( byte )0x008a,
        ( byte )0x00d4,( byte )0x00d2,( byte )0x006c,( byte )0x002a,
        ( byte )0x0036,( byte )0x00bd,( byte )0x00f5,( byte )0x00bc,
        ( byte )0x0074,( byte )0x009e,( byte )0x00cf,( byte )0x00d6,
        ( byte )0x006a,( byte )0x0092,( byte )0x0087,( byte )0x00c7,
        ( byte )0x005a,( byte )0x00c8,( byte )0x0086,( byte )0x00ae,
        ( byte )0x0000,( byte )0x00ac,( byte )0x002c,( byte )0x00ba,
        ( byte )0x003d,( byte )0x009c,( byte )0x00f0,( byte )0x004e
    } ;
    
    /**
     * 暗号表長.
     */
    public static final int TABLE_LENGTH = 256 ;
    
    /**
     * デフォルト基本コード.
     */
    private static final int DEF_BCODE = 0x065bb502 ;
    
    /**
     * 基本コード : 最小値.
     */
    private static final int MIN_BCODE = 0 ;
    
    /**
     * 基本コード : 最大値.
     */
    private static final int MAX_BCODE = 268435200 ;
    
    /**
     * 暗号コード : マスク情報.
     */
    private static final int CODE_MASK = 0x000000ff ;
    
    /**
     * KEY生成係数.
     */
    private static final double CREATE_KEY_DEFINE = 1.692358f ;
    
    /**
     * デフォルトバイナリリソースタイプ.
     */
    private static final ResourceType DEF_RES = new ResourceType() ;
    
    
    
    /**
     * コンストラクタ.
     */
    private ScIO(){}
    
    
    
    /**
     * 暗号処理.
     * <BR><BR>
     * 暗号処理を行います.
     * <BR>
     * @param out 暗号対象のバイナリを設定します.
     * @param code 対象の暗号コードを設定します.
     * @param offset 開始位置を設定します.
     * @param length データ長を設定します.
     * @exception InputException 入力例外.
     */
    public static final void input( byte[] out,int code,int offset,int length )
        throws InputException
    {
        BinResource res = null ;
        
        if( out == null || out.length <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            res = Resource.createBinResource( DEF_RES,out ) ;
            ScIO.exec( true,res,DEF_BCODE,DEF_C256,code,offset,length ) ;
        }catch( InputException e ){
            throw e ;
        }finally{
            
            if( res != null ){
                res.clear() ;
            }
            res = null ;
            
        }
    }
    
    /**
     * 暗号処理.
     * <BR><BR>
     * 暗号処理を行います.
     * <BR>
     * @param out 暗号対象のバイナリを設定します.
     * @param table 暗号表を設定します.<BR>
     *              この情報は[256]byteの条件で設定する必要があります.
     * @param code 対象の暗号コードを設定します.
     * @param offset 開始位置を設定します.
     * @param length データ長を設定します.
     * @exception InputException 入力例外.
     */
    public static final void input( byte[] out,byte[] table,int code,int offset,int length )
        throws InputException
    {
        BinResource res = null ;
        
        if( out == null || out.length <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            res = Resource.createBinResource( DEF_RES,out ) ;
            ScIO.exec( true,res,DEF_BCODE,table,code,offset,length ) ;
        }catch( InputException e ){
            throw e ;
        }finally{
            
            if( res != null ){
                res.clear() ;
            }
            res = null ;
            
        }
    }
    
    /**
     * 暗号処理.
     * <BR><BR>
     * 暗号処理を行います.
     * <BR>
     * @param out 暗号対象のバイナリを設定します.
     * @param bCd 対象の基本コードを設定します.<BR>
     *            設定可能な範囲は[0]から[268435200]まで設定可能です.
     * @param code 対象の暗号コードを設定します.
     * @param offset 開始位置を設定します.
     * @param length データ長を設定します.
     * @exception InputException 入力例外.
     */
    public static final void input( byte[] out,int bCd,int code,int offset,int length )
        throws InputException
    {
        BinResource res = null ;
        
        if( out == null || out.length <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            res = Resource.createBinResource( DEF_RES,out ) ;
            ScIO.exec( true,res,bCd,DEF_C256,code,offset,length ) ;
        }catch( InputException e ){
            throw e ;
        }finally{
            
            if( res != null ){
                res.clear() ;
            }
            res = null ;
            
        }
    }
    
    /**
     * 暗号処理.
     * <BR><BR>
     * 暗号処理を行います.
     * <BR>
     * @param out 暗号対象のバイナリを設定します.
     * @param bCd 対象の基本コードを設定します.<BR>
     *            設定可能な範囲は[0]から[268435200]まで設定可能です.
     * @param table 暗号表を設定します.<BR>
     *              この情報は[256]byteの条件で設定する必要があります.
     * @param code 対象の暗号コードを設定します.
     * @param offset 開始位置を設定します.
     * @param length データ長を設定します.
     * @exception InputException 入力例外.
     */
    public static final void input( byte[] out,int bCd,byte[] table,int code,int offset,int length )
        throws InputException
    {
        BinResource res = null ;
        
        if( out == null || out.length <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            res = Resource.createBinResource( DEF_RES,out ) ;
            ScIO.exec( true,res,bCd,table,code,offset,length ) ;
        }catch( InputException e ){
            throw e ;
        }finally{
            
            if( res != null ){
                res.clear() ;
            }
            res = null ;
            
        }
    }
    
    /**
     * 暗号処理.
     * <BR><BR>
     * 暗号処理を行います.
     * <BR>
     * @param out 暗号対象のバイナリを設定します.
     * @param code 対象の暗号コードを設定します.
     * @param offset 開始位置を設定します.
     * @param length データ長を設定します.
     * @exception InputException 入力例外.
     */
    public static final void input( BinResource out,int code,int offset,int length )
        throws InputException
    {
        ScIO.exec( true,out,DEF_BCODE,DEF_C256,code,offset,length ) ;
    }
    
    /**
     * 暗号処理.
     * <BR><BR>
     * 暗号処理を行います.
     * <BR>
     * @param out 暗号対象のバイナリを設定します.
     * @param table 暗号表を設定します.<BR>
     *              この情報は[256]byteの条件で設定する必要があります.
     * @param code 対象の暗号コードを設定します.
     * @param offset 開始位置を設定します.
     * @param length データ長を設定します.
     * @exception InputException 入力例外.
     */
    public static final void input( BinResource out,byte[] table,int code,int offset,int length )
        throws InputException
    {
        ScIO.exec( true,out,DEF_BCODE,table,code,offset,length ) ;
    }
    
    /**
     * 暗号処理.
     * <BR><BR>
     * 暗号処理を行います.
     * <BR>
     * @param out 暗号対象のバイナリを設定します.
     * @param bCd 対象の基本コードを設定します.<BR>
     *            設定可能な範囲は[0]から[268435200]まで設定可能です.
     * @param code 対象の暗号コードを設定します.
     * @param offset 開始位置を設定します.
     * @param length データ長を設定します.
     * @exception InputException 入力例外.
     */
    public static final void input( BinResource out,int bCd,int code,int offset,int length )
        throws InputException
    {
        ScIO.exec( true,out,bCd,DEF_C256,code,offset,length ) ;
    }
    
    /**
     * 暗号処理.
     * <BR><BR>
     * 暗号処理を行います.
     * <BR>
     * @param out 暗号対象のバイナリを設定します.
     * @param bCd 対象の基本コードを設定します.<BR>
     *            設定可能な範囲は[0]から[268435200]まで設定可能です.
     * @param table 暗号表を設定します.<BR>
     *              この情報は[256]byteの条件で設定する必要があります.
     * @param code 対象の暗号コードを設定します.
     * @param offset 開始位置を設定します.
     * @param length データ長を設定します.
     * @exception InputException 入力例外.
     */
    public static final void input( BinResource out,int bCd,byte[] table,int code,int offset,int length )
        throws InputException
    {
        ScIO.exec( true,out,bCd,table,code,offset,length ) ;
    }
    
    /**
     * 解析処理.
     * <BR><BR>
     * 暗号化された情報を解析します.
     * <BR>
     * @param out 暗号解析対象のバイナリを設定します.
     * @param code 対象の暗号コードを設定します.
     * @param offset 開始位置を設定します.
     * @param length データ長を設定します.
     * @exception InputException 入力例外.
     */
    public static final void output( byte[] out,int code,int offset,int length )
        throws InputException
    {
        BinResource res = null ;
        
        if( out == null || out.length <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            res = Resource.createBinResource( DEF_RES,out ) ;
            ScIO.exec( false,res,DEF_BCODE,DEF_C256,code,offset,length ) ;
        }catch( InputException e ){
            throw e ;
        }finally{
            
            if( res != null ){
                res.clear() ;
            }
            res = null ;
            
        }
    }
    
    /**
     * 解析処理.
     * <BR><BR>
     * 暗号化された情報を解析します.
     * <BR>
     * @param out 暗号解析対象のバイナリを設定します.
     * @param table 暗号表を設定します.<BR>
     *              この情報は[256]byteの条件で設定する必要があります.
     * @param code 対象の暗号コードを設定します.
     * @param offset 開始位置を設定します.
     * @param length データ長を設定します.
     * @exception InputException 入力例外.
     */
    public static final void output( byte[] out,byte[] table,int code,int offset,int length )
        throws InputException
    {
        BinResource res = null ;
        
        if( out == null || out.length <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            res = Resource.createBinResource( DEF_RES,out ) ;
            ScIO.exec( false,res,DEF_BCODE,table,code,offset,length ) ;
        }catch( InputException e ){
            throw e ;
        }finally{
            
            if( res != null ){
                res.clear() ;
            }
            res = null ;
            
        }
    }
    
    /**
     * 解析処理.
     * <BR><BR>
     * 暗号化された情報を解析します.
     * <BR>
     * @param out 暗号解析対象のバイナリを設定します.
     * @param bCd 対象の基本コードを設定します.<BR>
     *            設定可能な範囲は[0]から[268435200]まで設定可能です.
     * @param code 対象の暗号コードを設定します.
     * @param offset 開始位置を設定します.
     * @param length データ長を設定します.
     * @exception InputException 入力例外.
     */
    public static final void output( byte[] out,int bCd,int code,int offset,int length )
        throws InputException
    {
        BinResource res = null ;
        
        if( out == null || out.length <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            res = Resource.createBinResource( DEF_RES,out ) ;
            ScIO.exec( false,res,bCd,DEF_C256,code,offset,length ) ;
        }catch( InputException e ){
            throw e ;
        }finally{
            
            if( res != null ){
                res.clear() ;
            }
            res = null ;
            
        }
    }
    
    /**
     * 解析処理.
     * <BR><BR>
     * 暗号化された情報を解析します.
     * <BR>
     * @param out 暗号解析対象のバイナリを設定します.
     * @param bCd 対象の基本コードを設定します.<BR>
     *            設定可能な範囲は[0]から[268435200]まで設定可能です.
     * @param table 暗号表を設定します.<BR>
     *              この情報は[256]byteの条件で設定する必要があります.
     * @param code 対象の暗号コードを設定します.
     * @param offset 開始位置を設定します.
     * @param length データ長を設定します.
     * @exception InputException 入力例外.
     */
    public static final void output( byte[] out,int bCd,byte[] table,int code,int offset,int length )
        throws InputException
    {
        BinResource res = null ;
        
        if( out == null || out.length <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            res = Resource.createBinResource( DEF_RES,out ) ;
            ScIO.exec( false,res,bCd,table,code,offset,length ) ;
        }catch( InputException e ){
            throw e ;
        }finally{
            
            if( res != null ){
                res.clear() ;
            }
            res = null ;
            
        }
    }
    
    /**
     * 解析処理.
     * <BR><BR>
     * 暗号化された情報を解析します.
     * <BR>
     * @param out 暗号解析対象のバイナリを設定します.
     * @param code 対象の暗号コードを設定します.
     * @param offset 開始位置を設定します.
     * @param length データ長を設定します.
     * @exception InputException 入力例外.
     */
    public static final void output( BinResource out,int code,int offset,int length )
        throws InputException
    {
        ScIO.exec( false,out,DEF_BCODE,DEF_C256,code,offset,length ) ;
    }
    
    /**
     * 解析処理.
     * <BR><BR>
     * 暗号化された情報を解析します.
     * <BR>
     * @param out 暗号解析対象のバイナリを設定します.
     * @param table 暗号表を設定します.<BR>
     *              この情報は[256]byteの条件で設定する必要があります.
     * @param code 対象の暗号コードを設定します.
     * @param offset 開始位置を設定します.
     * @param length データ長を設定します.
     * @exception InputException 入力例外.
     */
    public static final void output( BinResource out,byte[] table,int code,int offset,int length )
        throws InputException
    {
        ScIO.exec( false,out,DEF_BCODE,table,code,offset,length ) ;
    }
    
    /**
     * 解析処理.
     * <BR><BR>
     * 暗号化された情報を解析します.
     * <BR>
     * @param out 暗号解析対象のバイナリを設定します.
     * @param bCd 対象の基本コードを設定します.<BR>
     *            設定可能な範囲は[0]から[268435200]まで設定可能です.
     * @param code 対象の暗号コードを設定します.
     * @param offset 開始位置を設定します.
     * @param length データ長を設定します.
     * @exception InputException 入力例外.
     */
    public static final void output( BinResource out,int bCd,int code,int offset,int length )
        throws InputException
    {
        ScIO.exec( false,out,bCd,DEF_C256,code,offset,length ) ;
    }
    
    /**
     * 解析処理.
     * <BR><BR>
     * 暗号化された情報を解析します.
     * <BR>
     * @param out 暗号解析対象のバイナリを設定します.
     * @param bCd 対象の基本コードを設定します.<BR>
     *            設定可能な範囲は[0]から[268435200]まで設定可能です.
     * @param table 暗号表を設定します.<BR>
     *              この情報は[256]byteの条件で設定する必要があります.
     * @param code 対象の暗号コードを設定します.
     * @param offset 開始位置を設定します.
     * @param length データ長を設定します.
     * @exception InputException 入力例外.
     */
    public static final void output( BinResource out,int bCd,byte[] table,int code,int offset,int length )
        throws InputException
    {
        ScIO.exec( false,out,bCd,table,code,offset,length ) ;
    }
    
    /**
     * 新しい暗号表を生成.
     * <BR><BR>
     * 新しいユニークでない暗号表を生成します.<BR>
     * 条件として、0-255までのテーブルをランダムに生成します.<BR>
     * このときの暗号表は、ランダムに割り当てられます.<BR>
     * そのため、ユニークな暗号表を利用する場合は、メソッド
     * [ScIO.getNewUniqueTable()]を利用してください.
     * <BR>
     * @return byte[] 新しい暗号表が返されます.
     */
    public static final byte[] getNewTable()
    {
        int i ;
        
        byte[] ret = null ;
        
        ret = new byte[ TABLE_LENGTH ] ;
        
        for( i = 0 ; i < TABLE_LENGTH ; i ++ ){
            ret[ i ] = ( byte )UtilCom.random( TABLE_LENGTH ) ;
        }
        
        return ret ;
    }
    
    /**
     * 新しいユニーク暗号表を生成.
     * <BR><BR>
     * 新しいユニーク暗号表を生成します.<BR>
     * 条件として、0-255までのテーブルをランダムに生成します.<BR>
     * このときの暗号表は、ユニークに割り当てられます.<BR>
     * そのため、ユニークでない暗号表を利用する場合は、メソッド
     * [ScIO.getNewTable()]を利用してください.
     * <BR>
     * @return byte[] 新しい暗号表が返されます.
     */
    public static final byte[] getNewUniqueTable()
    {
        int i ;
        int now ;
        
        SearchInt si = null ;
        byte[] ret = null ;
        
        si = new SearchInt() ;
        ret = new byte[ TABLE_LENGTH ] ;
        
        for( i = 0 ; i < TABLE_LENGTH ; i ++ ){
            
            now = UtilCom.random( TABLE_LENGTH ) ;
            
            if( si.isData( now ) == false ){
                si.add( now ) ;
                ret[ i ] = ( byte )( now & 0x000000ff ) ;
            }
            else{
                i -- ;
            }
        }
        
        if( si != null ){
            si.clear() ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * 暗号化/解析処理.
     * mode=true  : 暗号.
     * mode=false : 解析.
     */
    private static final void exec( boolean mode,BinResource out,int bCd,byte[] table,int code,int offset,int length )
        throws InputException
    {
        int i,j ;
        int len ;
        int key ;
        int now ;
        int bef ;
        int tmp ;
        
        BufferedBinResource buf = null ;
        
        if( 
            out == null || out.isUse() == false || ( len = out.size() ) <= 0 ||
            bCd < MIN_BCODE || bCd > MAX_BCODE ||
            table == null || table.length != TABLE_LENGTH ||
            offset < 0 || offset > len ||
            length <= 0 || length > len ||
            ( offset + length ) > len
        )
        {
            if( out == null || out.isUse() == false || ( len = out.size() ) <= 0 ){
                throw new InputException( "受け取り側のバイナリ条件は不正です" ) ;
            }
            else if( bCd < MIN_BCODE || bCd > MAX_BCODE ){
                throw new InputException( "対象の基本コード(" + bCd + ")は範囲外です" ) ;
            }
            else if( table == null || table.length != TABLE_LENGTH ){
                throw new InputException( "指定された暗号表は不正です" ) ;
            }
            else if(
                offset < 0 || offset > len ||
                length <= 0 || length > len ||
                ( offset + length ) > len
            )
            {
                throw new InputException(
                    "指定されたバイナリ範囲(target:" + len +
                    " offset:" + offset +
                    " length:" + length +
                    ")は範囲外です"
                ) ;
            }
            
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            // 基本コードを変換.
            key = (
                (
                    ( int )(
                        ( double )(
                            ( bCd & 0x000000ff ) -
                            ( ( bCd & 0x0000ff00 ) >> 8 ) +
                            ( ( bCd & 0x00ff0000 ) >> 16 ) -
                            ( ( bCd & 0xff000000 ) >> 24 )
                        ) * CREATE_KEY_DEFINE
                    )
                ) & 0x000000ff
            ) ;
            
            // 対象コードをマスク処理.
            code = code & 0x000000ff ;
            
            // 最初の条件を生成.
            if( ( code & 0x00000001 ) == 0 ){
                bef = (
                    (
                        ( ( (~code) ^ table[ key ] ) & 0x000000ff ) +
                        ( table[ length & 0x000000ff ] ) +
                        ( ( ~( offset - 1 ) ) & 0x000000ff )
                    ) & 0x000000ff
                ) ;
            }
            else{
                bef = (
                    (
                        ( ( table[ code ] ^ (~key) ) & 0x000000ff ) +
                        ( ( ~length ) & 0x000000ff ) +
                        ( table[ ( offset - 1 ) & 0x000000ff ] )
                    ) & 0x000000ff
                ) ;
            }
            
            // バイナリバッファを生成.
            buf = new BufferedBinResource( out ) ;
            
            /////////////
            // 暗号処理.
            /////////////
            if( mode == true ){
                
                // パターン-A.
                if( ( code & 0x00000001 ) == 0 ){
                
                    for( i = 0,j = offset ; i < length ; i ++,j ++ ){
                        
                        now = buf.get( j ) ;
                        
                        // 奇数の場合.
                        if( ( j & 0x00000001 ) == 1 ){
                            tmp = CodeCommon.flipBit0x3c(
                                ( ~( now ^ table[ ( bef & 0x000000ff ) ] ) )
                            ) & 0x000000ff ;
                        }
                        // 偶数の場合.
                        else{
                            tmp = CodeCommon.flipBit0xc3(
                                ( now ^ table[ ( bef & 0x000000ff ) ] )
                            ) & 0x000000ff ;
                        }
                        
                        buf.set( j,tmp ) ;
                        bef = CodeCommon.flipCode(
                            ( ( bef + now + table[ ( i & 0x000000ff ) ] ) ),table[ ( j & 0x000000ff ) ]
                        ) & 0x000000ff ;
                        
                    }
                    
                }
                // パターン-B.
                else{
                    
                    for( i = 0,j = offset ; i < length ; i ++,j ++ ){
                        
                        now = buf.get( j ) ;
                        
                        // 偶数の場合.
                        if( ( i & 0x00000001 ) == 0 ){
                            tmp = CodeCommon.flipBit0x3c(
                                ~( now ^ table[ ( bef & 0x000000ff ) ] )
                            ) & 0x000000ff ;
                        }
                        // 奇数の場合.
                        else{
                            tmp = CodeCommon.flipBit0xc3(
                                ( now ^ ( ~table[ ( bef & 0x000000ff ) ] ) )
                            ) & 0x000000ff ;
                        }
                        
                        buf.set( j,tmp ) ;
                        bef = CodeCommon.flipCode(
                            ( bef + now + table[ ( j & 0x000000ff ) ] ),table[ ( i & 0x000000ff ) ]
                        ) & 0x000000ff ;
                        
                    }
                    
                }
                
            }
            ////////////
            // 解析処理.
            ////////////
            else{
                
                // パターン-A.
                if( ( code & 0x00000001 ) == 0 ){
                
                    for( i = 0,j = offset ; i < length ; i ++,j ++ ){
                        
                        now = buf.get( j ) ;
                        
                        // 奇数の場合.
                        if( ( j & 0x00000001 ) == 1 ){
                            tmp = (
                                ( CodeCommon.flipBit0x3c( ~now ) ^ table[ ( bef & 0x000000ff ) ] )
                            ) & 0x000000ff ;
                        }
                        // 偶数の場合.
                        else{
                            tmp = (
                                ( CodeCommon.flipBit0xc3( now ) ^ table[ ( bef & 0x000000ff ) ] )
                            ) & 0x000000ff ;
                        }
                        
                        buf.set( j,tmp ) ;
                        bef = CodeCommon.flipCode(
                            ( ( bef + tmp + table[ ( i & 0x000000ff ) ] ) ),table[ ( j & 0x000000ff ) ]
                        ) & 0x000000ff ;
                        
                    }
                    
                }
                // パターン-B.
                else{
                    
                    for( i = 0,j = offset ; i < length ; i ++,j ++ ){
                        
                        now = buf.get( j ) ;
                        
                        // 偶数の場合.
                        if( ( i & 0x00000001 ) == 0 ){
                            tmp = (
                                ( CodeCommon.flipBit0x3c( ~now ) ^ table[ ( bef & 0x000000ff ) ] )
                            ) & 0x000000ff ;
                        }
                        // 奇数の場合.
                        else{
                            tmp = (
                                ( CodeCommon.flipBit0xc3( now ) ^ ( ~table[ ( bef & 0x000000ff ) ] ) )
                            ) & 0x000000ff ;
                        }
                        
                        buf.set( j,tmp ) ;
                        bef = CodeCommon.flipCode(
                            ( bef + tmp + table[ ( j & 0x000000ff ) ] ),table[ ( i & 0x000000ff ) ]
                        ) & 0x000000ff ;
                        
                    }
                    
                }
                
            }
            
            buf.clear() ;
            buf = null ;
            
        }catch( Exception e ){
            throw new InputException( e ) ;
        }finally{
            if( buf != null ){
                buf.clear() ;
            }
            buf = null ;
        }
        
    }
    
}

