/*
 * @(#)CodeBase32.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.conv;

import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;

import com.JRcServer.commons.def.BaseDef;
import com.JRcServer.commons.exception.AccessException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.exception.NotEqualInfoException;
import com.JRcServer.commons.exception.UnsupportException;
import com.JRcServer.commons.resource.BinMemoryResource;
import com.JRcServer.commons.resource.BinResource;
import com.JRcServer.commons.resource.BufferedBinResource;
import com.JRcServer.commons.util.UtilCom;

/**
 * 暗号 CodeBase32 実装オブジェクト.
 * <BR><BR>
 * 暗号 CodeBase32 実装オブジェクトをサポートします.
 *
 * @version 1.00, 2003/12/09
 * @author  Masahito Suzuki
 * @since   JRcCommons 1.00
 */
public class CodeBase32
{
    
    /**
     * 変換コード値.
     */
    private static final String CHARSET = BaseDef.UTF8 ;
    
    /**
     * ステップコードマスク.
     */
    private static final int STEP_MASK = 0x0000007f ;
    
    /**
     * ステップ条件マスク.
     */
    private static final int STEP_IF_MASK = 0x0000002 ;
    
    /**
     * セキュリティキー変換長.
     */
    private static final int CODE32_TO_CODE256CNV = 256 ;
    
    /**
     * 基本プライベートキー.
     */
    private static final byte[] BACK_PRIVATE_KEY = {
        ( byte )0xfe,( byte )0x01,( byte )0xed,( byte )0x12,
        ( byte )0xdc,( byte )0x23,( byte )0xcb,( byte )0x34,
        ( byte )0x45,( byte )0xba,( byte )0x56,( byte )0xa9,
        ( byte )0x67,( byte )0x98,( byte )0x78,( byte )0x87,
        ( byte )0x76,( byte )0x89,( byte )0x65,( byte )0x9a,
        ( byte )0x54,( byte )0xab,( byte )0x43,( byte )0xbc,
        ( byte )0xcd,( byte )0x32,( byte )0xde,( byte )0x21,
        ( byte )0xef,( byte )0x10,( byte )0xff,( byte )0x00
    } ;
    
    /**
     * メッセージダイジェスト : MD5.
     */
    private static final String MD5_DIGEST = "MD5" ;
    
    /**
     * MD5データ長.
     */
    private static final int MD5_LENGTH = 16 ;
    
    
    
    /**
     * チェックステップコードマスキング範囲.
     */
    public static final int CHECK_STEPMASK = 0x000000ff ;
    
    /**
     * 無効ステップコード.
     */
    public static final int NOT_STEP = -1 ;
    
    /**
     * セキュリティーキー長.
     */
    public static final int CODE32_KEY_LENGTH = 32 ;
    
    /**
     * マスク配列.
     */
    private static final int[] MASK_ARRAY = {
        0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,
        18,19,20,21,22,23,24,25,26,27,28,29,30,31,1
    } ;
    
    
    
    /**
     * プライベート格納キー.
     */
    private final byte[] m_privateKey = new byte[ CodeBase32.CODE32_KEY_LENGTH ] ;
    
    /**
     * パブリック格納キー.
     */
    private final byte[] m_publicKey = new byte[ CodeBase32.CODE32_KEY_LENGTH ] ;
    
    /**
     * CodeBase32ステップコード.
     */
    private int m_code32StepCode = 0 ;
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 情報を生成します.<BR>
     * この情報で生成した場合、デフォルトのプライベートキーを
     * ベースとした暗号処理を実施します.
     */
    public CodeBase32()
    {
        this.create() ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 情報を生成します.<BR>
     * この情報で生成した場合、プライベートキーを指定した
     * 暗号処理を実施します.
     * <BR>
     * @param privateKey 対象のプライベートキーを設定します.
     * @exception InputException 入力例外.
     */
    public CodeBase32( byte[] privateKey )
        throws InputException
    {
        try{
            this.create( privateKey ) ;
        }catch( InputException in ){
            throw in ;
        }
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 情報を生成します.<BR>
     * この情報で生成した場合、デフォルトのプライベートキーを
     * ベースとした暗号処理を実施します.
     */
    public final void create()
    {
        this.clear() ;
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 情報を生成します.<BR>
     * この情報で生成した場合、プライベートキーを指定した
     * 暗号処理を実施します.
     * <BR>
     * @param privateKey 対象のプライベートキーを設定します.
     * @exception InputException 入力例外.
     */
    public final void create( byte[] privateKey )
        throws InputException
    {
        if(
            privateKey == null ||
            privateKey.length != CodeBase32.CODE32_KEY_LENGTH
        )
        {
            throw new InputException( "対象のプライベートキーは不正です" ) ;
        }
        
        this.clear() ;
        System.arraycopy( privateKey,0,m_privateKey,0,CodeBase32.CODE32_KEY_LENGTH ) ;
        
    }
    
    /**
     * 情報のクリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public final void clear()
    {
        System.arraycopy( 
            CodeBase32.BACK_PRIVATE_KEY,
            0,m_privateKey,0,
            CodeBase32.CODE32_KEY_LENGTH
        ) ;
        
        m_code32StepCode = CodeBase32.NOT_STEP ;
    }
    
    /**
     * 暗号処理.
     * <BR><BR>
     * 暗号処理を実施します.
     * <BR>
     * @param key 対象のパブリックキー情報を設定します.<BR>
     *            パブリックキーを内部で自動生成する場合は、[ null ]を設定してください.
     * @param binary 暗号対象のバイナリ情報を設定します.
     * @param offset 暗号化開始位置を設定します.
     * @param size 暗号化開始位置[ offset ]から暗号を行うサイズを設定します.
     * @return int 暗号ステップコード値が返されます.<BR>
     *             暗号化されたバイナリ情報の解析処理には、暗号ステップコード値が必要と
     *             なります.<BR>
     *             また、[ CodeBase32#getStepCode() ]メソッドからも同一の情報が取得されます.
     * @exception InputException 入力例外.
     */
    public final int encryption( 
        byte[] key,byte[] binary,int offset,int size
     )
        throws InputException
    {
        return this.encryption( key,new BinMemoryResource( binary ),offset,size ) ;
    }
    
    /**
     * 暗号処理.
     * <BR><BR>
     * 暗号処理を実施します.
     * <BR>
     * @param key 対象のパブリックキー情報を設定します.<BR>
     *            パブリックキーを内部で自動生成する場合は、[ null ]を設定してください.
     * @param binary 暗号対象のバイナリリソースを設定します.
     * @param offset 暗号化開始位置を設定します.
     * @param size 暗号化開始位置[ offset ]から暗号を行うサイズを設定します.
     * @return int 暗号ステップコード値が返されます.<BR>
     *             暗号化されたバイナリ情報の解析処理には、暗号ステップコード値が必要と
     *             なります.<BR>
     *             また、[ CodeBase32#getStepCode() ]メソッドからも同一の情報が取得されます.
     * @exception InputException 入力例外.
     */
    public final int encryption( 
        byte[] key,BinResource binary,int offset,int size
     )
        throws InputException
    {
        int len ;
        int step ;
        
        byte[] baseKey = null ;
        byte[] code32Key = null ;
        byte[] cd32 = null ;
        byte[] cd256 = null ;
        
        BufferedBinResource buf = null ;
        
        if( 
            binary == null || offset < 0 ||
            size < 0 || binary.size() < offset + size
         )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        code32Key = m_publicKey ;
        
        buf = new BufferedBinResource( binary ) ;
        
        try{
            
            // CodeBase32キーが設定されていない場合、新規作成.
            if( key == null ){
                CodeBase32.createPublicKey( code32Key ) ;
                baseKey = new byte[ CodeBase32.CODE32_KEY_LENGTH ] ;
                System.arraycopy( code32Key,0,baseKey,0,CodeBase32.CODE32_KEY_LENGTH ) ;
            }else{
                baseKey = key ;
                System.arraycopy( key,0,code32Key,0,CodeBase32.CODE32_KEY_LENGTH ) ;
            }
            
            // 暗号キーをデフォルトキーで変換.
            cd32 = CodeBase32.encryptionCodeBase32Key( code32Key,m_privateKey ) ;
            
            // 暗号キーを256バイトに引き伸ばす.
            cd256 = CodeBase32.code32To256( cd32 ) ;
            cd32 = null ;
            
            // ステップコードの生成.
            step = CodeBase32.getStepNo( baseKey,buf,offset,size ) ;
            
            // ステップコードをマスキングしてメンバー変数にセット.
            m_code32StepCode = step & CodeBase32.CHECK_STEPMASK ;
            
            // ステップコードを利用可能な状態に変換.
            step = step & CodeBase32.STEP_MASK ;
            
            // 暗号キー( 256 )をデフォルトキーで変換.
            step = CodeBase32.cnv256( cd256,m_privateKey,step ) ;
            
            // 指定バイナリの暗号処理.
            CodeBase32.cnvBinary( buf,offset,size,cd256,step ) ;
            
        }catch( RuntimeException e ){
            throw e ;
        }finally{
            
            if( buf != null ){
                buf.clear() ;
            }
            
            baseKey = null ;
            cd32 = null ;
            cd256 = null ;
            code32Key = null ;
            buf = null ;
            
        }
        
        return m_code32StepCode ;
    }
    
    /**
     * 解析処理.
     * <BR><BR>
     * 暗号化された情報を解析します.
     * <BR>
     * @param key 対象のパブリックキー情報を設定します.
     * @param step 暗号ステップコードを設定します.<BR>
     *             この情報は、暗号処理時に生成されたステップコード値を設定する
     *             必要があります.
     * @param binary 暗号対象のバイナリ情報を設定します.
     * @param offset 暗号化開始位置を設定します.
     * @param size 暗号化開始位置[ offset ]から暗号するサイズを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final void analysis( 
        byte[] key,int step,byte[] binary,int offset,int size
     )
        throws InputException,AccessException
    {
        this.analysis( key,step,new BinMemoryResource( binary ),offset,size ) ;
    }
    
    /**
     * 解析処理.
     * <BR><BR>
     * 暗号化された情報を解析します.
     * <BR>
     * @param key 対象のパブリックキー情報を設定します.
     * @param step 暗号ステップコードを設定します.<BR>
     *             この情報は、暗号処理時に生成されたステップコード値を設定する
     *             必要があります.
     * @param binary 暗号対象のバイナリリソースを設定します.
     * @param offset 暗号化開始位置を設定します.
     * @param size 暗号化開始位置[ offset ]から暗号するサイズを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final void analysis( 
        byte[] key,int step,BinResource binary,int offset,int size
     )
        throws InputException,AccessException
    {
        int len ;
        int checkStep,setStep ;
        
        byte[] code32Key = null ;
        byte[] cd32 = null ;
        byte[] cd256 = null ;
        
        BufferedBinResource buf = null ;
        
        if( 
            key == null || binary == null || 
            offset < 0 || size < 0 ||
            binary.size() < offset + size
         )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        code32Key = m_publicKey ;
        
        buf = new BufferedBinResource( binary ) ;
        
        try{
            
            // CodeBase32キーをメンバー変数に登録.
            System.arraycopy( key,0,code32Key,0,CodeBase32.CODE32_KEY_LENGTH ) ;
            
            // 暗号キーをデフォルトキーで変換.
            cd32 = CodeBase32.encryptionCodeBase32Key( code32Key,m_privateKey ) ;
            
            // 暗号キーを256バイトに引き伸ばす.
            cd256 = CodeBase32.code32To256( cd32 ) ;
            cd32 = null ;
            
            // セットステップコードをマスク.
            setStep = step & CodeBase32.CHECK_STEPMASK ;
            
            // ステップコードを利用可能な状態に変換.
            step = step & CodeBase32.STEP_MASK ;
            
            // 暗号キー( 256 )をデフォルトキーで変換.
            step = CodeBase32.cnv256( cd256,m_privateKey,step ) ;
            
            // 指定バイナリの暗号解析処理.
            CodeBase32.cnvBinary( buf,offset,size,cd256,step ) ;
            
            // チェック用ステップコードの生成.
            checkStep = CodeBase32.getStepNo( key,buf,offset,size ) ;
            
            cd32 = null ;
            cd256 = null ;
            
            // 指定ステップコードが、解析結果のステップコードと一致しない場合.
            if( setStep != ( checkStep & CodeBase32.CHECK_STEPMASK ) ){
                
                // 一致しない場合は、情報内容が不正である.
                throw new AccessException(
                    "解析対象条件(setStep:" + setStep +
                    " : checkStep:" + checkStep +
                    ")は一致しません"
                ) ;
                
            }
        
        }catch( AccessException ac ){
            throw ac ;
        }catch( Exception e ){
            throw new AccessException( e ) ;
        }finally{
            
            if( buf != null ){
                buf.clear() ;
            }
            
            cd32 = null ;
            cd256 = null ;
            
        }
        
        // ステップコードをメンバー変数にセット.
        m_code32StepCode = setStep ;
        
    }
    
    /**
     * ユーザ名／パスワード情報から、Keyを生成.
     * <BR><BR>
     * ユーザ名／パスワード情報から、Key情報を生成します.
     * <BR>
     * @param user ユーザ名を指定します.
     * @param passwd パスワード情報を指定します.
     * @return byte[] 生成されたKeyが返されます.
     * @exception InputException 入力例外.
     * @exception NotEqualInfoException 情報不一致例外.
     */
    public static final byte[] getUserPasswdByKey( String user,String passwd )
        throws InputException,NotEqualInfoException
    {
        StringBuffer buf = null ;
        byte[] ret = null ;
        
        if( user == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            buf = new StringBuffer() ;
            buf.append( "-uSer+" ) ;
            buf.append( user ) ;
            buf.append( "+PasSwD-" ) ;
            buf.append(
                ( ( passwd == null || passwd.length() <= 0 ) ?
                    "nOpASSwoRd" : passwd
                )
            ) ;
            
            ret = CodeBase32.convertStringByCode32Key( buf.toString() ) ;
            
        }catch( InputException in ){
            throw in ;
        }catch( NotEqualInfoException ne ){
            throw ne ;
        }finally{
            buf = null ;
        }
        
        return ret ;
    }
    
    /**
     * 指定文字列から、key情報を生成.
     * <BR><BR>
     * 指定された文字列から、key情報を生成します.<BR>
     * また、文字列はAsciiコードである必要があります.
     * <BR>
     * @param string 変換対象の文字列を設定します.
     * @return byte[] 変換されたkeyコードが返されます.
     * @exception InputException 入力例外.
     */
    public static final byte[] convertStringByCode32Key( String string )
        throws InputException,NotEqualInfoException
    {
        int i,j,k ;
        int before ;
        int beforeTmp ;
        int stepRoop ;
        int len ;
        int stepLen ;
        int stepMask ;
        int originStep ;
        int step ;
        
        int next ;
        int tmp ;
        int tmpBin ;
        
        int[] mask_ary ;
        BufferedBinResource buf = null ;
        byte[] code = null ;
        byte[] ret = null ;
        
        if( string == null || string.length() <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        mask_ary = MASK_ARRAY ;
        
        try{
            
            //code = SHA1.convertBinary( code ) ;
            code = SHA1.convertBinary( string.getBytes( BaseDef.UTF8 ) ) ;
            code = CodeBase32.convertBinaryToUnicodeBinary(
                code,0,code.length
            ) ;
            
            //code = Asc64.convertAsc64ByEnc( 
            //    Asc64.convertAsciiByAsc64(
            //        Asc64.rounds(
            //            SHA1.convert(
            //                string.getBytes( BaseDef.UTF8 )
            //            )
            //            //string.getBytes( BaseDef.UTF8 )
            //        )
            //    )
            //) ;
            
            buf = new BufferedBinResource( new BinMemoryResource( code ) ) ;
            
            len = code.length ;
            step = CodeBase32.getStepNo( code,buf,0,len ) ;
            step = ( ( len & STEP_IF_MASK ) != 0 ) ? step - len : step + len ;
            originStep = ( step & 0x000000ff );
            
            ret = new byte[ CodeBase32.CODE32_KEY_LENGTH ] ;
            System.arraycopy( CodeBase32.BACK_PRIVATE_KEY,0,ret,0,CodeBase32.CODE32_KEY_LENGTH ) ;
            for( i = 0 ; i < CodeBase32.CODE32_KEY_LENGTH ; i ++ ) {
                ret[ i ] = ( byte )( ( ( ret[ i ] & 0x000000ff ) + step + i ) & 0x000000ff ) ;
            }
            stepLen = ( step & 0x0000001f ) ;
            stepMask = CodeBase32.CODE32_KEY_LENGTH - 1 ;
            
            for( stepRoop = 0,before = originStep ; stepRoop < stepLen ; stepRoop ++ ) {
                for(
                    i = 0,j = 0,k = ( stepRoop & stepMask ),
                    next = ( step & 0x00000007 ),tmp = 0 ;
                    i < len ;
                    i ++,j ++,k ++
                ) {
                    
                    j &= 0x00000007 ;
                    k = mask_ary[ k ] ;
                    
                    switch( j ){
                        
                        case 0 :
                            next += ( int )( ( ~code[ i ] ) & 0x000000fe ) ;
                            tmp = ( ( ( (next+1) + (~step) ) & 0x00000020 ) >> 4 ) ;
                            break ;
                        case 1 :
                            next += ( int )( ( code[ i ] ) & 0x000000df ) ;
                            tmp = ( ( ( (next) ^ originStep ) & 0x000000060 ) >> 5 ) ;
                            break ;
                        case 2 :
                            next += ( int )( ( ~code[ i ] ) & 0x0000007f ) ;
                            tmp = ( ( ( (next+1) + (~step) ) & 0x0000018 ) >> 3 ) ;
                            break ;
                        case 3 :
                            next += ( int )( ( ~code[ i ] ) & 0x000000fb ) ;
                            tmp = ( ( (next) ^ step ) & 0x00000002 ) ;
                            break ;
                        case 4 :
                            next += ( int )( ( code[ i ] ) & 0x000000ef ) ;
                            tmp = ( ( ( (next+1) + (~originStep) ) & 0x0000000c0 ) >> 6 ) ;
                            break ;
                        case 5 :
                            next += ( int )( ( ~code[ i ] ) & 0x000000bf ) ;
                            tmp = ( ( ( (next) + step ) & 0x00000006 ) >> 1 ) ;
                            break ;
                        case 6 :
                            next += ( int )( ( code[ i ] ) & 0x000000f7 ) ;
                            tmp = ( ( ( (next+1) + (~step) ) & 0x0000000c ) >> 2 ) ;
                            break ;
                        case 7 :
                            next += ( int )( ( ~code[ i ] ) & 0x000000fd ) ;
                            tmp = ( ( ( (next) + originStep ) & 0x00000020 ) >> 4 ) ;
                            break ;
                            
                    }
                    
                    step = ( step & next ) & 0x00000007 ;
                    tmpBin = ( byte )CodeCommon.flipCode( ( byte )( next & 0x000000ff ),step ) ;
                    
                    switch( tmp ){
                        
                        case 0 :
                            ret[ k ] += ( byte )
                                ( ( ( (~ret[ k ]) & 0x000000ff ) + (~tmpBin) ) & 0x000000ff ) ;
                            break ;
                        case 1 :
                            ret[ k ] += ( byte )
                                (
                                    ( ( ( ret[ k ] & 0x000000ff ) ^ tmpBin ) + originStep ) &
                                    0x000000ff
                                ) ;
                            break ;
                        case 2 :
                            ret[ k ] += ( byte )
                                ( ( ret[ k ] + ( tmpBin & 0x000000ff ) ) & 0x000000ff ) ;
                            break ;
                        case 3 :
                            ret[ k ] += ( byte )
                                (
                                    ( ( ret[ k ] + ( (~tmpBin) & 0x000000ff ) ) + originStep ) &
                                    0x000000ff
                                ) ;
                            break ;
                        
                    }
                    
                    beforeTmp = before ;
                    before = ( int )( ret[ k ] & 0x000000ff ) ;
                    ret[ k ] = ( byte )(
                        ( ( ( ret[ k ] & 0x000000ff ) + beforeTmp ) >> 1 ) &
                        0x000000ff ) ;
                    
                }
                
            }
            
        }catch( Exception e ){
            ret = null ;
            throw new NotEqualInfoException( e ) ;
        }finally{
            if( buf != null ){
                buf.clear() ;
            }
            code = null ;
            buf = null ;
        }
        
        return ret ;
    }
    
    /**
     * パブリックキーコードを発行.
     * <BR><BR>
     * 新しいパブリックキーコードを発行します.
     * <BR>
     * @return byte[] 発行されたパブリックキーコードを取得します.
     */
    public static final byte[] getPublicKey()
    {
        byte[] ret = null ;
        
        ret = new byte[ CodeBase32.CODE32_KEY_LENGTH ] ;
        CodeBase32.createPublicKey( ret ) ;
        
        return ret ;
    }
    
    /**
     * 設定されているプライベートキー情報の取得.
     * <BR><BR>
     * 設定されているプライベートキー情報を取得します.
     * <BR>
     * @return byte[] 設定されているプライベートキー情報が返されます.
     */
    public final byte[] getSettingPrivateKey()
    {
        return m_privateKey ;
    }
    
    /**
     * 設定されているパブリックキー情報の取得.
     * <BR><BR>
     * 設定されているパブリックキー情報を取得します.
     * <BR>
     * @return byte[] 設定されているパブリックキー情報が返されます.
     */
    public final byte[] getSettingPublicKey()
    {
        return m_publicKey ;
    }
    
    /**
     * 暗号ステップコード情報の取得.
     * <BR><BR>
     * 暗号化されたときのステップコード情報を取得します.<BR>
     * この情報は、暗号の解析を行うときに必要となります.
     * <BR>
     * @return int 設定されているステップコード値を取得します.<BR>
     *             暗号/解析処理を実施していない場合[CodeBase32.NOT_STEP]が返されます.
     */
    public final int getStepCode()
    {
        return m_code32StepCode ;
    }
    
    
    /**
     * パブリックキーコードを発行.
     */
    private static final void createPublicKey( byte[] cd32 )
    {
        int i ;
        int len ;
        
        len = CodeBase32.CODE32_KEY_LENGTH ;
        
        for( i = 0 ; i < len ; i ++ ){
            cd32[ i ] = ( byte )UtilCom.random( 0x00000100 ) ;
        }
    }
    
    /**
     * コンバート処理.
     */
    private static final byte convert( byte[] key,int no,byte pause )
    {
        /*
        byte ret ;
        
        if( ( no & 0x00000001 ) == 0 ){
            ret = ( byte )( ~ ( pause ^ key[ no ] ) ) ;
        }
        else{
            ret = ( byte )( pause ^ key[ no ] ) ;
        }
        
        return ret ;
        */
        switch( ( no & 0x00000001 ) ) {
            case 0 : return ( byte )( ~ ( pause ^ key[ no ] ) ) ;
            case 1 : return ( byte )( ~ ( pause ^ key[ no ] ) ) ;
        }
        return -1 ;
    }
    
    /**
     * 暗号キー( CodeBase32Key )をデフォルトキーで暗号化.
     */
    private static final byte[] encryptionCodeBase32Key( byte[] cd32,byte[] pri32Key )
        throws InputException
    {
        /*
        int i ;
        int len ;
        
        byte[] ret = null ;
        
        len = CodeBase32.CODE32_KEY_LENGTH ;
        ret = new byte[ len ] ;
        
        for( i = 0 ; i < len ; i ++ ){
            ret[ i ] = CodeBase32.convert( pri32Key,i,cd32[ i ] ) ;
        }
        
        return ret ;
        */
        
        int i,j,k ;
        int len ;
        
        byte[] low = null ;
        byte[] hight = null ;
        byte[] ret = null ;
        
        low = MD5.convertBinary( cd32 ) ;
        hight = SHA1.convertBinary( pri32Key ) ;
        
        ret = new byte[ CodeBase32.CODE32_KEY_LENGTH ] ;
        len = low.length ;
        
        for( i = 0,j = 0,k = len-1 ; i < len ; i ++,j += 2,k -- ) {
            //ret[ j ] = low[ i ] ;
            //ret[ j+1 ] = ( byte )( hight[ i ] ^ low[ k ] ) ;
            ret[ j ] = CodeBase32.convert( low,i,cd32[ j ] ) ;
            ret[ j+1 ] = CodeBase32.convert( hight,i,low[ k ] ) ;
        }
        //System.out.println( UtilCom.convertBinaryTo16String( ret,',',0,ret.length ) ) ;
        return ret ;
    }
    
    /**
     * CodeBase32キーコードを256バイト変換.
     */
    private static final byte[] code32To256( byte[] key )
    {
        int i,j,k ;
        int len ;
        int oneLen ;
        int befLen ;
        int next ;
        
        byte[] tmp = null ;
        byte[] binary = null ;
        MessageDigest md = null ;
        
        try{
            
            // 32バイトデータを256バイトデータに引き伸ばす.
            binary = new byte[ CODE32_TO_CODE256CNV ] ;
            len = CodeBase32.CODE32_KEY_LENGTH - 1 ;
            
            for( i = 1,k = 0,next = 0 ; i < len ; i ++ ){
                
                next = ( ( key[ i ] << 8 ) & 0x0000ff00 ) | key[ i + 1 ] ;
                
                binary[ k ]     = ( byte )(     next & 0x000000ff ) ;
                binary[ k + 1 ] = ( byte )( ( ( next & 0x000001fe ) >> 1 ) & 0x000000ff ) ;
                binary[ k + 2 ] = ( byte )( ( ( next & 0x000003fc ) >> 2 ) & 0x000000ff ) ;
                binary[ k + 3 ] = ( byte )( ( ( next & 0x000007f8 ) >> 3 ) & 0x000000ff ) ;
                binary[ k + 4 ] = ( byte )( ( ( next & 0x00000ff0 ) >> 4 ) & 0x000000ff ) ;
                binary[ k + 5 ] = ( byte )( ( ( next & 0x00001fe0 ) >> 5 ) & 0x000000ff ) ;
                binary[ k + 6 ] = ( byte )( ( ( next & 0x00003fc0 ) >> 6 ) & 0x000000ff ) ;
                binary[ k + 7 ] = ( byte )( ( ( next & 0x00007f80 ) >> 7 ) & 0x000000ff ) ;
                
                k += 8 ;
                
            }
            
            next = ( ( key[ len ] << 8 ) & 0x0000ff00 ) | key[ 0 ] ;
            
            binary[ k ]     = ( byte )( next & 0x000000ff ) ;
            binary[ k + 1 ] = ( byte )( ( ( next & 0x000001fe ) >> 1 ) & 0x000000ff ) ;
            binary[ k + 2 ] = ( byte )( ( ( next & 0x000003fc ) >> 2 ) & 0x000000ff ) ;
            binary[ k + 3 ] = ( byte )( ( ( next & 0x000007f8 ) >> 3 ) & 0x000000ff ) ;
            binary[ k + 4 ] = ( byte )( ( ( next & 0x00000ff0 ) >> 4 ) & 0x000000ff ) ;
            binary[ k + 5 ] = ( byte )( ( ( next & 0x00001fe0 ) >> 5 ) & 0x000000ff ) ;
            binary[ k + 6 ] = ( byte )( ( ( next & 0x00003fc0 ) >> 6 ) & 0x000000ff ) ;
            binary[ k + 7 ] = ( byte )( ( ( next & 0x00007f80 ) >> 7 ) & 0x000000ff ) ;
            
            // 生成されたバイナリをMD5に変換.
            oneLen = CodeBase32.MD5_LENGTH ;
            len = CODE32_TO_CODE256CNV ;
            md = MessageDigest.getInstance( CodeBase32.MD5_DIGEST ) ;
            
            for( i = 0,j = oneLen ; j < len ; i ++ ){
                
                md.update( binary,j,oneLen ) ;
                
                tmp = md.digest() ;
                oneLen = tmp.length ;
                oneLen = ( j+oneLen >= len ) ? len-j : oneLen ;
                System.arraycopy( tmp,0,binary,j,oneLen ) ;
                md.reset() ;
                
                j += oneLen ;
                
            }
            
        }catch( Exception e ){
            e.printStackTrace() ;
            binary = null ;
        }finally{
            md = null ;
        }
        
        return binary ;
    }
    
    /**
     * 対象バイナリ情報をUnicode変換.
     * [ binary ] -> [ string 注１ ] -> [ unicode( byte ) ]..
     *  注１: 文字列に変換する方法として、以下のような方法を
     *  用いています..
     *  binary [ 0x01 ] [ 0x02 ] [ 0x03 ].....
     *  string [ 010203..... ]
     */
    private static final byte[] convertBinaryToUnicodeBinary(
        byte[] binary,int offset,int size
     )
        throws UnsupportException
    {
        String tmp = null ;
        byte[] ret = null ;
        
        try{
            
            tmp = UtilCom.convertBinaryTo16String( binary,offset,size ) ;
            ret = tmp.getBytes( CodeBase32.CHARSET ) ;
            
        }catch( UnsupportedEncodingException ue ){
            throw new UnsupportException( ue ) ;
        }finally{
            tmp = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象のユニコード情報をバイナリに変換.
     * CodeBase32#convertBinaryToUnicodeBinary()で変換した内容を元に戻す.
     */
    private static final byte[] convertUnicodeBinaryToBinary(
        byte[] binary
     )
        throws UnsupportException
    {
        
        String uni = null ;
        byte[] ret ;
        
        try{
            
            uni = new String( binary,CodeBase32.CHARSET ) ;
            ret = UtilCom.convert16StringToBinary( false,uni ) ;
            
            return ret ;
        }catch( InputException in ){
            throw new UnsupportException( in ) ;
        }catch( UnsupportedEncodingException ue ){
            throw new UnsupportException( ue ) ;
        }finally{
            uni = null ;
        }
    }
    
    /**
     * ステップNo情報の取得.
     */
    private static final int getStepNo( byte[] publicKey,BufferedBinResource binary,int offset,int size )
    {
        int i,j ;
        int addCd ;
        int len ;
        
        int bin ;
        int ret ;
        
        ret = 0 ;
        len = offset + size ;
        
        addCd = ( int )(
            ( publicKey[
                ( int )( binary.get( offset + ( size / 2 ) ) & 0x0000001f )
            ] & 0x00000003 ) + 1
        ) ;
        
        for( i = offset,j = 0 ; i < len ; i += addCd,j += addCd ){
            
            bin = ( int )( ( ~ binary.get( i ) ) & 0x000000ff ) ;
            
            ret = ( 
                (   bin & 0x00000001 ) +
                ( ( bin & 0x00000002 ) >> 1 ) +
                ( ( bin & 0x00000004 ) >> 2 ) +
                ( ( bin & 0x00000008 ) >> 3 ) +
                ( ( bin & 0x00000010 ) >> 4 ) +
                ( ( bin & 0x00000020 ) >> 5 ) +
                ( ( bin & 0x00000040 ) >> 6 ) +
                ( ( bin & 0x00000080 ) >> 7 )
            ) + ( j & 0x000000ff ) + ret ;
            
        }
        
        if( ( ret & 0x00000001 ) == 0 ){
            
            for( i = 0 ; i < CodeBase32.CODE32_KEY_LENGTH ; i ++ ){
                
                bin = ( int )( ( ( publicKey[ i ] & 0x00000001 ) == 0 ) ?
                    ( ( ~ publicKey[ i ] ) & 0x000000ff ) :
                    ( publicKey[ i ] & 0x000000ff )
                ) ;
                
                ret += ( 
                    (   bin & 0x00000001 ) +
                    ( ( bin & 0x00000002 ) >> 1 ) +
                    ( ( bin & 0x00000004 ) >> 2 ) +
                    ( ( bin & 0x00000008 ) >> 3 ) +
                    ( ( bin & 0x00000010 ) >> 4 ) +
                    ( ( bin & 0x00000020 ) >> 5 ) +
                    ( ( bin & 0x00000040 ) >> 6 ) +
                    ( ( bin & 0x00000080 ) >> 7 )
                ) ;
                
            }
            
        }
        else{
            
            for( i = 0 ; i < CodeBase32.CODE32_KEY_LENGTH ; i ++ ){
                
                bin = ( int )( ( ( publicKey[ i ] & 0x00000001 ) == 0 ) ?
                    ( ( ~ publicKey[ i ] ) & 0x000000ff ) :
                    ( publicKey[ i ] & 0x000000ff )
                ) ;
                
                ret -= ( 
                    (   bin & 0x00000001 ) +
                    ( ( bin & 0x00000002 ) >> 1 ) +
                    ( ( bin & 0x00000004 ) >> 2 ) +
                    ( ( bin & 0x00000008 ) >> 3 ) +
                    ( ( bin & 0x00000010 ) >> 4 ) +
                    ( ( bin & 0x00000020 ) >> 5 ) +
                    ( ( bin & 0x00000040 ) >> 6 ) +
                    ( ( bin & 0x00000080 ) >> 7 )
                ) ;
                
            }
            
        }
        
        return ( int )( ( ( ~ret ) | 0x00000080 ) & 0x000000ff ) ;
    }
    
    /**
     * 生成256コード加工処理.
     */
    private static final int cnv256( byte[] code256,byte[] def32,int step )
    {
        int i,j ;
        int len ;
        int nowStep ;
        
        nowStep = step & 0x0000007f ;
        len = CodeBase32.CODE32_TO_CODE256CNV ;
        
        for( i = 0,j = 0 ; i < len ; i ++,j = ( ( j + 1 ) & 0x0000001f ) ){
            
            nowStep = ( nowStep & ( int )code256[ i ] ) & 0x00000007 ;
            code256[ i ] = ( byte )CodeCommon.flipCode( ( int )code256[ i ],nowStep ) ;
            code256[ i ] = CodeBase32.convert( def32,j,code256[ i ] ) ;
            
        }
        return nowStep ;
    }
    
    /**
     * 指定バイナリ情報の加工.
     */
    private static final void cnvBinary( 
        BufferedBinResource binary,int offset,int size,byte[] code256,int step
     )
    {
        int i,j ;
        int len ;
        int nowStep ;
        
        nowStep = step ;
        len = offset + size ;
        
        for( i = offset,j = 0 ; i < len ; i ++,j = ( ( j + 1 ) & 0x000000ff ) ){
            
            nowStep = ( ( nowStep & ( int )( code256[ j ] & 0x000000ff ) ) & 0x00000007 ) ;
            binary.set( i,( byte )CodeCommon.flipCode( ( byte )binary.get( i ),nowStep ) ) ;
            binary.set( i,CodeBase32.convert( code256,j,( byte )binary.get( i ) ) ) ;
            
        }
    }
    
    
}
