/*
 * @(#)Asc64.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.conv;

import com.JRcServer.commons.def.BaseDef;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.exception.NotEqualInfoException;

/**
 * ASC64情報変換処理群.
 * <BR><BR>
 * ASC64情報変換処理をサポートします.
 *
 * @version 1.00, 2004/01/04
 * @author  Masahito Suzuki
 * @since   JRcCommons 1.00
 */
public class Asc64
{
    
    /**
     * 対象文字列をASC64に丸める.
     * <BR><BR>
     * 対象文字列をASC64に丸めます.
     * <BR>
     * @param string 対象文字列を設定します.
     * @return byte[] 丸められた内容が返されます.
     * @exception Exception 例外.
     */
    public static final byte[] rounds( String string )
        throws Exception {
        
        int i ;
        int len ;
        
        byte[] ret = null ;
        
        if( string == null || string.length() <= 0 ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        ret = string.getBytes( BaseDef.UTF8 ) ;
        len = ret.length ;
        
        for( i = 0 ; i < len ; i ++ ) {
            ret[ i ] = ( byte )( ret[ i ] & 0x0000003f ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象バイナリをASC64に丸める.
     * <BR><BR>
     * 対象バイナリをASC64に丸めます.
     * <BR>
     * @param binary 対象バイナリを設定します.
     * @return byte[] 丸められた内容が返されます.
     * @exception InputException 入力例外.
     */
    public static final byte[] rounds( byte[] binary )
        throws InputException {
        
        int i ;
        int len ;
        byte[] ret = null ;
        
        if( binary == null || binary.length <= 0 ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        ret = binary ;
        len = ret.length ;
        
        for( i = 0 ; i < len ; i ++ ) {
            ret[ i ] = ( byte )( ret[ i ] & 0x0000003f ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * ASC64コード変換 ASCII -> ASC64.
     * <BR><BR>
     * 対象のASCIIコードをASC64コードに変換します.<BR>
     * また、変換対象となるASCIIコードは、以下の通りです.<BR>
     * <pre>
     * [0 - 9] : 半角数値.
     * [a - z] : 半角英数小文字
     * [A - Z] : 半角英数大文字
     * [-]     : 半角ハイフン.
     * [+]     : 半角プラス.
     * </pre>
     * <BR>
     * 上記以外の情報が含まれている場合[0x0000003f]で丸められます.
     * <BR>
     * @param asc 変換対象のASCIIコードを設定します.
     * @return int[] 変換されたASC64コードが返されます.
     * @exception InputException 入力例外.
     */
    public static final int[] convertAsciiByAsc64( byte[] asc )
        throws InputException
    {
        int i ;
        int len ;
        
        int[] ret = null ;
        
        if( asc == null || ( len = asc.length ) <= 0 ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        ret = new int[ len ] ;
        
        try{
            for( i = 0 ; i < len ; i ++ ){
                
                ret[ i ] = ( int )(
                    Asc64Def.convertASC_BY_ASC64(
                        ( char )( asc[ i ] & 0x000000ff )
                    ) & 0x0000003f
                ) ;
                
            }
        }catch( Exception e ){
            ret = null ;
            throw new InputException( e ) ;
        }
        
        return ret ;
    }
    
    /**
     * ASC64コード変換 ASC64 -> ASCII.
     * <BR><BR>
     * 対象のASC64コード情報を、ASCIIに変換します.
     * <BR>
     * @param asc64 対象のASC64コードを設定します.
     * @return String 変換されたASCIIコードが返されます.
     * @exception InputException 入力例外.
     * @exception NotEqualInfoException 情報不一致例外.
     */
    public static final String convertAsc64ByAscii( int[] asc64 )
        throws InputException,NotEqualInfoException
    {
        int i ;
        int len ;
        
        StringBuffer buf = null ;
        String ret = null ;
        
        if( asc64 == null || ( len = asc64.length ) <= 0 ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        buf = new StringBuffer( len ) ;
        
        try{
            
            for( i = 0 ; i < len ; i ++ ){
                
                
                buf.append( Asc64Def.ASC64_BY_ASC[ asc64[ i ] ] ) ;
                
            }
            
            ret = buf.toString() ;
            
        }catch( IndexOutOfBoundsException io ){
            
            throw new NotEqualInfoException(
                "対象の文字列に範囲外のパスワードコードが含まれています"
            ) ;
            
        }finally{
            buf = null ;
        }
        
        return ret ;
    }
    
    
    /**
     * 対象のASC64コードを暗号化.
     * <BR><BR>
     * 対象のASC64コードを暗号化します.
     * <BR>
     * @param asc64 暗号化対象のASC64コードを設定します.
     * @return byte[] 暗号化されたバイナリ情報が返されます.
     * @exception InputException 入力例外.
     */
    public static final byte[] convertAsc64ByEnc( int[] asc64 )
        throws InputException
    {
        int i,j,k ;
        int len ;
        int pause ;
        int binLen ;
        
        byte[] ret = null ;
        
        if( asc64 == null || ( len = asc64.length ) <= 0 ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        binLen = ( len * 2 ) / 8 ;
        binLen = len - binLen + 2 ;
        
        ret = new byte[ binLen ] ;
        
        pause = ( (~( len & 0x000000ff ) ) & 0x000000ff ) ;
        pause = CodeCommon.flipBit0xff( pause ) ;
        ret[ 0 ] = ( byte )pause ;
        
        pause = ( (~( ( len & 0x0000ff00 ) >> 8 ) ) & 0x000000ff ) ;
        pause = CodeCommon.flipBit0xff( pause ) ;
        ret[ 1 ] = ( byte )pause ;
        
        for( i = 0,j = 2,k = 0 ; i < len ; i ++,k ++ ){
            
            pause = asc64[ i ] ;
            
            switch( k ){
                
                case 0 :
                    
                    pause = ~pause ;
                    pause = CodeCommon.flipBit0x3f( pause ) ;
                    ret[ j ] = ( byte )( pause & 0x0000003f ) ;
                    
                    break ;
                    
                case 1 :
                    
                    ret[ j ] |= ( byte )((( pause & 0x00000003 ) << 6 ) & 0x000000c0 ) ;
                    j ++ ;
                    ret[ j ] = ( byte )((( pause & 0x0000003c ) >> 2 ) & 0x0000000f ) ;
                    
                    break ;
                    
                case 2 :
                    
                    pause = ( ( ~pause ) & 0x0000003f ) ;
                    ret[ j ] |= ( byte )((( pause & 0x0000000f ) << 4 ) & 0x000000f0 ) ;
                    j ++ ;
                    ret[ j ] = ( byte )((( pause & 0x00000030 ) >> 4 ) & 0x00000003 ) ;
                    
                    break ;
                    
                case 3 :
                    
                    pause = CodeCommon.flipBit0x3f( pause ) ;
                    ret[ j ] |= ( byte )((( pause & 0x0000003f ) << 2 ) & 0x000000fc ) ;
                    j ++ ;
                    k = -1 ;
                    
                    break ;
            }
            
        }
        
        return ret ;
    }
    
    /**
     * 対象の暗号化されたASC64コードを元に戻す.
     * <BR><BR>
     * 対象の暗号化されたASC64コードを元に戻します.
     * <BR>
     * @param enc 対象の暗号化されたASC64コードを設定します.
     * @return int[] 元に戻されたASC64コード情報が返されます.
     * @exception InputException 入力例外.
     */
    public static final int[] convertEncByAsc64( byte[] enc )
        throws InputException
    {
        int i,j,k ;
        int len ;
        int intLen ;
        int pause ;
        int binInt ;
        
        int[] ret = null ;
        
        if( enc == null || ( len = enc.length ) <= 2 ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        pause = ( int )( enc[ 0 ] & 0x000000ff ) ;
        pause = ( ( ~pause ) & 0x000000ff ) ;
        pause = CodeCommon.flipBit0xff( pause ) ;
        intLen = ( pause & 0x000000ff ) ;
        
        pause = ( int )( enc[ 1 ] & 0x000000ff ) ;
        pause = ( ( ~pause ) & 0x000000ff ) ;
        pause = CodeCommon.flipBit0xff( pause ) ;
        intLen |= ( ( ( pause & 0x000000ff ) << 8 ) & 0x0000ff00 ) ;
        
        ret = new int[ intLen ] ;
        
        for( i = 2,j = 0,k = 0 ; i < len ; i ++,k ++ ){
            
            binInt = ( int )enc[ i ] ;
            
            switch( k ){
                
                case 0 :
                    
                    pause = ( binInt & 0x0000003f ) ;
                    pause = ~pause ;
                    ret[ j ] = CodeCommon.flipBit0x3f( pause ) ;
                    j ++ ;
                    
                    pause = ((( binInt & 0x000000c0 ) >> 6 ) & 0x00000003 ) ;
                    break ;
                    
                case 1 :
                    
                    pause |= ((( binInt & 0x0000000f ) << 2 ) & 0x0000003c ) ;
                    ret[ j ] = pause ;
                    j ++ ;
                    
                    pause = ((( binInt & 0x000000f0 ) >> 4 ) & 0x0000000f ) ;
                    break ;
                    
                case 2 :
                    
                    pause |= ((( binInt & 0x00000003 ) << 4 ) & 0x00000030 ) ;
                    ret[ j ] = ( ( ~pause ) & 0x0000003f ) ;
                    j ++ ;
                    
                    pause = ((( binInt & 0x000000fc ) >> 2 ) & 0x0000003f ) ;
                    ret[ j ] = CodeCommon.flipBit0x3f( pause ) ;
                    j ++ ;
                    
                    k = -1 ;
                    break ;
                    
            }
            
        }
        
        return ret ;
    }
    
    /**
     * 対象文字列が有効範囲内であるかチェック.
     * <BR><BR>
     * 対象文字列が有効範囲内であるかチェックします.<BR>
     * また、変換対象となるASCIIコードは、以下の通りです.<BR>
     * <pre>
     * [0 - 9] : 半角数値.
     * [a - z] : 半角英数小文字
     * [A - Z] : 半角英数大文字
     * [-]     : 半角ハイフン.
     * [+]     : 半角プラス.
     * </pre>
     * <BR>
     * @param string チェック対象の文字列を設定します.<BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返される場合、範囲内です.<BR>
     *                 [false]が返される場合、範囲外です.
     */
    public static final boolean isAsc64( String string )
    {
        int i ;
        int len ;
        boolean ret ;
        
        try{
            
            len = string.length() ;
            
            for( i = 0,ret = true ; i < len ; i ++ ){
                
                if(
                    Asc64Def.convertASC_BY_ASC64(
                        string.charAt( i )
                    ) == Asc64Def.NOT_ASC_64
                )
                {
                    ret = false ;
                    break ;
                }
                
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * コンストラクタ.
     */
    private Asc64()
    {
    }

}

