/*
 * @(#)BaseConfig.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.commons.conf;

import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.exception.ReadSyntaxException;
import com.JRcServer.commons.io.FileAccessException;
import com.JRcServer.commons.io.IOCom;
import com.JRcServer.commons.thread.Synchronized;
import com.JRcServer.commons.util.read.ReadElement;

/**
 * 基本コンフィグオブジェクト.
 * <BR><BR>
 * 基本コンフィグオブジェクトです.<BR>
 * 基本的に、コンフィグ情報を呼び出す場合、
 * このオブジェクトを継承して行う必要があります.
 *
 * @version 1.00, 2005/12/25
 * @author  Masahito Suzuki
 * @since   JRcCommons 1.00
 */
public abstract class BaseConfig
{
    
    /**
     * コンフィグデータ管理.
     */
    protected ReadConfig m_conf = null ;
    
    /**
     * ファイル名.
     */
    protected String m_name = null ;
    
    /**
     * ファイル更新日付.
     */
    protected long m_lastUpdate = -1L ;
    
    /**
     * コンフィグ情報読み込みフラグ.
     */
    protected boolean m_isRead = false ;
    
    /**
     * 同期用.
     */
    protected final Synchronized m_sync = new Synchronized() ;
    
    
    
    /**
     * コンストラクタ.
     */
    public BaseConfig()
    {
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected void finalize() throws Exception
    {
        
        try{
            this.close() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 指定iniファイルをオープン.
     * <BR><BR>
     * 指定されたIniファイルをオープンします.
     * <BR>
     * @param name オープン対象のファイル名を設定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     * @exception ReadSyntaxException 読み込み構文例外.
     */
    public final void open( String name )
        throws InputException,FileAccessException,ReadSyntaxException
    {
        
        if( name == null || name.length() <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            this.close() ;
            
            m_sync.create() ;
            m_conf = new ReadConfig() ;
            
            synchronized( m_sync.get() ){
                this.newInit( name ) ;
            }
            
            m_lastUpdate = IOCom.getLastTime( name ) ;
            
        }catch( InputException in ){
            this.close() ;
            throw in ;
        }catch( FileAccessException fa ){
            this.close() ;
            throw fa ;
        }catch( ReadSyntaxException rs ){
            this.close() ;
            throw rs ;
        }
    }
    
    /**
     * オブジェクトクリア.
     * <BR><BR>
     * オブジェクト情報をクリアします.
     */
    public final void close()
    {
        try{
            synchronized( m_sync.get() ){
                if( m_conf != null ){
                    m_conf.clear() ;
                }
            }
        }catch( Exception e ){
        }
        
        m_isRead = false ;
        m_lastUpdate = -1L ;
        m_name = null ;
        m_conf = null ;
        
        m_sync.clear() ;
    }
    
    /**
     * コンフィグ情報再読込み.
     * <BR><BR>
     * 現在オープンされているコンフィグ情報を再読込みします.
     */
    public final void reload()
    {
        long time ;
        ReadConfig old = null ;
        
        try{
            synchronized( m_sync.get() ){
                
                try{
                    
                    if( m_lastUpdate < ( time = IOCom.getLastTime( m_name ) ) ){
                        
                        old = m_conf ;
                        m_conf = new ReadConfig() ;
                        m_isRead = false ;
                        
                        this.newInit( m_name ) ;
                        m_lastUpdate = time ;
                        
                        if( old != null ){
                            old.clear() ;
                        }
                        old = null ;
                        
                    }
                    
                }catch( Exception ee ){
                    if( old != null ){
                        m_conf = old ;
                        m_isRead = true ;
                    }
                }
            }
        }catch( Exception e ){
        }finally{
            old = null ;
        }
        
    }
    
    /**
     * コンフィグオブジェクトの取得.
     * <BR><BR>
     * コンフィグオブジェクトを取得します.
     * <BR>
     * @return CommonConfig コンフィグオブジェクトが返されます.<BR>
     *                      オブジェクトがオープンされていない場合[null]が返されます.
     */
    public final CommonConfig getConfig()
    {
        CommonConfig ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_conf ;
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象コンフィグファイル名を取得.
     * <BR><BR>
     * 対象のコンフィグファイル名を取得します.
     * <BR>
     * @return String コンフィグファイル名が返されます.
     */
    public final String getName()
    {
        String ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_name ;
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * ファイル最終更新時間を取得.
     * <BR><BR>
     * 現在読み込み中の対象ファイルの更新時間が返されます.
     * <BR>
     * @return long ファイル更新時間が返されます.
     */
    public final long getTime()
    {
        long ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_lastUpdate ;
            }
        }catch( Exception e ){
            ret = -1L ;
        }
        
        return ret ;
    }
    
    /**
     * 対象コンフィグで必要なセクション名群が有効であるか
     * チェック.
     * <BR><BR>
     * 対象コンフィグで必要なセクション名群が有効であるか
     * チェックします.
     * <BR>
     * @return boolean 有効であるかチェックします.<BR>
     *                 [true]が返された場合、有効です.<BR>
     *                 [false]が返された場合、有効ではありません.
     */
    public abstract boolean isUseSection() ;
    
    /**
     * 読み込みチェック.
     * <BR><BR>
     * 定義ファイルが正常に読み込まれているかチェックします.
     * <BR>
     * @return boolean 正常に読み込まれているかチェックします.<BR>
     *                 [true]が返された場合、正常に読み込まれています.<BR>
     *                 [false]が返された場合、正常に読み込まれていないか
     *                 クリアされています.
     */
    public final boolean isRead()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_isRead ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 対象セクション名存在チェック.
     * <BR><BR>
     * 対象のセクション名が存在するかチェックします.
     * @param section 対象のセクション名を設定します.
     * @return boolean 存在結果が返されます.<BR>
     *                 [true]が返された場合、セクション名は存在します.<BR>
     *                 [false]が返された場合、セクション名は存在しません.
     */
    public boolean isSection( String section )
    {
        boolean ret = false ;
        
        try{
            
            synchronized( m_sync.get() ){
                ret = m_conf.isSection( section ) ;
            }
            
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 対象コンフィグ情報を取得.
     * <BR><BR>
     * 対象のコンフィグ情報を取得します.<BR>
     * 別途読み込み方法を変える場合は、このメソッドを
     * オーバーライトしてください.
     * <BR>
     * @param name 対象のコンフィグ名を設定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     * @exception ReadSyntaxException 読み込み構文例外.
     */
    protected void newInit( String name )
        throws InputException,FileAccessException,ReadSyntaxException
    {
        try{
            synchronized( m_sync.get() ){
                m_conf.create( name ) ;
                m_name = name ;
                m_isRead = true ;
            }
        }catch( InputException in ){
            this.close() ;
            throw in ;
        }catch( FileAccessException fa ){
            this.close() ;
            throw fa ;
        }catch( ReadSyntaxException rs ){
            this.close() ;
            throw rs ;
        }
    }
    
    /**
     * 対象の情報を文字列で取得.
     * <BR><BR>
     * 対象の情報を文字列で取得します.
     * <BR>
     * @param section 対象のセクション名を設定します.
     * @param key 対象のキー名を設定します.
     * @param no 対象のキー名に対する項番を設定します.
     * @return String 対象の条件をStringで取得した結果が返されます.
     * @exception InputException 入力例外.
     */
    protected final String getString( String section,String key,int no )
        throws InputException
    {
        String ret = null ;
        ReadElement pause = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                pause = m_conf.get( section,key,no ) ;
                ret = pause.getNativeString( 0 ) ;
                ret = ( ret == null || ret.length() <= 0 ) ?
                    null : ret ;
                
            }
            
        }catch( NullPointerException nul ){
            ret = null ;
        }catch( InputException in ){
            throw in ;
        }finally{
            if( pause != null ){
                pause.clear() ;
            }
            pause = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象の情報を数値で取得.
     * <BR><BR>
     * 対象の情報を数値で取得します.
     * <BR>
     * @param section 対象のセクション名を設定します.
     * @param key 対象のキー名を設定します.
     * @param no 対象のキー名に対する項番を設定します.
     * @return int 対象の条件をintで取得した結果が返されます.
     * @exception InputException 入力例外.
     */
    protected final int getNumber( String section,String key,int no )
        throws InputException
    {
        int ret ;
        Integer tmp = null ;
        ReadElement pause = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                pause = m_conf.get( section,key,no ) ;
                tmp = pause.getInteger( 0 ) ;
                if( tmp != null ){
                    ret = tmp.intValue() ;
                }
                else{
                    ret = -1 ;
                }
                
            }
            
        }catch( NullPointerException nul ){
            ret = -1 ;
        }catch( InputException in ){
            throw in ;
        }finally{
            if( pause != null ){
                pause.clear() ;
            }
            pause = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象の情報をフラグで取得.
     * <BR><BR>
     * 対象の情報をフラグで取得します.
     * <BR>
     * @param section 対象のセクション名を設定します.
     * @param key 対象のキー名を設定します.
     * @param no 対象のキー名に対する項番を設定します.
     * @return boolean 対象の条件をフラグで取得した結果が返されます.
     * @exception InputException 入力例外.
     */
    protected final boolean getBoolean( String section,String key,int no )
        throws InputException
    {
        boolean ret ;
        Boolean tmp = null ;
        ReadElement pause = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                pause = m_conf.get( section,key,no ) ;
                tmp = pause.getBoolean( 0 ) ;
                if( tmp != null ){
                    ret = tmp.booleanValue() ;
                }
                else{
                    ret = false ;
                }
                
            }
            
        }catch( NullPointerException nul ){
            ret = false ;
        }catch( InputException in ){
            throw in ;
        }finally{
            if( pause != null ){
                pause.clear() ;
            }
            pause = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象情報長の取得.
     * <BR><BR>
     * 対象の情報長を取得します.
     * @param section 対象のセクション名を設定します.
     * @param key 対象のキー名を設定します.
     * @return int キー名に対する情報長が返されます.
     */
    protected final int getLength( String section,String key )
    {
        int ret ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                ret = m_conf.getElement( section,key ) ;
                
            }
            
        }catch( NullPointerException nul ){
            ret = 0 ;
        }catch( InputException in ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * コンフィグ情報を文字列に出力.
     * <BR><BR>
     * コンフィグ情報を文字列に出力します.
     * <BR>
     * @return String コンフィグ内容が返されます.
     */
    public final String toString()
    {
        int i,j,k ;
        int len ;
        int lenJ ;
        int lenK ;
        
        String enter = System.getProperty( "line.separator" ) ;
        
        String[] sec = null ;
        String[] keys = null ;
        StringBuffer buf = null ;
        ReadConfig conf = null ;
        ReadElement re = null ;
        String ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                
                conf = m_conf ;
                sec = conf.getNames() ;
                
                if( sec != null && ( len = sec.length ) > 0 ){
                    
                    buf = new StringBuffer() ;
                    re = new ReadElement() ;
                    
                    for( i = 0 ; i < len ; i ++ ){
                        
                        buf.append( "[" ) ;
                        buf.append( sec[ i ] ) ;
                        buf.append( "]" ) ;
                        buf.append( enter ) ;
                        
                        keys = conf.getNames( sec[ i ] ) ;
                        
                        if( keys != null && ( lenJ = keys.length ) > 0 ){
                            
                            for( j = 0 ; j < lenJ ; j ++ ){
                                
                                conf.get( re,sec[ i ],keys[ j ] ) ;
                                
                                lenK = re.size() ;
                                
                                for( k = 0 ; k < lenK ; k ++ ){
                                
                                    buf.append( "\t" ) ;
                                    buf.append( keys[ j ] ) ;
                                    buf.append( "[" ) ;
                                    buf.append( k ) ;
                                    buf.append( "]" ) ;
                                    buf.append( " = " ) ;
                                    buf.append( re.getString( k ) ) ;
                                    buf.append( enter ) ;
                                    
                                }
                                
                            }
                            
                        }
                        
                    }
                    
                }
                
            }
            
            if( buf != null ){
                ret = buf.toString() ;
            }
            
        }catch( Exception e ){
            ret = null ;
        }finally{
            
            if( re != null ){
                re.clear() ;
            }
            
            enter = null ;
            sec = null ;
            keys = null ;
            buf = null ;
            conf = null ;
            re = null ;
        }
        
        return ret ;
    }
    
}
