/*
 * @(#)JRcClient.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.client ;

import java.io.Serializable;
import java.net.InetAddress;

import com.JRcServer.commons.conv.SHA1;
import com.JRcServer.commons.def.BaseDef;
import com.JRcServer.commons.exception.AccessException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.io.IOCom;
import com.JRcServer.commons.serialize.SerializeUtil;

/**
 * JRcClient-シリアライズsession管理.
 * <BR><BR>
 * JRcClientのsessionをシリアライズで管理します.
 *  
 * @version 2006/09/10
 * @author  masahito suzuki
 * @since   JRcClient 1.00
 */
class JRcSerializableSession implements JRcClientSession,Serializable {
    
    static {
        serialVersionUID = SerializeUtil.serialVersionUID(
            JRcSerializableSession.class.getName()
        ) ;
    }
    
    /**
     * シリアライズUID.
     */
    private static final long serialVersionUID ;
    
    /**
     * ヘッダ名.
     */
    public static final String HEADER_NAME = ".j@" + JRcClientDriver.VERSION + "@cl";
    
    /**
     * 拡張子.
     */
    public static final String PLUS = ".jss" ;
    
    
    
    /**
     * コネクション先Host名.
     */
    private String hostName = null ;
    
    /**
     * コネクションポート番号.
     */
    private int port = -1 ;
    
    /**
     * アプリケーション名.
     */
    private String applicationName = null ;
    
    /**
     * セッションID.
     */
    private long sessionID = -1L ;
    
    /**
     * コンストラクタ.
     */
    public JRcSerializableSession() {
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected void finalize() throws Exception {
        this.clear() ;
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public void clear() {
        
        this.hostName = null ;
        this.port = -1 ;
        this.applicationName = null ;
        this.sessionID = -1L ;
        
    }
    
    /**
     * オブジェクトコピー.
     * <BR><BR>
     * オブジェクトをコピーします.
     * <BR>
     * @param o コピー元のオブジェクトを設定します.
     */
    public void copyObject( JRcClientSession o ) {
        
    	JRcSerializableSession s = ( JRcSerializableSession)o ;
        s.setHostName( this.hostName ) ;
        s.setPort( this.port ) ;
        s.setApplicationName( this.applicationName ) ;
        s.setSessionID( this.sessionID ) ;
        
    }
    
    /**
     * applicationName を取得.
     * <BR><BR>
     * @return applicationName が返されます.
     */
    public String getApplicationName() {
        return applicationName;
    }
    
    /**
     * applicationName を設定.
     * <BR><BR>
     * @param applicationName applicationName を設定します.
     */
    public void setApplicationName(String applicationName) {
        this.applicationName = applicationName;
    }
    
    /**
     * hostName を取得.
     * <BR><BR>
     * @return hostName が返されます.
     */
    public String getHostName() {
        return hostName;
    }
    
    /**
     * hostName を設定.
     * <BR><BR>
     * @param hostName hostName を設定します.
     */
    public void setHostName(String hostName) {
        this.hostName = hostName;
    }
    
    /**
     * port を取得.
     * <BR><BR>
     * @return port が返されます.
     */
    public int getPort() {
        return port;
    }
    
    /**
     * port を設定.
     * <BR><BR>
     * @param port port を設定します.
     */
    public void setPort(int port) {
        this.port = port;
    }
    
    /**
     * sessionID を取得.
     * <BR><BR>
     * @return sessionID が返されます.
     */
    public long getSessionID() {
        return sessionID;
    }
    
    /**
     * sessionID を設定.
     * <BR><BR>
     * @param sessionID sessionID を設定します.
     */
    public void setSessionID(long sessionID) {
        this.sessionID = sessionID;
    }
    
    /**
     * シリアライズ名を取得.
     * <BR><BR>
     * シリアライズ名を取得します.
     * <BR>
     * @return String シリアライズ名が返されます.
     */
    public String toString() {
        
        byte[] bin = null ;
        String ret = null ;
        
        try {
            
            bin = ( new StringBuffer().
                append( JRcClientDriver.VERSION ).
                append( "#" ).
                append( this.hostName ).
                append( ":" ).
                append( this.port ).
                append( "@" ).
                append( this.applicationName ).
                append( "~" ).
                append( BaseDef.LOGIN_USERNAME ).
                toString()
            ).getBytes( BaseDef.UTF8 ) ;
            
            ret = new StringBuffer().
                append( HEADER_NAME ).
                append( SHA1.convert( bin ) ).
                append( PLUS ).
                toString() ;
            
        } catch( Exception e ) {
            ret = "" ;
        } finally {
            bin = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * オブジェクト有効チェック.
     * <BR><BR>
     * オブジェクト有効チェックを行います.
     * <BR>
     * @return boolean チェック結果が返されます.
     */
    public boolean isUseObject() {
        
        return (
            applicationName == null ||
            applicationName.length() <= 0 ||
            hostName == null ||
            hostName.length() <= 0 ||
            port < 0 ||
            port > 65535
        ) ? false : true ;
        
    }
    
    
    /**
     * 指定条件でJRcClientSessionを取得.
     * <BR><BR>
     * 指定条件でJRcClientSessionを取得します.
     * <BR>
     * @param applicationName 対象のアプリケーション名を設定します
     * @param serverAddress 対象のサーバアドレスを設定します.
     * @param serverPort 対象のサーバポートを設定します.
     * @return JRcClientSession 生成されたJRcClientセッション情報が返されます.
     * @exception InputException 入力例外.
     */
    public static final JRcClientSession getJRcClientSession(
        String applicationName,InetAddress serverAddress,int serverPort )
        throws InputException {
        
        String dir = null ;
        String serializableName = null ;
        JRcSerializableSession ret = null ;
        
        if(
            applicationName == null ||
            ( applicationName = applicationName.trim().toLowerCase() ).length() <= 0 ||
            serverAddress == null || serverPort < 0 || serverPort > 65535
        ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try {
            
            ret = new JRcSerializableSession() ;
            ret.setApplicationName( applicationName ) ;
            ret.setHostName( serverAddress.getHostName() ) ;
            ret.setPort( serverPort ) ;
            
            dir = JRcClientDriver.JRC_CLIENT_DIR ;
            
            if( IOCom.isDirExists( dir ) == true ) {
                
                serializableName = new StringBuffer().
                    append( dir ).
                    append( BaseDef.FILE_SEPARATOR ).
                    append( ret.toString() ).
                    toString() ;
                
                if( IOCom.isFileExists( serializableName ) == true ) {
                    
                    ret = ( JRcSerializableSession )SerializeUtil.getCb32Serialize(
                        true,dir,serializableName ) ;
                        
                }
                
            }
            
        } catch( Exception e ) {
            
            ret = new JRcSerializableSession() ;
            ret.setApplicationName( applicationName ) ;
            ret.setHostName( serverAddress.getHostName() ) ;
            ret.setPort( serverPort ) ;
            
        } finally {
            
            dir = null ;
            serializableName = null ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * 指定条件でJRcClientSessionを保存.
     * <BR><BR>
     * 指定条件でJRcClientSessionを保存します.
     * <BR>
     * @param session 対象のJRcClientSessionを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final void setJRcClientSession( JRcClientSession session )
        throws InputException,AccessException {
        
        String dir = null ;
        String serializableName = null ;
        
        if( session == null ||
            session.getApplicationName() == null ||
            session.getHostName() == null ||
            session.getPort() < 0 || session.getPort() > 65535
        ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try {
            
            dir = JRcClientDriver.JRC_CLIENT_DIR ;
            
            if( IOCom.isDirExists( dir ) == false ) {
                IOCom.mkdirs( dir ) ;
            }
            
            serializableName = new StringBuffer().
                append( dir ).
                append( BaseDef.FILE_SEPARATOR ).
                append( session.toString() ).
                toString() ;
            
            SerializeUtil.putCb32Serialize(
                true,dir,serializableName,
                ( JRcSerializableSession )session ) ;
            
        } catch( AccessException ae ) {
            throw ae ;
        } catch( Exception e ) {
            throw new AccessException( e ) ;
        } finally {
            dir = null ;
            serializableName = null ;
        }
        
    }
}

