/*
 * @(#)JRcConnectionImple.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.client ;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.math.BigDecimal;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.net.SocketException;
import java.net.URLDecoder;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;

import com.JRcServer.JRCommandException;
import com.JRcServer.JRcConnectCommon;
import com.JRcServer.JRcErrorCodeDef;
import com.JRcServer.JRcParameter;
import com.JRcServer.JRcProtocolDef;
import com.JRcServer.JRcResponseBean;
import com.JRcServer.JRcResultDef;
import com.JRcServer.commons.conv.Base64;
import com.JRcServer.commons.def.BaseDef;
import com.JRcServer.commons.exception.AccessException;
import com.JRcServer.commons.exception.ConvertException;
import com.JRcServer.commons.exception.ExecutionException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.io.IOCom;
import com.JRcServer.commons.net.ConnectTimeoutException;
import com.JRcServer.commons.resource.BinResource;
import com.JRcServer.commons.resource.BinResourceOutputStream;
import com.JRcServer.commons.resource.ConvertResourceParam;
import com.JRcServer.commons.resource.Resource;
import com.JRcServer.commons.resource.ResourceType;
import com.JRcServer.commons.thread.Synchronized;
import com.JRcServer.commons.util.CharTable;
import com.JRcServer.commons.util.ConvertParam;
import com.JRcServer.commons.util.UtilCom;
import com.JRcServer.commons.util.array.ObjectArray;

/**
 * JRcConnectionImple.
 * <BR><BR>
 * JRcClientコネクション実装オブジェクト.
 *  
 * @version 2006/09/10
 * @author  masahito suzuki
 * @since   JRcClient 1.00
 */
class JRcConnectionImple implements JRcConnection {
    
    /**
     * コネクション待ち時間.
     */
    private static final int WAIT_CONNECT_TIME = 2500 ;
    
    /**
     * コネクション再オープン時間.
     */
    private static final long RECONNECT_TIME = 5000 ;
    
    /**
     * 送信バッファ長.
     */
    private static final int SEND_BUFFER = JRcConnectCommon.BUFFER_LENGTH ;
    
    /**
     * Base64変換ヘッダ.
     */
    private static final String BASE64_HEADER = "base64-" ;
    
    /**
     * ローカルファイル変換ヘッダ.
     */
    private static final String LFILE_HEADER = "file-" ;
    
    
    
    /**
     * セッション管理.
     */
    private JRcSerializableSession session = null ;
    
    /**
     * データ保持用リソースタイプ.
     */
    private ResourceType resType = null ;
    
    /**
     * コネクションクライアント.
     */
    private Socket socket = null ;
    
    /**
     * 受信タイムアウト値.
     */
    private int timeout = -1 ;
    
    /**
     * パラメータ管理テーブル.
     */
    private CharTable table = null ;
    
    /**
     * 呼び出しサービス名.
     */
    private String serviceName = null ;
    
    /**
     * 前回処理時間.
     */
    private long beforeExecTime = -1L ;
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized sync = new Synchronized() ;
    
    
    
    /**
     * コンストラクタ.
     */
    private JRcConnectionImple() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 条件を指定してオブジェクトを生成します.
     * <BR>
     * @param cacheMode キャッシュモードを設定します.<BR>
     *                  [true]を設定した場合、ファイルキャッシュをONにします.<BR>
     *                  [false]を設定した場合、メモリキャッシュをONにします.
     * @param applicationName 対象のアプリケーション名を設定します.<BR>
     *                        この内容は、セッション条件に影響します.<BR>
     *                        そのため、Uniqueな条件で設定してください.
     * @param addr コネクション先のアドレスを設定します.
     * @param port コネクション先のポート番号を設定します.
     * @param timeout コネクションタイムアウトを設定します.<BR>
     *                [0]以下を設定した場合、無限待機となります.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public JRcConnectionImple(
        boolean cacheMode,String applicationName,InetAddress addr,int port,int timeout )
        throws InputException,AccessException {
        
        try {
            
            this.sync.create() ;
            
            this.socket = new Socket() ;
            JRcConnectCommon.setSocketOption( this.socket ) ;
            this.socket.bind( null ) ;
            this.socket.connect(
                new InetSocketAddress( addr,port ),
                WAIT_CONNECT_TIME
            ) ;
            
            this.session = ( JRcSerializableSession )JRcSerializableSession.
                getJRcClientSession(
                        applicationName,addr,port ) ;
            
            if( cacheMode == true ) {
                resType = new ResourceType(
                    ResourceType.RESOURCE_TYPE_FILE,
                    JRcClientDriver.CACHE_DIR
                ) ;
            }
            else {
                resType = new ResourceType() ;
            }
            
            this.timeout = ( timeout <= 0 ) ? -1 : timeout ;
            this.table = new CharTable() ;
            this.serviceName = null ;
            
            this.beforeExecTime = System.currentTimeMillis() ;
            
        } catch( InputException in ) {
            this.close() ;
            throw in ;
        } catch( Exception e ) {
            this.close() ;
            throw new AccessException( e ) ;
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected void finalize() throws Exception {
        this.close() ;
    }
    
    /**
     * 現在のコネクションをクローズ.
     * <BR><BR>
     * 現在のコネクション条件をクローズします.
     */
    public void close() {
        
        if( this.table != null ) {
            try {
                this.table.clear() ;
            } catch( Exception e ) {
            }
        }
        
        if( this.session != null ) {
            
            try {
                JRcSerializableSession.setJRcClientSession(
                    session
                ) ;
            } catch( Exception e ) {
            }
            
        }
        
        if( this.socket != null ) {
            try {
                this.socket.close() ;
            } catch( Exception e ) {
            }
        }
        
        this.table = null ;
        this.session = null ;
        this.resType = null ;
        this.socket = null ;
        this.timeout = -1 ;
        this.serviceName = null ;
        
        this.sync.clear() ;
        
    }
    
    /**
     * 再コネクション.
     * <BR><BR>
     * 再コネクションを行います.
     * <BR>
     * @exception AccessException アクセス例外.
     */
    public synchronized void retry() throws AccessException {
        
        Socket soc = null ;
        
        if(
            this.session == null ||
            this.session.isUseObject() == false
        ) {
            throw new AccessException( "コネクション情報は存在しません" ) ;
        }
        
        try {
            
            soc = new Socket(
                InetAddress.getByName( this.session.getHostName() ),
                this.session.getPort()
            ) ;
            
            JRcConnectCommon.setSocketOption( soc ) ;
            
        } catch( Exception e ) {
            try {
                soc.close() ;
            } catch( Exception ee ) {
            }
            throw new AccessException( e ) ;
        }
        
        if( this.socket != null ) {
            try {
                this.socket.close() ;
            } catch( Exception e ) {
            }
        }
        
        this.socket = soc ;
        soc = null ;
        
    }
    
    /**
     * 切断処理.
     * <BR><BR>
     * 通信を切断します.
     */
    public void closeConnection() {
        
        if( this.socket != null ) {
            try {
                this.socket.close() ;
            } catch( Exception e ) {
            }
        }
        
        this.socket = null ;
    }
    
    /**
     * 新しいサービスでコネクション.
     * <BR><BR>
     * 新しいサービス名を設定してコネクションを生成します.<BR>
     * そのため、設定途中の条件が存在する場合、その情報は破棄されます.
     * <BR>
     * @param name 対象のサービス名を設定します.
     * @exception InputException 入力例外.
     */
    public synchronized void createService( String name )
        throws InputException {
        
        if( name == null || ( name = name.trim() ).length() <= 0 ) {
            throw new InputException( "引数は不正です" ) ;
        }
        else if( this.session == null || this.session.isUseObject() == false ) {
            throw new InputException( "オブジェクトは無効な状態です" ) ;
        }
        
        this.serviceName = name ;
        this.table.clear() ;
        
    }
    
    /**
     * 現在のサービス条件を破棄.
     * <BR><BR>
     * 現在のサービスを破棄します.
     */
    public synchronized void clearService() {
        
        if( this.session != null || this.session.isUseObject() == true ) {
            
            this.serviceName = null ;
            if( this.table != null ) {
                this.table.clear() ;
            }
            else {
                this.table = new CharTable() ;
            }
            
        }
        
    }
    
    /**
     * 現在の条件でサービスを呼び出す.
     * <BR><BR>
     * 設定されている現在の条件で、サービスを呼び出します.
     * <BR>
     * @return JRcResult 処理結果の内容が返されます.
     * @exception JRCommandException JRCommon例外.
     * @exception ExecutionException 実行例外.
     */
    public synchronized JRcResult execution()
        throws JRCommandException,ExecutionException {
        
        JRcResponseBean bean = null ;
        JRcResultImple ret = null ;
        
        try {
            
            // 状態が不正な場合.
            if( this.session == null || this.session.isUseObject() == false ) {
                throw new JRCommandException(
                    JRcErrorCodeDef.ERROR_ETC,
                    "コネクションは既にクローズされています"
                ) ;
            }
            
            // サービス名が登録されていない場合.
            if( this.serviceName == null || this.serviceName.length() <= 0 ) {
                throw new JRCommandException(
                    JRcErrorCodeDef.ERROR_ETC,
                    "問い合わせたいサービス名が指定されていません"
                ) ;
            }
            
            // コネクションが切断されている場合.
            // 前回実行時間から、指定秒以上経過している場合.
            if(
                this.isClosed() == true ||
                (
                    beforeExecTime != -1L &&
                    ( System.currentTimeMillis() - beforeExecTime >= RECONNECT_TIME )
                )
            ) {
                // 再コネクション.
                this.retry() ;
                beforeExecTime = System.currentTimeMillis() ;
            }
            
            // JRcServerに問い合わせ.
            bean = this.executionService() ;
            
            // 処理結果がnull.
            if( bean == null ) {
                
                throw new JRCommandException(
                    JRcErrorCodeDef.ERROR_ETC,
                    "処理結果の取得に失敗しました"
                ) ;
                
            }
            
            // 問い合わせた結果コードが[success]の場合.
            if( ( bean.getResult() & JRcResultDef.STATE_MASK ) == JRcResultDef.SUCCESS ) {
                
                // セッション情報が設定対象の場合.
                if( bean.getSessionID() >= 0L ) {
                    this.session.setSessionID( bean.getSessionID() ) ;
                }
                
                ret = new JRcResultImple( bean ) ;
                bean.destroy() ;
                bean = null ;
                
            }
            // 問い合わせた結果コードが[success]以外で結果メッセージありの場合.
            else if( bean.getResultMessage() != null ) {
                throw new JRCommandException(
                    bean.getResult(),
                    bean.getResultMessage()
                ) ;
            }
            // 問い合わせた結果コードが[WARNING]の場合.
            else if( ( bean.getResult() & JRcResultDef.STATE_MASK ) == JRcResultDef.WARNING ) {
                throw new JRCommandException(
                    bean.getResult(),
                    "以下の警告コードが返されました[0x" +
                    Integer.toHexString( bean.getResult() ) + "]"
                ) ;
            }
            // 問い合わせた結果コードが[ERROR]の場合.
            else if( ( bean.getResult() & JRcResultDef.STATE_MASK ) == JRcResultDef.ERROR ) {
                throw new JRCommandException(
                    bean.getResult(),
                    "以下の異常コードが返されました[0x" +
                    Integer.toHexString( bean.getResult() ) + "]"
                ) ;
            }
            
            beforeExecTime = System.currentTimeMillis() ;
            
        } catch( ConnectTimeoutException ct ) {
            this.closeConnection() ;
            throw new JRCommandException(
                JRcErrorCodeDef.ERROR_TIMEOUT,
                "タイムアウトにより切断します"
            ) ;
        } catch( JRCommandException je ) {
            throw je ;
        } catch( ExecutionException ee ) {
            this.closeConnection() ;
            throw ee ;
        } catch( AccessException ae ) {
            this.closeConnection() ;
            throw new JRCommandException(
                JRcErrorCodeDef.ERROR_ETC,
                ae
            ) ;
        } catch( Exception e ) {
            this.closeConnection() ;
            throw new ExecutionException(
                e,ExecutionException.LEVEL_STOP
            ) ;
        } finally {
            
            if( bean != null ) {
                bean.destroy() ;
            }
            bean = null ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * コマンドによる直接指定で呼び出し.
     * <BR><BR>
     * コマンドによる直接指定で呼び出します.<BR>
     * また、このメソッドの場合は、[createService()]や
     * パラメータ指定の条件は全て無視します.<BR>
     * 代わりに、指定された文字列をコマンドとして、サービス名、
     * パラメータの順に定義して呼び出します.<BR>
     * 呼び出す方法としては、以下の2点があります.<BR>
     * <PRE>
     * その１:通常指定.
     * hoge -param1 ジェイアールシー -param2 sv
     * 
     * その２:URL指定.
     * ？hoge＆param1=%E3%82%B8%E3%82%A7%E3%82%A4%E3%82%A2%E3%83%BC%E3%83%AB%E3%82%B7%E3%83%BC＆param2=sv
     * 
     * 　※先頭に必ず[？(半角)]が必要となります.
     *
     * </PRE>
     * 両方とも同様の[hoge]サービスを呼び出し、２つのパラメータ<BR>
     * [param1=ジェイアールシー],[param2=sv]を指定しています.<BR>
     * また、URLエンコードはUTF8のみサポートしているので、<BR>
     * それ以外の指定条件では設定できません.<BR>
     * (文字化けします)<BR>
     * <BR>
     * 設定されたパラメータの値はString以外の近い条件で設定します.<BR>
     * たとえば、Boolean変換したいものは、[true]や[false]を設定することで、<BR>
     * 変換対象となります.<BR>
     * その中でも特殊なものとしては、Timestamp登録です.<BR>
     * 以下に対応している日付フォーマット一覧です.<BR>
     * <BR>
     * ・yyyy/MM/dd,HH:mm:ss<BR>
     * ・yyyy-MM-dd,HH:mm:ss<BR>
     * ・yyyy年MM月dd日,HH:mm:ss<BR>
     * ・yyyy/MM/dd<BR>
     * ・yyyy-MM-dd<BR>
     * ・yyyy年MM月dd日<BR>
     * <BR>
     * もう１つ特殊なものとしては、Base64でのバイナリ指定があります.<BR>
     * 方法としては、データ定義の初めに[base64-]を指定して以降にBase64の
     * コードを指定します.<BR>
     * <PRE>
     * ＜例＞
     * hoge -param1 base64-44GY44GH44GE44GC44O844KL44GX44O8c3Y=
     * 
     * または
     * ?hoge&param1=base64-44GY44GH44GE44GC44O844KL44GX44O8c3Y=
     * </PRE>
     * 
     * このように定義しなければいけません.<BR>
     * <BR>
     * 最後に、ローカルファイルの内容を設定する方法として、以下のような<BR>
     * 方法で、設定できます.<BR>
     * <PRE>
     * ＜例＞
     * hoge -param1 file-/c:/dir/test.txt
     * 
     * または
     * ?hoge?param1=file-/c:/dir/test.txt
     * 
     * </PRE>
     * このように定義しなければいけません.<BR>
     * <BR>
     * @param command コマンドによる直接指定で呼び出します.
     * @return JRcResult 処理結果の内容が返されます.
     * @exception JRCommandException JRCommon例外.
     * @exception ExecutionException 実行例外.
     */
    public JRcResult execution( String command )
        throws JRCommandException,ExecutionException {
        
        String[] cmd = null ;
        ObjectArray ary = null ;
        
        if( command == null || ( command = command.trim() ).length() <= 0 ) {
            throw new JRCommandException(
                JRcErrorCodeDef.ERROR_ETC,
                "指定された条件は不正です>" + command
            ) ;
        }
        
        this.clearService() ;
        
        cmd = new String[]{ command } ;
        
        // URL形式の条件.
        if( cmd[ 0 ].startsWith( "?" ) == true ) {
            
            // URL形式の条件を通常のコマンド条件に変更.
            cmd[ 0 ] = cmd[ 0 ].substring( 1 ) ;
            ary = JRcConnectionImple.convertURLDecoder( cmd ) ;
            cmd = null ;
            
        }
        // 通常の形式で処理.
        else {
            
            ary = JRcConnectionImple.convertBase( cmd ) ;
            cmd = null ;
            
        }
        
        // コマンドを解析.
        this.commandAnalysisByCommand( ary ) ;
        
        return this.execution() ;
        
    }
    
    /**
     * 対象のパラメータを削除.
     * <BR><BR>
     * 対象のパラメータ情報を削除します.
     * <BR>
     * @param key 対象のKey名を設定します.
     */
    public synchronized void removeParameter( String key ) {
        
        if(
            this.session != null ||
            this.session.isUseObject() == true
        ) {
            
            if( this.table != null ) {
                try {
                    this.table.remove(
                        convertKey( key ) ) ;
                } catch( Exception e ) {
                }
            }
            
        }
        
    }
    
    /**
     * パラメータをバイナリで追加.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @param value 対象の内容を設定します.
     * @exception InputException 入力例外.
     */
    public synchronized void setParameterToBinary( String key,byte[] value )
        throws InputException {
        
        JRcParameter param = null ;
        
        if(
            ( key = convertKey( key ) ) == null ||
            value == null || value.length <= 0 ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        if( this.session == null || this.session.isUseObject() == false ) {
            throw new InputException( "オブジェクトは無効な状態です" ) ;
        }
            
        if( this.table != null ) {
            
            if( this.table.isData( key ) == true ) {
                throw new InputException(
                    "設定対象のキー名[" + key +
                    "]は既に存在します" ) ;
            }
            
            param = new JRcParameter() ;
            param.setBinary( value ) ;
            
            this.table.add( key,param ) ;
            
        }
        
    }
    
    /**
     * パラメータをバイナリで取得.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @return byte[] パラメータ情報が返されます.
     */
    public synchronized byte[] getParameterToBinary( String key ) {
        
        if(
            ( key = convertKey( key ) ) == null ||
            this.session == null ||
            this.session.isUseObject() == false ||
            this.table == null
        ) {
            return null ;
        }
        
        try {
            return (
                ( JRcParameter )this.table.get( key )
            ).getBinary() ;
        } catch( Exception e ) {
        }
        
        return null ;
        
    }
    
    /**
     * パラメータを文字列で追加.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @param value 対象の内容を設定します.
     * @exception InputException 入力例外.
     */
    public synchronized void setParameterToString( String key,String value )
        throws InputException {
        
        JRcParameter param = null ;
        
        if(
            ( key = convertKey( key ) ) == null ||
            value == null || value.length() <= 0 ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        if( this.session == null || this.session.isUseObject() == false ) {
            throw new InputException( "オブジェクトは無効な状態です" ) ;
        }
            
        if( this.table != null ) {
            
            if( this.table.isData( key ) == true ) {
                throw new InputException(
                    "設定対象のキー名[" + key +
                    "]は既に存在します" ) ;
            }
            
            param = new JRcParameter() ;
            param.setString( value ) ;
            
            this.table.add( key,param ) ;
            
        }
        
    }
    
    /**
     * パラメータを文字列で取得.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @return String パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public synchronized String getParameterToString( String key )
        throws ConvertException {
        
        if(
            ( key = convertKey( key ) ) == null ||
            this.session == null ||
            this.session.isUseObject() == false ||
            this.table == null
        ) {
            return null ;
        }
        
        try {
            return (
                ( JRcParameter )this.table.get( key )
            ).getString() ;
        } catch( ConvertException ce ) {
            throw ce ;
        } catch( Exception e ) {
        }
        
        return null ;
        
    }
    
    /**
     * パラメータをフラグで追加.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @param value 対象の内容を設定します.
     * @exception InputException 入力例外.
     */
    public synchronized void setParameterToBoolean( String key,Boolean value )
        throws InputException {
        
        JRcParameter param = null ;
        
        if(
            ( key = convertKey( key ) ) == null ||
            value == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        if( this.session == null || this.session.isUseObject() == false ) {
            throw new InputException( "オブジェクトは無効な状態です" ) ;
        }
            
        if( this.table != null ) {
            
            if( this.table.isData( key ) == true ) {
                throw new InputException(
                    "設定対象のキー名[" + key +
                    "]は既に存在します" ) ;
            }
            
            param = new JRcParameter() ;
            param.setBoolean( value ) ;
            
            this.table.add( key,param ) ;
            
        }
        
    }
    
    /**
     * パラメータをフラグで取得.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @return Boolean パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public synchronized Boolean getParameterToBoolean( String key )
        throws ConvertException {
        
        if(
            ( key = convertKey( key ) ) == null ||
            this.session == null ||
            this.session.isUseObject() == false ||
            this.table == null
        ) {
            return null ;
        }
        
        try {
            return (
                ( JRcParameter )this.table.get( key )
            ).getBoolean() ;
        } catch( ConvertException ce ) {
            throw ce ;
        } catch( Exception e ) {
        }
        
        return null ;
        
    }
    
    /**
     * パラメータを数値(Short)で追加.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @param value 対象の内容を設定します.
     * @exception InputException 入力例外.
     */
    public synchronized void setParameterToShort( String key,Short value )
        throws InputException {
        
        JRcParameter param = null ;
        
        if(
            ( key = convertKey( key ) ) == null ||
            value == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        if( this.session == null || this.session.isUseObject() == false ) {
            throw new InputException( "オブジェクトは無効な状態です" ) ;
        }
            
        if( this.table != null ) {
            
            if( this.table.isData( key ) == true ) {
                throw new InputException(
                    "設定対象のキー名[" + key +
                    "]は既に存在します" ) ;
            }
            
            param = new JRcParameter() ;
            param.setShort( value ) ;
            
            this.table.add( key,param ) ;
            
        }
        
    }
    
    /**
     * パラメータを数値(Short)で取得.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @return Short パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public synchronized Short getParameterToShort( String key )
        throws ConvertException {
        
        if(
            ( key = convertKey( key ) ) == null ||
            this.session == null ||
            this.session.isUseObject() == false ||
            this.table == null
        ) {
            return null ;
        }
        
        try {
            return (
                ( JRcParameter )this.table.get( key )
            ).getShort() ;
        } catch( ConvertException ce ) {
            throw ce ;
        } catch( Exception e ) {
        }
        
        return null ;
        
    }
    
    /**
     * パラメータを数値(Integer)で追加.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @param value 対象の内容を設定します.
     * @exception InputException 入力例外.
     */
    public synchronized void setParameterToInteger( String key,Integer value )
        throws InputException {
        
        JRcParameter param = null ;
        
        if(
            ( key = convertKey( key ) ) == null ||
            value == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        if( this.session == null || this.session.isUseObject() == false ) {
            throw new InputException( "オブジェクトは無効な状態です" ) ;
        }
            
        if( this.table != null ) {
            
            if( this.table.isData( key ) == true ) {
                throw new InputException(
                    "設定対象のキー名[" + key +
                    "]は既に存在します" ) ;
            }
            
            param = new JRcParameter() ;
            param.setInteger( value ) ;
            
            this.table.add( key,param ) ;
            
        }
        
    }
    
    /**
     * パラメータを数値(Integer)で取得.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @return Integer パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public synchronized Integer getParameterToInteger( String key )
        throws ConvertException {
        
        if(
            ( key = convertKey( key ) ) == null ||
            this.session == null ||
            this.session.isUseObject() == false ||
            this.table == null
        ) {
            return null ;
        }
        
        try {
            return (
                ( JRcParameter )this.table.get( key )
            ).getInteger() ;
        } catch( ConvertException ce ) {
            throw ce ;
        } catch( Exception e ) {
        }
        
        return null ;
        
    }
    
    /**
     * パラメータを数値(Long)で追加.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @param value 対象の内容を設定します.
     * @exception InputException 入力例外.
     */
    public synchronized void setParameterToLong( String key,Long value )
        throws InputException {
        
        JRcParameter param = null ;
        
        if(
            ( key = convertKey( key ) ) == null ||
            value == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        if( this.session == null || this.session.isUseObject() == false ) {
            throw new InputException( "オブジェクトは無効な状態です" ) ;
        }
            
        if( this.table != null ) {
            
            if( this.table.isData( key ) == true ) {
                throw new InputException(
                    "設定対象のキー名[" + key +
                    "]は既に存在します" ) ;
            }
            
            param = new JRcParameter() ;
            param.setLong( value ) ;
            
            this.table.add( key,param ) ;
            
        }
        
    }
    
    /**
     * パラメータを数値(Long)で取得.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @return Long パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public synchronized Long getParameterToLong( String key )
        throws ConvertException {
        
        if(
            ( key = convertKey( key ) ) == null ||
            this.session == null ||
            this.session.isUseObject() == false ||
            this.table == null
        ) {
            return null ;
        }
        
        try {
            return (
                ( JRcParameter )this.table.get( key )
            ).getLong() ;
        } catch( ConvertException ce ) {
            throw ce ;
        } catch( Exception e ) {
        }
        
        return null ;
        
    }
    
    /**
     * パラメータを浮動少数値(Float)で追加.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @param value 対象の内容を設定します.
     * @exception InputException 入力例外.
     */
    public synchronized void setParameterToFloat( String key,Float value )
        throws InputException {
        
        JRcParameter param = null ;
        
        if(
            ( key = convertKey( key ) ) == null ||
            value == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        if( this.session == null || this.session.isUseObject() == false ) {
            throw new InputException( "オブジェクトは無効な状態です" ) ;
        }
            
        if( this.table != null ) {
            
            if( this.table.isData( key ) == true ) {
                throw new InputException(
                    "設定対象のキー名[" + key +
                    "]は既に存在します" ) ;
            }
            
            param = new JRcParameter() ;
            param.setFloat( value ) ;
            
            this.table.add( key,param ) ;
            
        }
        
    }
    
    /**
     * パラメータを浮動少数値(Float)で取得.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @return Float パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public synchronized Float getParameterToFloat( String key )
        throws ConvertException {
        
        if(
            ( key = convertKey( key ) ) == null ||
            this.session == null ||
            this.session.isUseObject() == false ||
            this.table == null
        ) {
            return null ;
        }
        
        try {
            return (
                ( JRcParameter )this.table.get( key )
            ).getFloat() ;
        } catch( ConvertException ce ) {
            throw ce ;
        } catch( Exception e ) {
        }
        
        return null ;
        
    }
    
    /**
     * パラメータを浮動少数値(Double)で追加.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @param value 対象の内容を設定します.
     * @exception InputException 入力例外.
     */
    public synchronized void setParameterToDouble( String key,Double value )
        throws InputException {
        
        JRcParameter param = null ;
        
        if(
            ( key = convertKey( key ) ) == null ||
            value == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        if( this.session == null || this.session.isUseObject() == false ) {
            throw new InputException( "オブジェクトは無効な状態です" ) ;
        }
            
        if( this.table != null ) {
            
            if( this.table.isData( key ) == true ) {
                throw new InputException(
                    "設定対象のキー名[" + key +
                    "]は既に存在します" ) ;
            }
            
            param = new JRcParameter() ;
            param.setDouble( value ) ;
            
            this.table.add( key,param ) ;
            
        }
        
    }
    
    /**
     * パラメータを浮動少数値(Double)で取得.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @return Double パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public synchronized Double getParameterToDouble( String key )
        throws ConvertException {
        
        if(
            ( key = convertKey( key ) ) == null ||
            this.session == null ||
            this.session.isUseObject() == false ||
            this.table == null
        ) {
            return null ;
        }
        
        try {
            return (
                ( JRcParameter )this.table.get( key )
            ).getDouble() ;
        } catch( ConvertException ce ) {
            throw ce ;
        } catch( Exception e ) {
        }
        
        return null ;
        
    }
    
    /**
     * パラメータを浮動少数値(Decimal)で追加.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @param value 対象の内容を設定します.
     * @exception InputException 入力例外.
     */
    public synchronized void setParameterToDecimal( String key,BigDecimal value )
        throws InputException {
        
        JRcParameter param = null ;
        
        if(
            ( key = convertKey( key ) ) == null ||
            value == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        if( this.session == null || this.session.isUseObject() == false ) {
            throw new InputException( "オブジェクトは無効な状態です" ) ;
        }
            
        if( this.table != null ) {
            
            if( this.table.isData( key ) == true ) {
                throw new InputException(
                    "設定対象のキー名[" + key +
                    "]は既に存在します" ) ;
            }
            
            param = new JRcParameter() ;
            param.setDecimal( value ) ;
            
            this.table.add( key,param ) ;
            
        }
        
    }
    
    /**
     * パラメータを浮動少数値(Decimal)で取得.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @return BigDecimal パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public synchronized BigDecimal getParameterToDecimal( String key )
        throws ConvertException {
        
        if(
            ( key = convertKey( key ) ) == null ||
            this.session == null ||
            this.session.isUseObject() == false ||
            this.table == null
        ) {
            return null ;
        }
        
        try {
            return (
                ( JRcParameter )this.table.get( key )
            ).getDecimal() ;
        } catch( ConvertException ce ) {
            throw ce ;
        } catch( Exception e ) {
        }
        
        return null ;
        
    }
    
    /**
     * パラメータをタイムスタンプ(Timestamp)で追加.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @param value 対象の内容を設定します.
     * @exception InputException 入力例外.
     */
    public synchronized void setParameterToTimestamp( String key,Date value )
        throws InputException {
        
        JRcParameter param = null ;
        
        if(
            ( key = convertKey( key ) ) == null ||
            value == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        if( this.session == null || this.session.isUseObject() == false ) {
            throw new InputException( "オブジェクトは無効な状態です" ) ;
        }
            
        if( this.table != null ) {
            
            if( this.table.isData( key ) == true ) {
                throw new InputException(
                    "設定対象のキー名[" + key +
                    "]は既に存在します" ) ;
            }
            
            param = new JRcParameter() ;
            param.setTimestamp( value ) ;
            
            this.table.add( key,param ) ;
            
        }
        
    }
    
    /**
     * パラメータをタイムスタンプ(Timestamp)で取得.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @return Date パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public synchronized Date getParameterToTimestamp( String key )
        throws ConvertException {
        
        if(
            ( key = convertKey( key ) ) == null ||
            this.session == null ||
            this.session.isUseObject() == false ||
            this.table == null
        ) {
            return null ;
        }
        
        try {
            return (
                ( JRcParameter )this.table.get( key )
            ).getTimestamp() ;
        } catch( ConvertException ce ) {
            throw ce ;
        } catch( Exception e ) {
        }
        
        return null ;
        
    }
    
    /**
     * パラメータタイプを取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータタイプが返されます.
     * <BR>
     * @param key 対象のKey名を設定します.
     * @return int パラメータタイプが返されます.<BR>
     *             [JRcParameterType#TYPE_BINARY]が返された場合(byte[])情報です.<BR>
     *             [JRcParameterType#TYPE_STRING]が返された場合(String)情報です.<BR>
     *             [JRcParameterType#TYPE_BOOLEAN]が返された場合(Boolean)情報です.<BR>
     *             [JRcParameterType#TYPE_SHORT]が返された場合(Short)情報です.<BR>
     *             [JRcParameterType#TYPE_INTEGER]が返された場合(Integer)情報です.<BR>
     *             [JRcParameterType#TYPE_LONG]が返された場合(Long)情報です.<BR>
     *             [JRcParameterType#TYPE_FLOAT]が返された場合(Float)情報です.<BR>
     *             [JRcParameterType#TYPE_DOUBLE]が返された場合(Double)情報です.<BR>
     *             [JRcParameterType#TYPE_DECIMAL]が返された場合(BigDecimal)情報です.<BR>
     *             [JRcParameterType#TYPE_TIMESTAMP]が返された場合(Date)情報です.<BR>
     *             パラメータタイプが定義されていない場合かキー内容が存在しない場合、
     *             [-1]が返されます.
     */
    public synchronized int getParameterType( String key ) {
        
        if(
            ( key = convertKey( key ) ) == null ||
            this.session == null ||
            this.session.isUseObject() == false ||
            this.table == null
        ) {
            return -1 ;
        }
        
        try {
            return (
                ( JRcParameter )this.table.get( key )
            ).getType() ;
        } catch( Exception e ) {
        }
        
        return -1 ;
        
    }
    
    /**
     * パラメータキー名一覧を取得.
     * <BR><BR>
     * パラメータキー名一覧を取得します.
     * <BR>
     * @return String[] パラメータキー名一覧が返されます.
     */
    public synchronized String[] getParameterKeys() {
        
        if(
            this.session == null ||
            this.session.isUseObject() == false ||
            this.table == null
        ) {
            return null ;
        }
        
        return this.table.getNames() ;
        
    }
    
    /**
     * パラメータ数を取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータ数が返されます.
     * <BR>
     * @return int 設定されているパラメータ数が返されます.
     */
    public synchronized int getParameterSize() {
        
        if(
            this.session == null ||
            this.session.isUseObject() == false ||
            this.table == null
        ) {
            return 0 ;
        }
        
        return this.table.size() ;
        
    }
    
    /**
     * パラメータ名が存在するかチェック.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータ名が
     * 存在するかチェックします.
     * <BR>
     * @param key チェック対象Key名を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、対象Key名はパラメータに存在します.<BR>
     *                 [false]が返された場合、対象Key名はパラメータに存在しません.
     */
    public synchronized boolean isParameterName( String key ) {
        
        if(
            ( key = convertKey( key ) ) == null ||
            this.session == null ||
            this.session.isUseObject() == false ||
            this.table == null
        ) {
            return false ;
        }
        
        return this.table.isData( key ) ;
        
    }
    
    /**
     * JRcClientSessionオブジェクトを取得.
     * <BR><BR>
     * JRcClientSessionオブジェクトを取得します.
     * <BR>
     * @return JRcClientSession JRcClientSessionオブジェクトが返されます.
     */
    public synchronized JRcClientSession getJRcClientSession() {
        
        return (
            this.session == null ||
            this.session.isUseObject() == false
        ) ?
            null : this.session ;
        
    }
    
    /**
     * アプリケーション名を取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのアプリケーション名を取得します.
     * <BR>
     * @return String アプリケーション名が返されます.
     */
    public synchronized String getApplicationName() {
        
        return (
            this.session == null ||
            this.session.isUseObject() == false
        ) ?
            null : this.session.getApplicationName() ;
        
    }
    
    /**
     * セッションIDを取得.
     * <BR><BR>
     * 現在のセッションIDを取得します.
     * <BR>
     * @return long セッションIDが返されます.
     */
    public synchronized long getSessionID() {
        
        return (
            this.session == null ||
            this.session.isUseObject() == false
        ) ?
            -1L : this.session.getSessionID() ;
        
    }
    
    /**
     * 現在のコネクション先アドレスを取得.
     * <BR><BR>
     * 現在コネクションされているアドレス情報を取得します.
     * <BR>
     * @return InetAddress コネクション先アドレスが返されます.
     */
    public synchronized InetAddress getServerAddress() {
        
        InetAddress ret = null ;
        
        if (
            this.session == null ||
            this.session.isUseObject() == false
        ) {
            ret = null ;
        }
        else {
            try {
                ret = InetAddress.getByName(
                    this.session.getHostName()
                ) ;
            } catch( Exception e ) {
                ret = null ;
            }
        }
        
        return ret ;
        
    }
    
    /**
     * 現在のコネクション先ポート番号を取得.
     * <BR><BR>
     * 現在コネクションされているポート番号を取得します.
     * <BR>
     * @return int コネクション先ポート番号が返されます.
     */
    public synchronized int getServerPort() {
        
        return (
            this.session == null ||
            this.session.isUseObject() == false
        ) ?
            -1 : this.session.getPort() ;
        
    }
    
    /**
     * ローカルアドレスを取得.
     * <BR><BR>
     * ローカルアドレスを取得します.
     * <BR>
     * @return InetAddress ローカルアドレスが返されます.
     */
    public synchronized InetAddress getLocalAddress() {
        
        if (
            this.session == null ||
            this.session.isUseObject() == false ||
            this.socket == null
        ) {
            return null ;
        }
        
        return this.socket.getLocalAddress() ;
        
    }
    
    /**
     * ローカルポート番号を取得.
     * <BR><BR>
     * ローカルポート番号を取得します.
     * <BR>
     * @return int ローカルポート番号が返されます.
     */
    public synchronized int getLocalPort() {
        
        if (
            this.session == null ||
            this.session.isUseObject() == false ||
            this.socket == null
        ) {
            return -1 ;
        }
        
        return this.socket.getLocalPort() ;
        
    }
    
    /**
     * リソースタイプを取得.
     * <BR><BR>
     * リソースタイプを取得します.
     * <BR>
     * @return ResourceType 対象のリソースタイプが返されます.
     */
    public synchronized ResourceType getResourceType() {
        
        if (
            this.session == null ||
            this.session.isUseObject() == false
        ) {
            return null ;
        }
        
        return this.resType ;
        
    }
    
    /**
     * このオブジェクトが既にクローズされているかチェック.
     * <BR><BR>
     * このオブジェクトが既にクローズされているかチェックします.
     * <BR>
     * @return boolean クローズされているかチェックします.<BR>
     *                 [true]が返された場合、既にクローズされています.<BR>
     *                 [false]が返された場合、コネクション中です.
     */
    public synchronized boolean isClosed() {
        
        if (
            this.session == null ||
            this.session.isUseObject() == false ||
            this.socket == null
        ) {
            return true ;
        }
        
        return this.socket.isClosed() ;
        
    }
    
    
    
    /**
     * 指定キー名を小文字変換.
     */
    private static final String convertKey( String key ) {
        if(
            key == null ||
            ( key = key.trim().toLowerCase() ).length() <= 0
        ) {
            return null ;
        }
        
        return key ;
    }
    
    /**
     * 対象条件をKey,valueに変換.
     */
    private static final String[] convertKeyValue( String string ) {
        
        int pnt ;
        String[] ret = null ;
        
        pnt = string.indexOf( "=" ) ;
        ret = new String[ 2 ] ;
        ret[ 0 ] = string.substring( 0,pnt ) ;
        ret[ 1 ] = string.substring( pnt+1,string.length() ) ;
        
        return ret ;
        
    }
    
    /**
     * URL形式のコマンド内容をURLデコードする.
     */
    private static final ObjectArray convertURLDecoder( String[] command )
        throws JRCommandException {
        
        int i ;
        int len ;
        int pnt ;
        
        String cmd = null ;
        String str = null ;
        String[] kv = null ;
        ArrayList lst = null ;
        ObjectArray ret = null ;
        
        try {
            
            if( command[ 0 ] == null || ( command[ 0 ] = command[ 0 ].trim() ).length() <= 0 ) {
                throw new JRCommandException(
                    JRcErrorCodeDef.ERROR_ETC,
                    "コマンド内容は存在しません"
                ) ;
            }
            if( ( pnt = command[ 0 ].indexOf( "&" ) ) <= -1 ) {
                ret = new ObjectArray() ;
                ret.add( new String( command[ 0 ] ) ) ;
                command[ 0 ] = null ;
                return ret ;
            }
            
            cmd = command[ 0 ].substring( 0,pnt ) ;
            command[ 0 ] = command[ 0 ].substring( pnt+1,command[ 0 ].length() ) ;
            
            if( command[ 0 ] == null || ( command[ 0 ] = command[ 0 ].trim() ).length() <= 0 ) {
                ret = new ObjectArray() ;
                ret.add( cmd ) ;
            }
            else {
                
                lst = UtilCom.pauseString( command[ 0 ],"&" ) ;
                command[ 0 ] = null ;
                
                if( lst != null && ( len = lst.size() ) > 0 ) {
                    
                    ret = new ObjectArray() ;
                    ret.add( cmd ) ;
                    
                    for( i = 0 ; i < len ; i ++ ) {
                        
                        str = ( String )lst.get( i ) ;
                        
                        if( str == null || ( str = str.trim() ).length() <= 0 ) {
                            continue ;
                        }
                        
                        if( UtilCom.checkCountByStirng( str,"=" ) != 1 ) {
                            continue ;
                        }
                        
                        kv = JRcConnectionImple.convertKeyValue( str ) ;
                        if(
                            kv[ 0 ] == null || ( kv[ 0 ] = kv[ 0 ].trim() ).length() <= 0 ||
                            kv[ 1 ] == null || ( kv[ 1 ] = kv[ 1 ].trim() ).length() <= 0
                        ) {
                            continue ;
                        }
                        
                        kv[ 0 ] = kv[ 0 ].trim() ;
                        kv[ 1 ] = URLDecoder.decode( kv[ 1 ].trim(),BaseDef.UTF8 ) ;
                        
                        ret.add( kv ) ;
                        kv = null ;
                        
                    }
                    
                }
                else {
                    ret = new ObjectArray() ;
                    ret.add( cmd ) ;
                }
                
            }
            
        } catch( Exception e ) {
            throw new JRCommandException(
                JRcErrorCodeDef.ERROR_ETC,
                "コマンドの解析に失敗しました"
            ) ;
        } finally {
            if( lst != null ) {
                lst.clear() ;
            }
            cmd = null ;
            str = null ;
            kv = null ;
            lst = null ;
        }
        
        return ret ;
    }
    
    /**
     * 通常の形式で解析.
     */
    private static final ObjectArray convertBase( String[] command )
        throws JRCommandException {
        
        int i,j ;
        int len ;
        int lenj ;
        boolean flg ;
        boolean chk ;
        
        String str = null ;
        String[] kv = null ;
        ArrayList lst = null ;
        ObjectArray ret = null ;
        
        try {
            
            lst = UtilCom.pauseString( true,false,command[ 0 ]," \t" ) ;
            
            if( lst != null && ( len = lst.size() ) > 0 ) {
                
                ret = new ObjectArray() ;
                ret.add( lst.get( 0 ) ) ;
                command[ 0 ] = null ;
                
                for( i = 1,flg = false ; i < len ; i ++ ) {
                    
                    str = ( String )lst.get( i ) ;
                    
                    if( str.startsWith( "-" ) == true && ( lenj = str.length() ) > 1 ) {
                        
                        for( j = 1,chk = false ; j < lenj ; j ++ ) {
                            
                            if( str.charAt( j ) != '-' ) {
                                
                                str = str.substring( j ) ;
                                chk = true ;
                                break ;
                                
                            }
                            
                        }
                        
                        if( chk == true && ( str = str.trim().toLowerCase() ).length() > 0 ) {
                            
                            kv = new String[ 2 ] ;
                            kv[ 0 ] = str ;
                            flg = true ;
                            
                        }
                        else {
                            kv = null ;
                            flg = false ;
                        }
                        
                    }
                    else if( flg == true && kv != null ) {
                        
                        kv[ 1 ] = str ;
                        ret.add( kv ) ;
                        kv = null ;
                        flg = false ;
                        
                    }
                    else {
                        
                        flg = false ;
                        kv = null ;
                        
                    }
                    
                }
                
            }
            else {
                ret = new ObjectArray() ;
                ret.add( new String( command[ 0 ] ) ) ;
            }
            
        } catch( Exception e ) {
            throw new JRCommandException(
                JRcErrorCodeDef.ERROR_ETC,
                "コマンドの解析に失敗しました"
            ) ;
        } finally {
            if( lst != null ) {
                lst.clear() ;
            }
            str = null ;
            kv = null ;
            lst = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 変換可能な条件を探してパラメータに登録.
     */
    private final void convertPossibleParameter( String[] kv )
        throws JRCommandException,Exception {
        
        String key = null ;
        String value = null ;
        String file = null ;
        String base64 ;
        Date d = null ;
        SimpleDateFormat s = null ;
        
        key = kv[ 0 ] ;
        value = kv[ 1 ] ;
        
        // Base64.
        if( value.length() > BASE64_HEADER.length() ) {
            
            // Base64ヘッダが設定されている場合.
            if(
                BASE64_HEADER.equals(
                    value.substring( 0,BASE64_HEADER.length() ).toLowerCase()
                ) == true
            ) {
                
                base64 = value.substring( BASE64_HEADER.length() ) ;
                
                // 対象条件がBase64変換できるかチェック.
                if( Base64.isBase64( base64 ) == false ) {
                    throw new JRCommandException(
                        JRcErrorCodeDef.ERROR_ETC,
                        "Base64デコード処理に失敗しました"
                    ) ;
                }
                
                // Base64をバイナリとして設定.
                this.setParameterToBinary( key,Base64.decode( base64 ) ) ;
                
                return ;
                
            }
        }
        
        // LocalFile.
        if( value.length() > LFILE_HEADER.length() ) {
            
            // 読み込みローカルファイルヘッダが指定されている場合.
            if(
                LFILE_HEADER.equals(
                    value.substring( 0,LFILE_HEADER.length() ).toLowerCase()
                ) == true
            ) {
                
                // 読み込みファイル名を設定.
                file = value.substring( LFILE_HEADER.length() ) ;
                
                // 読み込みファイルがworkディレクトリ内の場合.
                if(
                    file.indexOf( ":" ) == -1 &&
                    file.startsWith( "/" ) == false &&
                    file.startsWith( "\\" ) == false
                ){
                    
                    // workディレクトリ以下で設定.
                    file = new StringBuffer().
                        append( JRcClientDriver.WORK_DIR ).
                        append( BaseDef.FILE_SEPARATOR ).
                        append( file ).
                        toString() ;
                    
                }
                
                // ファイル名を正しい条件で変換.
                file = IOCom.getName( file ) ;
                
                // ファイウが存在しない場合.
                if( IOCom.isFileExists( file ) == false ) {
                    throw new JRCommandException(
                        JRcErrorCodeDef.ERROR_ETC,
                        "対象ファイル[" + file + "]は存在しません"
                    ) ;
                }
                
                // ファイルを読み込みバイナリをパラメータに設定.
                this.setParameterToBinary( key,IOCom.getFile( file ) ) ;
                
                return ;
                
            }
        }
        
        // 数値系.
        try {
            
            // とりあえずLongで変換.
            long lng = Long.parseLong( value ) ;
            
            // Integerでもセットできる場合.
            try {
                int integer ;
                if( lng == ( long )( ( integer = Integer.parseInt( value ) ) ) ) {
                    this.setParameterToInteger( key,new Integer( integer ) ) ;
                    return ;
                }
            } catch( Exception e ) {
            }
            
            // Longで変換.
            this.setParameterToLong( key,new Long( lng ) ) ;
            return ;
            
        } catch( Exception e ) {
        }
        
        // 小数点.
        try {
            // 小数点系.
            if( value.indexOf( "." ) >= 0 ) {
                
                try {
                    Double v = new Double( value ) ;
                    this.setParameterToDouble( key,v ) ;
                    return ;
                } catch( Exception e ) {
                }
                
            }
        } catch( Exception e ) {
        }
        
        // Boolean設定.
        try {
            String v = value.toLowerCase() ;
            if( v.equals( "true" ) == true ) {
                this.setParameterToBoolean( key,new Boolean( true ) ) ;
                return ;
            }
            if( v.equals( "false" ) == true ) {
                this.setParameterToBoolean( key,new Boolean( false ) ) ;
                return ;
            }
        } catch( Exception e ) {
        }
        
        // 日付フォーマットの可能性がある場合.
        if(
            value.indexOf( "/" ) >= 0 || value.indexOf( "-" ) >= 0 ||
            value.indexOf( "年" ) >= 0 || value.indexOf( "月" ) >= 0 ||
            value.indexOf( "日" ) >= 0
        ) {
            
            // 日付フォーマット1+時間フォーマット.
            try {
                s = new SimpleDateFormat( "yyyy/MM/dd,HH:mm:ss" ) ;
                d = s.parse( value ) ;
                this.setParameterToTimestamp( key,d ) ;
                return ;
            } catch( Exception e ) {
            }
            // 日付フォーマット2+時間フォーマット.
            try {
                s = new SimpleDateFormat( "yyyy-MM-dd,HH:mm:ss" ) ;
                d = s.parse( value ) ;
                this.setParameterToTimestamp( key,d ) ;
                return ;
            } catch( Exception e ) {
            }
            // 日付フォーマット3+時間フォーマット.
            try {
                s = new SimpleDateFormat( "yyyy年MM月dd日,HH:mm:ss" ) ;
                d = s.parse( value ) ;
                this.setParameterToTimestamp( key,d ) ;
                return ;
            } catch( Exception e ) {
            }
            // 日付フォーマット1
            try {
                s = new SimpleDateFormat( "yyyy/MM/dd" ) ;
                d = s.parse( value ) ;
                this.setParameterToTimestamp( key,d ) ;
                return ;
            } catch( Exception e ) {
            }
            // 日付フォーマット2
            try {
                s = new SimpleDateFormat( "yyyy-MM-dd" ) ;
                d = s.parse( value ) ;
                this.setParameterToTimestamp( key,d ) ;
                return ;
            } catch( Exception e ) {
            }
            // 日付フォーマット2
            try {
                s = new SimpleDateFormat( "yyyy年MM月dd日" ) ;
                d = s.parse( value ) ;
                this.setParameterToTimestamp( key,d ) ;
                return ;
            } catch( Exception e ) {
            }
        }
        
        // 全てで登録できない場合、文字で登録.
        this.setParameterToString( key,value ) ;
        
    }
    
    /**
     * コマンドを解析.
     */
    private final void commandAnalysisByCommand( ObjectArray ary )
        throws JRCommandException {
        
        int i ;
        int len ;
        
        String[] kv = null ;
        
        if( ary == null || ( len = ary.size() ) <= 0 ) {
            throw new JRCommandException(
                JRcErrorCodeDef.ERROR_ETC,
                "コマンド内容は存在しません"
            ) ;
        }
        
        try {
            
            // コマンド名を設定.
            this.createService( ( String )ary.get( 0 ) ) ;
            
            // コマンドパラメータを設定.
            for( i = 1 ; i < len ; i ++ ) {
                
                kv = ( String[] )ary.get( i ) ;
                //this.setParameterToString( kv[ 0 ],kv[ 1 ] ) ;
                this.convertPossibleParameter( kv ) ;
                
            }
            
        } catch( JRCommandException je ) {
            throw je ;
        } catch( Exception e ) {
            throw new JRCommandException(
                JRcErrorCodeDef.ERROR_ETC,
                "コマンド条件設定中にエラーが発生しました"
            ) ;
        } finally {
            
            ary.clear() ;
            ary = null ;
            
        }
        
    }
    
    /**
     * JRcServerにサービス処理実行.
     */
    private final JRcResponseBean executionService()
        throws ConnectTimeoutException,ExecutionException,AccessException {
        
        boolean flg = false ;
        OutputStream outputStream = null ;
        InputStream inputStream = null ;
        BinResource binary = null ;
        JRcResponseBean ret = null ;
        
        try {
            
            // JRcServerにサービス送信.
            outputStream = new BufferedOutputStream(
                this.socket.getOutputStream()
            ) ;
            //outputStream = this.socket.getOutputStream() ;
            
            // データ送信.
            this.createSend( resType,outputStream ) ;
            outputStream.flush() ;
            //outputStream.close() ;
            outputStream = null ;
            
            // 受信待ち.
            inputStream = new BufferedInputStream(
                this.socket.getInputStream()
            ) ;
            //inputStream = this.socket.getInputStream() ;
            binary = JRcConnectCommon.receive(
                true,resType,sync,inputStream,timeout
            ) ;
            
            inputStream = null ;
            
            // 受信結果内容を解析.
            ret = this.getAnalysis( resType,binary ) ;
            binary.clear() ;
            binary = null ;
            
        } catch( SocketException se ) {
            //se.printStackTrace() ;
            System.out.println( se.getMessage() ) ;
            flg = true ;
            throw new ExecutionException(
                se,ExecutionException.LEVEL_STOP
            ) ;
        } catch( ConnectTimeoutException ct ) {
            //ct.printStackTrace() ;
            System.out.println( ct.getMessage() ) ;
            flg = true ;
            throw ct ;
        } catch( ExecutionException ee ) {
            //ee.printStackTrace() ;
            System.out.println( ee.getMessage() ) ;
            flg = true ;
            throw ee ;
        } catch( AccessException ae ) {
            //ae.printStackTrace() ;
            System.out.println( ae.getMessage() ) ;
            flg = true ;
            throw ae ;
        } catch( Exception e ) {
            //e.printStackTrace() ;
            System.out.println( e.getMessage() ) ;
            flg = true ;
            throw new AccessException( e ) ;
        } finally {
            
            if( flg == true ) {
                this.closeConnection() ;
                ret.destroy() ;
                ret = null ;
            }
            
            if( binary != null ) {
                binary.clear() ;
            }
            
            outputStream = null ;
            inputStream = null ;
            binary = null ;
            
        }
        
        return ret ;
    }
    
    
    /**
     * 指定条件の送信電文を生成.
     */
    private final void createSend( ResourceType type,OutputStream stream )
        throws AccessException {
        
        int i ;
        int len ;
        int cnt ;
        int paramNameLen ;
        
        byte[] bin = null ;
        BinResource binRes = null ;
        OutputStream binStream = null ;
        String[] paramNames = null ;
        JRcParameter param = null ;
        
        try {
            
            binRes = Resource.createBinResource( type,1 ) ;
            binStream = new BufferedOutputStream(
                new BinResourceOutputStream(
                    binRes
                )
            ) ;
            
            // セッションID.
            binStream.write( ConvertParam.convertLong( this.session.getSessionID() ) ) ;
            
            // アプリケーション名長.
            bin = this.session.getApplicationName().getBytes( BaseDef.UTF8 ) ;
            len = bin.length ;
            binStream.write( ConvertParam.convertInt( len ) ) ;
            
            // アプリケーション名.
            binStream.write( bin ) ;
            
            // サービス名.
            bin = serviceName.getBytes( BaseDef.UTF8 ) ;
            len = bin.length ;
            binStream.write( ConvertParam.convertInt( len ) ) ;
            
            // アプリケーション名.
            binStream.write( bin ) ;
            
            // パラメータ長を取得.
            len = this.table.size() ;
            
            // パラメータ長を設定.
            binStream.write( ConvertParam.convertInt( len ) ) ;
            
            // パラメータが存在する場合.
            if( len > 0 ) {
                
                // パラメータ名群を取得.
                paramNames = this.table.getNames() ;
                
                // 存在するパラメータ一覧を登録.
                for( i = 0 ; i < len ; i ++ ) {
                    
                    // 対象名のパラメータを取得.
                    param = ( JRcParameter )this.table.get( paramNames[ i ] ) ;
                    
                    // 情報が不正な場合.
                    if(
                        param == null ||
                        param.getBinary() == null || param.getBinary().length <= 0
                    ) {
                        // 長さ0のパラメータ名を設定.
                        binStream.write( ConvertParam.convertInt( len ) ) ;
                        
                        // 存在しないパラメータタイプを設定.
                        binStream.write( 0x000000ff ) ;
                        
                        // 長さ0のパラメータ内容を設定.
                        binStream.write( ConvertParam.convertInt( len ) ) ;
                        
                        continue ;
                    }
                    
                    // パラメータ名長を設定.
                    bin = paramNames[ i ].getBytes( BaseDef.UTF8 ) ;
                    paramNameLen = bin.length ;
                    binStream.write( ConvertParam.convertInt( paramNameLen ) ) ;
                    
                    // パラメータ名を設定.
                    binStream.write( bin ) ;
                    
                    // パラメータ内容を設定.
                    param.exportOutputStream( binStream ) ;
                    
                }
                
            }
            
            binStream.flush() ;
            binStream.close() ;
            binStream = null ;
            
            bin = new byte[ SEND_BUFFER ] ;
            
            // 送信条件に対して、プロトコルヘッダと情報長を設定する.
            len = JRcProtocolDef.RCV_HEADER_LENGTH + 4 + binRes.size() ;
            
            // プロトコルヘッダ.
            stream.write( JRcProtocolDef.RCV_HEADER ) ;
            
            // 情報長.
            stream.write( ConvertParam.convertInt( len ) ) ;
            
            // 生成したプロトコル内容を設定.
            for( cnt = 0 ;; ) {
                
                if( ( len = binRes.getBinary( bin,cnt ) ) <= 0 ) {
                    break ;
                }
                
                stream.write( bin,0,len ) ;
                cnt += len ;
                
            }
            
            //stream.write( binRes.getBinary() ) ;
            stream.flush() ;
            
        } catch( Exception e ) {
            throw new AccessException( e ) ;
        } finally {
            
            if( binStream != null ) {
                try {
                    binStream.close() ;
                } catch( Exception ee ) {
                }
            }
            
            binStream = null ;
            
        }
        
    }
    
    /**
     * 受信データを解析.
     */
    private final JRcResponseBean getAnalysis( ResourceType resType,BinResource binary )
        throws AccessException {
        
        int len ;
        int pnt ;
        
        byte[] bin = null ;
        BinResource data = null ;
        JRcResponseBean ret = null ;
        
        try {
            
            bin = binary.getBinary( 0,JRcProtocolDef.SEND_HEADER_LENGTH ) ;
            
            // プロトコルチェック.
            if(
                bin[ 0 ] != JRcProtocolDef.SEND_HEADER[ 0 ] ||
                bin[ 1 ] != JRcProtocolDef.SEND_HEADER[ 1 ] ||
                bin[ 2 ] != JRcProtocolDef.SEND_HEADER[ 2 ] ||
                bin[ 3 ] != JRcProtocolDef.SEND_HEADER[ 3 ] ||
                bin[ 4 ] != JRcProtocolDef.SEND_HEADER[ 4 ] ||
                bin[ 5 ] != JRcProtocolDef.SEND_HEADER[ 5 ]
            ) {
                throw new AccessException(
                    "受信プロトコルヘッダは不正です"
                ) ;
            }
            
            bin = null ;
            pnt = JRcProtocolDef.SEND_HEADER_LENGTH + 4 ;
            
            // データ格納条件を生成.
            ret = new JRcResponseBean() ;
            
            // データモードを取得.
            ret.setMode(
                ( ( binary.get( pnt ) & 0x00000001 ) == 1 ) ?
                    true : false
            ) ;
            pnt += 1 ;
            
            // キャラクタセット長を取得.
            len = ConvertResourceParam.convertInt( pnt,binary ) ;
            pnt += 4 ;
            
            // キャラクタセットを取得.
            if( len > 0 ) {
                
                ret.setCharset(
                    ConvertResourceParam.convertString(
                        pnt,len,binary,BaseDef.UTF8
                    )
                ) ;
                
                pnt += len ;
                
            }
            else {
                ret.setCharset( JRcResponseBean.DEFAULT_CHARSET ) ;
            }
            
            // Mimeタイプ長を取得.
            len = ConvertResourceParam.convertInt( pnt,binary ) ;
            pnt += 4 ;
            
            // Mimeタイプを取得.
            if( len > 0 ) {
                
                ret.setMimeType(
                    ConvertResourceParam.convertString(
                        pnt,len,binary,BaseDef.UTF8
                    )
                ) ;
                
                pnt += len ;
                
            }
            
            // 処理戻り値を取得.
            ret.setResult( ConvertResourceParam.convertInt( pnt,binary ) ) ;
            pnt += 4 ;
            
            // 戻り値メッセージ長.
            len = ConvertResourceParam.convertInt( pnt,binary ) ;
            pnt += 4 ;
            
            // 戻り値メッセージを取得.
            if( len > 0 ) {
                
                ret.setResultMessage(
                    ConvertResourceParam.convertString(
                        pnt,len,binary,BaseDef.UTF8
                    )
                ) ;
                
                pnt += len ;
                
            }
            
            // セッションIDを取得.
            ret.setSessionID( ConvertResourceParam.convertLong( pnt,binary ) ) ;
            pnt += 8 ;
            
            // データ長.
            len = ConvertResourceParam.convertInt( pnt,binary ) ;
            pnt += 4 ;
            
            // データを取得.
            if( len > 0 ) {
                data = Resource.createBinResource( resType,len ) ;
                binary.getBinary( data,pnt,0,len ) ;
                ret.setBinary( data ) ;
                pnt += len ;
            }
            
        } catch( AccessException ae ) {
            throw ae ;
        } catch( Exception e ) {
            throw new AccessException( e ) ;
        }
        
        return ret ;
        
    }
}

