/*
 * @(#)JRcConnection.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.client ;

import java.math.BigDecimal;
import java.net.InetAddress;
import java.util.Date;

import com.JRcServer.JRCommandException;
import com.JRcServer.commons.exception.AccessException;
import com.JRcServer.commons.exception.ConvertException;
import com.JRcServer.commons.exception.ExecutionException;
import com.JRcServer.commons.exception.InputException;

/**
 * JRcConnection.
 * <BR><BR>
 * JRcClientコネクションインターフェイス.
 *  
 * @version 2006/09/10
 * @author  masahito suzuki
 * @since   JRcClient 1.00
 */
public interface JRcConnection {
    
    /**
     * 現在のコネクションをクローズ.
     * <BR><BR>
     * 現在のコネクション条件をクローズします.
     */
    public void close() ;
    
    /**
     * 再コネクション.
     * <BR><BR>
     * 再コネクションを行います.
     * <BR>
     * @exception AccessException アクセス例外.
     */
    public void retry() throws AccessException ;
    
    /**
     * 切断処理.
     * <BR><BR>
     * 通信を切断します.
     */
    public void closeConnection() ;
    
    /**
     * 新しいサービスでコネクション.
     * <BR><BR>
     * 新しいサービス名を設定してコネクションを生成します.<BR>
     * そのため、設定途中の条件が存在する場合、その情報は破棄されます.
     * <BR>
     * @param name 対象のサービス名を設定します.
     * @exception InputException 入力例外.
     */
    public void createService( String name )
        throws InputException ;
    
    /**
     * 現在のサービス条件を破棄.
     * <BR><BR>
     * 現在のサービスを破棄します.
     */
    public void clearService() ;
    
    /**
     * 現在の条件でサービスを呼び出す.
     * <BR><BR>
     * 設定されている現在の条件で、サービスを呼び出します.
     * <BR>
     * @return JRcResult 処理結果の内容が返されます.
     * @exception JRCommandException JRCommon例外.
     * @exception ExecutionException 実行例外.
     */
    public JRcResult execution()
        throws JRCommandException,ExecutionException ;
    
    /**
     * コマンドによる直接指定で呼び出し.
     * <BR><BR>
     * コマンドによる直接指定で呼び出します.<BR>
     * また、このメソッドの場合は、[createService()]や
     * パラメータ指定の条件は全て無視します.<BR>
     * 代わりに、指定された文字列をコマンドとして、サービス名、
     * パラメータの順に定義して呼び出します.<BR>
     * 呼び出す方法としては、以下の2点があります.<BR>
     * <PRE>
     * その１:通常指定.
     * hoge -param1 ジェイアールシー -param2 sv
     * 
     * その２:URL指定.
     * ？hoge＆param1=%E3%82%B8%E3%82%A7%E3%82%A4%E3%82%A2%E3%83%BC%E3%83%AB%E3%82%B7%E3%83%BC＆param2=sv
     * 
     * 　※先頭に必ず[？(半角)]が必要となります.
     *
     * </PRE>
     * 両方とも同様の[hoge]サービスを呼び出し、２つのパラメータ<BR>
     * [param1=ジェイアールシー],[param2=sv]を指定しています.<BR>
     * また、URLエンコードはUTF8のみサポートしているので、<BR>
     * それ以外の指定条件では設定できません.<BR>
     * (文字化けします)<BR>
     * <BR>
     * 設定されたパラメータの値はString以外の近い条件で設定します.<BR>
     * たとえば、Boolean変換したいものは、[true]や[false]を設定することで、<BR>
     * 変換対象となります.<BR>
     * その中でも特殊なものとしては、Timestamp登録です.<BR>
     * 以下に対応している日付フォーマット一覧です.<BR>
     * <BR>
     * ・yyyy/MM/dd,HH:mm:ss<BR>
     * ・yyyy-MM-dd,HH:mm:ss<BR>
     * ・yyyy年MM月dd日,HH:mm:ss<BR>
     * ・yyyy/MM/dd<BR>
     * ・yyyy-MM-dd<BR>
     * ・yyyy年MM月dd日<BR>
     * <BR>
     * もう１つ特殊なものとしては、Base64でのバイナリ指定があります.<BR>
     * 方法としては、データ定義の初めに[base64-]を指定して以降にBase64の
     * コードを指定します.<BR>
     * <PRE>
     * ＜例＞
     * hoge -param1 base64-44GY44GH44GE44GC44O844KL44GX44O8c3Y=
     * 
     * または
     * ?hoge&param1=base64-44GY44GH44GE44GC44O844KL44GX44O8c3Y=
     * </PRE>
     * 
     * このように定義しなければいけません.<BR>
     * <BR>
     * 最後に、ローカルファイルの内容を設定する方法として、以下のような<BR>
     * 方法で、設定できます.<BR>
     * <PRE>
     * ＜例＞
     * hoge -param1 file:/c:/dir/test.txt
     * 
     * または
     * ?hoge?param1=file:/c:/dir/test.txt
     * 
     * </PRE>
     * このように定義しなければいけません.<BR>
     * <BR>
     * @param command コマンドによる直接指定で呼び出します.
     * @return JRcResult 処理結果の内容が返されます.
     * @exception JRCommandException JRCommon例外.
     * @exception ExecutionException 実行例外.
     */
    public JRcResult execution( String command )
        throws JRCommandException,ExecutionException ;
    
    /**
     * 対象のパラメータを削除.
     * <BR><BR>
     * 対象のパラメータ情報を削除します.
     * <BR>
     * @param key 対象のKey名を設定します.
     */
    public void removeParameter( String key ) ;
    
    /**
     * パラメータをバイナリで追加.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @param value 対象の内容を設定します.
     * @exception InputException 入力例外.
     */
    public void setParameterToBinary( String key,byte[] value )
        throws InputException ;
    
    /**
     * パラメータをバイナリで取得.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @return byte[] パラメータ情報が返されます.
     */
    public byte[] getParameterToBinary( String key ) ;
    
    /**
     * パラメータを文字列で追加.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @param value 対象の内容を設定します.
     * @exception InputException 入力例外.
     */
    public void setParameterToString( String key,String value )
        throws InputException ;
    
    /**
     * パラメータを文字列で取得.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @return String パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public String getParameterToString( String key )
        throws ConvertException ;
    
    /**
     * パラメータをフラグで追加.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @param value 対象の内容を設定します.
     * @exception InputException 入力例外.
     */
    public void setParameterToBoolean( String key,Boolean value )
        throws InputException ;
    
    /**
     * パラメータをフラグで取得.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @return Boolean パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public Boolean getParameterToBoolean( String key )
        throws ConvertException ;
    
    /**
     * パラメータを数値(Short)で追加.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @param value 対象の内容を設定します.
     * @exception InputException 入力例外.
     */
    public void setParameterToShort( String key,Short value )
        throws InputException ;
    
    /**
     * パラメータを数値(Short)で取得.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @return Short パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public Short getParameterToShort( String key )
        throws ConvertException ;
    
    /**
     * パラメータを数値(Integer)で追加.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @param value 対象の内容を設定します.
     * @exception InputException 入力例外.
     */
    public void setParameterToInteger( String key,Integer value )
        throws InputException ;
    
    /**
     * パラメータを数値(Integer)で取得.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @return Integer パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public Integer getParameterToInteger( String key )
        throws ConvertException ;
    
    /**
     * パラメータを数値(Long)で追加.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @param value 対象の内容を設定します.
     * @exception InputException 入力例外.
     */
    public void setParameterToLong( String key,Long value )
        throws InputException ;
    
    /**
     * パラメータを数値(Long)で取得.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @return Long パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public Long getParameterToLong( String key )
        throws ConvertException ;
    
    /**
     * パラメータを浮動少数値(Float)で追加.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @param value 対象の内容を設定します.
     * @exception InputException 入力例外.
     */
    public void setParameterToFloat( String key,Float value )
        throws InputException ;
    
    /**
     * パラメータを浮動少数値(Float)で取得.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @return Float パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public Float getParameterToFloat( String key )
        throws ConvertException ;
    
    /**
     * パラメータを浮動少数値(Double)で追加.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @param value 対象の内容を設定します.
     * @exception InputException 入力例外.
     */
    public void setParameterToDouble( String key,Double value )
        throws InputException ;
    
    /**
     * パラメータを浮動少数値(Double)で取得.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @return Double パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public Double getParameterToDouble( String key )
        throws ConvertException ;
    
    /**
     * パラメータを浮動少数値(Decimal)で追加.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @param value 対象の内容を設定します.
     * @exception InputException 入力例外.
     */
    public void setParameterToDecimal( String key,BigDecimal value )
        throws InputException ;
    
    /**
     * パラメータを浮動少数値(Decimal)で取得.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @return BigDecimal パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public BigDecimal getParameterToDecimal( String key )
        throws ConvertException ;
    
    /**
     * パラメータをタイムスタンプ(Timestamp)で追加.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @param value 対象の内容を設定します.
     * @exception InputException 入力例外.
     */
    public void setParameterToTimestamp( String key,Date value )
        throws InputException ;
    
    /**
     * パラメータをタイムスタンプ(Timestamp)で取得.
     * <BR><BR>
     * @param key 対象のKey名を設定します.
     * @return Date パラメータ情報が返されます.
     * @exception ConvertException コンバート例外.
     */
    public Date getParameterToTimestamp( String key )
        throws ConvertException ;
    
    /**
     * パラメータタイプを取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータタイプが返されます.
     * <BR>
     * @param key 対象のKey名を設定します.
     * @return int パラメータタイプが返されます.<BR>
     *             [JRcParameterType#TYPE_BINARY]が返された場合(byte[])情報です.<BR>
     *             [JRcParameterType#TYPE_STRING]が返された場合(String)情報です.<BR>
     *             [JRcParameterType#TYPE_BOOLEAN]が返された場合(Boolean)情報です.<BR>
     *             [JRcParameterType#TYPE_SHORT]が返された場合(Short)情報です.<BR>
     *             [JRcParameterType#TYPE_INTEGER]が返された場合(Integer)情報です.<BR>
     *             [JRcParameterType#TYPE_LONG]が返された場合(Long)情報です.<BR>
     *             [JRcParameterType#TYPE_FLOAT]が返された場合(Float)情報です.<BR>
     *             [JRcParameterType#TYPE_DOUBLE]が返された場合(Double)情報です.<BR>
     *             [JRcParameterType#TYPE_DECIMAL]が返された場合(BigDecimal)情報です.<BR>
     *             [JRcParameterType#TYPE_TIMESTAMP]が返された場合(Date)情報です.<BR>
     *             パラメータタイプが定義されていない場合かキー内容が存在しない場合、
     *             [-1]が返されます.
     */
    public int getParameterType( String key ) ;
    
    /**
     * パラメータキー名一覧を取得.
     * <BR><BR>
     * パラメータキー名一覧を取得します.
     * <BR>
     * @return String[] パラメータキー名一覧が返されます.
     */
    public String[] getParameterKeys() ;
    
    /**
     * パラメータ数を取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータ数が返されます.
     * <BR>
     * @return int 設定されているパラメータ数が返されます.
     */
    public int getParameterSize() ;
    
    /**
     * パラメータ名が存在するかチェック.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのパラメータ名が
     * 存在するかチェックします.
     * <BR>
     * @param key チェック対象Key名を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、対象Key名はパラメータに存在します.<BR>
     *                 [false]が返された場合、対象Key名はパラメータに存在しません.
     */
    public boolean isParameterName( String key ) ;
    
    /**
     * JRcClientSessionオブジェクトを取得.
     * <BR><BR>
     * JRcClientSessionオブジェクトを取得します.
     * <BR>
     * @return JRcClientSession JRcClientSessionオブジェクトが返されます.
     */
    public JRcClientSession getJRcClientSession() ;
    
    /**
     * アプリケーション名を取得.
     * <BR><BR>
     * リクエストを送ってきたクライアントからのアプリケーション名を取得します.
     * <BR>
     * @return String アプリケーション名が返されます.
     */
    public String getApplicationName() ;
    
    /**
     * セッションIDを取得.
     * <BR><BR>
     * 現在のセッションIDを取得します.
     * <BR>
     * @return long セッションIDが返されます.
     */
    public long getSessionID() ;
    
    /**
     * 現在のコネクション先アドレスを取得.
     * <BR><BR>
     * 現在コネクションされているアドレス情報を取得します.
     * <BR>
     * @return InetAddress コネクション先アドレスが返されます.
     */
    public InetAddress getServerAddress() ;
    
    /**
     * 現在のコネクション先ポート番号を取得.
     * <BR><BR>
     * 現在コネクションされているポート番号を取得します.
     * <BR>
     * @return int コネクション先ポート番号が返されます.
     */
    public int getServerPort() ;
    
    /**
     * ローカルアドレスを取得.
     * <BR><BR>
     * ローカルアドレスを取得します.
     * <BR>
     * @return InetAddress ローカルアドレスが返されます.
     */
    public InetAddress getLocalAddress() ;
    
    /**
     * ローカルポート番号を取得.
     * <BR><BR>
     * ローカルポート番号を取得します.
     * <BR>
     * @return int ローカルポート番号が返されます.
     */
    public int getLocalPort() ;
    
    /**
     * このオブジェクトが既にクローズされているかチェック.
     * <BR><BR>
     * このオブジェクトが既にクローズされているかチェックします.
     * <BR>
     * @return boolean クローズされているかチェックします.<BR>
     *                 [true]が返された場合、既にクローズされています.<BR>
     *                 [false]が返された場合、コネクション中です.
     */
    public boolean isClosed() ;
    
}

