/*
 * @(#)JRcClient.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.client ;

import java.net.InetAddress;

import com.JRcServer.JRcProtocolDef;
import com.JRcServer.commons.def.BaseDef;
import com.JRcServer.commons.exception.AccessException;
import com.JRcServer.commons.exception.InputException;
import com.JRcServer.commons.io.IOCom;

/**
 * JRcClientドライバ.
 * <BR><BR>
 * JRcServerに接続するためのドライバ.<BR>
 * <BR>
 * 基本的にJRcServerに接続するには、JRcClientDriver経由で
 * アクセスする必要があります.<BR>
 * まず初めに、JRcClientDriverを使い、※アプリケーション名と<BR>
 * 接続先のIPアドレス+ポート番号(デフォルトポート番号は32109を利用)<BR>
 * を指定します。<BR>
 * <BR>
 * 　　※アプリケーション名は接続先のセッション情報を識別するときの名前なので、<BR>
 * 　　　ユニーク名である必要があります.<BR>
 * <PRE>
 * 
 * JRcConnection conn = JRcClientDriver.getConnection( "hoge","127.0.0.1" ) ;
 *
 * </PRE>
 * 上記のようにしてJRcServerに接続します.<BR>
 * この処理が成功することで、JRcServerとコネクションを確立することができます.<BR>
 * また、前回に接続している場合にセッションを利用している場合は、セッションも確立されます.<BR>
 * そのとき、前回の接続先＋アプリケーション名でセッションを再定義されます.<BR>
 * <BR>
 * 次に、上記の[JRcConnection]を利用して、JRcServerに定義されているサービス名(コマンド名)を<BR>
 * 呼び出す例を示します.<BR>
 * <PRE>
 * 
 * BufferedReader r = null ;
 * String s = null ;
 * JRcResult result = null ;
 * 
 * // サービス名 list を呼び出す.
 * conn.createService( "list" ) ;
 * result = conn.execution() ;
 * 
 * r = new BufferedReader(
 * 　new InputStreamReader(
 * 　　result.getInputStream(),result.getCharset()
 * 　)
 * ) ;
 * 
 * for( ;; ) {
 * 　if( ( s = r.readLine() ) == null ) { break ; }
 * 　System.out.println( s ) ;
 * }
 * r.close() ;
 * r = null ;
 * result = null ;
 * 
 * // サービス名 help [hoge] を呼び出す.
 * conn.createService( "help" ) ;
 * conn.setParameterToString( "command","hoge" ) ;
 * result = conn.execution() ;
 *
 * r = new BufferedReader(
 * 　new InputStreamReader(
 * 　　result.getInputStream(),result.getCharset()
 * 　)
 * ) ;
 * 
 * for( ;; ) {
 * 　if( ( s = r.readLine() ) == null ) { break ; }
 * 　System.out.println( s ) ;
 * }
 * r.close() ;
 * r = null ;
 * result = null ;
 * 
 * </PRE>
 * 予約語である[list],[help]を用いた例です.<BR>
 * 予約語[list]は、利用可能なサービス名一覧をリストアップする処理で、<BR>
 * 予約語[help]は、利用可能なサービスの使い方を表示するための機能です.<BR>
 * 予約語[help]の場合には、パラメータ名を[command]として、その内容に調べたい<BR>
 * サービス名(ここでは[hoge]を指定している)を設定することで、ヘルプ情報を取得する<BR>
 * ことができます.<BR>
 * <BR>
 * そして、executionでサービスを呼び出した結果は、[JRcResult]を使って、取得できます.<BR>
 * またここでは、予約語を例題に説明していますが、登録されているサービスも同様の手順で<BR>
 * 呼び出すことができます.
 * <BR><BR>
 * 上記のような呼び方が、通常のプログラム内で呼び出す方法ですが、もう１つの実行方法として、<BR>
 * 以下のような方法もあります.
 * <BR>
 * <PRE>
 * 
 * result = conn.execution( "help -command hoge" ) ;
 * r = new BufferedReader(
 * 　new InputStreamReader(
 * 　　result.getInputStream(),result.getCharset()
 * 　)
 * ) ;
 * 
 * for( ;; ) {
 * 　if( ( s = r.readLine() ) == null ) { break ; }
 * 　System.out.println( s ) ;
 * }
 * r.close() ;
 * r = null ;
 * result = null ;
 * 
 * </PRE>
 * このように、コマンドとして呼び出すことができます.<BR>
 * 詳しくは、[com.JRcServer.client.JRcConnection]オブジェクトの[execution]メソッドを<BR>
 * 参照してください.
 * 
 * 
 * @version 2006/09/10
 * @author  masahito suzuki
 * @since   JRcClient 1.00
 */
public class JRcClientDriver {
    
    static {
        
        String string = null ;
        
        string = 
            BaseDef.USER_HOME +
            BaseDef.FILE_SEPARATOR +
            ".jrccl" ;
        
        JRC_CLIENT_DIR = string ;
        
        
        CACHE_DIR = string +
            BaseDef.FILE_SEPARATOR +
            ".cache" ;
        
        WORK_DIR = string +
            BaseDef.FILE_SEPARATOR +
            ".work" ;
        
    }
    
    /**
     * JRcClientディレクトリ名.
     */
    public static final String JRC_CLIENT_DIR ;
    
    /**
     * JRcClientキャッシュディレクトリ名.
     */
    public static final String CACHE_DIR ;
    
    /**
     * JRcClientワークディレクトリ名.
     */
    public static final String WORK_DIR ;
    
    /**
     * JRcClientバージョン.
     */
    public static final String VERSION = "1.00" ;
    
    /**
     * ドライバ同期.
     */
    public static final Object SYNC = new Object() ;
    
    /**
     * コンストラクタ.
     */
    private JRcClientDriver() {
        
    }
    
    /**
     * コネクション処理.
     * <BR><BR>
     * JRcServerにコネクションします.<BR>
     * このメソッドでは、JRcServerのデフォルトポート番号(32109)でアクセスします.
     * <BR><BR>
     * @param applicationName 対象のアプリケーション名を設定します.<BR>
     *                        この内容は、セッション条件に影響します.<BR>
     *                        そのため、なるべくUniqueな条件で設定してください.
     * @param addr コネクション先のアドレスを設定します.
     * @return JRcConnection コネクションオブジェクトが返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final JRcConnection getConnection( String applicationName,String addr )
        throws InputException,AccessException
    {
        JRcConnection ret = null ;
        
        try {
            ret = JRcClientDriver.getConnection(
                false,applicationName,InetAddress.getByName( addr ),
                JRcProtocolDef.DEFAULT_JRCSERVER_PORT,
                JRcProtocolDef.DEFAULT_JRCSERVER_TIMEOUT ) ;
                
        } catch( InputException in ) {
            throw in ;
        } catch( AccessException ae ) {
            throw ae ;
        } catch( Exception e ) {
            throw new AccessException( e ) ;
        }
        
        return ret ;
    }
    
    /**
     * コネクション処理.
     * <BR><BR>
     * JRcServerにコネクションします.
     * <BR><BR>
     * @param applicationName 対象のアプリケーション名を設定します.<BR>
     *                        この内容は、セッション条件に影響します.<BR>
     *                        そのため、なるべくUniqueな条件で設定してください.
     * @param addr コネクション先のアドレスを設定します.
     * @param port コネクション先のポート番号を設定します.
     * @return JRcConnection コネクションオブジェクトが返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final JRcConnection getConnection( String applicationName,String addr,int port )
        throws InputException,AccessException
    {
        JRcConnection ret = null ;
        
        try {
            ret = JRcClientDriver.getConnection(
                false,applicationName,InetAddress.getByName( addr ),port,
                JRcProtocolDef.DEFAULT_JRCSERVER_TIMEOUT ) ;
                
        } catch( InputException in ) {
            throw in ;
        } catch( AccessException ae ) {
            throw ae ;
        } catch( Exception e ) {
            throw new AccessException( e ) ;
        }
        
        return ret ;
    }
    
    /**
     * コネクション処理.
     * <BR><BR>
     * JRcServerにコネクションします.
     * <BR><BR>
     * @param applicationName 対象のアプリケーション名を設定します.<BR>
     *                        この内容は、セッション条件に影響します.<BR>
     *                        そのため、Uniqueな条件で設定してください.
     * @param addr コネクション先のアドレスを設定します.
     * @param port コネクション先のポート番号を設定します.
     * @param timeout コネクションタイムアウトを設定します.<BR>
     *                [0]以下を設定した場合、無限待機となります.
     * @return JRcConnection コネクションオブジェクトが返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final JRcConnection getConnection(
        String applicationName,String addr,int port,int timeout )
        throws InputException,AccessException
    {
        JRcConnection ret = null ;
        
        try {
            ret = JRcClientDriver.getConnection(
                false,applicationName,InetAddress.getByName( addr ),port,timeout ) ;
                
        } catch( InputException in ) {
            throw in ;
        } catch( AccessException ae ) {
            throw ae ;
        } catch( Exception e ) {
            throw new AccessException( e ) ;
        }
        
        return ret ;
    }
    
    /**
     * コネクション処理.
     * <BR><BR>
     * JRcServerにコネクションします.<BR>
     * このメソッドでは、JRcServerのデフォルトポート番号(32109)でアクセスします.
     * <BR><BR>
     * @param applicationName 対象のアプリケーション名を設定します.<BR>
     *                        この内容は、セッション条件に影響します.<BR>
     *                        そのため、なるべくUniqueな条件で設定してください.
     * @param addr コネクション先のアドレスを設定します.
     * @return JRcConnection コネクションオブジェクトが返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final JRcConnection getConnection( String applicationName,InetAddress addr )
        throws InputException,AccessException
    {
        return JRcClientDriver.getConnection(
            false,applicationName,addr,
            JRcProtocolDef.DEFAULT_JRCSERVER_PORT,
            JRcProtocolDef.DEFAULT_JRCSERVER_TIMEOUT ) ;
    }
    
    /**
     * コネクション処理.
     * <BR><BR>
     * JRcServerにコネクションします.
     * <BR><BR>
     * @param applicationName 対象のアプリケーション名を設定します.<BR>
     *                        この内容は、セッション条件に影響します.<BR>
     *                        そのため、なるべくUniqueな条件で設定してください.
     * @param addr コネクション先のアドレスを設定します.
     * @param port コネクション先のポート番号を設定します.
     * @return JRcConnection コネクションオブジェクトが返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final JRcConnection getConnection( String applicationName,InetAddress addr,int port )
        throws InputException,AccessException
    {
        return JRcClientDriver.getConnection(
            false,applicationName,addr,port,
            JRcProtocolDef.DEFAULT_JRCSERVER_TIMEOUT ) ;
    }
    
    /**
     * コネクション処理.
     * <BR><BR>
     * JRcServerにコネクションします.
     * <BR><BR>
     * @param applicationName 対象のアプリケーション名を設定します.<BR>
     *                        この内容は、セッション条件に影響します.<BR>
     *                        そのため、Uniqueな条件で設定してください.
     * @param addr コネクション先のアドレスを設定します.
     * @param port コネクション先のポート番号を設定します.
     * @param timeout コネクションタイムアウトを設定します.<BR>
     *                [0]以下を設定した場合、無限待機となります.
     * @return JRcConnection コネクションオブジェクトが返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final JRcConnection getConnection(
        String applicationName,InetAddress addr,int port,int timeout )
        throws InputException,AccessException
    {
        return JRcClientDriver.getConnection(
            false,applicationName,addr,port,timeout ) ;
    }
    
    /**
     * コネクション処理.
     * <BR><BR>
     * JRcServerにコネクションします.<BR>
     * このメソッドでは、JRcServerのデフォルトポート番号(32109)でアクセスします.
     * <BR><BR>
     * @param cacheMode キャッシュモードを設定します.<BR>
     *                  [true]を設定した場合、ファイルキャッシュをONにします.<BR>
     *                  [false]を設定した場合、メモリキャッシュをONにします.
     * @param applicationName 対象のアプリケーション名を設定します.<BR>
     *                        この内容は、セッション条件に影響します.<BR>
     *                        そのため、なるべくUniqueな条件で設定してください.
     * @param addr コネクション先のアドレスを設定します.
     * @return JRcConnection コネクションオブジェクトが返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final JRcConnection getConnection( boolean cacheMode,String applicationName,String addr )
        throws InputException,AccessException
    {
        JRcConnection ret = null ;
        
        try {
            ret = JRcClientDriver.getConnection(
                cacheMode,applicationName,InetAddress.getByName( addr ),
                JRcProtocolDef.DEFAULT_JRCSERVER_PORT,
                JRcProtocolDef.DEFAULT_JRCSERVER_TIMEOUT ) ;
                
        } catch( InputException in ) {
            throw in ;
        } catch( AccessException ae ) {
            throw ae ;
        } catch( Exception e ) {
            throw new AccessException( e ) ;
        }
        
        return ret ;
    }
    
    /**
     * コネクション処理.
     * <BR><BR>
     * JRcServerにコネクションします.
     * <BR><BR>
     * @param cacheMode キャッシュモードを設定します.<BR>
     *                  [true]を設定した場合、ファイルキャッシュをONにします.<BR>
     *                  [false]を設定した場合、メモリキャッシュをONにします.
     * @param applicationName 対象のアプリケーション名を設定します.<BR>
     *                        この内容は、セッション条件に影響します.<BR>
     *                        そのため、なるべくUniqueな条件で設定してください.
     * @param addr コネクション先のアドレスを設定します.
     * @param port コネクション先のポート番号を設定します.
     * @return JRcConnection コネクションオブジェクトが返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final JRcConnection getConnection( boolean cacheMode,String applicationName,String addr,int port )
        throws InputException,AccessException
    {
        JRcConnection ret = null ;
        
        try {
            ret = JRcClientDriver.getConnection(
                cacheMode,applicationName,InetAddress.getByName( addr ),port,
                JRcProtocolDef.DEFAULT_JRCSERVER_TIMEOUT ) ;
                
        } catch( InputException in ) {
            throw in ;
        } catch( AccessException ae ) {
            throw ae ;
        } catch( Exception e ) {
            throw new AccessException( e ) ;
        }
        
        return ret ;
    }
    
    /**
     * コネクション処理.
     * <BR><BR>
     * JRcServerにコネクションします.
     * <BR><BR>
     * @param cacheMode キャッシュモードを設定します.<BR>
     *                  [true]を設定した場合、ファイルキャッシュをONにします.<BR>
     *                  [false]を設定した場合、メモリキャッシュをONにします.
     * @param applicationName 対象のアプリケーション名を設定します.<BR>
     *                        この内容は、セッション条件に影響します.<BR>
     *                        そのため、Uniqueな条件で設定してください.
     * @param addr コネクション先のアドレスを設定します.
     * @param port コネクション先のポート番号を設定します.
     * @param timeout コネクションタイムアウトを設定します.<BR>
     *                [0]以下を設定した場合、無限待機となります.
     * @return JRcConnection コネクションオブジェクトが返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final JRcConnection getConnection(
        boolean cacheMode,String applicationName,String addr,int port,int timeout )
        throws InputException,AccessException
    {
        JRcConnection ret = null ;
        
        try {
            ret = JRcClientDriver.getConnection(
                cacheMode,applicationName,InetAddress.getByName( addr ),port,timeout ) ;
                
        } catch( InputException in ) {
            throw in ;
        } catch( AccessException ae ) {
            throw ae ;
        } catch( Exception e ) {
            throw new AccessException( e ) ;
        }
        
        return ret ;
    }
    
    /**
     * コネクション処理.
     * <BR><BR>
     * JRcServerにコネクションします.<BR>
     * このメソッドでは、JRcServerのデフォルトポート番号(32109)でアクセスします.
     * <BR><BR>
     * @param cacheMode キャッシュモードを設定します.<BR>
     *                  [true]を設定した場合、ファイルキャッシュをONにします.<BR>
     *                  [false]を設定した場合、メモリキャッシュをONにします.
     * @param applicationName 対象のアプリケーション名を設定します.<BR>
     *                        この内容は、セッション条件に影響します.<BR>
     *                        そのため、なるべくUniqueな条件で設定してください.
     * @param addr コネクション先のアドレスを設定します.
     * @return JRcConnection コネクションオブジェクトが返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final JRcConnection getConnection( boolean cacheMode,String applicationName,InetAddress addr )
        throws InputException,AccessException
    {
        return JRcClientDriver.getConnection(
            cacheMode,applicationName,addr,
            JRcProtocolDef.DEFAULT_JRCSERVER_PORT,
            JRcProtocolDef.DEFAULT_JRCSERVER_TIMEOUT ) ;
    }
    
    /**
     * コネクション処理.
     * <BR><BR>
     * JRcServerにコネクションします.
     * <BR><BR>
     * @param cacheMode キャッシュモードを設定します.<BR>
     *                  [true]を設定した場合、ファイルキャッシュをONにします.<BR>
     *                  [false]を設定した場合、メモリキャッシュをONにします.
     * @param applicationName 対象のアプリケーション名を設定します.<BR>
     *                        この内容は、セッション条件に影響します.<BR>
     *                        そのため、なるべくUniqueな条件で設定してください.
     * @param addr コネクション先のアドレスを設定します.
     * @param port コネクション先のポート番号を設定します.
     * @return JRcConnection コネクションオブジェクトが返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final JRcConnection getConnection( boolean cacheMode,String applicationName,InetAddress addr,int port )
        throws InputException,AccessException
    {
        return JRcClientDriver.getConnection(
            cacheMode,applicationName,addr,port,
            JRcProtocolDef.DEFAULT_JRCSERVER_TIMEOUT ) ;
    }
    
    /**
     * コネクション処理.
     * <BR><BR>
     * JRcServerにコネクションします.
     * <BR><BR>
     * @param cacheMode キャッシュモードを設定します.<BR>
     *                  [true]を設定した場合、ファイルキャッシュをONにします.<BR>
     *                  [false]を設定した場合、メモリキャッシュをONにします.
     * @param applicationName 対象のアプリケーション名を設定します.<BR>
     *                        この内容は、セッション条件に影響します.<BR>
     *                        そのため、Uniqueな条件で設定してください.
     * @param addr コネクション先のアドレスを設定します.
     * @param port コネクション先のポート番号を設定します.
     * @param timeout コネクションタイムアウトを設定します.<BR>
     *                [0]以下を設定した場合、無限待機となります.
     * @return JRcConnection コネクションオブジェクトが返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final JRcConnection getConnection(
        boolean cacheMode,String applicationName,InetAddress addr,int port,int timeout )
        throws InputException,AccessException
    {
        
        // 作業ディレクトリが存在しない場合、生成.
        JRcClientDriver.createDirectory() ;
        
        // コネクション情報を生成.
        return new JRcConnectionImple(
            cacheMode,applicationName,addr,port,timeout ) ;
        
    }
    
    
    /**
     * 基本ディレクトリ名が存在しない場合生成.
     */
    private static final void createDirectory()
        throws InputException,AccessException {
        
        if( IOCom.isDirExists( CACHE_DIR ) == false ) {
            IOCom.mkdirs( CACHE_DIR ) ;
        }
        
        if( IOCom.isDirExists( WORK_DIR ) == false ) {
            IOCom.mkdirs( WORK_DIR ) ;
        }
        
    }
    
}

