/*
 * @(#)JRcClientConsole.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package com.JRcServer.client ;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.FileOutputStream;
import java.io.InputStreamReader;
import java.io.PrintStream;
import java.util.ArrayList;

import com.JRcServer.JRCommandException;
import com.JRcServer.JRcErrorCodeDef;
import com.JRcServer.JRcProtocolDef;
import com.JRcServer.commons.conv.Base64;
import com.JRcServer.commons.def.BaseDef;
import com.JRcServer.commons.io.IOCom;
import com.JRcServer.commons.resource.BinResourceInputStream;
import com.JRcServer.commons.util.UtilCom;

/**
 * JRcClientConsole.
 * <BR><BR>
 * JRcClientコンソール操作用オブジェクト.
 *  
 * @version 2006/09/11
 * @author  masahito suzuki
 * @since   JRcClient 1.00
 */
public class JRcClientConsole {
    
    /**
     * デフォルトタイムアウト値.
     */
    private static final int DEFAULT_JRCSERVER_TIMEOUT = 5000 ;
    
    /**
     * アプリケーション名区切り条件.
     */
    private static final String PARSE_APPNAME = "@" ;
    
    /**
     * ポート番号設定ヘッダ.
     */
    private static final String HEAD_PORT = ":" ;
    
    /**
     * タイムアウト設定ヘッダ.
     */
    private static final String HEAD_TIMEOUT = "/" ;
    
    /**
     * 条件オプション : 出力結果をファイルに保存.
     */
    private static final String OUTPUT_FILE = ">>" ;
    
    /**
     * アプリケーション名.
     */
    private String applicationName = null ;
    
    /**
     * main.
     * <BR><BR>
     * @param args Java起動引数が設定されます.
     * @exception Exception 例外.
     */
    public static final void main( String[] args )
        throws Exception {
        
        int ret ;
        
        //new InitPackage().init() ;
        
        ret = ( new JRcClientConsole() ).console( args ) ;
        
        //new InitPackage().destroy() ;
        
        System.exit( ret ) ;
    }
    
    /**
     * コンソール実行処理.
     * <BR><BR>
     * コンソール実行処理を行います.
     * <BR>
     * @param args Java起動引数が設定されます.
     * @return int 処理結果が返されます.
     * @exception Exception 例外.
     */
    private final int console( String[] args )
        throws Exception {
        
        int ret = 0 ;
        JRcConnection conn = null ;
        BufferedReader input = null ;
        
        try {
            
            // オプション指定が存在しない場合.
            if( args == null || args.length <= 0 ) {
                
                System.err.println( this.getHelp() ) ;
                ret = -1 ;
                
            }
            // オプションが存在する場合.
            else {
                
                // 起動処理.
                conn = this.argsAnalysis( args ) ;
                
                // コネクションが確立された場合.
                if( conn != null ) {
                    
                    // キー入力情報を取得.
                    input = new BufferedReader(
                        new InputStreamReader( System.in )
                    ) ;
                    
                    // 処理ループ.
                    for( ;; ) {
                        
                        try {
                            
                            // コンソール入力処理.
                            if( this.waitConsole( conn,input )  == false ) {
                                
                                // 処理終了の場合.
                                if( conn != null ) {
                                    conn.close() ;
                                }
                                if( input != null ) {
                                    try {
                                        input.close() ;
                                    } catch( Exception e ) {
                                    }
                                }
                                
                                conn = null ;
                                input = null ;
                                
                                break ;
                                
                            }
                            
                        } catch( JRCommandException je ) {
                            
                            System.err.println( je.getMessage() ) ;
                            
                        }
                    }
                    
                }
                
            }
            
        } catch( JRCommandException je ) {
            
            ret = -2 ;
            System.err.println( je.getMessage() ) ;
            
        } catch( Exception e ) {
            
            ret = -2 ;
            System.out.println( e.getMessage() ) ;
        }
        
        return ret ;
    }
    
    /**
     * コンソール入力待ち処理.
     * <BR><BR>
     * コンソール入力待ち処理を実施します.
     * <BR>
     * @param conn 対象のコネクションオブジェクトを設定します.
     * @param reader 対象の入力オブジェクトを設定します.
     * @return boolean 処理クローズ結果が返されます.
     * @exception JRCommandException コマンド例外.
     */
    private final boolean waitConsole( JRcConnection conn,BufferedReader reader )
        throws JRCommandException {
        
        String command = null ;
        JRcResult result = null ;
        
        System.out.print( applicationName ) ;
        System.out.print( ">" ) ;
        
        try {
            
            // コンソール入力待ち.
            command = reader.readLine() ;
            
            if( command == null || ( command = command.trim() ).length() <= 0 ) {
                return true ;
            }
            
            // 終了コマンドを検知.
            if( JRcClientConsole.isExitCommand( command ) == true ) {
                return false ;
            }
            
            // コマンド内容を実行.
            result = this.executionService( conn,command ) ;
            
            // 結果情報を取得.
            this.outputResult( command,result ) ;
            result.close() ;
            result = null ;
            
        } catch( JRCommandException je ) {
            throw je ;
        } catch( Exception e ) {
            throw new JRCommandException(
                JRcErrorCodeDef.ERROR_ETC,
                "コンソール処理でエラーが発生しました[message:" +
                e.getMessage() + "]"
            ) ;
        } finally {
            if( result != null ) {
                result.close() ;
            }
            result = null ;
            command = null ;
        }
        
        return true ;
        
    }
    
    /**
     * 起動引数を解析してJRcClientを起動.
     * <BR><BR>
     * 起動引数を解析してJRcClientを起動します.
     * <BR>
     * @param args 対象の起動引数を設定します.
     * @param length 起動引数長を設定します.
     * @return JRcConnection コネクションオブジェクトが返されます.
     * @exception JRCommandException コマンド例外.
     */
    private final JRcConnection argsAnalysis( String[] args )
        throws JRCommandException {
        
        int i ;
        int len ;
        int pnt ;
        int tpnt ;
        boolean scFlg = false ;
        
        String param = null ;
        String addr = null ;
        int port = -1 ;
        String appName = null ;
        int timeout = -1 ;
        String cmd = null ;
        StringBuffer buf = null ;
        
        JRcConnection ret = null ;
        
        try {
            
            len = args.length ;
            
            // パラメータを解析.
            param = args[ 0 ].trim() ;
            
            // アプリケーション名.
            if(
                ( pnt = param.indexOf( PARSE_APPNAME ) ) == -1 ||
                ( appName = param.substring( 0,pnt ).trim().toLowerCase() ).length() <= 0
            ) {
                if( param.length() == 0 ) {
                    throw new JRCommandException(
                        JRcErrorCodeDef.ERROR_ETC,
                        "アプリケーション名は指定されていないか存在しません"
                    ) ;
                }
                else {
                    throw new JRCommandException(
                        JRcErrorCodeDef.ERROR_ETC,
                        "IPアドレスは指定されていないか存在しません"
                    ) ;
                }
            }
            
            pnt += 1 ;
            
            // IPアドレス.
            if(
                ( tpnt = param.indexOf( HEAD_PORT,pnt ) ) != -1 &&
                ( addr = param.substring( pnt,tpnt ).trim() ).length() > 0
            ) {
                // 次の条件にポート番号が存在する.
                pnt = tpnt + 1  ;
                
                // タイムアウト条件が存在するかチェック.
                if( ( tpnt = param.indexOf( HEAD_TIMEOUT,pnt ) ) != -1 ) {
                    port = Integer.parseInt( param.substring( pnt,tpnt ).trim() ) ;
                    timeout = Integer.parseInt( param.substring( tpnt+1 ).trim() ) ;
                }
                else {
                    port = Integer.parseInt( param.substring( pnt ).trim() ) ;
                }
                
            }
            else if( 
                ( tpnt = param.indexOf( HEAD_TIMEOUT,pnt ) ) != -1 &&
                ( addr = param.substring( pnt,tpnt ).trim() ).length() > 0
            ) {
                timeout = Integer.parseInt( param.substring( tpnt+1 ).trim() ) ;
            }
            else {
                
                if( ( addr = param.substring( pnt ).trim() ).length() <= 0 ) {
                    throw new JRCommandException(
                        JRcErrorCodeDef.ERROR_ETC,
                        "IPアドレスは指定されていないか存在しません"
                    ) ;
                }
                
            }
            
            // 入力引数が複数存在する.
            if( ( len = args.length ) > 1 ) {
                
                // 単発コマンドとして設定.
                buf = new StringBuffer( 8192 ) ;
                for( i = 1 ; i < len ; i ++ ) {
                    if( i != 1 ) {
                        buf.append( " " ) ;
                    }
                    if(
                        args[ i ] != null &&
                        ( args[ i ] = args[ i ].trim() ).length() > 0
                    ) {
                        buf.append( args[ i ] ) ;
                    }
                }
                cmd = buf.toString() ;
                buf = null ;
                scFlg = true ;
                
            }
            
            // 必須条件が設定されていない場合.
            if(
                addr == null || addr.length() <= 0 ||
                appName == null || appName.length() <= 0
            ) {
                if( addr == null || addr.length() <= 0 ) {
                    throw new JRCommandException(
                        JRcErrorCodeDef.ERROR_ETC,
                        "IPアドレスは指定されていないか存在しません"
                    ) ;
                }
                else {
                    throw new JRCommandException(
                        JRcErrorCodeDef.ERROR_ETC,
                        "アプリケーション名は指定されていないか存在しません"
                    ) ;
                }
            }
            
            // 単発起動フラグがONであるが、コマンド情報が不正な場合.
            if( scFlg == true && ( cmd == null || ( cmd = cmd.trim() ).length() <= 0 ) ){
                throw new JRCommandException(
                    JRcErrorCodeDef.ERROR_ETC,
                    "単発コマンド起動に対して条件が指定されていません"
                ) ;
            }
            // デフォルト指定条件が設定されていない場合.
            if( port < 0 || port > 65535 ) {
                port = JRcProtocolDef.DEFAULT_JRCSERVER_PORT ;
            }
            
            // 受信タイムアウト値が設定されていない場合.
            if( timeout < 0 ) {
                timeout = DEFAULT_JRCSERVER_TIMEOUT ;
            }
            else {
                // 入力は秒単位.
                timeout = timeout * 1000 ;
            }
            
            // コネクションを生成.
            ret = JRcClientDriver.getConnection( false,appName,addr,port,timeout ) ;
            
            // 単発起動条件の場合.
            if( scFlg == true ) {
                
                JRcResult result = null ;
                
                // 単発処理実行.
                result = this.executionService( ret,cmd ) ;
                this.outputResult( cmd,result ) ;
                
                // コネクションをクローズ.
                ret.close() ;
                
                result = null ;
                ret = null ;
                
            }
            
            this.applicationName = appName ;
            
        } catch( JRCommandException je ) {
            throw je ;
        } catch( Exception e ) {
            throw new JRCommandException(
                JRcErrorCodeDef.ERROR_ETC,
                "コネクション処理でエラーが発生しました"
            ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象条件のサービス名を起動.
     * <BR><BR>
     * 起動条件のサービス名を起動します.
     * <BR>
     * @param conn 対象のコネクションオブジェクトを設定します.
     * @param command サービス起動コマンドを設定します.
     * @return JRcResult 処理結果の戻り情報が返されます.
     * @exception JRCommandException コマンド例外.
     */
    private final JRcResult executionService( JRcConnection conn,String command )
        throws JRCommandException {
        
        int pnt ;
        
        JRcResult ret = null ;
        
        if( ( pnt = command.indexOf( OUTPUT_FILE ) ) != -1 ) {
            command = command.substring( 0,pnt ).trim() ;
        }
        
        try {
            ret = conn.execution( command ) ;
        } catch( JRCommandException je ) {
            throw je ;
        } catch( Exception e ) {
            throw new JRCommandException(
                JRcErrorCodeDef.ERROR_ETC,
                "コネクション処理でエラーが発生しました"
            ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * 処理結果の内容を指定条件に出力.
     * <BR><BR>
     * 処理結果の内容を指定条件に出力します.
     * <BR>
     * @param command 対象のサービスコマンドを設定します.
     * @param result 対象の条件を設定します.
     * @exception JRCommandException コマンド例外.
     */
    private final void outputResult( String command,JRcResult result )
        throws JRCommandException {
        
        int data ;
        
        String file = null ;
        String dir = null ;
        
        BufferedReader br = null ;
        BufferedInputStream bi = null ;
        PrintStream ps = null;
        BufferedOutputStream bo = null ;
        
        try {
            
            // 出力先のファイル名を取得.
            file = JRcClientConsole.getOutputFileName( command ) ;
            
            // 結果コードを出力する内容は、ファイル名.
            if( file != null ) {
                
                // ディレクトリ名を取得.
                dir = JRcClientConsole.getDirectory( file ) ;
                
                // ファイルのディレクトリ名が存在しない場合.
                if( IOCom.isDirExists( dir ) == false ) {
                    IOCom.mkdirs( dir ) ;
                }
                
                // 出力データ件数が0件の場合.
                if( result.getSize() <= 0 ) {
                    
                    // 空のファイルを作成.
                    IOCom.setFileByString( file,BaseDef.ENTER_SEPARATOR ) ;
                    System.out.println( " " ) ;
                    return ;
                    
                }
                // 出力データ件数が0件以上の場合.
                else {
                    
                    // 出力条件がバイナリの場合.
                    if(
                        result.getCharset() == null ||
                        BaseDef.ISO_8859_1.equals( result.getCharset() ) == true
                    ){
                        bi = new BufferedInputStream(
                            result.getInputStream()
                        ) ;
                        bo = new BufferedOutputStream(
                            new FileOutputStream( file )
                        ) ;
                    }
                    // 出力条件が文字列の場合.
                    else {
                        
                        br = new BufferedReader(
                            new InputStreamReader(
                                result.getInputStream(),
                                result.getCharset()
                            )
                        ) ;
                        
                        ps = new PrintStream(
                            new BufferedOutputStream(
                                new FileOutputStream( file )
                            )
                        ) ;
                        
                    }
                    
                }
                
            }
            // 結果コードを出力する内容は、コンソール先.
            else {
                
                // 出力データ件数が0件の場合.
                if( result.getSize() <= 0 ) {
                    
                    // 0内容を出力.
                    System.out.println( " " ) ;
                    return ;
                    
                }
                // 出力データ件数が0件以上の場合.
                else {
                    
                    // 出力条件がバイナリの場合.
                    if(
                        result.getCharset() == null ||
                        BaseDef.ISO_8859_1.equals( result.getCharset() ) == true
                    ){
                        // Base64で出力.
                        System.out.println(
                            Base64.encode(
                                (
                                    ( BinResourceInputStream )result.getInputStream()
                                ).getBinResource()
                            )
                        ) ;
                        return ;
                    }
                    // 出力条件が文字列の場合.
                    else {
                        
                        br = new BufferedReader(
                            new InputStreamReader(
                                result.getInputStream(),
                                result.getCharset()
                            )
                        ) ;
                        
                        ps = System.out ;
                        
                    }
                    
                }
                
            }
            
            // バイナリで出力.
            if( bi != null && bo != null ) {
                
                while( ( data = bi.read() ) >= 0 ){
                    bo.write( data ) ;
                }
                
                bi.close() ;
                bo.flush() ;
                bo.close() ;
                
                bi = null ;
                bo = null ;
                
                System.out.println( " " ) ;
                
            }
            // 文字列で出力.
            else {
                
                String str = null ;
                while( ( str = br.readLine() ) != null ) {
                    ps.println( str ) ;
                }
                
                str = null ;
                br.close() ;
                ps.flush() ;
                
                if( ps != System.out ) {
                    ps.close() ;
                }
                
                br = null ;
                ps = null ;
                
                System.out.println( " " ) ;
                
            }
            
            
        } catch( Exception e ) {
            throw new JRCommandException(
                JRcErrorCodeDef.ERROR_ETC,
                "受信情報出力処理でエラーが発生しました[message:" +
                e.getMessage() + "]"
            ) ;
        } finally{
            if( br != null ) {
                try {
                    br.close() ;
                } catch( Exception ee ) {
                }
            }
            if( bi != null ) {
                try {
                    bi.close() ;
                } catch( Exception ee ) {
                }
            }
            if( ps != null ) {
                if( ps != System.out ) {
                    try {
                        ps.close() ;
                    } catch( Exception ee ) {
                    }
                }
            }
            if( bo != null ) {
                try {
                    bo.close() ;
                } catch( Exception ee ) {
                }
            }
            
            file = null ;
            dir = null ;
            br = null ;
            bi = null ;
            ps = null;
            bo = null ;
            
        }
        
    }
    
    
    /**
     * コマンドヘルプ表示.
     * <BR><BR>
     * コマンドヘルプ内容を表示します.
     * <BR>
     * @return String コマンドヘルプ内容が返されます.
     */
    private final String getHelp() {
        
        return new StringBuffer( 4096 ).
            append( "JRcClient(" ).
            append( JRcClientDriver.VERSION ).
            append( ")" ).append( BaseDef.ENTER_SEPARATOR ).
            append( "                                                            " ).append( BaseDef.ENTER_SEPARATOR ).
            append( " jrccl [ap]@[ip][:[po]/[tm]] [cmd]                          " ).append( BaseDef.ENTER_SEPARATOR ).
            append( "                                                            " ).append( BaseDef.ENTER_SEPARATOR ).
            append( "    jrccl : JRcClientコンソールコマンド名.                  " ).append( BaseDef.ENTER_SEPARATOR ).
            append( "     ap   : アプリケーション名。この名前はアプリケーション  " ).append( BaseDef.ENTER_SEPARATOR ).
            append( "            固有名を設定する必要があります(なるべくならば). " ).append( BaseDef.ENTER_SEPARATOR ).
            append( "     ip   : コネクション先アドレス.                         " ).append( BaseDef.ENTER_SEPARATOR ).
            append( "     po   : コネクション先ポート番号.                       " ).append( BaseDef.ENTER_SEPARATOR ).
            append( "            設定しない場合はデフォルトポート番号(32109).    " ).append( BaseDef.ENTER_SEPARATOR ).
            append( "     tm   : タイムアウト値を設定します.                     " ).append( BaseDef.ENTER_SEPARATOR ).
            append( "            [0]を設定した場合は、無限待ちとなります         " ).append( BaseDef.ENTER_SEPARATOR ).
            append( "            設定しない場合は、デフォルト値(300)             " ).append( BaseDef.ENTER_SEPARATOR ).
            append( "     cmd  : 単発で起動するコマンド内容を設定します          " ).append( BaseDef.ENTER_SEPARATOR ).
            append( "                                                            " ).append( BaseDef.ENTER_SEPARATOR ).
            append( " また、起動引数設定後[cmd]で単発のサービス呼び出しが、      " ).append( BaseDef.ENTER_SEPARATOR ).
            append( " できます.                                                  " ).append( BaseDef.ENTER_SEPARATOR ).
            append( "        jrccl [ap]@[ip][:[po]/[tm]] help -command hoge      " ).append( BaseDef.ENTER_SEPARATOR ).
            append( "        jrccl [ap]@[ip][:[po]/[tm]] ?help&command=hoge      " ).append( BaseDef.ENTER_SEPARATOR ).
            append( "                                                            " ).append( BaseDef.ENTER_SEPARATOR ).
            append( " ここでは、[help]コマンドを利用して、[hoge]サービスの       " ).append( BaseDef.ENTER_SEPARATOR ).
            append( " ヘルプ内容を表示します.                                    " ).append( BaseDef.ENTER_SEPARATOR ).
            append( " また、単発起動でない場合は、コネクション処理が終わった後に " ).append( BaseDef.ENTER_SEPARATOR ).
            append( " [>]が表示されるので、コマンドを入力します.                 " ).append( BaseDef.ENTER_SEPARATOR ).
            toString() ;
        
    }
    
    /**
     * 対象内容に書き込みファイル名が存在するかチェック.
     */
    private static final String getOutputFileName( String command ) {
        
        int pnt ;
        
        ArrayList lst = null ;
        String ret = null ;
        
        // コマンド名に、ファイル出力条件が設定されている.
        if( ( pnt = command.indexOf( OUTPUT_FILE ) ) != -1 ) {
            
            try {
                
                // 定義では、ファイル出力条件の後にファイル名が
                // 設定されているため、その条件を取得.
                command = command.substring(
                    pnt + OUTPUT_FILE.length()
                ).trim() ;
                
                lst = UtilCom.pauseString( true,false,command," \t&" ) ;
                if( lst != null && lst.size() > 0 ) {
                    ret = ( String )lst.get( 0 ) ;
                }
                
                // 読み込みファイルがworkディレクトリ内の場合.
                if(
                    ret.indexOf( ":" ) == -1 &&
                    ret.startsWith( "/" ) == false &&
                    ret.startsWith( "\\" ) == false
                ){
                    
                    // workディレクトリ以下で設定.
                    ret = new StringBuffer().
                        append( JRcClientDriver.WORK_DIR ).
                        append( BaseDef.FILE_SEPARATOR ).
                        append( ret ).
                        toString() ;
                    
                }
                
                // ファイル名を正しい条件で変換.
                ret = IOCom.getName( ret ) ;
                
            } catch( Exception e ) {
                ret = null ;
            } finally{
                if( lst != null ) {
                    lst.clear() ;
                }
                lst = null ;
            }
            
        }
        
        return ret ;
    }
    
    /**
     * 指定ファイルに対するディレクトリ名を取得.
     */
    private static final String getDirectory( String file ) {
        
        int pnt ;
        int p ;
        
        String ret = null ;
        
        for( pnt = 0 ;; ) {
            if( ( p = file.indexOf( BaseDef.FILE_SEPARATOR,pnt ) ) == -1 ) {
                ret = file.substring( 0,pnt ) ;
                break ;
            }
            pnt = p + BaseDef.FILE_SEPARATOR.length() ;
        }
        
        return ret ;
        
    }
    
    /**
     * 終了メッセージチェック.
     */
    private static final boolean isExitCommand( String command ) {
        
        if(
            command == null ||
            ( command = command.trim().toLowerCase() ).length() <= 0
        ) {
            return false ;
        }
        
        if(
            "end".equals( command ) == true ||
            "exit".equals( command ) == true ||
            "close".equals( command ) == true ||
            "bye".equals( command ) == true
        ) {
            return true ;
        }
        
        return false ;
        
    }
    
    
}

