#!/usr/bin/perl
#------------------------------------------------------------------------------
#    59Tracker, weblog software for personal publisher.
#    Copyright (C) 2004-2010 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

use strict;
use warnings;

use Lib::Logger;
use Lib::Error;
use Lib::User;
use Lib::Mail;
use Lib::Conf;
use Lib::Language;

my $logger = Lib::Logger->new();
my $error = Lib::Error->new();
my $conf = Lib::Conf->new();

sub act() {
    my ($act, $cgi) = @_;

    my $sessionid = $cgi->cookie('cookie59t');
    if ($sessionid eq '') {
        $error->print_error('00003e');
    }
    my $user = Lib::User->new();
    my $session_userid = $user->check_session($sessionid);
    if ($session_userid eq '') {
        $error->print_error('00003e');
    }

    if ($act eq 'exec_setting') {
        _exec_setting($session_userid, $cgi);
    }
    else {
        _view_setting($session_userid, $cgi);
    }
}

sub _view_setting() {
    my ($session_userid, $cgi) = @_;

    # func: password, account, leave
    my $func = $cgi->param('func');
    if ($func eq 'password') {
        _input_password($func);
    }
    elsif ($func eq 'bookmarklet') {
        _view_bookmarklet($func);
    }
    elsif ($func eq 'invite') {
        _input_invite($func);
    }
    elsif ($func eq 'leave') {
        _view_leave($func);
    }
    else {
        _view_account('account', $session_userid);
    }
}

sub _exec_setting() {
    my ($session_userid, $cgi) = @_;

    # func: password, account, bookmarklet, invite, leave
    my $func = $cgi->param('func');
    if ($func eq 'password') {
        _update_password($session_userid, $cgi);
    }
    elsif ($func eq 'account') {
        _update_account($session_userid, $cgi);
    }
    elsif ($func eq 'invite') {
        _post_invite($session_userid, $cgi);
    }
    elsif ($func eq 'leave') {
        _update_leave($session_userid, $cgi);
    }
    else {
        $error->print_error('00002e');
    }
}

sub _input_password() {
    my ($func) = @_;
    my $userdata;
    $userdata->{password} = '';
    $userdata->{password2} = '';
    _show_form($func, $userdata, '');
}

sub _update_password() {
    my ($session_userid, $cgi) = @_;

    my $userdata;
    $userdata ->{password} = $cgi->param('password');
    $userdata ->{password2} = $cgi->param('password2');

    my $language = Lib::Language->new();
    if ($userdata->{password} eq '') {
        _show_form('password', $userdata, $language->get_usererror('input-password'));
    }
    if ($userdata->{password2} eq '') {
        _show_form('password', $userdata, $language->get_usererror('input-repassword'));
    }
    if ($userdata->{password} ne $userdata->{password2}) {
        _show_form('password', $userdata, $language->get_usererror('confirm-password'));
    }
    if (length($userdata->{password}) < 4 || length($userdata->{password}) > 9) {
        _show_form('password', $userdata, $language->get_usererror('password-length'));
    }
    if ($userdata->{password} =~ /[^0-9a-zA-Z]/) {
        _show_form('password', $userdata, $language->get_usererror('password-format'));
    }

    my $user = Lib::User->new();
    $user->load($session_userid);
    $user->set_password($userdata ->{password});
    $user->update();

    print "Location: ./?act=view_setting&func=password", "\n\n";
}

sub _show_form {
    my ($func, $data, $msg) = @_;

    my $language = Lib::Language->new();
    my $caption = $language->get_statictext('setting') . ':';
    my $mainform = '';
    if ($func eq 'password') {
        $caption .= $language->get_statictext('change-password');
        $mainform = _password_form($data);
    }
    elsif ($func eq 'account') {
        $caption .= $language->get_statictext('change-account');
        $mainform = _account_form($data);
    }
    elsif ($func eq 'bookmarklet') {
        $caption .= $language->get_statictext('bookmarklet');
        $mainform = _bookmarklet_form();
    }
    elsif ($func eq 'invite') {
        $caption .= $language->get_statictext('invite-friend');
        $mainform = _invite_form($data);
    }
    elsif ($func eq 'leave') {
        $caption .= $language->get_statictext('leave');
        $mainform = _leave_form();
    }

    my $docrootdir = $conf->get_docroot_dir();
    my $sitetitle = $conf->get_site_title();
    my $subtitle = $conf->get_sub_title();

    my $homelabel = $language->get_statictext('home');

    my $path = <<"END_PATH";
<a class="path" href="./">$homelabel</a>&nbsp;
&gt;&nbsp;
$caption
END_PATH

    if ($msg ne '') {
        $msg = "<p><div id=\"message\">$msg</div></p>";
    }

    my $systemdir = $conf->get_system_dir();

    my $changepassword = $language->get_statictext('change-password');
    my $changeaccount = $language->get_statictext('change-account');
    my $bookmarklet = $language->get_statictext('bookmarklet');
    my $invitefriend = $language->get_statictext('invite-friend');
    my $leave = $language->get_statictext('leave');

    open my $templatefh, '<', "$systemdir/tmpl/setting.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$SITETITLE\$/$sitetitle/g;
    $template =~ s/\$SUBTITLE\$/$subtitle/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$MSG\$/$msg/g;
    $template =~ s/\$CHANGEPASSWORD\$/$changepassword/g;
    $template =~ s/\$CHANGEACCOUNT\$/$changeaccount/g;
    $template =~ s/\$BOOKMARKLET\$/$bookmarklet/g;
    $template =~ s/\$INVITEFRIEND\$/$invitefriend/g;
    $template =~ s/\$LEAVE\$/$leave/g;
    $template =~ s/\$MAINFORM\$/$mainform/g;

    print "Content-Type: text/html\n\n";
    print $template;

    exit;
}

sub _password_form() {
    my ($userdata) = @_;

    my $language = Lib::Language->new();
    my $newpassword = $language->get_statictext('new-password');
    my $repassword = $language->get_statictext('re-password');
    my $changebutton = $language->get_statictext('change');

    my $mainform = <<"END_MAINFORM";
<form method="post" action="./">
<input type="hidden" name="act" value="exec_setting">
<input type="hidden" name="func" value="password">
$newpassword :<br />
<input type="password" name="password" size="50" value="$userdata->{password}"><br />
$repassword :<br />
<input type="password" name="password2" size="50" value="$userdata->{password2}"><br />
<p>
<input type="submit" value="$changebutton">
</p>
</form>
END_MAINFORM

    return $mainform;
}

sub _view_account() {
    my ($func, $session_userid) = @_;

    my $user = Lib::User->new();
    $user->load($session_userid);

    my $userdata;
    $userdata->{mailaddr} = $user->get_mailaddr();
    $userdata->{hpurl}    = $user->get_hpurl();
    $userdata->{username} = $user->get_username();
    $userdata->{ad1}      = $user->get_ad1();
    $userdata->{ad2}      = $user->get_ad2();
    $userdata->{ad3}      = $user->get_ad3();
    $userdata->{parts}    = $user->get_parts();

    _show_form($func, $userdata, '');
}

sub _update_account() {
    my ($session_userid, $cgi) = @_;
#    $logger->write("_update_account : session_userid = $session_userid");   # for debug

    my $userdata;
    $userdata->{mailaddr} = $cgi->param('mailaddr');
    $userdata->{hpurl}    = $cgi->param('hpurl');
    $userdata->{username} = $cgi->param('username');
    $userdata->{ad1}      = $cgi->param('ad1');
    $userdata->{ad2}      = $cgi->param('ad2');
    $userdata->{ad3}      = $cgi->param('ad3');
    $userdata->{parts}    = $cgi->param('parts');

    _check_account($userdata);

    my $user = Lib::User->new();
    $user->load($session_userid);

    $user->set_mailaddr($userdata->{mailaddr});
    $user->set_hpurl($userdata->{hpurl});
    $user->set_username($userdata->{username});
    $user->set_ad1($userdata->{ad1});
    $user->set_ad2($userdata->{ad2});
    $user->set_ad3($userdata->{ad3});
    $user->set_parts($userdata->{parts});
    $user->update();

    print "Location: ./?act=view_setting&func=account", "\n\n";
}

sub _check_account() {
    my ($userdata) = @_;

    my $language = Lib::Language->new();
    if ($userdata->{mailaddr} eq '') {
        _show_form('account', $userdata, $language->get_usererror('input-mailaddr'));
    }
    if ($userdata->{mailaddr} !~ /[-_0-9a-z]+\@[-_0-9a-z]+\.[-_0-9a-z.]+/) {
        _show_form('account', $userdata, $language->get_usererror('mailaddr-format'));
    }

    if ($userdata->{username} eq '') {
        _show_form('account', $userdata, $language->get_usererror('input-name'));
    }
}

sub _account_form() {
    my ($userdata) = @_;

    my $language = Lib::Language->new();
    my $mailaddr = $language->get_statictext('mail-address');
    my $homepageurl = $language->get_statictext('homepage-url');
    my $username = $language->get_statictext('user-name');
    my $adcode1 = $language->get_statictext('ad-code1');
    my $adcode2 = $language->get_statictext('ad-code2');
    my $adcode3 = $language->get_statictext('ad-code3');
    my $blogparts = $language->get_statictext('blogparts');
    my $updatebutton = $language->get_statictext('update');

    my $form = <<"END_FORM";
<form method="post" action="./">
<input type="hidden" name="act" value="exec_setting">
<input type="hidden" name="func" value="account">
$mailaddr :<br />
<input type="text" name="mailaddr" size="100" value="$userdata->{mailaddr}"><br />
$homepageurl :<br />
<input type="text" name="hpurl" size="100" value="$userdata->{hpurl}"><br />
$username :<br />
<input type="text" name="username" size="100" value="$userdata->{username}"><br />
$adcode1 :<br />
<textarea name="ad1" rows="3" cols="75">$userdata->{ad1}</textarea><br />
$adcode2 :<br />
<textarea name="ad2" rows="3" cols="75">$userdata->{ad2}</textarea><br />
$adcode3 :<br />
<textarea name="ad3" rows="3" cols="75">$userdata->{ad3}</textarea><br />
$blogparts :<br />
<textarea name="parts" rows="3" cols="75">$userdata->{parts}</textarea><br />
<p>
<input type="submit" value="$updatebutton">
</p>
</form>
END_FORM

    return $form;
}

sub _input_invite() {
    my ($func) = @_;
    my $invitedata;
    $invitedata->{mailaddr} = '';
    $invitedata->{message} = '';
    _show_form($func, $invitedata, '');
}

sub _invite_form() {
    my ($invitedata) = @_;

    my $language = Lib::Language->new();
    my $mailaddr = $language->get_statictext('mail-address');
    my $message = $language->get_statictext('message');
    my $invitebutton = $language->get_statictext('invite');

    my $form = <<"END_FORM";
<form method="post" action="./">
<input type="hidden" name="act" value="exec_setting">
<input type="hidden" name="func" value="invite">
$mailaddr :<br />
<input type="text" name="mailaddr" size="100" value="$invitedata->{mailaddr}"><br />
$message :<br />
<textarea name="message" rows="7" cols="75">$invitedata->{message}</textarea><br>
<p>
<input type="submit" value="$invitebutton">
</p>
</form>
END_FORM

    return $form;
}

sub _post_invite() {
    my ($session_userid, $cgi) = @_;

    my $invitedata;
    $invitedata->{mailaddr} = $cgi->param('mailaddr');
    $invitedata->{message}  = $cgi->param('message');

    my $language = Lib::Language->new();
    if ($invitedata->{mailaddr} eq '') {
        _show_form('invite', $invitedata, $language->get_usererror('input-mailaddr'));
    }
    if ($invitedata->{mailaddr} !~ /[-_0-9a-z]+\@[-_0-9a-z]+\.[-_0-9a-z.]+/) {
        _show_form('invite', $invitedata, $language->get_usererror('mailaddr-format'));
    }
    if ($invitedata->{message} eq '') {
        _show_form('invite', $invitedata, $language->get_usererror('input-message'));
    }
    $invitedata->{message} =~ s/,/enc_conma/g;

    my $systemdir = $conf->get_system_dir();
    my $docrootdir = $conf->get_docroot_dir();
    my $sitetitle = $conf->get_site_title();

    # 10000 ～ 99999 の招待コードを生成
    my $invitecode = int(rand(89999)) + 10000;

    # 招待コードを保存 (招待コード,メールアドレス)
    my $invitefile = "$systemdir/invite.txt";
    open my $invitefh, '>', "$invitefile";
    print $invitefh "$invitecode,$invitedata->{mailaddr}\n";
    close $invitefh;
    chmod 0766, $invitefile;

    my $url = "$docrootdir/?funct=signup&invitecode=$invitecode";

    # send invite mail to user.
    my $invitenotice = $language->get_message('invite-notice');
    my $invitemail1 = $language->get_message('invite-mail1');
    my $invitemail2 = $language->get_message('invite-mail2');
    my $invitemail3 = $language->get_message('invite-mail3');

    $invitenotice =~ s/\$SESSION_USER\$/$session_userid/g;
    $invitenotice =~ s/\$SITETITLE\$/$sitetitle/g;
    $invitemail1  =~ s/\$SESSION_USER\$/$session_userid/g;
    $invitemail1  =~ s/\$SITETITLE\$/$sitetitle/g;
    $invitemail3  =~ s/\$SESSION_USER\$/$session_userid/g;

    my $subject = $invitenotice;
    my $body = <<"END_BODY";
$invitemail1
$invitemail2

$url

$invitemail3 :
$invitedata->{message}

END_BODY

    my $mail = Lib::Mail->new();
    $mail->send($subject, $body, $invitedata->{mailaddr});

    # send invite information mail to webmaster.
    my $inviteinfo = $language->get_message('invite-info');
    my $inviteinfobody = $language->get_message('invite-infobody');
    $inviteinfo     =~ s/\$MAILADDR\$/$invitedata->{mailaddr}/g;
    $inviteinfobody =~ s/\$SESSION_USER\$/$session_userid/g;
    $inviteinfobody =~ s/\$MAILADDR\$/$invitedata->{mailaddr}/g;
    $subject = $inviteinfo;
    $body = $inviteinfobody;
    $mail->send($subject, $body, $conf->get_admin_address());

    print "Location: ./?act=view_setting&func=invite", "\n\n";
}

sub _view_leave() {
    my ($func) = @_;
    my $data;
    _show_form($func, $data, '');
}

sub _leave_form() {
    my $language = Lib::Language->new();
    my $leavebutton = $language->get_statictext('leave');
    my $leavequestion = $language->get_message('leave-question');

    my $sitetitle = $conf->get_site_title();
    $leavequestion =~ s/\$SITETITLE\$/$sitetitle/g;

    my $form = <<"END_FORM";
<p>
$leavequestion
</p>

<form method="post" action="./">
  <input type="hidden" name="a" value="exec_setting">
  <input type="hidden" name="func" value="leave">
  <input type="submit" value="$leavebutton">
</form>
END_FORM

    return $form;
}

# ユーザーアカウントを無効化
sub _update_leave() {
    my ($session_userid, $cgi) = @_;

    my $user = Lib::User->new();
    $user->load($session_userid);
    my $mailaddr = $user->get_mailaddr();

    # status invalidate.
    $user->set_status(0);
    $user->update();

    # delete session.
    my $sessionid = $cgi->cookie('cookie59t');
    $user->logout($sessionid);

    my $mail = Lib::Mail->new();

    my $sitetitle = $conf->get_site_title();
    my $adminaddr = $conf->get_admin_address();

    my $language = Lib::Language->new();
    my $leavenotice = $language->get_message('leave-notice');
    $leavenotice =~ s/\$SITETITLE\$/$sitetitle/g;
    my $leavemail1 = $language->get_message('leave-mail1');
    my $leavemail2 = $language->get_message('leave-mail2');

    # send leave information mail to user.
    my $subject = $language->get_statictext('leave-subject');
    my $body = <<"END_BODY";
$leavenotice

$leavemail1
$leavemail2
mailto:$adminaddr
END_BODY

    $mail->send($subject, $body, $mailaddr);

    my $leaveinfo = $language->get_message('leave-info');
    $leaveinfo =~ s/\$SESSION_USER\$/$session_userid/g;

    my $maillabel = $language->get_statictext('mail-address');

    # send leave information mail to webmaster.
    $subject = $leaveinfo;
    $body = <<"END_BODY";
$leaveinfo

$maillabel : $mailaddr
END_BODY

    $mail->send($subject, $body, $conf->get_admin_address());

    my $homelabel = $language->get_statictext('home');

    my $caption = $leavenotice;
    my $path = <<"END_PATH";
<a class="path" href="./">$homelabel</a>&nbsp;
&gt;&nbsp;
$caption
END_PATH

    my $msg = "<p><b><big>$leavenotice</big></b></p>";

    my $systemdir = $conf->get_system_dir();
    my $subtitle = $conf->get_sub_title();

    open my $templatefh, '<', "$systemdir/tmpl/message.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$SITETITLE\$/$sitetitle/g;
    $template =~ s/\$SUBTITLE\$/$subtitle/g;
    $template =~ s/\$MSG\$/$msg/g;

    print "Content-Type: text/html\n\n";
    print $template;
}

sub _view_bookmarklet() {
    my ($func) = @_;
    my $data;
    _show_form($func, $data, '');
}

sub _bookmarklet_form() {
    my $docrootdir = $conf->get_docroot_dir();

    my $language = Lib::Language->new();
    my $bookmarkletguide = $language->get_message('bookmarklet-guide');
    my $bookmarklabel = $language->get_statictext('bookmark');
    my $bookmarklet = <<"END_BOOKMARKLET";
<p>
<a href="javascript:void(window.open('$docrootdir/?act=view_editor&url='+escape(document.location.href),'_self'))">$bookmarklabel</a>
</p>
END_BOOKMARKLET

    my $form = <<"END_FORM";
<p>
$bookmarkletguide
</p>

$bookmarklet
END_FORM

    return $form;
}

1;
